/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: SlsGenericRequestQueue.hxx,v $
 *
 *  $Revision: 1.8 $
 *
 *  last change: $Author: rt $ $Date: 2006/01/11 17:26:06 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

#ifndef SD_SLIDESORTER_GENERIC_REQUEST_QUEUE_HXX
#define SD_SLIDESORTER_GENERIC_REQUEST_QUEUE_HXX

#include "SlsRequestPriorityClass.hxx"

#include "taskpane/SlideSorterCacheDisplay.hxx"
#include <drawdoc.hxx>

#ifndef _OSL_MUTEX_HXX_
#include "osl/mutex.hxx"
#endif

#ifndef INCLUDED_SET
#include <set>
#define INCLUDED_SET
#endif

#ifndef INCLUDED_ALGORITHM
#include <algorithm>
#define INCLUDED_ALGORITHM
#endif

// Uncomment the define below to have some more OSL_TRACE messages.
#ifdef DEBUG
// #define VERBOSE
#endif


namespace sd { namespace slidesorter { namespace cache {

/** This class extends the actual request data with additional information
    that is used by the priority queues.
*/
template<class RequestData>
class Request 
{
public:
    Request (
        RequestData* pData, sal_Int32 nPriority, RequestPriorityClass eClass)
        : mpData(pData), mnPriorityInClass(nPriority), meClass(eClass) 
    {}
    /** Sort requests according to priority classes and then to priorities.
    */
    class Comparator { public:
        bool operator() (const Request& rRequest1, const Request& rRequest2)
        {
            if (rRequest1.meClass == rRequest2.meClass)
                return (rRequest1.mnPriorityInClass > rRequest2.mnPriorityInClass);
            else
                return (rRequest1.meClass < rRequest2.meClass);
        }
    };
    /** Request data is compared arbitrarily by their addresses in memory.
        This just establishes an order so that the STL containers are happy.
        The order is not semantically interpreted.
    */
    class DataComparator { public:
        DataComparator (const Request&rRequest):mrData(*rRequest.mpData){}
        DataComparator (const RequestData&rData):mrData(rData){}
        bool operator() (const Request& rRequest)
        { return &mrData == rRequest.mpData; }
    private: const RequestData& mrData;
    };
    
    RequestData* mpData;
    sal_Int32 mnPriorityInClass;
    RequestPriorityClass meClass;
};





/** The request queue stores requests that are described by the RequestData
    sorted according to priority class and then priority.
*/
template<class RequestData>
class GenericRequestQueue
{
public:
    typedef Request<RequestData> RequestType;

    GenericRequestQueue (void);

    /** Insert a request with highest or lowest priority in its priority
        class.  When the request is already present then it is first
        removed.  This effect is then a re-prioritization.
        @param rRequestData
            The request.
        @param eRequestClass
            The priority class in which to insert the request with highest
            or lowest priority.
        @param bInsertWithHighestPriority
            When this flag is <TRUE/> the request is inserted with highes
            priority in its class.  When <FALSE/> the request is inserted
            with lowest priority.
    */
    void AddRequest (
        RequestData& rRequestData, 
        RequestPriorityClass eRequestClass,
        bool bInsertWithHighestPriority = false);

    /** Remove the specified request from the queue.
        @param rRequestData
            It is OK when the specified request is not a member of the
            queue.
        @return
            Returns <TRUE/> when the request has been successfully been
            removed from the queue.  Otherwise, e.g. because the request was
            not a member of the queue, <FALSE/> is returned.
    */
    bool RemoveRequest (const RequestData& rRequestData);

    /** Change the priority class of the specified request.
    */
    void ChangeClass (RequestData& rRequestData, RequestPriorityClass eNewRequestClass);

    /** Get the request with the highest priority int the highest priority class.
    */
    RequestData& GetFront (void);

    // For debugging.
    RequestPriorityClass GetFrontPriorityClass (void);

    /** Really a synonym for RemoveRequest(GetFront());
    */
    void PopFront (void);

    /** Returns <TRUE/> when there is no element in the queue.
    */
    bool IsEmpty (void);

    /** Remove all requests from the queue.  This resets the minimum and
        maximum priorities to their default values.
    */
    void Clear (void);

    /** Return the mutex that guards the access to the priority queue.
    */
    ::osl::Mutex& GetMutex (void);

private:
    ::osl::Mutex maMutex;
    typedef ::std::set<
        Request<RequestData>,
        typename Request<RequestData>::Comparator> RequestQueueContainer;
    RequestQueueContainer maRequestQueue;
    /** A lower bound of the lowest priority of all elements in the queues.
        The start value is 0.  It is assigned and then decreased every time
        when an element is inserted or marked as the request with lowest
        priority.
    */
    int mnMinimumPriority;
    /** An upper bound of the highest priority of all elements in the queues.
        The start value is 1.  It is assigned and then increased every time
        when an element is inserted or marked as the request with highest
        priority.
    */
    int mnMaximumPriority;
};


//=============================================================================
// Implementation


//=====  GenericRequestQueue  =================================================

template<class RequestData>
GenericRequestQueue<RequestData>::GenericRequestQueue (void)
    : mnMinimumPriority(0),
      mnMaximumPriority(1)
{
}




template<class RequestData>
    void GenericRequestQueue<RequestData>::AddRequest (
        RequestData& rRequestData,
        RequestPriorityClass eRequestClass,
        bool bInsertWithHighestPriority)
{
    ::osl::MutexGuard aGuard (maMutex);

    OSL_ASSERT(eRequestClass>=MIN__CLASS && eRequestClass<=MAX__CLASS);
    
    // If the request is already a member of the queue then remove it so
    // that the following insertion will use the new prioritization.
    bool bRemoved = RemoveRequest(rRequestData);
    
    // The priority of the request inside its priority class is defined by
    // the page number.  This ensures a strict top-to-bottom, left-to-right order.
    sal_Int32 nPriority ( - (rRequestData.GetPage()->GetPageNum()-1) / 2);
    RequestType aRequest (&rRequestData, nPriority, eRequestClass);
    maRequestQueue.insert(aRequest);

    SSCD_SET_REQUEST_CLASS(rRequestData.GetPage(),eRequestClass);

#ifdef VERBOSE
    OSL_TRACE("%s request for page %d with priority class %d",
        bRemoved?"replaced":"added",
        (rRequestData.GetPage()->GetPageNum()-1)/2,
        eRequestClass);
#endif
}




template<class RequestData>
    bool GenericRequestQueue<RequestData>::RemoveRequest (
        const RequestData& rRequest)
{
    bool bRequestWasRemoved (false);
    ::osl::MutexGuard aGuard (maMutex);

    while(true)
    {
        typename RequestQueueContainer::iterator aRequestIterator = ::std::find_if (
            maRequestQueue.begin(), 
            maRequestQueue.end(),
            typename RequestType::DataComparator(rRequest));
        if (aRequestIterator != maRequestQueue.end())
        {
            if (aRequestIterator->mnPriorityInClass == mnMinimumPriority+1)
                mnMinimumPriority++;
            else if (aRequestIterator->mnPriorityInClass == mnMaximumPriority-1)
                mnMaximumPriority--;
            maRequestQueue.erase (aRequestIterator);
            bRequestWasRemoved = true;

            if (bRequestWasRemoved)
            {
                SSCD_SET_STATUS(rRequest.GetPage(),NONE);
            }
        }
        else
            break;
    }

    return bRequestWasRemoved;
}




template<class RequestData>
    void GenericRequestQueue<RequestData>::ChangeClass (
        RequestData& rRequestData,
        RequestPriorityClass eNewRequestClass)
{
    ::osl::MutexGuard aGuard (maMutex);

    OSL_ASSERT(eNewRequestClass>=MIN__CLASS && eNewRequestClass<=MAX__CLASS);

    typename RequestQueueContainer::iterator iRequest (
        ::std::find_if (
            maRequestQueue.begin(),
            maRequestQueue.end(),
            typename RequestType::DataComparator(rRequestData)));
    if (iRequest!=maRequestQueue.end() && iRequest->meClass!=eNewRequestClass)
    {
        AddRequest(rRequestData, eNewRequestClass, true);
        SSCD_SET_REQUEST_CLASS(rRequestData.GetPage(),eNewRequestClass);
    }
}




template<class RequestData>
    RequestData& GenericRequestQueue<RequestData>::GetFront (void)
{
    ::osl::MutexGuard aGuard (maMutex);

    if (maRequestQueue.empty())
        throw ::com::sun::star::uno::RuntimeException(
            ::rtl::OUString(RTL_CONSTASCII_USTRINGPARAM(
                "GenericRequestQueue::GetFront(): queue is empty")),
            NULL);
           
    return *maRequestQueue.begin()->mpData;
}




template<class RequestData>
    RequestPriorityClass GenericRequestQueue<RequestData>::GetFrontPriorityClass (void)
{
    ::osl::MutexGuard aGuard (maMutex);

    if (maRequestQueue.empty())
        throw ::com::sun::star::uno::RuntimeException(
            ::rtl::OUString(RTL_CONSTASCII_USTRINGPARAM(
                "GenericRequestQueue::GetFrontPriorityClass(): queue is empty")),
            NULL);

    return maRequestQueue.begin()->meClass;
}




template<class RequestData>
    void GenericRequestQueue<RequestData>::PopFront (void)
{
    ::osl::MutexGuard aGuard (maMutex);

    if ( ! maRequestQueue.empty())
    {
        SSCD_SET_STATUS(maRequestQueue.begin()->mpData->GetPage(),NONE);

        maRequestQueue.erase (maRequestQueue.begin());

        // Reset the priority counter if possible.
        if (maRequestQueue.empty())
        {
            mnMinimumPriority = 0;
            mnMaximumPriority = 1;
        }
    }
}




template<class RequestData>
    bool GenericRequestQueue<RequestData>::IsEmpty (void)
{
    ::osl::MutexGuard aGuard (maMutex);
    return maRequestQueue.empty();
}




template<class RequestData>
    void GenericRequestQueue<RequestData>::Clear (void)
{
    ::osl::MutexGuard aGuard (maMutex);

    maRequestQueue.clear();
    mnMinimumPriority = 0;
    mnMaximumPriority = 1;
}




template<class RequestData>
    ::osl::Mutex& GenericRequestQueue<RequestData>::GetMutex (void)
{
    return maMutex;
}


} } } // end of namespace ::sd::slidesorter::cache

#endif
