/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: strmstd.cxx,v $
 *
 *  $Revision: 1.2 $
 *
 *  last change: $Author: rt $ $Date: 2005/09/09 14:34:58 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

#include <stdio.h>
#include <io.h>
#include <errno.h>
#include <limits.h>

#include <stream.hxx>

// -----------------------------------------------------------------------

// --------------
// - StreamData -
// --------------

class StreamData
{
public:
    FILE*   fPtr;

            StreamData() { fPtr = 0; }
};

/*************************************************************************
|*
|*    SvFileStream::SvFileStream()
|*
|*    Beschreibung      STREAM.SDW
|*    Ersterstellung    OV 08.06.94
|*    Letzte Aenderung  OV 08.06.94
|*
*************************************************************************/

SvFileStream::SvFileStream( const String& rFileName, StreamMode nOpenMode )
{
    bIsOpen             = FALSE;
    nLockCounter        = 0;
    bIsWritable         = FALSE;
    pInstanceData       = new StreamData;

    SetBufferSize( 128 );
    Open( rFileName, nOpenMode );
}

/*************************************************************************
|*
|*    SvFileStream::SvFileStream()
|*
|*    Beschreibung      STREAM.SDW
|*    Ersterstellung    OV 08.06.94
|*    Letzte Aenderung  OV 08.06.94
|*
*************************************************************************/

SvFileStream::SvFileStream()
{
    bIsOpen             = FALSE;
    nLockCounter        = 0;
    bIsWritable         = FALSE;
    pInstanceData       = new StreamData;

    SetBufferSize( 128 );
}

/*************************************************************************
|*
|*    SvFileStream::~SvFileStream()
|*
|*    Beschreibung      STREAM.SDW
|*    Ersterstellung    OV 08.06.94
|*    Letzte Aenderung  OV 08.06.94
|*
*************************************************************************/

SvFileStream::~SvFileStream()
{
    Close();
    if ( pInstanceData )
        delete pInstanceData;
}

/*************************************************************************
|*
|*    SvFileStream::GetFileHandle()
|*
|*    Beschreibung      STREAM.SDW
|*    Ersterstellung    OV 08.06.94
|*    Letzte Aenderung  OV 08.06.94
|*
*************************************************************************/

ULONG SvFileStream::GetFileHandle() const
{
    return (ULONG)pInstanceData->fPtr;
}

/*************************************************************************
|*
|*    SvFileStream::GetData()
|*
|*    Beschreibung      STREAM.SDW, Prueft nicht Eof; IsEof danach rufbar
|*    Ersterstellung    OV 15.06.94
|*    Letzte Aenderung  OV 15.06.94
|*
*************************************************************************/

ULONG SvFileStream::GetData( char* pData, ULONG nSize )
{
    ULONG nCount = 0L;
    if ( IsOpen() )
    {
        nCount= (ULONG)fread((void*)pData,sizeof(char),(size_t)nSize,pInstanceData->fPtr);
        if ( ferror(pInstanceData->fPtr) )
            SetError( SVSTREAM_READ_ERROR );

    }
    return nCount;
}

/*************************************************************************
|*
|*    SvFileStream::PutData()
|*
|*    Beschreibung      STREAM.SDW
|*    Ersterstellung    OV 15.06.94
|*    Letzte Aenderung  OV 15.06.94
|*
*************************************************************************/

ULONG SvFileStream::PutData( const char* pData, ULONG nSize )
{
    ULONG nCount = 0L;
    if ( IsOpen() )
    {
        nCount= (ULONG)fwrite((void*)pData,1,(size_t)nSize,pInstanceData->fPtr);
        if ( ferror(pInstanceData->fPtr) )
            SetError( SVSTREAM_READ_ERROR );
        else if( !nCount )
            SetError( SVSTREAM_DISK_FULL );

    }
    return nCount;
}

/*************************************************************************
|*
|*    SvFileStream::SeekPos()
|*
|*    Beschreibung      STREAM.SDW
|*    Ersterstellung    OV 15.06.94
|*    Letzte Aenderung  OV 15.06.94
|*
*************************************************************************/

ULONG SvFileStream::SeekPos( ULONG nPos )
{
    if ( IsOpen() )
    {
        if ( nPos != STREAM_SEEK_TO_END )
            fseek( pInstanceData->fPtr, (long)nPos, SEEK_SET );
        else
            fseek( pInstanceData->fPtr, 0L, SEEK_END );

        if ( ferror( pInstanceData->fPtr ) )
            SetError( SVSTREAM_SEEK_ERROR );

        return (ULONG)ftell( pInstanceData->fPtr );
    }
    SetError( SVSTREAM_GENERALERROR );
    return 0L;
}


/*************************************************************************
|*
|*    SvFileStream::FlushData()
|*
|*    Beschreibung      STREAM.SDW
|*    Ersterstellung    OV 15.06.94
|*    Letzte Aenderung  OV 15.06.94
|*
*************************************************************************/

void SvFileStream::FlushData()
{
    if ( IsOpen() )
        fflush( pInstanceData->fPtr );
}

/*************************************************************************
|*
|*    SvFileStream::LockRange()
|*
|*    Beschreibung      STREAM.SDW
|*    Ersterstellung    OV 15.06.94
|*    Letzte Aenderung  OV 15.06.94
|*
*************************************************************************/

BOOL SvFileStream::LockRange( ULONG nByteOffset, ULONG nBytes )
{
    return FALSE;
}

/*************************************************************************
|*
|*    SvFileStream::UnlockRange()
|*
|*    Beschreibung      STREAM.SDW
|*    Ersterstellung    OV 15.06.94
|*    Letzte Aenderung  OV 15.06.94
|*
*************************************************************************/

BOOL SvFileStream::UnlockRange( ULONG nByteOffset, ULONG nBytes )
{
    return FALSE;
}

/*************************************************************************
|*
|*    SvFileStream::LockFile()
|*
|*    Beschreibung      STREAM.SDW
|*    Ersterstellung    OV 15.06.94
|*    Letzte Aenderung  OV 15.06.94
|*
*************************************************************************/

BOOL SvFileStream::LockFile()
{
    BOOL bRetVal = FALSE;
    if ( !nLockCounter )
    {
        if ( LockRange( 0L, LONG_MAX ) )
        {
            nLockCounter = 1;
            bRetVal = TRUE;
        }
    }
    else
    {
        nLockCounter++;
        bRetVal = TRUE;
    }
    return bRetVal;
}

/*************************************************************************
|*
|*    SvFileStream::UnlockFile()
|*
|*    Beschreibung      STREAM.SDW
|*    Ersterstellung    OV 15.06.94
|*    Letzte Aenderung  OV 15.06.94
|*
*************************************************************************/

BOOL SvFileStream::UnlockFile()
{
    BOOL bRetVal = FALSE;
    if ( nLockCounter > 0)
    {
        if ( nLockCounter == 1)
        {
            if ( UnlockRange( 0L, LONG_MAX ) )
            {
                nLockCounter = 0;
                bRetVal = TRUE;
            }
        }
        else
        {
            nLockCounter--;
            bRetVal = TRUE;
        }
    }
    return bRetVal;
}

/*************************************************************************
|*
|*    SvFileStream::Open()
|*
|*    Beschreibung      STREAM.SDW
|*    Ersterstellung    OV 15.06.94
|*    Letzte Aenderung  OV 15.06.94
|*
*************************************************************************/

/* ***********************************************************************

- Wenn das Beschreiben der Datei nicht moeglich ist, wird die Datei
  fuer Lesezugriff geoeffnet.

stdio open modes:

'r'     Open for reading only
'w'     Create for writing. If a file by that name already exists, it
        will be overwritten
'r+'    Open an existing file for update ( reading & writing )
'w+'    Create a new file for update( reading & writing). If a file
        by that name already exists, it will be overwritten

*********************************************************************** */

void SvFileStream::Open( const String& rFilename, StreamMode nOpenMode )
{
	String aParsedFilename;
#ifndef WIN
	if	( Folder::IsAvailable() && (rFilename.Search("{") < 9) )
	{
		String		aVirtualPart;
		String		aRealPart;
		String		aVirtualPath;
		ItemIDPath	aVirtualURL;
		ULONG		nDivider = 0;

		String		aVirtualString(rFilename);

		for (int x=aVirtualString.Len(); x>0; x--)
		{
			if (aVirtualString.Copy(x,1).Compare("}")==COMPARE_EQUAL)
			{
				nDivider = x;
				break;
			}
		}

		aVirtualPart = aVirtualString.Copy(0,nDivider+1);
		aRealPart = aVirtualString.Copy(nDivider+2);

		aVirtualURL  = aVirtualPart;
		aVirtualPath = aVirtualURL.GetHostNotationPath();

		DirEntry aTempDirEntry(aVirtualPath);

		aTempDirEntry += aRealPart;

		aParsedFilename = aTempDirEntry.GetFull();
	}
	else
#endif
    {
		aParsedFilename = rFilename;
	}

    FILE* fTmp;
    Close();
    SvStream::ClearBuffer();

    SvStream::ClearError();
    BOOL bCreate = (nOpenMode & STREAM_NOCREATE) == 0;

    eStreamMode = nOpenMode;
    eStreamMode &= ~STREAM_TRUNC; // beim ReOpen nicht cutten

	String aFilename = aParsedFilename;
	FSysRedirector::DoRedirect( aFilename );

    // Existiert Datei ?
    BOOL bFileExists = TRUE;
    {
        fTmp = fopen( (const char*)aFilename,"rb");
        if ( fTmp == NULL )
            bFileExists = FALSE;
        else
            fclose( fTmp );
    }

    if ( !bCreate && !bFileExists )
    {
        // Datei nicht da und wir duerfen keine neue erzeugen
        SetError( SVSTREAM_FILE_NOT_FOUND );
        return;
    }

    if ( bCreate && !bFileExists )
    {
        // Datei nicht da, aber wir duerfen eine neue erzeugen
        fTmp = fopen( (const char*)aFilename,"wb");
        if ( fTmp == NULL )
        {
            SetError( SVSTREAM_ACCESS_DENIED );
            return;
        }
        else
            fclose( fTmp );
    }

    if ( nOpenMode & STREAM_TRUNC )
    {
        // Datei abschneiden
        fTmp = fopen( (const char*)aFilename,"w+b");
        if ( fTmp == NULL )
        {
            SetError( SVSTREAM_ACCESS_DENIED );
            return;
        }
        else
            fclose( fTmp );
    }

    bIsWritable = FALSE;
    bIsOpen     = TRUE;

    if ( nOpenMode & STREAM_WRITE )
    {
        fTmp = fopen( (const char*)aFilename,"r+b");
        if ( fTmp != NULL )
        {
            pInstanceData->fPtr = fTmp;
            bIsWritable = TRUE;
            return;
        }
    }


    // read only oeffnen
    fTmp = fopen( (const char*)aFilename,"rb" );
    if ( fTmp != NULL )
    {
        pInstanceData->fPtr = fTmp;
        return;
    }
    bIsOpen = FALSE;
    SetError( SVSTREAM_GENERALERROR );
}

/*************************************************************************
|*
|*    SvFileStream::Close()
|*
|*    Beschreibung      STREAM.SDW
|*    Ersterstellung    OV 15.06.94
|*    Letzte Aenderung  OV 15.06.94
|*
*************************************************************************/

void SvFileStream::Close()
{
    if ( IsOpen() )
    {
        if ( nLockCounter )
        {
            nLockCounter = 1;
            UnlockFile();
        }
        Flush();
        fclose( pInstanceData->fPtr );
    }
    bIsOpen     = FALSE;
    nLockCounter= 0;
    bIsWritable = FALSE;
    SvStream::ClearBuffer();
    SvStream::ClearError();
}

/*************************************************************************
|*
|*    SvFileStream::ResetError()
|*
|*    Beschreibung      STREAM.SDW; Setzt Filepointer auf Dateianfang
|*    Ersterstellung    OV 15.06.94
|*    Letzte Aenderung  OV 15.06.94
|*
*************************************************************************/

void SvFileStream::ResetError()
{
    SvStream::ClearError();
    if ( IsOpen() )
    {
        long nPos = ftell( pInstanceData->fPtr );
        rewind( pInstanceData->fPtr );
        fseek( pInstanceData->fPtr, nPos, SEEK_SET );
    }
}

/*************************************************************************
|*
|*    SvFileStream::IsA()
|*
|*    Beschreibung      STREAM.SDW
|*    Ersterstellung    OV 15.06.94
|*    Letzte Aenderung  OV 15.06.94
|*
*************************************************************************/

USHORT SvFileStream::IsA() const
{
    return ID_FILESTREAM;
}

/*************************************************************************
|*
|*    SvFileStream::SetSize()
|*
|*    Beschreibung      STREAM.SDW
|*    Ersterstellung    OV 15.06.94
|*    Letzte Aenderung  OV 15.06.94
|*
*************************************************************************/

void SvFileStream::SetSize( ULONG nSize )
{
    if ( IsOpen() )
    {
        int nHandle = fileno( pInstanceData->fPtr );
        int nRet = chsize( nHandle, (long)nSize );
        if( nRet == -1 )
        {
            switch ( errno )
            {
                case EACCES:
                    SetError( SVSTREAM_LOCKING_VIOLATION );
                    break;
                case EBADF:
                    SetError( SVSTREAM_INVALID_HANDLE );
                    break;
                case ENOSPC:
                    SetError( SVSTREAM_DISK_FULL );
                    break;
                default:
                    SetError( SVSTREAM_GENERALERROR );
            }
        }
    }
}
