#
# Copyright (C) 2005 Chris Halls <halls@debian.org>
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of version 2.1 of the GNU Lesser General Public
# License as published by the Free Software Foundation.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

"""Unit test for apt_proxy_conf.py"""

from twisted.trial import unittest
from StringIO import StringIO

from apt_proxy.apt_proxy_conf import apConfig, ConfigError

class EmptyConfigTest(unittest.TestCase):
    def setUp(self):
        self.c = apConfig(StringIO())
    def testEmptyConfig(self):
        self.assertEquals(len(self.c.backends),0)
    def testDefaultPort(self):
        self.assertEquals(self.c.port,9999)

config1="""
[DEFAULT]
debug=all:9
port=8989
address=1.2.3.4 5.6.7.8
timeout = 888
bandwidth_limit = 2323
http_proxy = somehost:9876

[backend1]
backends = ftp://a.b.c
timeout = 999
bandwidth_limit = 3434
http_proxy = user:secret@otherhost:webcache

[backend2]
backends = 
  http://x.y.z
  rsync://d.e.f
  file://g.h.i

# This backend will not be added
[backend3]
backends = invalid://aa.bb
"""

class ConfigTest(unittest.TestCase):
    def setUp(self):
        self.c = apConfig(StringIO(config1))
    def testPort(self):
        self.assertEquals(self.c.port,8989)
    def testAddress(self):
        self.assertEquals(self.c.address,['1.2.3.4','5.6.7.8'])
    def testBackendCount(self):
        self.assertEquals(len(self.c.backends),2)
    def testBackend1(self):
        self.assertEquals(self.c.backends['backend1'].backends,['ftp://a.b.c'])
        self.assertEquals(self.c.backends['backend1'].name,'backend1')
        self.assertEquals(self.c.backends['backend1'].dynamic,False)
    def testBackend2(self):
        self.assertEquals(self.c.backends['backend2'].backends,
           ['http://x.y.z','rsync://d.e.f','file://g.h.i'])
    def testTimeout(self):
        self.assertEquals(self.c.timeout,888)
    def testTimeout1(self):
        self.assertEquals(self.c.backends['backend1'].timeout,999)
    def testTimeout2(self):
        self.assertEquals(self.c.backends['backend2'].timeout,888) # Default section timeout
    def testDynamicBackend(self):
        self.c.addBackend(None, 'dynamic1', ['http://a.b.c'])
        self.assertEquals(self.c.backends['dynamic1'].backends,['http://a.b.c'])
        self.assertEquals(self.c.backends['dynamic1'].name,'dynamic1')
        self.assertEquals(self.c.backends['dynamic1'].dynamic,True)
        self.assertEquals(self.c.backends['dynamic1'].timeout,888)
    def testBandwidthLimit(self):
        self.assertEquals(self.c.bandwidth_limit, 2323)
        self.assertEquals(self.c.backends['backend1'].bandwidth_limit,3434)
        self.assertEquals(self.c.backends['backend2'].bandwidth_limit,2323)
    def testHttpProxy(self):
        self.assertEquals(self.c.http_proxy.host, 'somehost')
        self.assertEquals(self.c.http_proxy.port, 9876)
        self.assertEquals(self.c.http_proxy.user, None)
        self.assertEquals(self.c.http_proxy.password, None)
    def testHttpProxyUserPass(self):
        self.assertEquals(self.c.backends['backend1'].http_proxy.host, 'otherhost')
        self.assertEquals(self.c.backends['backend1'].http_proxy.port, 'webcache')
        self.assertEquals(self.c.backends['backend1'].http_proxy.user, 'user')
        self.assertEquals(self.c.backends['backend1'].http_proxy.password, 'secret')

class BrokenTimeoutTest(unittest.TestCase):
    def testBrokenTimeout(self):
        self.assertRaises(ConfigError, apConfig, StringIO("[Default]\ntimeout = "))

class DefaultsTest(unittest.TestCase):
    def setUp(self):
        self.c = apConfig(StringIO(""))
    def testDefaultReadLimit(self):
        self.assertEquals(self.c.bandwidth_limit, None)
    def testDefaultHttpProxy(self):
        self.assertEquals(self.c.http_proxy, None)
