// load_grouppolicy.cc
//
//  Copyright 2001 Daniel Burrows
//
//  Routines to parse grouping policies
//
// This uses a bunch of little functions to parse stuff.  Sue me.
//
// The parsers take an arglist and a chain argument, and return either a
// new pkg_grouppolicy_factory, or (if an error was encountered) NULL.
// (note: the chain argument will be NULL if the policy is the tail of a list)
// Yes, I don't believe in C++ exceptions.

#include "load_grouppolicy.h"

#include <generic/pkg_hier.h>

#include "aptitude.h"

#include "pkg_grouppolicy.h"
#include "strhash.h"
#include "pkg_ver_item.h"
#include "dep_item.h"

#include <generic/apt.h>

#include <apt-pkg/error.h>

#include <map>

using namespace std;

typedef vector<string> arglist;
typedef pkg_grouppolicy_factory * (*pkg_grouppolicy_parser)(const arglist &,
							    pkg_grouppolicy_factory *);
typedef map<string, pkg_grouppolicy_parser> parser_map;

static pkg_grouppolicy_factory *parse_section_policy(const arglist &args,
						     pkg_grouppolicy_factory *chain)
{
  int split_mode=pkg_grouppolicy_section_factory::split_none;
  bool passthrough=false;

  if(args.size()>=1)
    {
      if(!strcasecmp(args[0].c_str(), "none"))
	split_mode=pkg_grouppolicy_section_factory::split_none;
      else if(!strcasecmp(args[0].c_str(), "topdir"))
	split_mode=pkg_grouppolicy_section_factory::split_topdir;
      else if(!strcasecmp(args[0].c_str(), "subdir"))
	split_mode=pkg_grouppolicy_section_factory::split_subdir;
      else
	{
	  _error->Error(_("Bad section name '%s' (use 'none', 'topdir', or 'subdir')"), args[0].c_str());
	  return NULL;
	}
    }

  if(args.size()>=2)
    {
      if(!strcasecmp(args[1].c_str(), "passthrough"))
	passthrough=true;
      else if(!strcasecmp(args[1].c_str(), "nopassthrough"))
	passthrough=false;
      else
	{
	  _error->Error(_("Bad passthrough setting '%s' (use 'passthrough' or 'nopassthrough')"),
			args[1].c_str());
	  return NULL;
	}
    }

  if(args.size()>2)
    {
      _error->Error(_("Too many arguments to by-section grouping policy"));
      return NULL;
    }

  if(!chain)
    chain=new pkg_grouppolicy_end_factory;

  return new pkg_grouppolicy_section_factory(split_mode, passthrough, chain);
}

static pkg_grouppolicy_factory *parse_status_policy(const arglist &args,
						    pkg_grouppolicy_factory *chain)
{
  if(args.size()!=0)
    {
      _error->Error(_("By-status grouping policies take no arguments"));
      return NULL;
    }

  if(!chain)
    chain=new pkg_grouppolicy_end_factory;

  return new pkg_grouppolicy_status_factory(chain);
}

static pkg_grouppolicy_factory *parse_filter_policy(const arglist &args,
						    pkg_grouppolicy_factory *chain)
{
  if(args.size()!=1)
    {
      _error->Error(_("Exactly one filter name must be provided as an argument to a filter policy"));
      return NULL;
    }

  if(!strcasecmp(args[0].c_str(), "missing"))
    {
      if(!chain)
	chain=new pkg_grouppolicy_end_factory;

      return new pkg_grouppolicy_filter_factory(pkg_missing_filter, chain);
    }
  else
    {
      _error->Error(_("Bad filter type '%s' (valid types are: 'missing')"),
		      args[0].c_str());
      return NULL;
    }
}

static pkg_grouppolicy_factory *parse_mode_policy(const arglist &args,
						  pkg_grouppolicy_factory *chain)
{
  if(args.size()!=0)
    {
      _error->Error(_("By-mode grouping policies take no arguments"));
      return NULL;
    }

  if(!chain)
    chain=new pkg_grouppolicy_end_factory;

  return new pkg_grouppolicy_mode_factory(chain);
}

static pkg_grouppolicy_factory *parse_firstchar_policy(const arglist &args,
						       pkg_grouppolicy_factory *chain)
{
  if(args.size()!=0)
    {
      _error->Error(_("First-character grouping policies take no arguments"));
      return NULL;
    }

  if(!chain)
    chain=new pkg_grouppolicy_end_factory;

  return new pkg_grouppolicy_firstchar_factory(chain);
}

static pkg_grouppolicy_factory *parse_ver_policy(const arglist &args,
						 pkg_grouppolicy_factory *chain)
{
  if(args.size()!=0)
    {
      _error->Error(_("Version-generating grouping policies take no arguments"));
      return NULL;
    }

  if(chain)
    {
      _error->Error(_("Version-generating grouping policies must be the tail of a chain"));
      return NULL;
    }

  return new pkg_grouppolicy_ver_factory;
}

static pkg_grouppolicy_factory *parse_dep_policy(const arglist &args,
						 pkg_grouppolicy_factory *chain)
{
  if(args.size()!=0)
    {
      _error->Error(_("Dep-generating grouping policies take no arguments"));
      return NULL;
    }

  if(chain)
    {
      _error->Error(_("Dep-generating grouping policies must be the tail of a chain"));
      return NULL;
    }

  return new pkg_grouppolicy_dep_factory;
}

static pkg_grouppolicy_factory *parse_priority_policy(const arglist &args,
						      pkg_grouppolicy_factory *chain)
{
  if(args.size()!=0)
    {
      _error->Error(_("By-priority grouping policies take no arguments"));
      return NULL;
    }

  if(!chain)
    chain=new pkg_grouppolicy_end_factory;

  return new pkg_grouppolicy_priority_factory(chain);
}

static pkg_grouppolicy_factory *parse_hier_policy(const arglist &args,
						  pkg_grouppolicy_factory *chain)
{
  if(!chain)
    chain=new pkg_grouppolicy_end_factory;

  if(!args.empty())
    {
      // FIXME: who deletes this??
      pkg_hier *hier=new pkg_hier;

      for(arglist::size_type i=0; i<args.size(); ++i)
	hier->input_file(args[i]);

      return new pkg_grouppolicy_hier_factory(hier, chain, true);
    }
  else
    return new pkg_grouppolicy_hier_factory(get_user_pkg_hier(), chain, false);
}

static pkg_grouppolicy_factory *parse_task_policy(const arglist &args,
						  pkg_grouppolicy_factory *chain)
{
  if(args.size()!=0)
    {
      _error->Error(_("Task grouping policies take no arguments"));
      return NULL;
    }

  if(!chain)
    chain=new pkg_grouppolicy_end_factory;

  return new pkg_grouppolicy_task_factory(chain);
}

static parser_map parse_types;

static void init_parse_types()
{
  static bool initted_parse_types=false;

  if(!initted_parse_types)
    {
      parse_types["section"]=parse_section_policy;
      parse_types["status"]=parse_status_policy;
      parse_types["action"]=parse_mode_policy;
      parse_types["filter"]=parse_filter_policy;
      parse_types["firstchar"]=parse_firstchar_policy;

      parse_types["versions"]=parse_ver_policy;
      parse_types["deps"]=parse_dep_policy;
      parse_types["priority"]=parse_priority_policy;
      parse_types["hier"]=parse_hier_policy;
      parse_types["task"]=parse_task_policy;

      initted_parse_types=true;
    }
}

// Parses a chain of grouping policies from the given string and returns them.
pkg_grouppolicy_factory *parse_grouppolicy(string s)
{
  init_parse_types();

  typedef vector<pair<string, arglist> > temp_parse_list;
  temp_parse_list parsed;

  string::size_type i=0;

  while(i<s.size())
    {
      pair<string, arglist> current;

      // Find the first name.  Use tolower for nice case-insensitivity.
      while(i<s.size() && s[i]!=',' && s[i]!='(')
	current.first+=tolower(s[i++]);

      if(current.first.size()==0)
	{
	  _error->Error(_("Invalid zero-length grouping policy name"));
	  return NULL;
	}

      if(i<s.size() && s[i]=='(') // Parse argument list
	{
	  while(i<s.size() && s[i]!=')')
	    {
	      ++i; // Clear out the leading '(' or ','
	      string curarg;
	      while(i<s.size() && s[i]!=',' && s[i]!=')')
		curarg+=s[i++];

	      current.second.push_back(curarg);
	    }

	  if(!(i<s.size() && s[i]==')')) // Unexpected EOT, bail
	    {
	      _error->Error(_("Unmatched '(' in grouping policy description"));
	      return NULL;
	    }

	  ++i; // Clear out the ')'
	}

      parsed.push_back(current);
      // Ew, takes kinda long.  Probably not an issue now, but creating current
      // as a new item in parsed would be faster.

      if(i<s.size() && s[i]==',')
	i++; // Clear out trailing commas.
    }

  // Now run through the parsed stuff from back-to-front and instantiate it.
  pkg_grouppolicy_factory *rval=NULL;
  temp_parse_list::reverse_iterator j=parsed.rbegin();

  while(j!=parsed.rend())
    {
      // Look up the parse function
      parser_map::iterator found=parse_types.find(j->first);

      // Die gracefully if it's bad.
      if(found==parse_types.end())
	{
	  _error->Error(_("Invalid grouping policy type '%s'"),
			  j->first.c_str());
	  delete rval;
	  return NULL;
	}

      // Apply it to the argument list
      pkg_grouppolicy_factory *new_rval=found->second(j->second,
						      rval);

      // Check for failure
      if(!new_rval)
	{
	  delete rval;
	  return NULL;
	}

      rval=new_rval;
      ++j;
    }

  if(!rval)
    rval=new pkg_grouppolicy_end_factory;

  return rval;
}
