/***************************************************************************
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation; either version 2 of the License, or      *
 *  (at your option) any later version.                                    *
 *                                                                         *
 ***************************************************************************/

#include <stdio.h>
#include <aspell.h>

// QT headers
#include <qtimer.h>
#include <qcolor.h>
#include <qlistbox.h>

// Kadu headers
#include "message_box.h"
#include "chat.h"
#include "chat_manager.h"
#include "misc.h"
#include "modules.h"
#include "config_dialog.h"
#include "kadu.h"
#include "custom_input.h"

// private headers
#include "spellchecker.h"

// private definitions
#define MODULE_SPELLCHECKER_VERSION 0.19

// HTML mark for mispelled words
const char* endMark = "</span>";
SpellChecker* spellcheck;

extern "C" int spellchecker_init()
{
	spellcheck = new SpellChecker();

	// use configuration settings to create spellcheckers for languages
	if ( !spellcheck->buildCheckers() )
		/*
			Deleting spellcheck here produces Segmentation Foult.
			I guess that this is becouse any objects created in this module (library)
			is deleted automatically by deleting library object (by ModulesManager),
			while this is deleted over here... or I don't know why :)
		*/
		return 1;
	else
	{
		QObject::connect(chat_manager, SIGNAL(chatCreated(const UserGroup*)), spellcheck, SLOT(chatCreated(const UserGroup*)));
		return 0;
	}
}

extern "C" void spellchecker_close()
{
	if ( spellcheck )
	{
		QObject::disconnect(chat_manager, SIGNAL(chatCreated(const UserGroup*)), spellcheck, SLOT(chatCreated(const UserGroup*)));
		delete spellcheck;
	}
}

SpellChecker::SpellChecker() : QObject()
{

	// this timer will wake up spell checker to analyze chat windows'
	// input and mark spelling mistakes
	myWakeupTimer = new QTimer( this );
	connect( myWakeupTimer, SIGNAL(timeout()), this, SLOT(executeChecking()) );

	// prepare configuration of spellchecker
	spellConfig = new_aspell_config();
	aspell_config_replace( spellConfig, "encoding", "utf-8" );

	// load configuration file
	config = new ConfigFile( ggPath( QString("spellchecker.conf") ) );

	// Checker tab in Kadu config dialog
	ConfigDialog::addTab("ASpell", dataPath("kadu/modules/data/spellchecker/config.png"));
	ConfigDialog::addVGroupBox("ASpell", "ASpell", QT_TRANSLATE_NOOP("@default", "Misspelled words marking options"));
		ConfigDialog::addColorButton(config, "ASpell", "Misspelled words marking options", QT_TRANSLATE_NOOP("@default", "Color"), "Color", QColor("#FF0101"));
		ConfigDialog::addCheckBox(config, "ASpell", "Misspelled words marking options", QT_TRANSLATE_NOOP("@default", "Bold"), "Bold");
		ConfigDialog::addCheckBox(config, "ASpell", "Misspelled words marking options", QT_TRANSLATE_NOOP("@default", "Italic"), "Italic");
		ConfigDialog::addCheckBox(config, "ASpell", "Misspelled words marking options", QT_TRANSLATE_NOOP("@default", "Underline"), "Underline");
		ConfigDialog::addCheckBox(config, "ASpell", "ASpell", QT_TRANSLATE_NOOP("@default", "Ignore accents"), "Accents");
		ConfigDialog::addCheckBox(config, "ASpell", "ASpell", QT_TRANSLATE_NOOP("@default", "Ignore case"), "Case");

	ConfigDialog::registerSlotOnCreateTab("ASpell", this, SLOT(onCreateConfig()));
	ConfigDialog::registerSlotOnCloseTab("ASpell", this, SLOT(onDestroyConfig()));
	ConfigDialog::registerSlotOnApplyTab("ASpell", this, SLOT(onUpdateConfig()));

	ConfigDialog::addGrid("ASpell", "ASpell", "lists", 3);
		ConfigDialog::addGrid("ASpell", "lists", "list1", 1);
			ConfigDialog::addLabel("ASpell", "list1", QT_TRANSLATE_NOOP("@default", "Available languages"));
			ConfigDialog::addListBox("ASpell", "list1", "available");

		ConfigDialog::addGrid("ASpell", "lists", "list2", 1);
			ConfigDialog::addPushButton("ASpell", "list2", "", "AddToNotifyList","","forward");
			ConfigDialog::addPushButton("ASpell", "list2", "", "RemoveFromNotifyList","","back");

		ConfigDialog::addGrid("ASpell", "lists", "list3", 1);
			ConfigDialog::addLabel("ASpell", "list3", QT_TRANSLATE_NOOP("@default", "Checked"));
			ConfigDialog::addListBox("ASpell", "list3", "checked");

	ConfigDialog::connectSlot("ASpell", "", SIGNAL(clicked()), this, SLOT(configForward()), "forward");
	ConfigDialog::connectSlot("ASpell", "", SIGNAL(clicked()), this, SLOT(configBackward()), "back");
	ConfigDialog::connectSlot("ASpell", "available", SIGNAL(doubleClicked(QListBoxItem *)), this, SLOT(configForward2(QListBoxItem *)));
	ConfigDialog::connectSlot("ASpell", "checked", SIGNAL(doubleClicked(QListBoxItem *)), this, SLOT(configBackward2(QListBoxItem *)));

	// load mark settings
	buildMarkTag();
}

SpellChecker::~SpellChecker()
{
	// Removing configuration tab
	ConfigDialog::disconnectSlot("ASpell", "", SIGNAL(clicked()), this, SLOT(configForward()), "forward");
	ConfigDialog::disconnectSlot("ASpell", "", SIGNAL(clicked()), this, SLOT(configBackward()), "back");
	ConfigDialog::disconnectSlot("ASpell", "available", SIGNAL(doubleClicked(QListBoxItem *)), this, SLOT(configForward2(QListBoxItem *)));
	ConfigDialog::disconnectSlot("ASpell", "checked", SIGNAL(doubleClicked(QListBoxItem *)), this, SLOT(configBackward2(QListBoxItem *)));
	ConfigDialog::removeControl("ASpell", "Color");
	ConfigDialog::removeControl("ASpell", "Bold");
	ConfigDialog::removeControl("ASpell", "Italic");
	ConfigDialog::removeControl("ASpell", "Underline");
	ConfigDialog::removeControl("ASpell", "Ignore accents");
	ConfigDialog::removeControl("ASpell", "Ignore case");
	ConfigDialog::removeControl("ASpell", "Misspelled words marking options");
	ConfigDialog::removeControl("ASpell", "Available languages");
	ConfigDialog::removeControl("ASpell", "available");
	ConfigDialog::removeControl("ASpell", "list1");
	ConfigDialog::removeControl("ASpell", "", "forward");
	ConfigDialog::removeControl("ASpell", "", "back");
	ConfigDialog::removeControl("ASpell", "list2");
	ConfigDialog::removeControl("ASpell", "Checked");
	ConfigDialog::removeControl("ASpell", "checked");
	ConfigDialog::removeControl("ASpell", "list3");
	ConfigDialog::removeControl("ASpell", "lists");
	ConfigDialog::removeTab("ASpell");
	ConfigDialog::unregisterSlotOnCreateTab("ASpell", this, SLOT(onCreateConfig()));
	ConfigDialog::unregisterSlotOnApplyTab("ASpell", this, SLOT(onUpdateConfig()));
	ConfigDialog::unregisterSlotOnCloseTab("ASpell", this, SLOT(onDestroyConfig()));

	// Disabling spellcheker
	myWakeupTimer->stop();
	disconnect( myWakeupTimer, SIGNAL(timeout()), this, SLOT(executeChecking()));

	ChatList openChats = chat_manager->chats();
	for ( ChatList::Iterator it = openChats.begin(); it != openChats.end(); it++ )
		cleanMessage( *it );

	delete_aspell_config( spellConfig );
	delete config;

	delete myWakeupTimer;

	for ( Checkers::Iterator it = checkers.begin(); it != checkers.end(); it++ )
		delete_aspell_speller( it.data() );
}

QStringList SpellChecker::notCheckedLanguages()
{
	QStringList result;
	AspellDictInfoList* dlist;
	AspellDictInfoEnumeration* dels;
	const AspellDictInfo* entry;

	/* the returned pointer should _not_ need to be deleted */
	dlist = get_aspell_dict_info_list( spellConfig );

	dels = aspell_dict_info_list_elements(dlist);
	while ( (entry = aspell_dict_info_enumeration_next( dels )) != 0)
	{
		if ( checkers.find( entry->name ) == checkers.end() )
			result.push_back( entry->name );
	}
	delete_aspell_dict_info_enumeration(dels);

	return result;
}

QStringList SpellChecker::checkedLanguages()
{
	QStringList result;
	for ( Checkers::Iterator it = checkers.begin(); it != checkers.end(); it++ )
		result.append( it.key() );

	return result;
}

bool SpellChecker::addCheckedLang( QString& name )
{
	if ( checkers.find( name ) != checkers.end() )
		return true;

	aspell_config_replace( spellConfig, "lang", name.ascii() );

	// create spell checker using prepared configuration
	AspellCanHaveError* possibleErr = new_aspell_speller( spellConfig );
	if ( aspell_error_number( possibleErr ) != 0 )
	{
		MessageBox::msg( aspell_error_message( possibleErr ) );
		for ( Checkers::Iterator it = checkers.begin(); it != checkers.end(); it++ )
			delete_aspell_speller( it.data() );

		return false;
	}
	else
		checkers[name] = to_aspell_speller( possibleErr );

	if ( checkers.size() == 1 )
	{
		ChatList openChats = chat_manager->chats();
		for ( ChatList::Iterator it = openChats.begin(); it != openChats.end(); it++ )
			chatCreated( (*it)->users() );
	}

	return true;
}

void SpellChecker::removeCheckedLang( QString& name )
{
	Checkers::Iterator checker = checkers.find(name);
	if ( checker != checkers.end() )
	{
		delete_aspell_speller( checker.data() );
		checkers.erase(name);
	}
}

bool SpellChecker::buildCheckers()
{
	for ( Checkers::Iterator it = checkers.begin(); it != checkers.end(); it++ )
		delete_aspell_speller( it.data() );

	checkers.clear();

	// load languages to check from configuration
	QString checkedStr = config->readEntry( "ASpell", "Checked", "pl");
	QStringList checkedList = QStringList::split( ',', checkedStr );

	if ( config->readBoolEntry( "ASpell", "Accents", false ) )
		aspell_config_replace( spellConfig, "ignore-accents", "true" );
	else
		aspell_config_replace( spellConfig, "ignore-accents", "false" );

	if ( config->readBoolEntry( "ASpell", "Case", false ) )
		aspell_config_replace( spellConfig, "ignore-case", "true" );
	else
		aspell_config_replace( spellConfig, "ignore-case", "false" );

	// create aspell checkers for each language
	for (unsigned int i = 0; i < checkedList.count(); i++)
	{
		if ( !addCheckedLang( checkedList[i] ) )
		{
			delete_aspell_config( spellConfig );
			delete config;
			return false;
		}
	}
	return true;
}

void SpellChecker::buildMarkTag()
{
	ChatList openChats = chat_manager->chats();
	for ( ChatList::Iterator it = openChats.begin(); it != openChats.end(); it++ )
		cleanMessage( *it );

	beginMark = "<span style=\"";

	if ( config->readBoolEntry( "ASpell", "Bold", false ) )
		beginMark += "font-weight:600;";
	if ( config->readBoolEntry( "ASpell", "Italic", false ) )
		beginMark += "font-style:italic;";
	if ( config->readBoolEntry( "ASpell", "Underline", false ) )
		beginMark += "text-decoration:underline;";
	QColor colorMark("#FF0101");
	colorMark = config->readColorEntry( "ASpell", "Color", &colorMark );
	beginMark += "color:" + colorMark.name() + "\">";
}

void SpellChecker::changeMarkColor( const QColor& col )
{
	QString normalized = col.name();
	normalized[2] = '1';
	normalized[4] = '2';
	normalized[6] = '3';
	config->writeEntry( "ASpell", "Color", QColor( normalized ) );
}

void SpellChecker::chatCreated( const UserGroup* interlocutors )
{
	if ( checkers.size() > 0 )
	{
		if ( !myWakeupTimer->isActive() )
			myWakeupTimer->start( 200 );

		connect( chat_manager->findChat( interlocutors ), SIGNAL(messageSendRequested(Chat*)), this, SLOT(cleanMessage(Chat*)) );
	}
}

void SpellChecker::cleanMessage( Chat* chat )
{
	HtmlDocument parsedHtml;
	parsedHtml.parseHtml( chat->edit()->text() );
	bool change = false;

	for( int i = 0; i < parsedHtml.countElements(); i++)
	{
		if ( isTagMyOwn( parsedHtml, i ) )
		{
			parsedHtml.setElementValue( i, "" );
			i++;
			parsedHtml.setElementValue( i + 1, "" );
			i++;
			change = true;
		}
	}

	// if we have changed contents of chat window, than update it
	if ( change )
		updateChat( chat->edit(), parsedHtml.generateHtml() );
}

void SpellChecker::executeChecking()
{
	if ( chat_manager->chats().size() == 0 || checkers.size() == 0 )
		myWakeupTimer->stop();

	// iterate through open chats and check their inputs
	ChatList openChats = chat_manager->chats();
	for ( ChatList::Iterator it = openChats.begin(); it != openChats.end(); it++ )
	{
		HtmlDocument parsedHtml;
		parsedHtml.parseHtml( (*it)->edit()->text() );
		bool change = false;

		for( int i = 0; i < parsedHtml.countElements(); i++)
		{
			if (parsedHtml.isTagElement( i ))
				continue;

			QString text = parsedHtml.elementText( i );
			bool inWhite = true;
			int lastBegin = -1, lastEnd = -1;

			for(unsigned int j = 0; j < text.length(); j++)
			{
				if ( inWhite )
				{
					if ( text[j].isLetter() )
					{
						inWhite = false;
						lastBegin = j;
					}
				}
				else
				{
					// if we are at the end of current word
					if ( !text[j].isLetter() || j == text.length() - 1 )
					{
						inWhite = true;
						if ( text[j].isLetter() && j == text.length() - 1)
						{
							lastEnd = j + 1;

							if ( i + 1 < parsedHtml.countElements() && isTagMyOwn( parsedHtml, i + 1 ) )
							{
								parsedHtml.splitElement( i, lastBegin, lastEnd - lastBegin );
								parsedHtml.setElementValue( i + 2, parsedHtml.elementText(i) +
								parsedHtml.elementText(i+2), false );
								parsedHtml.setElementValue( i, "");

								continue;
							}
						}
						else
							lastEnd = j;

						QString word = text.mid( lastBegin, lastEnd - lastBegin );
						QCString wordUtf8 = word.utf8();

						// run checkers for all languages to check if this word is
						// valud in some of them
						bool isWordValid = checkers.size() == 0;
						for ( Checkers::Iterator it = checkers.begin(); it != checkers.end(); it++ )
						{
							if ( aspell_speller_check( it.data(), wordUtf8, -1 ) )
							{
								isWordValid = true;
								break;
							}
						}

						if ( !isWordValid )
						{
							// current word isn't in dictionary, so take a look at it
							parsedHtml.splitElement( i, lastBegin, lastEnd - lastBegin );

							// check if this word isn't already marked as misspelled
							if ( (i == 0 || !isTagMyOwn( parsedHtml, i - 1 )) &&
									i < parsedHtml.countElements() - 1 &&
									!parsedHtml.isTagElement( i + 1 ) )
							{
								parsedHtml.insertTag( i, beginMark );
								parsedHtml.insertTag( i + 2, endMark );
								change = true;
							}
							else if ( i > 0 && i < parsedHtml.countElements() )
							{
								// word is currently marked, but we must check if we don't
								// have some extra charactes inserted between this word
								// and the endmark
								if ( !parsedHtml.isTagElement( i + 1 ) )
								{
									parsedHtml.setElementValue( i + 2, parsedHtml.elementText( i + 1 ),
																							false );
									parsedHtml.setElementValue( i + 1, endMark, true );
									change = true;
								}
							}

							break;
						}
						else
						{
							// current word is correct, so remove possible tags
							if ( i > 0 && isTagMyOwn( parsedHtml, i - 1 ) &&
									 i < parsedHtml.countElements() - 1 &&
									 parsedHtml.isTagElement( i + 1 ) )
							{
								// we bring word back to not marked
								parsedHtml.setElementValue( i - 1, "");
								parsedHtml.setElementValue( i + 1, "");

								// trigger chat update
								change = true;
							}
						}
					}
				}
			}
		}
		// if we have changed contents of chat window, than update it
		if ( change )
			updateChat( (*it)->edit(), parsedHtml.generateHtml() );
	}
}

void SpellChecker::updateChat( CustomInput* edit, QString text )
{
	int currentY, currentX;
	edit->getCursorPosition(&currentY, &currentX);
	edit->setUpdatesEnabled( false );
	edit->setText( text );
	// set cursor to initial position
	edit->setCursorPosition(currentY, currentX);
	edit->setUpdatesEnabled( true );
}

// check some part of element content to distiguish it from
// tags generated by external code
bool SpellChecker::isTagMyOwn( HtmlDocument& doc, int idx)
{
	unsigned int len = beginMark.length();
	if ( doc.isTagElement ( idx ) )
	{
		QString text = doc.elementText( idx );
		return text.length() == len && text[len - 3] == beginMark[len - 3] &&
			text[len - 5] == beginMark[len - 5] &&
			text[len - 7] == beginMark[len - 7];
	}
	else
		return false;
}

void SpellChecker::onCreateConfig()
{
	QListBox* avail = ConfigDialog::getListBox("ASpell", "available");
	QListBox* check = ConfigDialog::getListBox("ASpell", "checked");
	avail->setSelectionMode(QListBox::Single);
	check->setSelectionMode(QListBox::Single);
	avail->insertStringList(notCheckedLanguages());
	check->insertStringList(checkedLanguages());
	modules_manager->moduleIncUsageCount("spellchecker");
}

void SpellChecker::onDestroyConfig()
{
	onUpdateConfig();
	modules_manager->moduleDecUsageCount("spellchecker");
}

void SpellChecker::onUpdateConfig()
{
	config->writeEntry("ASpell", "Checked", checkedLanguages().join(","));
	config->sync();
	buildMarkTag();
}

void SpellChecker::configForward()
{
	QListBox* avail = ConfigDialog::getListBox("ASpell", "available");
	QListBoxItem* it = avail->selectedItem();
	if (it)
		configForward2(it);
}

void SpellChecker::configBackward()
{
	QListBox* check = ConfigDialog::getListBox("ASpell", "checked");
	QListBoxItem* it = check->selectedItem();
	if (it)
		configBackward2(it);
}

void SpellChecker::configForward2(QListBoxItem* it)
{
	QListBox* avail = ConfigDialog::getListBox("ASpell", "available");
	QListBox* check = ConfigDialog::getListBox("ASpell", "checked");
	QString langName = it->text();
	if ( addCheckedLang(langName) )
	{
		check->insertItem(langName);
		avail->removeItem(avail->currentItem());
	}
}

void SpellChecker::configBackward2(QListBoxItem* it)
{
	QListBox* avail = ConfigDialog::getListBox("ASpell", "available");
	QListBox* check = ConfigDialog::getListBox("ASpell", "checked");
	QString langName = it->text();
	avail->insertItem(langName);
	check->removeItem(check->currentItem());
	removeCheckedLang(langName);
}
