//
// System.Web.UI.WebControls.GridView.cs
//
// Authors:
//	Lluis Sanchez Gual (lluis@novell.com)
//
// (C) 2005 Novell, Inc (http://www.novell.com)
//
// Permission is hereby granted, free of charge, to any person obtaining
// a copy of this software and associated documentation files (the
// "Software"), to deal in the Software without restriction, including
// without limitation the rights to use, copy, modify, merge, publish,
// distribute, sublicense, and/or sell copies of the Software, and to
// permit persons to whom the Software is furnished to do so, subject to
// the following conditions:
// 
// The above copyright notice and this permission notice shall be
// included in all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
// EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
// MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
// NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
// LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
// OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
// WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
//

#if NET_2_0

using System;
using System.Collections;
using System.Collections.Specialized;
using System.ComponentModel;
using System.Web.UI;
using System.Security.Permissions;
using System.Text;
using System.IO;
using System.Reflection;

namespace System.Web.UI.WebControls
{
	[DesignerAttribute ("System.Web.UI.Design.WebControls.GridViewDesigner, " + Consts.AssemblySystem_Design, "System.ComponentModel.Design.IDesigner")]
	[ControlValuePropertyAttribute ("SelectedValue")]
	[DefaultEventAttribute ("SelectedIndexChanged")]
	[AspNetHostingPermissionAttribute (SecurityAction.LinkDemand, Level = AspNetHostingPermissionLevel.Minimal)]
	[AspNetHostingPermissionAttribute (SecurityAction.InheritanceDemand, Level = AspNetHostingPermissionLevel.Minimal)]
	public class GridView: CompositeDataBoundControl, ICallbackEventHandler, ICallbackContainer
	{
		Table table;
		GridViewRowCollection rows;
		GridViewRow headerRow;
		GridViewRow footerRow;
		GridViewRow bottomPagerRow;
		GridViewRow topPagerRow;
		
		IOrderedDictionary currentEditRowKeys;
		IOrderedDictionary currentEditNewValues;
		IOrderedDictionary currentEditOldValues;
		
		ITemplate pagerTemplate;
		ITemplate emptyDataTemplate;
		
		PropertyDescriptor[] cachedKeyProperties;
			
		// View state
		DataControlFieldCollection columns;
		PagerSettings pagerSettings;
		
		TableItemStyle alternatingRowStyle;
		TableItemStyle editRowStyle;
		TableItemStyle emptyDataRowStyle;
		TableItemStyle footerStyle;
		TableItemStyle headerStyle;
		TableItemStyle pagerStyle;
		TableItemStyle rowStyle;
		TableItemStyle selectedRowStyle;
		DataKeyArray keys;
		DataKey oldEditValues;
		AutoGeneratedFieldProperties[] autoFieldProperties;
		readonly string[] emptyKeys = new string[0];
		
		private static readonly object PageIndexChangedEvent = new object();
		private static readonly object PageIndexChangingEvent = new object();
		private static readonly object RowCancelingEditEvent = new object();
		private static readonly object RowCommandEvent = new object();
		private static readonly object RowCreatedEvent = new object();
		private static readonly object RowDataBoundEvent = new object();
		private static readonly object RowDeletedEvent = new object();
		private static readonly object RowDeletingEvent = new object();
		private static readonly object RowEditingEvent = new object();
		private static readonly object RowUpdatedEvent = new object();
		private static readonly object RowUpdatingEvent = new object();
		private static readonly object SelectedIndexChangedEvent = new object();
		private static readonly object SelectedIndexChangingEvent = new object();
		private static readonly object SortedEvent = new object();
		private static readonly object SortingEvent = new object();
		
		// Control state
		int pageIndex;
		int pageCount = -1;
		int selectedIndex = -1;
		int editIndex = -1;
		SortDirection sortDirection = SortDirection.Ascending;
		string sortExpression;
		
		public GridView ()
		{
		}
		
		public event EventHandler PageIndexChanged {
			add { Events.AddHandler (PageIndexChangedEvent, value); }
			remove { Events.RemoveHandler (PageIndexChangedEvent, value); }
		}
		
		public event GridViewPageEventHandler PageIndexChanging {
			add { Events.AddHandler (PageIndexChangingEvent, value); }
			remove { Events.RemoveHandler (PageIndexChangingEvent, value); }
		}
		
		public event GridViewCancelEditEventHandler RowCancelingEdit {
			add { Events.AddHandler (RowCancelingEditEvent, value); }
			remove { Events.RemoveHandler (RowCancelingEditEvent, value); }
		}
		
		public event GridViewCommandEventHandler RowCommand {
			add { Events.AddHandler (RowCommandEvent, value); }
			remove { Events.RemoveHandler (RowCommandEvent, value); }
		}
		
		public event GridViewRowEventHandler RowCreated {
			add { Events.AddHandler (RowCreatedEvent, value); }
			remove { Events.RemoveHandler (RowCreatedEvent, value); }
		}
		
		public event GridViewRowEventHandler RowDataBound {
			add { Events.AddHandler (RowDataBoundEvent, value); }
			remove { Events.RemoveHandler (RowDataBoundEvent, value); }
		}
		
		public event GridViewDeletedEventHandler RowDeleted {
			add { Events.AddHandler (RowDeletedEvent, value); }
			remove { Events.RemoveHandler (RowDeletedEvent, value); }
		}
		
		public event GridViewDeleteEventHandler RowDeleting {
			add { Events.AddHandler (RowDeletingEvent, value); }
			remove { Events.RemoveHandler (RowDeletingEvent, value); }
		}
		
		public event GridViewEditEventHandler RowEditing {
			add { Events.AddHandler (RowEditingEvent, value); }
			remove { Events.RemoveHandler (RowEditingEvent, value); }
		}
		
		public event GridViewUpdatedEventHandler RowUpdated {
			add { Events.AddHandler (RowUpdatedEvent, value); }
			remove { Events.RemoveHandler (RowUpdatedEvent, value); }
		}
		
		public event GridViewUpdateEventHandler RowUpdating {
			add { Events.AddHandler (RowUpdatingEvent, value); }
			remove { Events.RemoveHandler (RowUpdatingEvent, value); }
		}
		
		public event EventHandler SelectedIndexChanged {
			add { Events.AddHandler (SelectedIndexChangedEvent, value); }
			remove { Events.RemoveHandler (SelectedIndexChangedEvent, value); }
		}
		
		public event GridViewSelectEventHandler SelectedIndexChanging {
			add { Events.AddHandler (SelectedIndexChangingEvent, value); }
			remove { Events.RemoveHandler (SelectedIndexChangingEvent, value); }
		}
		
		public event EventHandler Sorted {
			add { Events.AddHandler (SortedEvent, value); }
			remove { Events.RemoveHandler (SortedEvent, value); }
		}
		
		public event GridViewSortEventHandler Sorting {
			add { Events.AddHandler (SortingEvent, value); }
			remove { Events.RemoveHandler (SortingEvent, value); }
		}
		
		protected virtual void OnPageIndexChanged (EventArgs e)
		{
			if (Events != null) {
				EventHandler eh = (EventHandler) Events [PageIndexChangedEvent];
				if (eh != null) eh (this, e);
			}
		}
		
		protected virtual void OnPageIndexChanging (GridViewPageEventArgs e)
		{
			if (Events != null) {
				GridViewPageEventHandler eh = (GridViewPageEventHandler) Events [PageIndexChangingEvent];
				if (eh != null) eh (this, e);
			}
		}
		
		protected virtual void OnRowCancelingEdit (GridViewCancelEditEventArgs e)
		{
			if (Events != null) {
				GridViewCancelEditEventHandler eh = (GridViewCancelEditEventHandler) Events [RowCancelingEditEvent];
				if (eh != null) eh (this, e);
			}
		}
		
		protected virtual void OnRowCommand (GridViewCommandEventArgs e)
		{
			if (Events != null) {
				GridViewCommandEventHandler eh = (GridViewCommandEventHandler) Events [RowCommandEvent];
				if (eh != null) eh (this, e);
			}
		}
		
		protected virtual void OnRowCreated (GridViewRowEventArgs e)
		{
			if (Events != null) {
				GridViewRowEventHandler eh = (GridViewRowEventHandler) Events [RowCreatedEvent];
				if (eh != null) eh (this, e);
			}
		}
		
		protected virtual void OnRowDataBound (GridViewRowEventArgs e)
		{
			if (Events != null) {
				GridViewRowEventHandler eh = (GridViewRowEventHandler) Events [RowDataBoundEvent];
				if (eh != null) eh (this, e);
			}
		}
		
		protected virtual void OnRowDeleted (GridViewDeletedEventArgs e)
		{
			if (Events != null) {
				GridViewDeletedEventHandler eh = (GridViewDeletedEventHandler) Events [RowDeletedEvent];
				if (eh != null) eh (this, e);
			}
		}
		
		protected virtual void OnRowDeleting (GridViewDeleteEventArgs e)
		{
			if (Events != null) {
				GridViewDeleteEventHandler eh = (GridViewDeleteEventHandler) Events [RowDeletingEvent];
				if (eh != null) eh (this, e);
			}
		}
		
		protected virtual void OnRowEditing (GridViewEditEventArgs e)
		{
			if (Events != null) {
				GridViewEditEventHandler eh = (GridViewEditEventHandler) Events [RowEditingEvent];
				if (eh != null) eh (this, e);
			}
		}
		
		protected virtual void OnRowUpdated (GridViewUpdatedEventArgs e)
		{
			if (Events != null) {
				GridViewUpdatedEventHandler eh = (GridViewUpdatedEventHandler) Events [RowUpdatedEvent];
				if (eh != null) eh (this, e);
			}
		}
		
		protected virtual void OnRowUpdating (GridViewUpdateEventArgs e)
		{
			if (Events != null) {
				GridViewUpdateEventHandler eh = (GridViewUpdateEventHandler) Events [RowUpdatingEvent];
				if (eh != null) eh (this, e);
			}
		}
		
		protected virtual void OnSelectedIndexChanged (EventArgs e)
		{
			if (Events != null) {
				EventHandler eh = (EventHandler) Events [SelectedIndexChangedEvent];
				if (eh != null) eh (this, e);
			}
		}
		
		protected virtual void OnSelectedIndexChanging (GridViewSelectEventArgs e)
		{
			if (Events != null) {
				GridViewSelectEventHandler eh = (GridViewSelectEventHandler) Events [SelectedIndexChangingEvent];
				if (eh != null) eh (this, e);
			}
		}
		
		protected virtual void OnSorted (EventArgs e)
		{
			if (Events != null) {
				EventHandler eh = (EventHandler) Events [SortedEvent];
				if (eh != null) eh (this, e);
			}
		}
		
		protected virtual void OnSorting (GridViewSortEventArgs e)
		{
			if (Events != null) {
				GridViewSortEventHandler eh = (GridViewSortEventHandler) Events [SortingEvent];
				if (eh != null) eh (this, e);
			}
		}
		
		
		[WebCategoryAttribute ("Paging")]
		[DefaultValueAttribute (false)]
		public bool AllowPaging {
			get {
				object ob = ViewState ["AllowPaging"];
				if (ob != null) return (bool) ob;
				return false;
			}
			set {
				ViewState ["AllowPaging"] = value;
				RequireBinding ();
			}
		}
		
		[WebCategoryAttribute ("Behavior")]
		[DefaultValueAttribute (false)]
		public bool AllowSorting {
			get {
				object ob = ViewState ["AllowSorting"];
				if (ob != null) return (bool) ob;
				return false;
			}
			set {
				ViewState ["AllowSorting"] = value;
				RequireBinding ();
			}
		}
		
	    [WebCategoryAttribute ("Styles")]
		[PersistenceMode (PersistenceMode.InnerProperty)]
		[NotifyParentProperty (true)]
		[DesignerSerializationVisibility (DesignerSerializationVisibility.Content)]
		public virtual TableItemStyle AlternatingRowStyle {
			get {
				if (alternatingRowStyle == null) {
					alternatingRowStyle = new TableItemStyle ();
					if (IsTrackingViewState)
						alternatingRowStyle.TrackViewState();
				}
				return alternatingRowStyle;
			}
		}

		[WebCategoryAttribute ("Behavior")]
		[DefaultValueAttribute (false)]
		public virtual bool AutoGenerateEditButton {
			get {
				object ob = ViewState ["AutoGenerateEditButton"];
				if (ob != null) return (bool) ob;
				return false;
			}
			set {
				ViewState ["AutoGenerateEditButton"] = value;
				RequireBinding ();
			}
		}

		[WebCategoryAttribute ("Behavior")]
		[DefaultValueAttribute (false)]
		public virtual bool AutoGenerateDeleteButton {
			get {
				object ob = ViewState ["AutoGenerateDeleteButton"];
				if (ob != null) return (bool) ob;
				return false;
			}
			set {
				ViewState ["AutoGenerateDeleteButton"] = value;
				RequireBinding ();
			}
		}

		[WebCategoryAttribute ("Behavior")]
		[DefaultValueAttribute (false)]
		public virtual bool AutoGenerateSelectButton {
			get {
				object ob = ViewState ["AutoGenerateSelectButton"];
				if (ob != null) return (bool) ob;
				return false;
			}
			set {
				ViewState ["AutoGenerateSelectButton"] = value;
				RequireBinding ();
			}
		}

		[WebCategoryAttribute ("Behavior")]
		[DefaultValueAttribute (true)]
		public virtual bool AutoGenerateColumns {
			get {
				object ob = ViewState ["AutoGenerateColumns"];
				if (ob != null) return (bool) ob;
				return true;
			}
			set {
				ViewState ["AutoGenerateColumns"] = value;
				RequireBinding ();
			}
		}
		
		[UrlPropertyAttribute]
		[WebCategoryAttribute ("Appearance")]
		[DefaultValueAttribute ("")]
		[EditorAttribute ("System.Web.UI.Design.ImageUrlEditor, " + Consts.AssemblySystem_Design, "System.Drawing.Design.UITypeEditor, " + Consts.AssemblySystem_Drawing)]
		public virtual string BackImageUrl {
			get {
				object ob = ViewState ["BackImageUrl"];
				if (ob != null) return (string) ob;
				return string.Empty;
			}
			set {
				ViewState ["BackImageUrl"] = value;
				RequireBinding ();
			}
		}

		[DesignerSerializationVisibilityAttribute (DesignerSerializationVisibility.Hidden)]
		[BrowsableAttribute (false)]
		public virtual GridViewRow BottomPagerRow {
			get {
				EnsureDataBound ();
				return bottomPagerRow;
			}
		}
	
		[WebCategoryAttribute ("Accessibility")]
		[DefaultValueAttribute ("")]
		[LocalizableAttribute (true)]
		public string Caption {
			get {
				object ob = ViewState ["Caption"];
				if (ob != null) return (string) ob;
				return string.Empty;
			}
			set {
				ViewState ["Caption"] = value;
				RequireBinding ();
			}
		}
		
		[WebCategoryAttribute ("Accessibility")]
		[DefaultValueAttribute (TableCaptionAlign.NotSet)]
		public virtual TableCaptionAlign CaptionAlign
		{
			get {
				object o = ViewState ["CaptionAlign"];
				if(o != null) return (TableCaptionAlign) o;
				return TableCaptionAlign.NotSet;
			}
			set {
				ViewState ["CaptionAlign"] = value;
				RequireBinding ();
			}
		}

		[WebCategoryAttribute ("Layout")]
		[DefaultValueAttribute (-1)]
		public virtual int CellPadding
		{
			get {
				object o = ViewState ["CellPadding"];
				if (o != null) return (int) o;
				return -1;
			}
			set {
				ViewState ["CellPadding"] = value;
				RequireBinding ();
			}
		}

		[WebCategoryAttribute ("Layout")]
		[DefaultValueAttribute (0)]
		public virtual int CellSpacing
		{
			get {
				object o = ViewState ["CellSpacing"];
				if (o != null) return (int) o;
				return 0;
			}
			set {
				ViewState ["CellSpacing"] = value;
				RequireBinding ();
			}
		}
		
		[EditorAttribute ("System.Web.UI.Design.WebControls.DataControlFieldTypeEditor, " + Consts.AssemblySystem_Design, "System.Drawing.Design.UITypeEditor, " + Consts.AssemblySystem_Drawing)]
		[MergablePropertyAttribute (false)]
		[PersistenceModeAttribute (PersistenceMode.InnerProperty)]
		[DefaultValueAttribute (null)]
		[WebCategoryAttribute ("Misc")]
		public virtual DataControlFieldCollection Columns {
			get {
				if (columns == null) {
					columns = new DataControlFieldCollection ();
					columns.FieldsChanged += new EventHandler (OnFieldsChanged);
					if (IsTrackingViewState)
						((IStateManager)columns).TrackViewState ();
				}
				return columns;
			}
		}

		[DefaultValueAttribute (null)]
		[WebCategoryAttribute ("Data")]
		[TypeConverter (typeof(StringArrayConverter))]
		[EditorAttribute ("System.Web.UI.Design.WebControls.DataFieldEditor, " + Consts.AssemblySystem_Design, "System.Drawing.Design.UITypeEditor, " + Consts.AssemblySystem_Drawing)]
		public virtual string[] DataKeyNames
		{
			get {
				object o = ViewState ["DataKeyNames"];
				if (o != null) return (string[]) o;
				return emptyKeys;
			}
			set {
				ViewState ["DataKeyNames"] = value;
				RequireBinding ();
			}
		}
		
		[BrowsableAttribute (false)]
		[DesignerSerializationVisibilityAttribute (DesignerSerializationVisibility.Hidden)]
		public virtual DataKeyArray DataKeys {
			get {
				EnsureDataBound ();
				return keys;
			}
		}

		[WebCategoryAttribute ("Misc")]
		[DefaultValueAttribute (-1)]
		public int EditIndex {
			get {
				return editIndex;
			}
			set {
				editIndex = value;
				RequireBinding ();
			}
		}
	
	    [WebCategoryAttribute ("Styles")]
		[PersistenceMode (PersistenceMode.InnerProperty)]
		[NotifyParentProperty (true)]
		[DesignerSerializationVisibility (DesignerSerializationVisibility.Content)]
		public virtual TableItemStyle EditRowStyle {
			get {
				if (editRowStyle == null) {
					editRowStyle = new TableItemStyle ();
					if (IsTrackingViewState)
						editRowStyle.TrackViewState();
				}
				return editRowStyle;
			}
		}
		
	    [WebCategoryAttribute ("Styles")]
		[PersistenceMode (PersistenceMode.InnerProperty)]
		[NotifyParentProperty (true)]
		[DesignerSerializationVisibility (DesignerSerializationVisibility.Content)]
		public virtual TableItemStyle EmptyDataRowStyle {
			get {
				if (emptyDataRowStyle == null) {
					emptyDataRowStyle = new TableItemStyle ();
					if (IsTrackingViewState)
						emptyDataRowStyle.TrackViewState();
				}
				return emptyDataRowStyle;
			}
		}
		
		[DefaultValue (null)]
		[TemplateContainer (typeof(GridView), BindingDirection.OneWay)]
		[PersistenceMode (PersistenceMode.InnerProperty)]
	    [Browsable (false)]
		public ITemplate EmptyDataTemplate {
			get { return emptyDataTemplate; }
			set { emptyDataTemplate = value; RequireBinding (); }
		}
		
		[LocalizableAttribute (true)]
		[WebCategoryAttribute ("Appearance")]
		[DefaultValueAttribute ("")]
		public virtual string EmptyDataText {
			get {
				object ob = ViewState ["EmptyDataText"];
				if (ob != null) return (string) ob;
				return string.Empty;
			}
			set {
				ViewState ["EmptyDataText"] = value;
				RequireBinding ();
			}
		}
	
		[WebCategoryAttribute ("Behavior")]
		[DefaultValueAttribute (false)]
		public virtual bool EnableSortingAndPagingCallbacks {
			get {
				object ob = ViewState ["EnableSortingAndPagingCallbacks"];
				if (ob != null) return (bool) ob;
				return false;
			}
			set {
				ViewState ["EnableSortingAndPagingCallbacks"] = value;
				RequireBinding ();
			}
		}
	
		[DesignerSerializationVisibilityAttribute (DesignerSerializationVisibility.Hidden)]
		[BrowsableAttribute (false)]
		public virtual GridViewRow FooterRow {
			get {
				if (footerRow == null)
					footerRow = CreateRow (0, 0, DataControlRowType.Footer, DataControlRowState.Normal);
				return footerRow;
			}
		}
	
	    [WebCategoryAttribute ("Styles")]
		[PersistenceMode (PersistenceMode.InnerProperty)]
		[NotifyParentProperty (true)]
		[DefaultValue (null)]
		[DesignerSerializationVisibility (DesignerSerializationVisibility.Content)]
		public virtual TableItemStyle FooterStyle {
			get {
				if (footerStyle == null) {
					footerStyle = new TableItemStyle ();
					if (IsTrackingViewState)
						footerStyle.TrackViewState();
				}
				return footerStyle;
			}
		}
		
		[WebCategoryAttribute ("Appearance")]
		[DefaultValueAttribute (GridLines.Both)]
		public virtual GridLines GridLines {
			get {
				object ob = ViewState ["GridLines"];
				if (ob != null) return (GridLines) ob;
				return GridLines.Both;
			}
			set {
				ViewState ["GridLines"] = value;
			}
		}

		[DesignerSerializationVisibilityAttribute (DesignerSerializationVisibility.Hidden)]
		[BrowsableAttribute (false)]
		public virtual GridViewRow HeaderRow {
			get {
				if (headerRow == null)
					headerRow = CreateRow (0, 0, DataControlRowType.Header, DataControlRowState.Normal);
				return headerRow;
			}
		}
	
	    [WebCategoryAttribute ("Styles")]
		[PersistenceMode (PersistenceMode.InnerProperty)]
		[NotifyParentProperty (true)]
		[DefaultValue (null)]
		[DesignerSerializationVisibility (DesignerSerializationVisibility.Content)]
		public virtual TableItemStyle HeaderStyle {
			get {
				if (headerStyle == null) {
					headerStyle = new TableItemStyle ();
					if (IsTrackingViewState)
						headerStyle.TrackViewState();
				}
				return headerStyle;
			}
		}
		
		[Category ("Layout")]
		[DefaultValueAttribute (HorizontalAlign.NotSet)]
		public virtual HorizontalAlign HorizontalAlign {
			get {
				object ob = ViewState ["HorizontalAlign"];
				if (ob != null) return (HorizontalAlign) ob;
				return HorizontalAlign.NotSet;
			}
			set {
				ViewState ["HorizontalAlign"] = value;
				RequireBinding ();
			}
		}

		[BrowsableAttribute (false)]
		[DesignerSerializationVisibilityAttribute (DesignerSerializationVisibility.Hidden)]
		public int PageCount {
			get {
				if (pageCount != -1) return pageCount;
				EnsureDataBound ();
				return pageCount;
			}
		}

		[WebCategoryAttribute ("Paging")]
		[BrowsableAttribute (true)]
		[DefaultValueAttribute (0)]
		public int PageIndex {
			get {
				return pageIndex;
			}
			set {
				pageIndex = value;
				RequireBinding ();
			}
		}
	
		[DefaultValueAttribute (10)]
		[WebCategoryAttribute ("Paging")]
		public int PageSize {
			get {
				object ob = ViewState ["PageSize"];
				if (ob != null) return (int) ob;
				return 10;
			}
			set {
				ViewState ["PageSize"] = value;
				RequireBinding ();
			}
		}
	
		[WebCategoryAttribute ("Paging")]
		[DesignerSerializationVisibilityAttribute (DesignerSerializationVisibility.Content)]
		[NotifyParentPropertyAttribute (true)]
		[PersistenceModeAttribute (PersistenceMode.InnerProperty)]
		public PagerSettings PagerSettings {
			get {
				if (pagerSettings == null) {
					pagerSettings = new PagerSettings (this);
					if (IsTrackingViewState)
						((IStateManager)pagerSettings).TrackViewState ();
				}
				return pagerSettings;
			}
		}
	
	    [WebCategoryAttribute ("Styles")]
		[PersistenceMode (PersistenceMode.InnerProperty)]
		[NotifyParentProperty (true)]
		[DesignerSerializationVisibility (DesignerSerializationVisibility.Content)]
		public virtual TableItemStyle PagerStyle {
			get {
				if (pagerStyle == null) {
					pagerStyle = new TableItemStyle ();
					if (IsTrackingViewState)
						pagerStyle.TrackViewState();
				}
				return pagerStyle;
			}
		}
		
		
		[DefaultValue (null)]
		/* DataControlPagerCell isnt specified in the docs */
		//[TemplateContainer (typeof(DataControlPagerCell), BindingDirection.OneWay)]
		[PersistenceMode (PersistenceMode.InnerProperty)]
	    [Browsable (false)]
		public ITemplate PagerTemplate {
			get { return pagerTemplate; }
			set { pagerTemplate = value; RequireBinding (); }
		}
		
		[DefaultValueAttribute ("")]
		[WebCategoryAttribute ("Accessibility")]
//		[TypeConverterAttribute (typeof(System.Web.UI.Design.DataColumnSelectionConverter)]
		public virtual string RowHeaderColumn {
			get {
				object ob = ViewState ["RowHeaderColumn"];
				if (ob != null) return (string) ob;
				return string.Empty;
			}
			set {
				ViewState ["RowHeaderColumn"] = value;
				RequireBinding ();
			}
		}
		
		[DesignerSerializationVisibilityAttribute (DesignerSerializationVisibility.Hidden)]
		[BrowsableAttribute (false)]
		public virtual GridViewRowCollection Rows {
			get {
				EnsureDataBound ();
				return rows;
			}
		}
		
	    [WebCategoryAttribute ("Styles")]
		[PersistenceMode (PersistenceMode.InnerProperty)]
		[NotifyParentProperty (true)]
		[DesignerSerializationVisibility (DesignerSerializationVisibility.Content)]
		public virtual TableItemStyle RowStyle {
			get {
				if (rowStyle == null) {
					rowStyle = new TableItemStyle ();
					if (IsTrackingViewState)
						rowStyle.TrackViewState();
				}
				return rowStyle;
			}
		}
		
		[BrowsableAttribute (false)]
		[DesignerSerializationVisibilityAttribute (DesignerSerializationVisibility.Hidden)]
		public virtual DataKey SelectedDataKey {
			get {
				if (selectedIndex >= 0 && selectedIndex < DataKeys.Count) {
					return DataKeys [selectedIndex];
				} else
					return null;
			}
		}
		
		[BindableAttribute (true)]
		[DefaultValueAttribute (-1)]
		public int SelectedIndex {
			get {
				return selectedIndex;
			}
			set {
				if (selectedIndex >= 0 && selectedIndex < Rows.Count) {
					int oldIndex = selectedIndex;
					selectedIndex = -1;
					Rows [oldIndex].RowState = GetRowState (oldIndex);
				}
				selectedIndex = value;
				if (selectedIndex >= 0 && selectedIndex < Rows.Count) {
					Rows [selectedIndex].RowState = GetRowState (selectedIndex);
				}
			}
		}
	
		[BrowsableAttribute (false)]
		[DesignerSerializationVisibilityAttribute (DesignerSerializationVisibility.Hidden)]
		public virtual GridViewRow SelectedRow {
			get {
				if (selectedIndex >= 0 && selectedIndex < Rows.Count) {
					return Rows [selectedIndex];
				} else
					return null;
			}
		}
		
	    [WebCategoryAttribute ("Styles")]
		[PersistenceMode (PersistenceMode.InnerProperty)]
		[NotifyParentProperty (true)]
		[DesignerSerializationVisibility (DesignerSerializationVisibility.Content)]
		public virtual TableItemStyle SelectedRowStyle {
			get {
				if (selectedRowStyle == null) {
					selectedRowStyle = new TableItemStyle ();
					if (IsTrackingViewState)
						selectedRowStyle.TrackViewState();
				}
				return selectedRowStyle;
			}
		}
		
		[BrowsableAttribute (false)]
		public virtual object SelectedValue {
			get {
				if (SelectedDataKey != null)
					return SelectedDataKey.Value;
				else
					return null;
			}
		}
		
		[WebCategoryAttribute ("Appearance")]
		[DefaultValueAttribute (false)]
		public virtual bool ShowFooter {
			get {
				object ob = ViewState ["ShowFooter"];
				if (ob != null) return (bool) ob;
				return false;
			}
			set {
				ViewState ["ShowFooter"] = value;
				RequireBinding ();
			}
		}
	
		[WebCategoryAttribute ("Appearance")]
		[DefaultValueAttribute (true)]
		public virtual bool ShowHeader {
			get {
				object ob = ViewState ["ShowHeader"];
				if (ob != null) return (bool) ob;
				return true;
			}
			set {
				ViewState ["ShowHeader"] = value;
				RequireBinding ();
			}
		}
		
		[PersistenceModeAttribute (PersistenceMode.InnerProperty)]
		[BrowsableAttribute (false)]
		[DefaultValueAttribute (SortDirection.Ascending)]
		[DesignerSerializationVisibilityAttribute (DesignerSerializationVisibility.Hidden)]
		public virtual SortDirection SortDirection {
			get { return sortDirection; }
		}
		
		[BrowsableAttribute (false)]
		[DesignerSerializationVisibilityAttribute (DesignerSerializationVisibility.Hidden)]
		public virtual string SortExpression {
			get { return sortExpression; }
		}
		
		[DesignerSerializationVisibilityAttribute (DesignerSerializationVisibility.Hidden)]
		[BrowsableAttribute (false)]
		public virtual GridViewRow TopPagerRow {
			get {
				EnsureDataBound ();
				return topPagerRow;
			}
		}
	
		[WebCategoryAttribute ("Accessibility")]
		[DefaultValueAttribute (true)]
		public virtual bool UseAccessibleHeader {
			get {
				object ob = ViewState ["UseAccessibleHeader"];
				if (ob != null) return (bool) ob;
				return true;
			}
			set {
				ViewState ["UseAccessibleHeader"] = value;
				RequireBinding ();
			}
		}
	
		public virtual bool IsBindableType (Type type)
		{
			return type.IsPrimitive || type == typeof(string) || type == typeof(DateTime) || type == typeof(Guid);
		}
		
		protected override DataSourceSelectArguments CreateDataSourceSelectArguments ()
		{
			return base.CreateDataSourceSelectArguments ();
		}
		
		protected virtual ICollection CreateColumns (PagedDataSource dataSource, bool useDataSource)
		{
			ArrayList fields = new ArrayList ();
			
			if (AutoGenerateEditButton || AutoGenerateDeleteButton || AutoGenerateSelectButton) {
				CommandField field = new CommandField ();
				field.ShowEditButton = AutoGenerateEditButton;
				field.ShowDeleteButton = AutoGenerateDeleteButton;
				field.ShowSelectButton = AutoGenerateSelectButton;
				fields.Add (field);
			}
			
			if (AutoGenerateColumns) {
				if (useDataSource)
					autoFieldProperties = CreateAutoFieldProperties (dataSource);
	
				if (autoFieldProperties != null) {
					foreach (AutoGeneratedFieldProperties props in autoFieldProperties)
						fields.Add (CreateAutoGeneratedColumn (props));
				}
			}
			
			fields.AddRange (Columns);
			
			return fields;
		}
		
		protected virtual AutoGeneratedField CreateAutoGeneratedColumn (AutoGeneratedFieldProperties fieldProperties)
		{
			return new AutoGeneratedField (fieldProperties);
		}
		
		AutoGeneratedFieldProperties[] CreateAutoFieldProperties (PagedDataSource source)
		{
			if(source == null) return null;
			
			PropertyDescriptorCollection props = source.GetItemProperties (new PropertyDescriptor[0]);
			Type prop_type;
			
			ArrayList retVal = new ArrayList();
			
			if (props == null)
			{
				object fitem = null;
				prop_type = null;
				PropertyInfo prop_item =  source.DataSource.GetType().GetProperty("Item",
						  BindingFlags.Instance | BindingFlags.Static |
						  BindingFlags.Public, null, null,
						  new Type[] { typeof(int) }, null);
				
				if (prop_item != null) {
					prop_type = prop_item.PropertyType;
				}
				
				if (prop_type == null || prop_type == typeof(object)) {
					IEnumerator en = source.GetEnumerator();
					if (en.MoveNext())
						fitem = en.Current;
					if (fitem != null)
						prop_type = fitem.GetType();
				}
				
				if (fitem != null && fitem is ICustomTypeDescriptor) {
					props = TypeDescriptor.GetProperties(fitem);
				} else if (prop_type != null) {
					if (IsBindableType (prop_type)) {
						AutoGeneratedFieldProperties field = new AutoGeneratedFieldProperties ();
						((IStateManager)field).TrackViewState();
						field.Name = "Item";
						field.DataField = BoundField.ThisExpression;
						field.Type = prop_type;
						retVal.Add (field);
					} else {
						props = TypeDescriptor.GetProperties (prop_type);
					}
				}
			}
			
			if (props != null && props.Count > 0)
			{
				foreach (PropertyDescriptor current in props) {
					if (IsBindableType (current.PropertyType)) {
						AutoGeneratedFieldProperties field = new AutoGeneratedFieldProperties ();
						((IStateManager)field).TrackViewState();
						field.Name = current.Name;
						field.DataField = current.Name;
						field.IsReadOnly = current.IsReadOnly;
						field.Type = current.PropertyType;
						retVal.Add (field);
					}
				}
			}

			if (retVal.Count > 0)
				return (AutoGeneratedFieldProperties[]) retVal.ToArray (typeof(AutoGeneratedFieldProperties));
			else
				return new AutoGeneratedFieldProperties [0];
		}
		
		protected virtual GridViewRow CreateRow (int rowIndex, int dataSourceIndex, DataControlRowType rowType, DataControlRowState rowState)
		{
			GridViewRow row = new GridViewRow (rowIndex, dataSourceIndex, rowType, rowState);
			OnRowCreated (new GridViewRowEventArgs (row));
			return row;
		}
		
		void RequireBinding ()
		{
			if (Initialized) {
				RequiresDataBinding = true;
				pageCount = -1;
			}
		}
		
		protected virtual Table CreateChildTable ()
		{
			Table table = new Table ();
			table.Caption = Caption;
			table.CaptionAlign = CaptionAlign;
			table.CellPadding = CellPadding;
			table.CellSpacing = CellSpacing;
			table.HorizontalAlign = HorizontalAlign;
			table.BackImageUrl = BackImageUrl;
			return table;
		}
	
		protected override int CreateChildControls (IEnumerable data, bool dataBinding)
		{
			PagedDataSource dataSource;

			if (dataBinding) {
				DataSourceView view = GetData ();
				dataSource = new PagedDataSource ();
				dataSource.DataSource = data;
				
				if (AllowPaging) {
					dataSource.AllowPaging = true;
					dataSource.PageSize = PageSize;
					dataSource.CurrentPageIndex = PageIndex;
					if (view.CanPage) {
						dataSource.AllowServerPaging = true;
						if (view.CanRetrieveTotalRowCount)
							dataSource.VirtualCount = SelectArguments.TotalRowCount;
						else {
							dataSource.DataSourceView = view;
							dataSource.DataSourceSelectArguments = SelectArguments;
							dataSource.SetItemCountFromPageIndex (PageIndex + PagerSettings.PageButtonCount);
						}
					}
				}
				
				pageCount = dataSource.PageCount;
			}
			else
			{
				dataSource = new PagedDataSource ();
				dataSource.DataSource = data;
				if (AllowPaging) {
					dataSource.AllowPaging = true;
					dataSource.PageSize = PageSize;
					dataSource.CurrentPageIndex = PageIndex;
				}
			}

			bool showPager = AllowPaging && (PageCount > 1);
			
			Controls.Clear ();
			table = CreateChildTable ();
			Controls.Add (table);
				
			ArrayList list = new ArrayList ();
			ArrayList keyList = new ArrayList ();
			
			// Creates the set of fields to show
			
			ICollection fieldCollection = CreateColumns (dataSource, dataBinding);
			DataControlField[] fields = new DataControlField [fieldCollection.Count];
			fieldCollection.CopyTo (fields, 0);

			foreach (DataControlField field in fields) {
				field.Initialize (AllowSorting, this);
				if (EnableSortingAndPagingCallbacks)
					field.ValidateSupportsCallback ();
			}

			// Main table creation
			
			if (showPager && PagerSettings.Position == PagerPosition.Top || PagerSettings.Position == PagerPosition.TopAndBottom) {
				topPagerRow = CreatePagerRow (fields.Length, dataSource);
				table.Rows.Add (topPagerRow);
			}

			if (ShowHeader) {
				headerRow = CreateRow (0, 0, DataControlRowType.Header, DataControlRowState.Normal);
				table.Rows.Add (headerRow);
				InitializeRow (headerRow, fields);
			}
			
			foreach (object obj in dataSource) {
				DataControlRowState rstate = GetRowState (list.Count);
				GridViewRow row = CreateRow (list.Count, list.Count, DataControlRowType.DataRow, rstate);
				row.DataItem = obj;
				list.Add (row);
				table.Rows.Add (row);
				InitializeRow (row, fields);
				if (dataBinding) {
//					row.DataBind ();
					OnRowDataBound (new GridViewRowEventArgs (row));
					if (EditIndex == row.RowIndex)
						oldEditValues = new DataKey (GetRowValues (row, false, true));
					keyList.Add (new DataKey (CreateRowDataKey (row), DataKeyNames));
				} else {
					if (EditIndex == row.RowIndex)
						oldEditValues = new DataKey (new OrderedDictionary ());
					keyList.Add (new DataKey (new OrderedDictionary (), DataKeyNames));
				}

				if (list.Count >= PageSize)
					break;
			}
			
			if (list.Count == 0)
				table.Rows.Add (CreateEmptyrRow (fields.Length));

			if (ShowFooter) {
				footerRow = CreateRow (0, 0, DataControlRowType.Footer, DataControlRowState.Normal);
				table.Rows.Add (footerRow);
				InitializeRow (footerRow, fields);
			}

			if (showPager && PagerSettings.Position == PagerPosition.Bottom || PagerSettings.Position == PagerPosition.TopAndBottom) {
				bottomPagerRow = CreatePagerRow (fields.Length, dataSource);
				table.Rows.Add (bottomPagerRow);
			}

			rows = new GridViewRowCollection (list);
			keys = new DataKeyArray (keyList);
			
			if (dataBinding)
				DataBind (false);

			return dataSource.DataSourceCount;
		}

		[MonoTODO]
		protected override Style CreateControlStyle ()
		{
			return base.CreateControlStyle ();
		}
		
		DataControlRowState GetRowState (int index)
		{
			DataControlRowState rstate = (index % 2) == 0 ? DataControlRowState.Normal : DataControlRowState.Alternate;
			if (index == SelectedIndex) rstate |= DataControlRowState.Selected;
			if (index == EditIndex) rstate |= DataControlRowState.Edit;
			return rstate;
		}
		
		GridViewRow CreatePagerRow (int fieldCount, PagedDataSource dataSource)
		{
			GridViewRow row = CreateRow (-1, -1, DataControlRowType.Pager, DataControlRowState.Normal);
			InitializePager (row, fieldCount, dataSource);
			return row;
		}
		
		protected virtual void InitializePager (GridViewRow row, int columnSpan, PagedDataSource dataSource)
		{
			TableCell cell = new TableCell ();
			cell.ColumnSpan = columnSpan;
			
			if (pagerTemplate != null)
				pagerTemplate.InstantiateIn (cell);
			else
				cell.Controls.Add (PagerSettings.CreatePagerControl (dataSource.CurrentPageIndex, dataSource.PageCount));
			
			row.Cells.Add (cell);
		}
		
		GridViewRow CreateEmptyrRow (int fieldCount)
		{
			GridViewRow row = CreateRow (-1, -1, DataControlRowType.EmptyDataRow, DataControlRowState.Normal);
			TableCell cell = new TableCell ();
			cell.ColumnSpan = fieldCount;
			
			if (emptyDataTemplate != null)
				emptyDataTemplate.InstantiateIn (cell);
			else
				cell.Text = EmptyDataText;
			
			row.Cells.Add (cell);
			return row;
		}
		
		protected virtual void InitializeRow (GridViewRow row, DataControlField[] fields)
		{
			DataControlCellType ctype;
			bool accessibleHeader = false;

			switch (row.RowType) {
				case DataControlRowType.Header:
					ctype = DataControlCellType.Header; 
					accessibleHeader = UseAccessibleHeader;
					break;
				case DataControlRowType.Footer:
					ctype = DataControlCellType.Footer;
					break;
				default:
					ctype = DataControlCellType.DataCell;
					break;
			}
			
			for (int n=0; n<fields.Length; n++) {
				DataControlField field = fields [n];
				DataControlFieldCell cell;
				if (((field is BoundField) && ((BoundField)field).DataField == RowHeaderColumn) || accessibleHeader)
					cell = new DataControlFieldHeaderCell (field, accessibleHeader ? TableHeaderScope.Column : TableHeaderScope.Row);
				else
					cell = new DataControlFieldCell (field);
				row.Cells.Add (cell);
				field.InitializeCell (cell, ctype, row.RowState, row.RowIndex);
			}
		}
		
		IOrderedDictionary CreateRowDataKey (GridViewRow row)
		{
			if (cachedKeyProperties == null) {
				PropertyDescriptorCollection props = TypeDescriptor.GetProperties (row.DataItem);
				cachedKeyProperties = new PropertyDescriptor [DataKeyNames.Length];
				for (int n=0; n<DataKeyNames.Length; n++) { 
					PropertyDescriptor p = props [DataKeyNames[n]];
					if (p == null)
						new InvalidOperationException ("Property '" + DataKeyNames[n] + "' not found in object of type " + row.DataItem.GetType());
					cachedKeyProperties [n] = p;
				}
			}
			
			OrderedDictionary dic = new OrderedDictionary ();
			foreach (PropertyDescriptor p in cachedKeyProperties)
				dic [p.Name] = p.GetValue (row.DataItem);
			return dic;
		}
		
		IOrderedDictionary GetRowValues (GridViewRow row, bool includeReadOnlyFields, bool includePrimaryKey)
		{
			OrderedDictionary dic = new OrderedDictionary ();
			ExtractRowValues (dic, row, includeReadOnlyFields, includePrimaryKey);
			return dic;
		}
		
		protected virtual void ExtractRowValues (IOrderedDictionary fieldValues, GridViewRow row, bool includeReadOnlyFields, bool includePrimaryKey)
		{
			foreach (TableCell cell in row.Cells) {
				DataControlFieldCell c = cell as DataControlFieldCell;
				if (c != null)
					c.ContainingField.ExtractValuesFromCell (fieldValues, c, row.RowState, includeReadOnlyFields);
			}
			if (!includePrimaryKey && DataKeyNames != null)
				foreach (string key in DataKeyNames)
					fieldValues.Remove (key);
		}
		
		protected override HtmlTextWriterTag TagKey {
			get {
				if (EnableSortingAndPagingCallbacks)
					return HtmlTextWriterTag.Div;
				else
					return HtmlTextWriterTag.Table;
			}
		}
		
		public sealed override void DataBind ()
		{
			DataSourceView view = GetData ();
			if (AllowPaging && view.CanPage) {
				SelectArguments.StartRowIndex = PageIndex * PageSize;
				SelectArguments.MaximumRows = PageSize;
				if (view.CanRetrieveTotalRowCount)
					SelectArguments.RetrieveTotalRowCount = true;
			}

			if (sortExpression != "") {
				if (sortDirection == SortDirection.Ascending)
					SelectArguments.SortExpression = sortExpression;
				else
					SelectArguments.SortExpression = sortExpression + " DESC";
			}
			
			cachedKeyProperties = null;
			base.DataBind ();
		}
		
		protected internal override void PerformDataBinding (IEnumerable data)
		{
			base.PerformDataBinding (data);
		}

		[MonoTODO]
		protected internal virtual void PrepareControlHierarchy ()
		{
			throw new NotImplementedException ();
		}
		
		protected internal override void OnInit (EventArgs e)
		{
			Page.RegisterRequiresControlState (this);
			base.OnInit (e);
		}
		
		void OnFieldsChanged (object sender, EventArgs args)
		{
			RequireBinding ();
		}
		
		protected override void OnDataPropertyChanged ()
		{
			base.OnDataPropertyChanged ();
			RequireBinding ();
		}
		
		protected override void OnDataSourceViewChanged (object sender, EventArgs e)
		{
			base.OnDataSourceViewChanged (sender, e);
			RequireBinding ();
		}
		
		protected override bool OnBubbleEvent (object source, EventArgs e)
		{
			GridViewCommandEventArgs args = e as GridViewCommandEventArgs;
			if (args != null) {
				OnRowCommand (args);
				ProcessEvent (args.CommandName, args.CommandArgument as string);
			}
			return base.OnBubbleEvent (source, e);
		}
		
		// This is prolly obsolete
		protected virtual void RaisePostBackEvent (string eventArgument)
		{
			int i = eventArgument.IndexOf ('$');
			if (i != -1)
				ProcessEvent (eventArgument.Substring (0, i), eventArgument.Substring (i + 1));
			else
				ProcessEvent (eventArgument, null);
		}
		
		void ProcessEvent (string eventName, string param)
		{
			switch (eventName)
			{
				case DataControlCommands.PageCommandName:
					int newIndex = -1;
					switch (param) {
						case DataControlCommands.FirstPageCommandArgument:
							newIndex = 0;
							break;
						case DataControlCommands.LastPageCommandArgument:
							newIndex = PageCount - 1;
							break;
						case DataControlCommands.NextPageCommandArgument:
							if (PageIndex < PageCount - 1) newIndex = PageIndex + 1;
							break;
						case DataControlCommands.PreviousPageCommandArgument:
							if (PageIndex > 0) newIndex = PageIndex - 1;
							break;
						default:
							newIndex = int.Parse (param) - 1;
							break;
					}
					ShowPage (newIndex);
					break;
					
				case DataControlCommands.FirstPageCommandArgument:
					ShowPage (0);
					break;

				case DataControlCommands.LastPageCommandArgument:
					ShowPage (PageCount - 1);
					break;
					
				case DataControlCommands.NextPageCommandArgument:
					if (PageIndex < PageCount - 1)
						ShowPage (PageIndex + 1);
					break;

				case DataControlCommands.PreviousPageCommandArgument:
					if (PageIndex > 0)
						ShowPage (PageIndex - 1);
					break;
					
				case DataControlCommands.SelectCommandName:
					SelectRow (int.Parse (param));
					break;
					
				case DataControlCommands.EditCommandName:
					EditRow (int.Parse (param));
					break;
					
				case DataControlCommands.UpdateCommandName:
					UpdateRow (EditIndex, true);
					break;
					
				case DataControlCommands.CancelCommandName:
					CancelEdit ();
					break;
					
				case DataControlCommands.DeleteCommandName:
					DeleteRow (int.Parse (param));
					break;
					
				case DataControlCommands.SortCommandName:
					Sort (param);
					break;
			}
		}
		
		void Sort (string newSortExpression)
		{
			SortDirection newDirection;
			if (sortExpression == newSortExpression) {
				if (sortDirection == SortDirection.Ascending)
					newDirection = SortDirection.Descending;
				else
					newDirection = SortDirection.Ascending;
			} else
				newDirection = sortDirection;
			
			Sort (newSortExpression, newDirection);
		}
		
		public void Sort (string newSortExpression, SortDirection newSortDirection)
		{
			GridViewSortEventArgs args = new GridViewSortEventArgs (newSortExpression, newSortDirection);
			OnSorting (args);
			if (args.Cancel) return;
			
			sortExpression = args.SortExpression;
			sortDirection = args.SortDirection;
			RequireBinding ();
			
			OnSorted (EventArgs.Empty);
		}
		
		void SelectRow (int index)
		{
			GridViewSelectEventArgs args = new GridViewSelectEventArgs (index);
			OnSelectedIndexChanging (args);
			if (!args.Cancel) {
				SelectedIndex = args.NewSelectedIndex;
				OnSelectedIndexChanged (EventArgs.Empty);
			}
		}
		
		void ShowPage (int newIndex)
		{
			GridViewPageEventArgs args = new GridViewPageEventArgs (newIndex);
			OnPageIndexChanging (args);
			if (!args.Cancel) {
				EndRowEdit ();
				PageIndex = args.NewPageIndex;
				OnPageIndexChanged (EventArgs.Empty);
			}
		}
		
		void EditRow (int index)
		{
			GridViewEditEventArgs args = new GridViewEditEventArgs (index);
			OnRowEditing (args);
			if (!args.Cancel) {
				EditIndex = args.NewEditIndex;
			}
		}
		
		void CancelEdit ()
		{
			GridViewCancelEditEventArgs args = new GridViewCancelEditEventArgs (EditIndex);
			OnRowCancelingEdit (args);
			if (!args.Cancel) {
				EndRowEdit ();
			}
		}

		[MonoTODO ("Support two-way binding expressions")]
		public virtual void UpdateRow (int rowIndex, bool causesValidation)
		{
			if (causesValidation)
				Page.Validate ();
			
			if (rowIndex != EditIndex) throw new NotSupportedException ();
			
			currentEditOldValues = oldEditValues.Values;

			GridViewRow row = Rows [rowIndex];
			currentEditRowKeys = DataKeys [rowIndex].Values;
			currentEditNewValues = GetRowValues (row, false, false);
			
			GridViewUpdateEventArgs args = new GridViewUpdateEventArgs (EditIndex, currentEditRowKeys, currentEditOldValues, currentEditNewValues);
			OnRowUpdating (args);
			if (!args.Cancel) {
				DataSourceView view = GetData ();
				if (view == null) throw new HttpException ("The DataSourceView associated to data bound control was null");
				view.Update (currentEditRowKeys, currentEditNewValues, currentEditOldValues, new DataSourceViewOperationCallback (UpdateCallback));
			} else
				EndRowEdit ();
		}

        bool UpdateCallback (int recordsAffected, Exception exception)
		{
			GridViewUpdatedEventArgs dargs = new GridViewUpdatedEventArgs (recordsAffected, exception, currentEditRowKeys, currentEditOldValues, currentEditNewValues);
			OnRowUpdated (dargs);

			if (!dargs.KeepInEditMode)				
				EndRowEdit ();

			return dargs.ExceptionHandled;
		}
		
		public void DeleteRow (int rowIndex)
		{
			GridViewRow row = Rows [rowIndex];
			currentEditRowKeys = DataKeys [rowIndex].Values;
			currentEditNewValues = GetRowValues (row, true, true);
			
			GridViewDeleteEventArgs args = new GridViewDeleteEventArgs (rowIndex, currentEditRowKeys, currentEditNewValues);
			OnRowDeleting (args);

			if (!args.Cancel) {
				RequireBinding ();
				DataSourceView view = GetData ();
				if (view != null)
					view.Delete (currentEditRowKeys, currentEditNewValues, new DataSourceViewOperationCallback (DeleteCallback));
				else {
					GridViewDeletedEventArgs dargs = new GridViewDeletedEventArgs (0, null, currentEditRowKeys, currentEditNewValues);
					OnRowDeleted (dargs);
				}
			}
		}

        bool DeleteCallback (int recordsAffected, Exception exception)
		{
			GridViewDeletedEventArgs dargs = new GridViewDeletedEventArgs (recordsAffected, exception, currentEditRowKeys, currentEditNewValues);
			OnRowDeleted (dargs);
			return dargs.ExceptionHandled;
		}
		
		void EndRowEdit ()
		{
			EditIndex = -1;
			oldEditValues = new DataKey (new OrderedDictionary ());
			currentEditRowKeys = null;
			currentEditOldValues = null;
			currentEditNewValues = null;
		}

		protected internal override void LoadControlState (object ob)
		{
			if (ob == null) return;
			object[] state = (object[]) ob;
			base.LoadControlState (state[0]);
			pageIndex = (int) state[1];
			pageCount = (int) state[2];
			selectedIndex = (int) state[3];
			editIndex = (int) state[4];
			sortExpression = (string) state[5];
			sortDirection = (SortDirection) state[6];
		}
		
		protected internal override object SaveControlState ()
		{
			object bstate = base.SaveControlState ();
			return new object[] {
				bstate, pageIndex, pageCount, selectedIndex, editIndex, sortExpression, sortDirection
			};
		}
		
		protected override void TrackViewState()
		{
			base.TrackViewState();
			if (columns != null) ((IStateManager)columns).TrackViewState();
			if (pagerSettings != null) ((IStateManager)pagerSettings).TrackViewState();
			if (alternatingRowStyle != null) ((IStateManager)alternatingRowStyle).TrackViewState();
			if (footerStyle != null) ((IStateManager)footerStyle).TrackViewState();
			if (headerStyle != null) ((IStateManager)headerStyle).TrackViewState();
			if (pagerStyle != null) ((IStateManager)pagerStyle).TrackViewState();
			if (rowStyle != null) ((IStateManager)rowStyle).TrackViewState();
			if (selectedRowStyle != null) ((IStateManager)selectedRowStyle).TrackViewState();
			if (editRowStyle != null) ((IStateManager)editRowStyle).TrackViewState();
			if (emptyDataRowStyle != null) ((IStateManager)emptyDataRowStyle).TrackViewState();
			if (keys != null) ((IStateManager)keys).TrackViewState();
			if (autoFieldProperties != null) {
				foreach (IStateManager sm in autoFieldProperties)
					sm.TrackViewState ();
			}
		}

		protected override object SaveViewState()
		{
			object[] states = new object [14];
			states[0] = base.SaveViewState();
			states[1] = (columns == null ? null : ((IStateManager)columns).SaveViewState());
			states[2] = (pagerSettings == null ? null : ((IStateManager)pagerSettings).SaveViewState());
			states[3] = (alternatingRowStyle == null ? null : ((IStateManager)alternatingRowStyle).SaveViewState());
			states[4] = (footerStyle == null ? null : ((IStateManager)footerStyle).SaveViewState());
			states[5] = (headerStyle == null ? null : ((IStateManager)headerStyle).SaveViewState());
			states[6] = (pagerStyle == null ? null : ((IStateManager)pagerStyle).SaveViewState());
			states[7] = (rowStyle == null ? null : ((IStateManager)rowStyle).SaveViewState());
			states[8] = (selectedRowStyle == null ? null : ((IStateManager)selectedRowStyle).SaveViewState());
			states[9] = (editRowStyle == null ? null : ((IStateManager)editRowStyle).SaveViewState());
			states[10] = (emptyDataRowStyle == null ? null : ((IStateManager)emptyDataRowStyle).SaveViewState());
			states[11] = (keys == null ? null : ((IStateManager)keys).SaveViewState());
			states[12] = (oldEditValues == null ? null : ((IStateManager)oldEditValues).SaveViewState());
			
			if (autoFieldProperties != null) {
				object[] data = new object [autoFieldProperties.Length];
				bool allNull = true;
				for (int n=0; n<data.Length; n++) {
					data [n] = ((IStateManager)autoFieldProperties [n]).SaveViewState ();
					if (data [n] != null) allNull = false;
				}
				if (!allNull) states [13] = data;
			}

			for (int i = states.Length - 1; i >= 0; i--) {
				if (states [i] != null)
					return states;
			}

			return null;
		}

		protected override void LoadViewState (object savedState)
		{
			if (savedState == null) {
				base.LoadViewState (null);
				return;
			}

			object [] states = (object []) savedState;
			
			if (states[13] != null) {
				object[] data = (object[]) states [13];
				autoFieldProperties = new AutoGeneratedFieldProperties [data.Length];
				for (int n=0; n<data.Length; n++) {
					IStateManager p = new AutoGeneratedFieldProperties ();
					p.TrackViewState ();
					p.LoadViewState (data [n]);
					autoFieldProperties [n] = (AutoGeneratedFieldProperties) p;
				}
			}

			base.LoadViewState (states[0]);
			EnsureChildControls ();
			
			if (states[1] != null) ((IStateManager)Columns).LoadViewState (states[1]);
			if (states[2] != null) ((IStateManager)PagerSettings).LoadViewState (states[2]);
			if (states[3] != null) ((IStateManager)AlternatingRowStyle).LoadViewState (states[3]);
			if (states[4] != null) ((IStateManager)FooterStyle).LoadViewState (states[4]);
			if (states[5] != null) ((IStateManager)HeaderStyle).LoadViewState (states[5]);
			if (states[6] != null) ((IStateManager)PagerStyle).LoadViewState (states[6]);
			if (states[7] != null) ((IStateManager)RowStyle).LoadViewState (states[7]);
			if (states[8] != null) ((IStateManager)SelectedRowStyle).LoadViewState (states[8]);
			if (states[9] != null) ((IStateManager)EditRowStyle).LoadViewState (states[9]);
			if (states[10] != null) ((IStateManager)EmptyDataRowStyle).LoadViewState (states[10]);
			if (states[11] != null) ((IStateManager)DataKeys).LoadViewState (states[11]);
			if (states[12] != null && oldEditValues != null) ((IStateManager)oldEditValues).LoadViewState (states[12]);
		}
		
		string ICallbackEventHandler.RaiseCallbackEvent (string eventArgs)
		{
			return RaiseCallbackEvent (eventArgs);
		}
		
		protected virtual string RaiseCallbackEvent (string eventArgs)
		{
			string[] clientData = eventArgs.Split ('|');
			pageIndex = int.Parse (clientData[0]);
			sortExpression = HttpUtility.UrlDecode (clientData[1]);
			if (sortExpression == "") sortExpression = null;
			RequireBinding ();
			
			RaisePostBackEvent (clientData[2]);
			EnsureDataBound ();
			
			StringWriter sw = new StringWriter ();
			sw.Write (PageIndex.ToString() + '|' + SortExpression + '|');

			HtmlTextWriter writer = new HtmlTextWriter (sw);
			RenderGrid (writer);
			return sw.ToString ();
		}
		
		string ICallbackContainer.GetCallbackScript (IButtonControl control, string argument)
		{
			return GetCallbackScript (control, argument);
		}
		
		protected virtual string GetCallbackScript (IButtonControl control, string argument)
		{
			if (EnableSortingAndPagingCallbacks)
				return "javascript:GridView_ClientEvent (\"" + ClientID + "\",\"" + control.CommandName + "$" + control.CommandArgument + "\"); return false;";
			else
				return null;
		}
		
		[MonoTODO]
		protected override void OnPagePreLoad (object sender, EventArgs e)
		{
			base.OnPagePreLoad (sender, e);
		}
		
		protected internal override void OnPreRender (EventArgs e)
		{
			base.OnPreRender (e);
			
			if (EnableSortingAndPagingCallbacks)
			{
				if (!Page.ClientScript.IsClientScriptIncludeRegistered (typeof(GridView), "GridView.js")) {
					string url = Page.ClientScript.GetWebResourceUrl (typeof(GridView), "GridView.js");
					Page.ClientScript.RegisterClientScriptInclude (typeof(GridView), "GridView.js", url);
				}
				
				string cgrid = ClientID + "_data";
				string script = string.Format ("var {0} = new Object ();\n", cgrid);
				script += string.Format ("{0}.pageIndex = {1};\n", cgrid, ClientScriptManager.GetScriptLiteral (PageIndex));
				script += string.Format ("{0}.sortExp = {1};\n", cgrid, ClientScriptManager.GetScriptLiteral (SortExpression == null ? "" : SortExpression));
				script += string.Format ("{0}.uid = {1};\n", cgrid, ClientScriptManager.GetScriptLiteral (UniqueID));
				Page.ClientScript.RegisterStartupScript (typeof(TreeView), this.UniqueID, script, true);
				
				// Make sure the basic script infrastructure is rendered
		        Page.ClientScript.GetCallbackEventReference (this, "null", "", "null");
				Page.ClientScript.GetPostBackClientHyperlink (this, "");
			}
		}
		
		protected internal override void Render (HtmlTextWriter writer)
		{
			if (EnableSortingAndPagingCallbacks)
				base.RenderBeginTag (writer);

			RenderGrid (writer);
			
			if (EnableSortingAndPagingCallbacks)
				base.RenderEndTag (writer);
		}
		
		void RenderGrid (HtmlTextWriter writer)
		{
			switch (GridLines) {
				case GridLines.Horizontal:
					writer.AddAttribute (HtmlTextWriterAttribute.Rules, "rows");
					writer.AddAttribute (HtmlTextWriterAttribute.Border, "1");
					break;
				case GridLines.Vertical:
					writer.AddAttribute (HtmlTextWriterAttribute.Rules, "cols");
					writer.AddAttribute (HtmlTextWriterAttribute.Border, "1");
					break;
				case GridLines.Both:
					writer.AddAttribute (HtmlTextWriterAttribute.Rules, "all");
					writer.AddAttribute (HtmlTextWriterAttribute.Border, "1");
					break;
				default:
					writer.AddAttribute (HtmlTextWriterAttribute.Border, "0");
					break;
			}
			
			writer.AddAttribute (HtmlTextWriterAttribute.Cellspacing, "0");
			writer.AddStyleAttribute (HtmlTextWriterStyle.BorderCollapse, "collapse");
			table.RenderBeginTag (writer);
			
			foreach (GridViewRow row in table.Rows)
			{
				switch (row.RowType) {
					case DataControlRowType.Header:
						if (headerStyle != null)headerStyle.AddAttributesToRender (writer, row);
						break;
					case DataControlRowType.Footer:
						if (footerStyle != null) footerStyle.AddAttributesToRender (writer, row);
						break;
					case DataControlRowType.Pager:
						if (pagerStyle != null) pagerStyle.AddAttributesToRender (writer, row);
						break;
					case DataControlRowType.EmptyDataRow:
						if (emptyDataRowStyle != null) emptyDataRowStyle.AddAttributesToRender (writer, row);
						break;
					default:
						break;
				}

				if ((row.RowState & DataControlRowState.Normal) != 0 && rowStyle != null)
					rowStyle.AddAttributesToRender (writer, row);
				if ((row.RowState & DataControlRowState.Alternate) != 0 && alternatingRowStyle != null)
					alternatingRowStyle.AddAttributesToRender (writer, row);
				if ((row.RowState & DataControlRowState.Edit) != 0 && editRowStyle != null)
					editRowStyle.AddAttributesToRender (writer, row);
				if ((row.RowState & DataControlRowState.Selected) != 0 && selectedRowStyle != null)
					selectedRowStyle.AddAttributesToRender (writer, row);
				
				row.RenderBeginTag (writer);
				
				foreach (TableCell cell in row.Cells) {
					DataControlFieldCell fcell = cell as DataControlFieldCell;
					if (fcell != null) {
						Style cellStyle = null;
						switch (row.RowType) {
							case DataControlRowType.Header: cellStyle = fcell.ContainingField.HeaderStyle; break;
							case DataControlRowType.Footer: cellStyle = fcell.ContainingField.FooterStyle; break;
							default: cellStyle = fcell.ContainingField.ItemStyle; break;
						}
						if (cellStyle != null)
							cellStyle.AddAttributesToRender (writer, cell);
					}
					cell.Render (writer);
				}
				row.RenderEndTag (writer);
			}
			table.RenderEndTag (writer);
		}
	}
}

#endif
