/*
 * Copyright (c) 2004 Jean-François Wauthy (pollux@xfce.org)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <gtk/gtkversion.h>
#if GTK_CHECK_VERSION(2, 4, 0)
#undef GTK_DISABLE_DEPRECATED
#endif

#include <gmodule.h>
#include <cups/ipp.h>
#include <cups/ppd.h>
#include <cups/cups.h>
#include <cups/language.h>
#include <libxfcegui4/libxfcegui4.h>
#include <libxfce4util/libxfce4util.h>
#include <printing-system.h>

static GList *printers = NULL;

/*******************************************/
/* CUPS requests                           */
/* (widely inspired from the libgnomecups) */
/*******************************************/
static ipp_t *
cups_request_new (int operation_id)
{
  ipp_t *request;
  cups_lang_t *language;

  language = cupsLangDefault ();
  request = ippNew ();
  request->request.op.operation_id = operation_id;
  request->request.op.request_id = 1;

  ippAddString (request, IPP_TAG_OPERATION, IPP_TAG_CHARSET, "attributes-charset", NULL, "utf-8");

  ippAddString (request, IPP_TAG_OPERATION, IPP_TAG_LANGUAGE, "attributes-natural-language", NULL, language->language);
  cupsLangFree (language);

  return request;
}

static ipp_t *
cups_request_new_for_printer (int operation_id, const gchar * printer)
{
  ipp_t *request;
  gchar *printer_uri;

  g_return_val_if_fail (printer, NULL);

  printer_uri = g_strdup_printf ("ipp://%s/printers/%s", cupsServer (), printer);
  request = cups_request_new (operation_id);

  ippAddString (request, IPP_TAG_OPERATION, IPP_TAG_URI, "printer-uri", NULL, printer_uri);
  g_free (printer_uri);

  return request;
}

static ipp_t *
cups_request_execute (ipp_t * request, const gchar * path)
{
  http_t *HTTP = NULL;
  ipp_status_t status;

  HTTP = httpConnectEncrypt (cupsServer (), ippPort (), cupsEncryption ());
  if (!HTTP) {
    ippDelete (request);
    g_warning ("Unable to connect CUPS server");
    return NULL;
  }

  request = cupsDoRequest (HTTP, request, path);
  httpClose (HTTP);
  status = cupsLastError ();
  if (!request) {
    g_warning ("CUPS server couldn't execute request");
    return NULL;
  }

  if (status > IPP_OK_CONFLICT)
    g_warning ("failed request with status %d", status);

  return request;
}


/***************/
/* Main window */
/***************/
static GList *
get_printers_list ()
{
  GList *list = NULL;
  int i;
  cups_dest_t *dests;
  int num_dests;

  num_dests = cupsGetDests (&dests);    // Printer names and options

  if (num_dests > 0) {
    for (i = 0; i < num_dests; i++) {
      Printer *printer;
      ipp_t *request;
      ipp_attribute_t *attr;

      printer = g_new0 (Printer, 1);

      if (dests[i].instance)
	printer->name = g_strdup_printf ("%s/%s", dests[i].name, dests[i].instance);
      else
	printer->name = g_strdup (dests[i].name);

      list = g_list_append (list, printer);

      request = cups_request_new_for_printer (IPP_GET_PRINTER_ATTRIBUTES, dests[i].name);
      request = cups_request_execute (request, "/printers/");

      if (!request)
        continue;
      if (request->state == IPP_ERROR || request->state == IPP_IDLE) {
        ippDelete (request);
        continue;
      }

      attr = ippFindAttribute (request, "printer-info", IPP_TAG_TEXT);
      if (!attr || strlen (attr->values[0].string.text) == 0) {
        attr = ippFindAttribute (request, "printer-make-and-model", IPP_TAG_TEXT);
        if (attr)
          printer->alias = g_strdup (attr->values[0].string.text);
        else
          printer->alias = g_strdup ("");
      }
      else
        printer->alias = g_strdup (attr->values[0].string.text);

      attr = ippFindAttribute (request, "printer-type", IPP_TAG_ENUM);
      if (attr && (attr->values[0].integer & CUPS_PRINTER_CLASS))
        printer->type = PRINTER_TYPE_CLASS;
      else
        printer->type = PRINTER_TYPE_PRINTER;


      ippDelete (request);
    }
  } else
    g_warning ("no printer in the list, may be the CUPS server isn't running or you haven't configured any printer");

  cupsFreeDests (num_dests, dests);

  return list;
}

/* Returns the printer list  */
/* returned pointer must be freed */
G_MODULE_EXPORT GList * printing_system_get_printers_list_impl ()
{
  return get_printers_list ();
}

/* Customize main window */
G_MODULE_EXPORT void
printing_system_main_window_customize_impl (ManagerDlg * dlg)
{
  gtk_widget_set_sensitive (dlg->menuitem_properties, FALSE);
}

G_MODULE_EXPORT gboolean printing_system_set_default_printer_impl (const gchar * printer)
{
  /* code from XPP */
  int j;
  int num_dests = 0;
  cups_dest_t *dests = NULL;

  num_dests = cupsGetDests (&dests);
  for (j = 0; j < num_dests; j++)
    if (strcmp (dests[j].name, printer) == 0)
      dests[j].is_default = 1;
    else
      dests[j].is_default = 0;
  cupsSetDests (num_dests, dests);
  cupsFreeDests (num_dests, dests);

  return TRUE;
}

G_MODULE_EXPORT Printer * printing_system_get_default_printer_impl (GList * printers)
{
  /* TODO : find a way to work fine with CupsGetDefault and set_default */
  Printer *printer = NULL;
  int j;
  int num_dests = 0;
  cups_dest_t *dests = NULL;

  num_dests = cupsGetDests (&dests);
  for (j = 0; j < num_dests; j++)
    if (dests[j].is_default == 1)
      printer = printer_lookup_byname (printers, dests[j].name);
  cupsSetDests (num_dests, dests);
  cupsFreeDests (num_dests, dests);

  return printer;
}

/****************/
/* Queue dialog */
/****************/

/* Refresh the job list */
/* returned pointer must be freed */
G_MODULE_EXPORT GList * printing_system_get_jobs_list_impl (const gchar * printer)
{
  int i;
  int num_jobs;
  cups_job_t *jobs;
  GList *list = NULL;

  /* Obtain job list from cups server */
  num_jobs = cupsGetJobs (&jobs, printer, 0, 0);

  for (i = 0; i < num_jobs; i++) {
    Job *job;
    gchar string_creation_time[10] = "";
    gchar string_processing_time[10] = "";
    struct tm *local_time;

    job = g_new0 (Job, 1);

    job->name = g_strdup (jobs[i].title);
    job->id = jobs[i].id;
    job->user = g_strdup (jobs[i].user);
    job->state = g_strdup (jobs[i].state != IPP_JOB_PENDING ? _("printing") : _("pending"));
    job->size = g_strdup_printf ("%dKb", jobs[i].size);
    job->priority = g_strdup_printf ("%d", jobs[i].priority);
    local_time = localtime (&(jobs[i].creation_time));
    strftime (string_creation_time, sizeof (string_creation_time), "%H:%M:%S", local_time);
    job->creation_time = g_strdup (string_creation_time);
    if (jobs[i].state == IPP_JOB_PROCESSING) {
      local_time = localtime (&(jobs[i].processing_time));
      strftime (string_processing_time, sizeof (string_processing_time), "%H:%M:%S", local_time);
      job->processing_time = g_strdup (string_processing_time);
    }
    
    list = g_list_append (list, job);
      
  }

  cupsFreeJobs (num_jobs, jobs);

  return list;
}

/* Remove job from printer_name queue */
G_MODULE_EXPORT gboolean printing_system_remove_job_impl (const gchar * printer, gint id)
{
  if (cupsCancelJob (printer, id) == 0) {
    xfce_err (ippErrorString (cupsLastError ()));
    return FALSE;
  }

  return TRUE;
}

/* Customize the queue dialog */
G_MODULE_EXPORT void
printing_system_queue_dialog_customize_impl (QueueDlg * dialog)
{
}

/*********************/
/* Properties dialog */
/*********************/
/* Show the properties dialog */
G_MODULE_EXPORT void
printing_system_properties_dialog_run_impl (const gchar * printer)
{
}

/****************/
/* Print dialog */
/****************/
static void
printer_name_free (gchar *printer_name, gpointer user_data)
{
  if (printer_name)
    g_free (printer_name);
}

static gboolean
destroy_print_dialog_cb (GtkWidget * widget, GdkEvent * event, gpointer user_data)
{
  g_list_foreach (printers, (GFunc) printer_free, NULL);
  g_list_free (printers);

  return FALSE;
}

/* Customize the print dialog */
G_MODULE_EXPORT void
printing_system_print_dialog_customize_impl (PrintDlg * dialog)
{
  GList *printer_el;
  GList *list = NULL;
  Printer *printer;
  GtkWidget *hbox;
  GtkWidget *label;

  printers = get_printers_list ();

  g_signal_connect (G_OBJECT (dialog->dialog), "destroy_event", G_CALLBACK (destroy_print_dialog_cb), printers);

  printer_el = g_list_first (printers);
  while (printer_el) {
    gchar *list_entry;

    printer = printer_el->data;

    list_entry = g_strdup_printf ("%s (%s)", printer->alias, printer->name);
    list = g_list_append (list, list_entry);

    printer_el = g_list_next (printer_el);
  }

  hbox = gtk_hbox_new (FALSE, 5);

  label = gtk_label_new (_("Print to :"));
  gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, TRUE, 0);
  dialog->customization = (gpointer) gtk_combo_new ();
  gtk_box_pack_start (GTK_BOX (hbox), GTK_WIDGET (dialog->customization), TRUE, TRUE, 0);

  gtk_combo_set_popdown_strings (GTK_COMBO (dialog->customization), list);
  gtk_combo_set_value_in_list (GTK_COMBO (dialog->customization), TRUE, FALSE);

  /* select default printer */
  printer = printing_system_get_default_printer_impl (printers);
  if (printer) {
    gchar *printer_name;

    printer_name = g_strdup_printf ("%s (%s)", printer->alias, printer->name);
    gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (dialog->customization)->entry), printer_name);
    g_free (printer_name);
  }

  if (g_list_length (list) == 0) {
    gtk_widget_set_sensitive (GTK_WIDGET (dialog->customization), FALSE);
    gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (dialog->customization)->entry), _("No printer in the list"));
  }

  gtk_widget_show_all (hbox);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog->dialog)->vbox), hbox, TRUE, TRUE, 0);

  gtk_box_reorder_child (GTK_BOX (GTK_DIALOG (dialog->dialog)->vbox), hbox, 0);

  g_list_foreach (list, (GFunc) printer_name_free, NULL);
  g_list_free (list);
}

/************/
/* Printing */
/************/
G_MODULE_EXPORT gboolean printing_system_print_file_impl (PrintDlg * dialog, const gchar * file)
{
  gchar *base_name = NULL;
  const gchar* printer_selected;
  gchar *printer_name = NULL;
  gchar *printer_instance = NULL;
  int i, j, length;
  int num_dests;
  cups_dest_t *dests;
  cups_dest_t *dest;
  gboolean ret = FALSE;

  base_name = g_path_get_basename (dialog->original_name);

  /* extract printer name and instance name */
  printer_selected = gtk_entry_get_text (GTK_ENTRY (GTK_COMBO (dialog->customization)->entry));
  length = strlen (printer_selected);

  for (i = length; i >= 0 && printer_selected[i] != '('; i--);
  i++; /* remove last '(' */
  for (j = i; j < length && printer_selected[j] != '/' && printer_selected[j] != ')'; j++);
  printer_name = g_strndup (&printer_selected[i], j - i);
  j++; /* remove the '/' */
  if (j < length)
    printer_instance = g_strndup (&printer_selected[j], length - j - 1);

  num_dests = cupsGetDests(&dests);
  dest = cupsGetDest(printer_name, printer_instance, num_dests, dests);

  if (cupsPrintFile (printer_name, file, base_name, dest->num_options, dest->options)) {
    unlink (file);
    ret = TRUE;
  }
  else {
    xfce_err (ippErrorString (cupsLastError ()));
  }

  cupsFreeDests(num_dests, dests);
  g_free (printer_name);
  g_free (printer_instance);
  g_free (base_name);

  return ret;
}

/****************/
/* About dialog */
/****************/
G_MODULE_EXPORT void
printing_system_about_dialog_run_impl (GtkWidget * widget, gpointer data)
{
  XfceAboutInfo *info;
  GtkWidget *dialog;
  GdkPixbuf *icon = NULL;
  ManagerDlg *dlg;

  dlg = (ManagerDlg *) data;

  info = xfce_about_info_new ("xfprint-cups-plugin", VERSION,
                              _("CUPS plugin for Xfprint"),
                              XFCE_COPYRIGHT_TEXT ("2004", "Xfce4 Team"), XFCE_LICENSE_BSD);
  xfce_about_info_set_homepage (info, "http://www.xfce.org/");

  /* Credits */
  xfce_about_info_add_credit (info, "Jean-Francois Wauthy", "pollux@xfce.org", _("Developer"));
  xfce_about_info_add_credit (info, "Francois Le Clainche", "fleclainche@wanadoo.fr", _("Icon designer"));

  //  icon = xfce_themed_icon_load ("xfprint-cups-plugin" , 48);

  dialog = xfce_about_dialog_new (GTK_WINDOW (dlg->window), info, icon);

  gtk_window_set_default_size (GTK_WINDOW (dialog), 500, 400);
  xfce_about_info_free (info);

  gtk_dialog_run (GTK_DIALOG (dialog));

  gtk_widget_destroy (dialog);
}

/**********/
/* Others */
/**********/

/* Module informations initialization function */
G_MODULE_EXPORT void
printing_system_info_init_impl (PrintingSystemInfo * psinfo)
{
  psinfo->name = g_strdup ("CUPS");
  psinfo->description = g_strdup ("CUPS printing system support for Xfprint");
  psinfo->version = g_strdup (VERSION);
  psinfo->author = g_strdup ("Jean-Francois Wauthy");
  psinfo->homepage = g_strdup ("http://www.xfce.org");
}
