/*
 * snes9express
 * misc.cc
 * Copyright (C) 1998-2002, David Nordlund
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * For further details, please read the included COPYING file,
 * or go to http://www.gnu.org/copyleft/gpl.html
 */
#include <iostream>
#include <fstream>
#include <cstdio>
#include <string>
#include <algorithm>
#include <unistd.h>
#include <sys/stat.h>
#include <errno.h>
#include "defines.h"
#include "frend.h"
#include "misc.h"


s9x_OptionManager::s9x_OptionManager()
{
}

s9x_OptionManager::~s9x_OptionManager()
{
  options.clear();
}

void s9x_OptionManager::addOption(fr_Option& o)
{
    options.push_back(&o);
}

void s9x_OptionManager::SetToDefaults()
{
  std::list<fr_Option*>::iterator i = options.begin(), e=options.end();
  for(; i!=e; i++)
    {
      fr_Option* o = *i;
      o->SetToDefault();
    }
}

void s9x_OptionManager::SiftArgs(fr_ArgList& L)
{
  std::list<fr_Option*>::iterator i = options.begin(), e=options.end();
  for(; i!=e; i++)
    {
      fr_Option* o = *i;
      o->SiftArgs(L);
    }
}

void s9x_OptionManager::CompileArgs(fr_ArgList& L)
{
  std::list<fr_Option*>::iterator i = options.begin(), e=options.end();
  for(; i!=e; i++)
    {
      fr_Option* o = *i;
      o->CompileArgs(L);
    }
}


s9x_Notepage::s9x_Notepage(fr_Notebook*parent, const char*name):
fr_Notepage(parent, name)
{
	//SetPadding(9, 9);
}

void s9x_Notepage::ApplySkin(s9x_Skin*S)
{
  std::string iconID = getTabIconID();
  s9x_SkinSection* section = S?S->getSection("icons"):NULL;
  fr_Image* icon = section?section->getImage(iconID):NULL;
  setIcon(icon);
  if(icon)
    setLabelVisibility(isCurrentPage());
  else
    setLabelVisibility(true);
}

std::string s9x_Notepage::getTabIconID() const
{
  std::string iconID(Name);
  std::string::iterator i = iconID.begin(), e = iconID.end();
  std::transform(i, e, i, tolower);
  return iconID;
}

void s9x_Notepage::pageChanged(bool to_me)
{
  if(hasIcon())  
    setLabelVisibility(to_me);
}


s9x_File::s9x_File(const std::string& filename)
{
  filepath = fr_HomeDir() + '/' + S9X_SUBDIR + '/' + filename;
  filepath_cstr = filepath.c_str();
  stat();
}

s9x_File::~s9x_File()
{
}

int s9x_File::stat(struct stat *buf)
{
  int r = ::stat(filepath_cstr, &statbuf);
  existant = (r==0);
  if((buf)&&(buf != &statbuf))
    memcpy(buf, &statbuf, sizeof(statbuf));
  return r;
}

/**
 * Open for reading.
 * @param i  the ifstream to open
 */
void s9x_File::open(std::ifstream& i)
{
  i.open(filepath_cstr, std::ifstream::in);
  if(!i.is_open())
    throw "Could not read file:\n" + filepath + "\n" + fr_syserr();
}

/**
 * Open for writing.  Create if does not exist. Create directories along the way.
 * @param o  the ofstream to open
 */
void s9x_File::open(std::ofstream& o)
{
  createS9xDir();
  o.open(filepath_cstr, std::ofstream::out);
  if(!o.is_open())
    throw "Could not open file for writing:\n" + filepath + "\n" + fr_syserr();
}

void s9x_File::close(std::ifstream& f)
{
  f.close();
  stat();
}

void s9x_File::close(std::ofstream& f)
{
  f.close();
  stat();
}

void s9x_File::remove()
{
  unlink(filepath_cstr);
  removeS9xDir();
}

std::string operator+(const std::string& s, const s9x_File& f)
{
  return s + f.filepath;
}

std::string operator+(const s9x_File& f, const std::string& s)
{
  return f.filepath + s;
}

std::string operator+(const char* s, const s9x_File& f)
{
  return s + f.filepath;
}

/**
 * create the S9X_SUBDIR if it does not yet exist.
 */
void createS9xDir()
{
  std::string homedir = fr_HomeDir();
  std::string s9xdir = homedir + '/' + S9X_SUBDIR;

  if(fr_DirExists(s9xdir))
    return;
  
  if(!fr_DirExists(homedir))
    throw "Could not create directory:\n" + s9xdir + "\n$HOME does not exist.";

  mode_t m = S_IRUSR|S_IWUSR|S_IXUSR|S_IRGRP|S_IXGRP|S_IROTH|S_IXOTH;
  if(mkdir(s9xdir.c_str(), m)!=0)
    throw "Could not create directory:\n" + s9xdir + "\n\n" + fr_syserr();
}

/**
 * remove the S9X_SUBDIR if it contains no files
 */
void removeS9xDir()
{
  std::string homedir = fr_HomeDir();
  std::string s9xdir = homedir + '/' + S9X_SUBDIR;

  if(fr_DirExists(s9xdir))
    rmdir(s9xdir.c_str());
}

