// -*- verilog -*-
//
//  USRP - Universal Software Radio Peripheral
//
//  Copyright (C) 2003,2004 Matt Ettus
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//


`include "../../firmware/include/fpga_regs_common.v"
`include "../../firmware/include/fpga_regs_standard.v"

// Bidirectional registers

module bidir_reg
  ( inout wire [15:0] tristate,
    input wire [15:0] oe,
    input wire [15:0] reg_val );

   // This would be much cleaner if all the tools
   // supported "for generate"........

   assign 	 tristate[0] = oe[0] ? reg_val[0] : 1'bz;
   assign 	 tristate[1] = oe[1] ? reg_val[1] : 1'bz;
   assign 	 tristate[2] = oe[2] ? reg_val[2] : 1'bz;
   assign 	 tristate[3] = oe[3] ? reg_val[3] : 1'bz;
   assign 	 tristate[4] = oe[4] ? reg_val[4] : 1'bz;
   assign 	 tristate[5] = oe[5] ? reg_val[5] : 1'bz;
   assign 	 tristate[6] = oe[6] ? reg_val[6] : 1'bz;
   assign 	 tristate[7] = oe[7] ? reg_val[7] : 1'bz;
   assign 	 tristate[8] = oe[8] ? reg_val[8] : 1'bz;
   assign 	 tristate[9] = oe[9] ? reg_val[9] : 1'bz;
   assign 	 tristate[10] = oe[10] ? reg_val[10] : 1'bz;
   assign 	 tristate[11] = oe[11] ? reg_val[11] : 1'bz;
   assign 	 tristate[12] = oe[12] ? reg_val[12] : 1'bz;
   assign 	 tristate[13] = oe[13] ? reg_val[13] : 1'bz;
   assign 	 tristate[14] = oe[14] ? reg_val[14] : 1'bz;
   assign 	 tristate[15] = oe[15] ? reg_val[15] : 1'bz;
   
endmodule // bidir_reg


// Serial Control Bus from Cypress chip

module serial_io
  ( input serial_clock,
    input serial_data_in,
    input enable,
    input reset,
    inout wire serial_data_out,
    output reg [31:0] ch0rx_freq, output reg [31:0] ch1rx_freq, output reg [31:0] ch2rx_freq, output reg [31:0] ch3rx_freq,
    // output reg [31:0] ch0tx_freq, output reg [31:0] ch1tx_freq, output reg [31:0] ch2tx_freq, output reg [31:0] ch3tx_freq,
    output reg [7:0] interp_rate,
    output reg [7:0] decim_rate,
    output reg [7:0] tx_sample_rate,
    output reg [7:0] rx_sample_rate,
    output reg [15:0] adc_offset_0,
    output reg [15:0] adc_offset_1,
    output reg [15:0] adc_offset_2,
    output reg [15:0] adc_offset_3,
    output reg [7:0] master_controls,
    //output reg enable_tx,
    //output reg enable_rx,
    inout wire [15:0] io_0, io_1, io_2, io_3,
    output reg [7:0] settings,
    output reg [19:0] rxmux, output reg [19:0] txmux
    );
		
   reg [31:0] holding;
   reg 	      is_read;
   reg [7:0]  ser_ctr;
   reg 	      write_done;
   reg [6:0]  addr;
   wire [31:0] status = 32'h1234_5678;

   reg [15:0] io_0_oe,io_1_oe,io_2_oe,io_3_oe,io_0_reg,io_1_reg,io_2_reg,io_3_reg;

   bidir_reg bidir_reg_0 (.tristate(io_0),.oe(io_0_oe),.reg_val(io_0_reg));
   bidir_reg bidir_reg_1 (.tristate(io_1),.oe(io_1_oe),.reg_val(io_1_reg));
   bidir_reg bidir_reg_2 (.tristate(io_2),.oe(io_2_oe),.reg_val(io_2_reg));
   bidir_reg bidir_reg_3 (.tristate(io_3),.oe(io_3_oe),.reg_val(io_3_reg));
      
   assign serial_data_out = is_read ? holding[31] : 1'bz;

   always @(posedge serial_clock, posedge reset, negedge enable)
     if(reset)
       ser_ctr <= #1 0;
     else if(~enable)
       ser_ctr <= #1 0;
     else if(ser_ctr == 39)
       ser_ctr <= #1 0;
     else
       ser_ctr <= #1 ser_ctr + 1;

   always @(posedge serial_clock, posedge reset, negedge enable)
     if(reset)
       is_read <= #1 1'b0;
     else if(~enable)
       is_read <= #1 1'b0;
     else if((ser_ctr == 7)&&(addr[6]==1))
       is_read <= #1 1'b1;
   
   always @(posedge serial_clock, posedge reset)
     if(reset)
       begin
	  addr <= #1 7'b0;
	  holding <= #1 32'b0;
	  write_done <= #1 1'b0;
       end
     else if(~enable)
       begin
	  addr <= #1 7'b0;
	  holding <= #1 32'b0;
	  write_done <= #1 1'b0;
       end
     else 
       begin
	  if(~is_read && (ser_ctr == 39))
	    write_done <= #1 1'b1;
	  else
	    write_done <= #1 1'b0;
	  //	  if((ser_ctr == 7)&&(addr[6]==1))
	  if(is_read & (ser_ctr==8))
	    holding <= #1 (addr == 7'd1) ? {io_1,io_0} :
		       (addr == 7'd2) ? {io_3,io_2} :
		       status;
	  else if(ser_ctr >= 8)
	    holding <= #1 {holding[30:0],serial_data_in};
	  else if(ser_ctr < 8)
	    addr <= #1 {addr[5:0],serial_data_in};
       end // else: !if(~enable)
          
   //Asynch reset
   always @(negedge enable, posedge reset)
     if(reset)
       begin
	  ch0rx_freq <= #1 32'b0; ch1rx_freq <= #1 32'b0; ch2rx_freq <= #1 32'b0; ch3rx_freq <= #1 32'b0;
	  // ch0tx_freq <= #1 32'b0; ch1tx_freq <= #1 32'b0; ch2tx_freq <= #1 32'b0; ch3tx_freq <= #1 32'b0;
	  adc_offset_0 <= #1 16'b0; adc_offset_1 <= #1 16'b0; adc_offset_2 <= #1 16'b0; adc_offset_3 <= #1 16'b0;
	  interp_rate <= #1 8'b0; decim_rate <= #1 8'b0;
	  tx_sample_rate <= #1 8'b0; rx_sample_rate <= #1 8'b0;
	  master_controls <= #1 8'b0;
	  io_0_oe <= #1 16'b0; io_1_oe <= #1 16'b0; io_2_oe <= #1 16'b0; io_3_oe <= #1 16'b0;
	  io_0_reg <= #1 16'b0; io_1_reg <= #1 16'b0; io_2_reg <= #1 16'b0; io_3_reg <= #1 16'b0;
	  settings <= #1 8'b0;
	  rxmux <= #1 20'b0; txmux <= #1 20'b0;
       end // if (reset)
     else if(write_done)
       case(addr[5:0])
	 `FR_RX_FREQ_0 : ch0rx_freq <= #1 holding[31:0];
	 `FR_RX_FREQ_1 : ch1rx_freq <= #1 holding[31:0];
	 `FR_RX_FREQ_2 : ch2rx_freq <= #1 holding[31:0];
	 `FR_RX_FREQ_3 : ch3rx_freq <= #1 holding[31:0];
	 //`FR_TX_FREQ_0 : ch0tx_freq <= #1 holding[31:0];
	 //`FR_TX_FREQ_1 : ch1tx_freq <= #1 holding[31:0];
	 //`FR_TX_FREQ_2 : ch2tx_freq <= #1 holding[31:0];
	 //`FR_TX_FREQ_3 : ch3tx_freq <= #1 holding[31:0];
	 `FR_INTERP_RATE : interp_rate <= #1 holding[7:0];
	 `FR_DECIM_RATE : decim_rate <= #1 holding[7:0];
	 `FR_TX_SAMPLE_RATE_DIV: tx_sample_rate <= #1 holding[7:0];
	 `FR_RX_SAMPLE_RATE_DIV: rx_sample_rate <= #1 holding[7:0];
	 
	 `FR_ADC_OFFSET_3_2 : {adc_offset_3,adc_offset_2} <= #1 holding[31:0];
	 `FR_ADC_OFFSET_1_0 : {adc_offset_1,adc_offset_0} <= #1 holding[31:0];
	 `FR_MASTER_CTRL : master_controls <= #1 holding[7:0];
	 
	 // Upper 16 bits are mask for lower 16
	 `FR_OE_0 : io_0_oe
	   <= #1 (io_0_oe & ~holding[31:16]) | (holding[15:0] & holding[31:16] );
	 `FR_OE_1 : io_1_oe
	   <= #1 (io_1_oe & ~holding[31:16]) | (holding[15:0] & holding[31:16] );
	 `FR_OE_2 : io_2_oe
	   <= #1 (io_2_oe & ~holding[31:16]) | (holding[15:0] & holding[31:16] );
	 `FR_OE_3 : io_3_oe
	   <= #1 (io_3_oe & ~holding[31:16]) | (holding[15:0] & holding[31:16] );
	 `FR_IO_0 : io_0_reg
	   <= #1 (io_0_reg & ~holding[31:16]) | (holding[15:0] & holding[31:16] );
	 `FR_IO_1 : io_1_reg
	   <= #1 (io_1_reg & ~holding[31:16]) | (holding[15:0] & holding[31:16] );
	 `FR_IO_2 : io_2_reg
	   <= #1 (io_2_reg & ~holding[31:16]) | (holding[15:0] & holding[31:16] );
	 `FR_IO_3 : io_3_reg
	   <= #1 (io_3_reg & ~holding[31:16]) | (holding[15:0] & holding[31:16] );

	 `FR_MODE : settings <= #1 holding[7:0];

	 `FR_RX_MUX : rxmux <= #1 holding [19:0];
	 `FR_TX_MUX : txmux <= #1 holding [19:0];
	 	 
       endcase // case(addr[5:0])
   
endmodule // serial_io

