;;;***************************************************************
;;;An ACL2 Library of Floating Point Arithmetic

;;;David M. Russinoff
;;;Advanced Micro Devices, Inc.
;;;February, 1998
;;;***************************************************************

(in-package "ACL2")

;(include-book "floor")
(include-book "bits")

;end of stuff for bits

(defun COMP1 (x n)
  (1- (- (expt 2 n) x)))

(defun DEC1 (x n)
  (rem (1- (+ (expt 2 n) x)) (expt 2 n)))

(defthm LOGAND-0
    (equal (logand 0 y) 0))

(defthm LOGAND-0-2
    (equal (logand x 0) 0))

(defthm LOGAND-DEF
  (implies (and (integerp x) (>= x 0)
                (integerp y) (>= y 0))
           (= (logand x y)
              (+ (* 2 (logand (fl (/ x 2)) (fl (/ y 2))))
                 (logand (rem x 2) (rem y 2)))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable logand)
           :use ((:instance rem012)
                 (:instance rem012 (x y))
                 (:instance binary-logand (i x) (j y))
                 (:instance floor-fl (m x) (n 2))
                 (:instance floor-fl (m y) (n 2))
                 (:instance bitn-def-4)
                 (:instance bitn-def-4 (x y))))))

(defthm LOGAND-NAT
  (implies (and (integerp i) (>= i 0)
                (integerp j) (>= j 0))
           (and (integerp (logand i j))
                (>= (logand i j) 0)))
  :rule-classes ())

(defthm LOGAND-REM
  (implies (and (integerp x) (>= x 0)
                (integerp y) (>= y 0))
           (= (rem (logand x y) 2)
              (logand (rem x 2) (rem y 2))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable logand fl)
           :use ((:instance logand-def)
                 (:instance rem012)
                 (:instance rem012 (x y))
                 (:instance rem+ 
                            (m (logand (rem x 2) (rem y 2)))
                            (n 2)
                            (a (logand (fl (/ x 2)) (fl (/ y 2)))))
                 (:instance logand-nat (i (fl (/ x 2))) (j (fl (/ y 2))))
                 (:instance rem< (m (logand (rem x 2) (rem y 2))) (n 2))))))

(defthm LOGAND-FL
  (implies (and (integerp x) (>= x 0)
                (integerp y) (>= y 0))
           (= (fl (/ (logand x y) 2))
              (logand (fl (/ x 2)) (fl (/ y 2)))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable logand fl)
           :use ((:instance logand-def)
                 (:instance logand-rem)
                 (:instance rem-fl (m (logand x y)) (n 2))
                 (:instance logand-nat (i x) (j y))))))

(local
 (defthm logand-1
   (implies (integerp x)
            (equal (logand -1 x) x))))

(local
 (defthm logand-1-2
   (implies (integerp x)
            (equal (logand x -1) x))))

(local
 (defthm den-int
   (implies (integerp x)
            (equal (denominator x) 1))
   :hints (("Goal" :in-theory (disable rational-implies2)
            :use ((:instance lowest-terms
                             (n (denominator x))
                             (r x)
                             (q 1))
                  (:instance rational-implies1)
                  (:instance rational-implies2))))))

(local
 (defthm num-int
   (implies (integerp x)
            (equal (numerator x) x))
   :hints (("Goal" :in-theory (disable rational-implies2)
            :use ((:instance den-int)
                  (:instance rational-implies1)
                  (:instance rational-implies2))))))

(local
 (defthm floor*2
   (implies (integerp x)
            (equal (floor (* 2 x) 2) x))))

(local
 (defthm den-2x+1/2-1
   (implies (integerp x)
            (equal (denominator (/ (1+ (* 2 x)) 2))
                   (* 2 (- (numerator (/ (1+ (* 2 x)) 2))
                           (* x (denominator (/ (1+ (* 2 x)) 2)))))))
   :rule-classes ()
   :hints (("Goal" :in-theory (disable rational-implies2)
            :use ((:instance rational-implies1 (x (/ (1+ (* 2 x)) 2)))
                  (:instance rational-implies2 (x (/ (1+ (* 2 x)) 2))))))))

(local
 (defthm den-2x+1/2-2
    (implies (integerp x)
	     (equal (* 2 (numerator (/ (1+ (* 2 x)) 2)))
		    (* (denominator (/ (1+ (* 2 x)) 2))
		       (1+ (* 2 x)))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable rational-implies2)
		  :use ((:instance rational-implies1 (x (/ (1+ (* 2 x)) 2)))
			(:instance rational-implies2 (x (/ (1+ (* 2 x)) 2)))
			(:instance den-2x+1/2-1))))))

(local
 (defthm worst-hack-yet
    (implies (and (rationalp n)
		  (rationalp d)
		  (rationalp s)
		  (rationalp q)
		  (= (* 2 n) (* d s))
		  (= d (* 2 q)))
	     (= n (* s q)))
  :rule-classes ()))

(local
 (defthm den-2x+1/2-3
    (implies (integerp x)
	     (equal (numerator (/ (1+ (* 2 x)) 2))
		    (* (1+ (* 2 x))
		       (- (numerator (/ (1+ (* 2 x)) 2))
			  (* x (denominator (/ (1+ (* 2 x)) 2)))))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable rational-implies2)
		  :use ((:instance den-2x+1/2-1)
			(:instance den-2x+1/2-2)
			(:instance worst-hack-yet
				   (n (numerator (/ (1+ (* 2 x)) 2)))
				   (d (denominator (/ (1+ (* 2 x)) 2)))
				   (s (1+ (* 2 x)))
				   (q (- (numerator (/ (1+ (* 2 x)) 2))
					 (* x (denominator (/ (1+ (* 2 x)) 2)))))))))))

(local
 (defthm den-2x+1/2-4
    (implies (integerp x)
	     (equal (- (numerator (/ (1+ (* 2 x)) 2))
		       (* x (denominator (/ (1+ (* 2 x)) 2))))
		    1))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable rational-implies2)
		  :use ((:instance lowest-terms
				   (n (- (numerator (/ (1+ (* 2 x)) 2))
					 (* x (denominator (/ (1+ (* 2 x)) 2)))))
				   (r (1+ (* 2 x)))
				   (q 2)
				   (x (/ (1+ (* 2 x)) 2)))
			(:instance den-2x+1/2-1)
			(:instance den-2x+1/2-3)
			(:instance rational-implies1 (x (/ (1+ (* 2 x)) 2)))
			(:instance rational-implies2 (x (/ (1+ (* 2 x)) 2))))))))

(local
 (defthm den-2x+1/2
    (implies (integerp x)
	     (equal (denominator (+ 1/2 x)) 2))
  :hints (("Goal" :use ((:instance den-2x+1/2-1)
			(:instance den-2x+1/2-4))))))

(local
 (defthm num-2x+1/2
    (implies (integerp x)
	     (equal (numerator (+ 1/2 x)) (1+ (* 2 x))))
  :hints (("Goal" :in-theory (disable rational-implies2)
		  :use ((:instance rational-implies2 (x (/ (1+ (* 2 x)) 2)))
			(:instance den-2x+1/2))))))
;moved induct-nat to basic


(local
 (defthm floor*2+1-1
    (implies (and (integerp x) (>= x 0))
	     (equal (nonnegative-integer-quotient (1+ (* 2 x)) 2)
		    x))
  :rule-classes ()
  :hints (("Goal" :induct (induct-nat x)))))

(local
 (defthm floor*2+1-2
    (implies (and (integerp x) (> x 0))
	     (equal (nonnegative-integer-quotient (1- (* 2 x)) 2)
		    (1- x)))
  :rule-classes ()
  :hints (("Goal" :induct (induct-nat x)))))

(local
 (defthm floor*2+1
    (implies (integerp x)
	     (equal (floor (1+ (* 2 x)) 2) x))
  :rule-classes ()
  :hints (("Goal" :use ((:instance floor*2+1-2)
			(:instance floor*2+1-2 (x (- x))))))))


(defthm floor-logand
    (implies (and (integerp i)
		  (integerp j))
	     (= (floor (logand i j) 2)
		(logand (floor i 2) (floor j 2))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable logand floor evenp floor*2)
		  :use ((:instance binary-logand)
			(:instance floor*2 (x (LOGAND (FLOOR I 2) (FLOOR J 2))))
			(:instance floor*2+1 (x (LOGAND (FLOOR I 2) (FLOOR J 2))))))))

;moved x-or-x/2 stuff to basic


(local
 (defthm floor-lognot-1
    (implies (integerp n)
	     (= (floor (lognot (* 2 n)) 2)
		(lognot (floor (* 2 n) 2))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable floor)
		  :use ((:instance floor*2+1 (x (lognot n)))
			(:instance floor*2 (x n)))))))

(local
 (defthm floor-lognot-2
    (implies (integerp n)
	     (= (floor (lognot (1- (* 2 n))) 2)
		(lognot (floor (1- (* 2 n)) 2))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable floor)
		  :use ((:instance floor*2+1 (x (1- n)))
			(:instance floor*2 (x (- n))))))))


(defthm floor-lognot
    (implies (integerp i)
	     (= (floor (lognot i) 2)
		(lognot (floor i 2))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable floor lognot)
		  :use ((:instance floor-lognot-1 (n (/ i 2)))
			(:instance floor-lognot-2 (n (/ (1+ i) 2)))
			(:instance x-or-x/2 (x i))))))


(defthm floor-logior
    (implies (and (integerp i)
		  (integerp j))
	     (= (floor (logior i j) 2)
		(logior (floor i 2) (floor j 2))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable logand floor lognot)
		  :use ((:instance floor-lognot)
			(:instance floor-lognot (i j))
			(:instance floor-logand (i (lognot i)) (j (lognot j)))
			(:instance floor-lognot (i (logand (lognot i) (lognot j))))))))

(local
 (defthm fl-logior-1
    (implies (integerp i)
	     (iff (>= i 0) (< (lognot i) 0)))
  :rule-classes ()))

(local
 (defthm fl-logior-2
    (implies (and (integerp i) (< i 0)
		  (integerp j) (< j 0))
	     (and (integerp (logand i j))
		  (< (logand i j) 0)))
  :rule-classes ()))

(defthm LOGIOR-NAT
  (implies (and (integerp i) (>= i 0)
                (integerp j) (>= j 0))
           (and (integerp (logior i j))
                (>= (logior i j) 0)))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable lognot logand)
           :use ((:instance fl-logior-1)
                 (:instance fl-logior-1 (i j))
                 (:instance fl-logior-1 (i (logand (lognot i) (lognot j))))
                 (:instance fl-logior-2 (i (lognot i)) (j (lognot j)))))))

(defthm LOGIOR-FL
  (implies (and (integerp i) (>= i 0)
                (integerp j) (>= j 0))
           (= (fl (/ (logior i j) 2))
              (logior (fl (/ i 2)) (fl (/ j 2)))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable logior floor)
           :use ((:instance floor-logior)
                 (:instance floor-fl (m (logior i j)) (n 2))
                 (:instance floor-fl (m i) (n 2))
                 (:instance floor-fl (m j) (n 2))
                 (:instance logior-nat)))))

(defthm REM-2*I
  (implies (integerp i)
           (equal (rem (* 2 i) 2) 0))
  :rule-classes ()
  :hints (("Goal" :use ((:instance rem (x (* 2 i)) (y 2))
			(:instance rem012 (x (* 2 i)))))))

(local
 (defthm rem-logior-2
   (implies (integerp i)
            (equal (rem (1+ (* 2 i)) 2) (- 1 (* 2 (- (truncate (1+ (* 2 i)) 2) i)))))
   :rule-classes ()
   :hints (("Goal" :use ((:instance rem (x (1+ (* 2 i))) (y 2)))))))

(defthm REM-2*I+1
  (implies (integerp i)
           (not (equal (rem (1+ (* 2 i)) 2) 0)))
  :rule-classes ()
  :hints (("Goal" :use ((:instance bit+-4 (x (- (truncate (1+ (* 2 i)) 2) i)))
			(:instance rem-logior-2)))))

(local
 (defthm rem-logior-4
    (implies (integerp n)
	     (not (= (rem (lognot (* 2 n)) 2) 0)))
  :rule-classes ()
  :hints (("Goal" :use ((:instance rem-2*i+1 (i (1- (- n)))))))))

(local
 (defthm rem-logior-5
    (implies (integerp n)
	     (= (rem (lognot (1+ (* 2 n))) 2) 0))
  :rule-classes ()
  :hints (("Goal" :use ((:instance rem-2*i (i (1- (- n)))))))))

(local
 (defthm rem-logior-6
    (implies (integerp n) (integerp (1- n)))
  :rule-classes ()))

(local
 (defthm rem-logior-7
    (implies (rationalp x)
	     (= (1- (+ 1/2 x))
		(+ -1/2 x)))
  :rule-classes ()))

(local
 (defthm rem-logior-8
    (implies (and (rationalp x)
		  (integerp (+ 1/2 x)))
	     (integerp (+ -1/2 x)))
  :hints (("Goal" :use ((:instance rem-logior-6 (n (+ 1/2 x)))
			(:instance rem-logior-7))))))


(defthm rem-logior-9
    (implies (integerp i)
	     (iff (= (rem (lognot i) 2) 0)
		  (not (= (rem i 2) 0))))
  :hints (("Goal" :in-theory (disable lognot)
		  :use ((:instance x-or-x/2 (x i))
			(:instance rem-2*i (i (/ i 2)))
			(:instance rem-logior-4 (n (/ i 2)))
			(:instance rem-2*i+1 (i (1- (/ (1+ i) 2))))
			(:instance rem-logior-5 (n (1- (/ (1+ i) 2))))))))

(local
 (defthm evenp-logand-1
    (implies (and (integerp i)
		  (integerp j)
		  (or (evenp i) (evenp j)))
	     (= (logand i j)
		(* 2 (logand (floor i 2) (floor j 2)))))
  :rule-classes ()))

(local
 (defthm evenp-logand-2
    (implies (integerp x)
	     (iff (evenp x)
		  (= (rem x 2) 0)))
  :rule-classes ()
  :hints (("Goal" :use ((:instance x-or-x/2)
			(:instance rem-2*i (i (/ x 2)))
			(:instance rem-2*i+1 (i (1- (/ (1+ x) 2)))))))))

(local
 (defthm evenp-logand-3
    (implies (and (integerp i)
		  (integerp j)
		  (or (= (rem i 2) 0) (= (rem j 2) 0)))
	     (= (rem (logand i j) 2) 0))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable logand floor evenp)
		  :use ((:instance evenp-logand-1)
			(:instance evenp-logand-2 (x i))
			(:instance evenp-logand-2 (x j))
			(:instance rem-2*i (i (logand (floor i 2) (floor j 2)))))))))

(local
 (defthm evenp-logand-4
    (implies (and (integerp i)
		  (integerp j)
		  (not (or (evenp i) (evenp j)))
		  (or (= i -1) (= j -1)))
	     (or (= (logand i j) i)
		 (= (logand i j) j)))
  :rule-classes ()))

(local
 (defthm evenp-logand-5
    (implies (and (integerp i)
		  (integerp j)
		  (not (or (= (rem i 2) 0) (= (rem j 2) 0)))
		  (or (= i -1) (= j -1)))
	     (not (= (rem (logand i j) 2) 0)))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable logand evenp)
		  :use ((:instance evenp-logand-4)
			(:instance evenp-logand-2 (x i))
			(:instance evenp-logand-2 (x j)))))))

(local
 (defthm evenp-logand-6
    (implies (and (integerp i)
		  (integerp j)
		  (not (or (evenp i) (evenp j)))
		  (not (or (= i -1) (= j -1))))
	     (= (logand i j)
		(1+ (* 2 (logand (floor i 2) (floor j 2))))))
  :rule-classes ()))

(local
 (defthm evenp-logand-7
    (implies (and (integerp i)
		  (integerp j)
		  (not (or (= (rem i 2) 0) (= (rem j 2) 0)))
		  (not (or (= i -1) (= j -1))))
	     (not (= (rem (logand i j) 2) 0)))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable logand evenp floor)
		  :use ((:instance evenp-logand-6)
			(:instance evenp-logand-2 (x i))
			(:instance evenp-logand-2 (x j))
			(:instance rem-2*i+1 (i (logand (floor i 2) (floor j 2)))))))))

(local
 (defthm evenp-logand-8
    (implies (and (integerp i)
		  (integerp j)
		  (not (or (= (rem i 2) 0) (= (rem j 2) 0))))
	     (not (= (rem (logand i j) 2) 0)))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable logand evenp floor)
		  :use ((:instance evenp-logand-5)
			(:instance evenp-logand-7))))))


(defthm evenp-logand
    (implies (and (integerp i)
		  (integerp j))
	     (iff (or (= (rem i 2) 0) (= (rem j 2) 0))
		  (= (rem (logand i j) 2) 0)))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable logand evenp floor)
		  :use ((:instance evenp-logand-3)
			(:instance evenp-logand-8)))))


(defthm rem-logior-10
    (implies (and (integerp i)
		  (integerp j))
	     (iff (and (= (rem i 2) 0) (= (rem j 2) 0))
		  (= (rem (logior i j) 2) 0)))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable logand lognot)
		  :use ((:instance rem-logior-9 (i (logand (lognot i) (lognot j))))
			(:instance evenp-logand (i (lognot i)) (j (lognot j)))
			(:instance rem-logior-9)
			(:instance rem-logior-9 (i j))))))

(defthm LOGIOR-REM
  (implies (and (integerp i) (>= i 0)
                (integerp j) (>= j 0))
           (= (rem (logior i j) 2)
              (logior (rem i 2) (rem j 2))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable logior)
           :use ((:instance rem-logior-10)
                 (:instance rem012 (x i))
                 (:instance rem012 (x j))
                 (:instance logior-nat)
                 (:instance rem012 (x (logior i j)))))))

(defthm LOGIOR-DEF
    (implies (and (integerp i) (>= i 0)
		  (integerp j) (>= j 0))
	     (= (logior i j)
		(+ (* 2 (logior (fl (/ i 2)) (fl (/ j 2))))
		   (logior (rem i 2) (rem j 2)))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable logior)
		  :use ((:instance logior-nat)
			(:instance rem-fl (m (logior i j)) (n 2))
			(:instance logior-rem)
			(:instance logior-fl)))))



;from logxor-def.lisp

(local
(defthm logorc1-rem-1
    (implies (and (integerp i) (integerp j))
	     (iff (= (rem (logorc1 i j) 2) 0)
		  (and (= (rem (lognot i) 2) 0)
		       (= (rem j 2) 0))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable logior lognot)
		  :use ((:instance rem-logior-10 (i (lognot i))))))))

(local(defthm logorc1-rem
    (implies (and (integerp i) (>= i 0)
		  (integerp j))
	     (iff (= (rem (logorc1 i j) 2) 0)
		  (and (= (rem i 2) 1)
		       (= (rem j 2) 0))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable logior lognot)
		  :use ((:instance rem-logior-9)
			(:instance logorc1-rem-1)
			(:instance rem012 (x i)))))))

(local(defthm logeqv-rem-1
    (implies (and (integerp i) (>= i 0)
		  (integerp j) (>= j 0))
	     (iff (= (rem (logeqv i j) 2) 0)
		  (or (= (rem (logorc1 i j) 2) 0)
		      (= (rem (logorc1 j i) 2) 0))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable logorc1 logand)
		  :use ((:instance evenp-logand (i (logorc1 i j)) (j (logorc1 j i))))))))

(local(defthm logeqv-rem
    (implies (and (integerp i) (>= i 0)
		  (integerp j) (>= j 0))
	     (iff (= (rem (logeqv i j) 2) 0)
		  (not (= (logxor (rem i 2) (rem j 2))
			  0))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable logorc1 logeqv)
		  :use ((:instance logeqv-rem-1)
			(:instance logorc1-rem)
			(:instance logorc1-rem (i j) (j i))
			(:instance rem012 (x i))
			(:instance rem012 (x j)))))))

(local(defthm logxor-rem-1
    (implies (and (integerp i) (>= i 0)
		  (integerp j) (>= j 0))
	     (iff (= (rem (logxor i j) 2)
		     0)
		  (= (logxor (rem i 2) (rem j 2))
		     0)))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable logorc1 logeqv)
		  :use ((:instance logeqv-rem)
			(:instance rem-logior-9 (i (logeqv i j))))))))

(local(defthm logand>=0
    (implies (and (integerp i)
		  (>= i 0)
		  (integerp j))
	     (>= (logand i j) 0))
  :rule-classes ()))

(local(defthm logior<0
    (implies (and (integerp i)
		  (integerp j)
		  (< i 0))
	     (< (logior i j) 0))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable logand)
		  :use ((:instance logand>=0 (i (lognot i)) (j (lognot j))))))))

(local (defthm logand<0
    (implies (and (integerp i)
		  (integerp j)
		  (< i 0)
		  (< j 0))
	     (< (logand i j) 0))
  :rule-classes ()))

(defthm LOGXOR-NAT
    (implies (and (integerp i) (>= i 0)
		  (integerp j) (>= j 0))
	     (and (integerp (logxor i j))
		  (>= (logxor i j) 0)))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable logand logior)
		  :use ((:instance logior<0 (i (lognot i)))
			(:instance logior<0 (i (lognot j)) (j i))
			(:instance logand<0 (i (logorc1 i j)) (j (logorc1 j i)))))))

(defthm LOGXOR-REM
    (implies (and (integerp i) (>= i 0)
		  (integerp j) (>= j 0))
	     (= (rem (logxor i j) 2)
		(logxor (rem i 2) (rem j 2))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable logxor)
		  :use ((:instance logxor-rem-1)
			(:instance logxor-nat)
			(:instance rem012 (x (logxor i j)))
			(:instance rem012 (x i))
			(:instance rem012 (x j))))))

(defthm floor-logorc1
    (implies (and (integerp i)
		  (integerp j))
	     (= (floor (logorc1 i j) 2)
		(logorc1 (floor i 2) (floor j 2))))
  :rule-classes()
  :hints (("Goal" :in-theory (disable lognot logior floor)
		  :use ((:instance floor-logior (i (lognot i)))
			(:instance floor-lognot)))))

(defthm floor-logeqv
    (implies (and (integerp i)
		  (integerp j))
	     (= (floor (logeqv i j) 2)
		(logeqv (floor i 2) (floor j 2))))
  :rule-classes()
  :hints (("Goal" :in-theory (disable logand logorc1 floor)
		  :use ((:instance floor-logorc1)
			(:instance floor-logorc1 (i j) (j i))
			(:instance floor-logand (i (logorc1 i j)) (j (logorc1 j i)))))))

(defthm floor-logxor
    (implies (and (integerp i)
		  (integerp j))
	     (= (floor (logxor i j) 2)
		(logxor (floor i 2) (floor j 2))))
  :rule-classes()
  :hints (("Goal" :in-theory (disable lognot logeqv floor)
		  :use ((:instance floor-lognot (i (logeqv i j)))
			(:instance floor-logeqv)))))

(defthm LOGXOR-FL
    (implies (and (integerp i)
		  (integerp j)
		  (>= i 0)
		  (>= j 0))
	     (= (fl (/ (logxor i j) 2))
		(logxor (fl (/ i 2)) (fl (/ j 2)))))
  :rule-classes()
  :hints (("Goal" :in-theory (disable logxor floor)
		  :use ((:instance floor-logxor)
			(:instance floor-fl (m i) (n 2))
			(:instance floor-fl (m j) (n 2))
			(:instance logxor-nat)
			(:instance floor-fl (m (logxor i j)) (n 2))))))

(defthm LOGXOR-DEF
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0))
	     (= (logxor x y)
		(+ (* 2 (logxor (fl (/ x 2)) (fl (/ y 2))))
		   (logxor (rem x 2) (rem y 2)))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable logxor)
		  :use ((:instance logxor-fl (i x) (j y))
			(:instance logxor-rem (i x) (j y))
			(:instance logxor-nat (i x) (j y))
			(:instance rem-fl (m (logxor x y)) (n 2))))))



;from loglemmas

(local
(defun log-induct (x y)
  (if (and (integerp x) (>= x 0)
	   (integerp y) (>= y 0))
      (if (or (= x 0) (= y 0))
	  ()
	(log-induct (fl (/ x 2)) (fl (/ y 2))))
    ())))

(defthm BIT-BASIC-A
    (implies (and (integerp x) (>= x 0))
	     (= (logand x 0)
		0))
  :rule-classes ())

(defthm BIT-BASIC-B
    (implies (and (integerp x) (>= x 0))
	     (= (logior x 0)
		x))
  :rule-classes ())

(defthm BIT-BASIC-C
    (implies (and (integerp x)
		  (integerp y))
	     (= (logand x y) (logand y x)))
  :rule-classes ())

(defthm BIT-BASIC-D
    (implies (and (integerp x)
		  (integerp y))
	     (= (logior x y) (logior y x)))
  :rule-classes ()
  :hints (("Goal" :use ((:instance bit-basic-c (x (lognot x)) (y (lognot y)))))))

(local
(defun log-induct-3 (x y z)
  (if (and (integerp x) (>= x 0)
	   (integerp y) (>= y 0)
	   (integerp z) (>= z 0))
      (if (or (= x 0) (= y 0) (= z 0))
	  ()
	(log-induct-3 (fl (/ x 2)) (fl (/ y 2)) (fl (/ z 2))))
    ())))

(defthm LOGAND-FL-REWRITE
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0))
	     (equal (fl (* 1/2 (logand x y)))
		    (logand (fl (/ x 2)) (fl (/ y 2)))))
  :hints (("Goal" :use ((:instance logand-fl)))))

(defthm LOGAND-REM-REWRITE
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0))
	     (equal (rem (logand x y) 2)
		    (logand (rem x 2) (rem y 2))))
  :hints (("Goal" :use ((:instance logand-rem)))))

(defthm LOGAND-NAT-REWRITE
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0))
	     (>= (logand x y) 0))
  :hints (("Goal" :use ((:instance logand-nat (i x) (j y))))))

(local
 (defthm fl-rem-equal
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (= (fl (/ x 2)) (fl (/ y 2)))
		  (= (rem x 2) (rem y 2)))
	     (= x y))
  :rule-classes ()
  :hints (("Goal" :use ((:instance rem-fl (m x) (n 2))
			(:instance rem-fl (m y) (n 2)))))))

(local
(defthm bit-basic-e-1
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp z) (>= z 0))
	     (EQUAL (LOGAND (LOGAND (REM X 2) (REM Y 2))
			    (REM Z 2))
		    (LOGAND (REM X 2)
			    (LOGAND (REM Y 2) (REM Z 2)))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance rem012)
			(:instance rem012 (x y))
			(:instance rem012 (x z)))))))

(defthm BIT-BASIC-E
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp z) (>= z 0))
	     (= (logand (logand x y) z)
		(logand x (logand y z))))
  :rule-classes ()
  :hints (("Goal" :induct (log-induct-3 x y z))
	  ("Subgoal *1/2.1" :use ((:instance bit-basic-e-1)
				  (:instance fl-rem-equal
					      (x (logand (logand x y) z))
					      (y (logand x (logand y z))))))))

(in-theory (disable logior))

(local
(defthm bit-basic-f-1
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp z) (>= z 0))
	     (EQUAL (LOGIOR (LOGIOR (REM X 2) (REM Y 2))
			    (REM Z 2))
		    (LOGIOR (REM X 2)
			    (LOGIOR (REM Y 2) (REM Z 2)))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance rem012)
			(:instance rem012 (x y))
			(:instance rem012 (x z)))))))

(local
(defthm logior-fl-rewrite
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0))
	     (equal (fl (* 1/2 (logior x y)))
		    (logior (fl (/ x 2)) (fl (/ y 2)))))
  :hints (("Goal" :use ((:instance logior-fl (i x) (j y)))))))

(defthm LOGIOR-REM-REWRITE
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0))
	     (equal (rem (logior x y) 2)
		    (logior (rem x 2) (rem y 2))))
  :hints (("Goal" :use ((:instance logior-rem (i x) (j y))))))

(defthm LOGIOR-NAT-REWRITE
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0))
	     (>= (logior x y) 0))
  :hints (("Goal" :use ((:instance logior-nat (i x) (j y))))))

(defthm LOGIOR-0
    (implies (integerp x)
	     (equal (logior x 0) x))
  :hints (("Goal" :in-theory (enable logior))))

(defthm LOGIOR-0-2
    (implies (integerp x)
	     (equal (logior 0 x) x))
  :hints (("Goal" :in-theory (enable logior))))

(defthm BIT-BASIC-F
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp z) (>= z 0))
	     (= (logior (logior x y) z)
		(logior x (logior y z))))
  :rule-classes ()
  :hints (("Goal" :induct (log-induct-3 x y z))
	  ("Subgoal *1/2.1" :use ((:instance bit-basic-f-1)
				  (:instance fl-rem-equal
					      (x (logior (logior x y) z))
					      (y (logior x (logior y z))))))))

(local
(defthm bit-basic-g-1
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp z) (>= z 0))
	     (equal (logior (rem x 2)
			    (logand  (rem y 2) (rem z 2)))
		    (logand (logior (rem x 2) (rem y 2))
			    (logior (rem x 2) (rem z 2)))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance rem012)
			(:instance rem012 (x y))
			(:instance rem012 (x z)))))))

(local
(defthm bit-basic-g-2
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0))
	     (equal (logand (logior (rem x 2) (rem y 2)) (rem x 2))
		    (rem x 2)))
  :rule-classes ()
  :hints (("Goal" :use ((:instance rem012)
			(:instance rem012 (x y)))))))

(local
(defthm logand-x-x-1
    (implies (and (integerp x) (>= x 0))
	     (equal (logand (rem x 2) (rem x 2)) (rem x 2)))
  :rule-classes ()
  :hints (("Goal" :use ((:instance rem012))))))

(defthm LOGAND-X-X
    (implies (and (integerp x) (>= x 0))
	     (equal (logand x x) x))
  :hints (("Goal" :induct (log-induct x x))
	  ("Subgoal *1/2.1" :use ((:instance logand-x-x-1)
				  (:instance fl-rem-equal (y (logand x x)))))))

(local
(defthm bit-basic-g-3
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0))
	     (= (logand (logior x y) x)
		x))
  :rule-classes ()
  :hints (("Goal" :induct (log-induct x y))
	  ("Subgoal *1/2.2" :use ((:instance bit-basic-g-2)
				  (:instance fl-rem-equal
					      (y (logand (logior x y) x))))))))

(defthm BIT-BASIC-G
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp z) (>= z 0))
	     (= (logior x (logand y z))
		(logand (logior x y) (logior x z))))
  :rule-classes ()
  :hints (("Goal" :induct (log-induct-3 x y z))
	  ("Subgoal *1/2.1" :use ((:instance bit-basic-g-1)
				  (:instance fl-rem-equal
					      (x (logior x (logand y z)))
					      (y (logand (logior x y) (logior x z))))))
	  ("Subgoal *1/1" :use ((:instance bit-basic-g-3)
				(:instance bit-basic-c (y (logior x z)))
				(:instance bit-basic-g-3 (y z))))))

(local
(defthm bit-basic-h-1
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp z) (>= z 0))
	     (equal (logand (rem x 2)
			    (logior  (rem y 2) (rem z 2)))
		    (logior (logand (rem x 2) (rem y 2))
			    (logand (rem x 2) (rem z 2)))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance rem012)
			(:instance rem012 (x y))
			(:instance rem012 (x z)))))))

(defthm BIT-BASIC-H
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp z) (>= z 0))
	     (= (logand x (logior y z))
		(logior (logand x y) (logand x z))))
  :rule-classes ()
  :hints (("Goal" :induct (log-induct-3 x y z))
	  ("Subgoal *1/2.1" :use ((:instance bit-basic-h-1)
				  (:instance fl-rem-equal
					      (x (logand x (logior y z)))
					      (y (logior (logand x y) (logand x z))))))))

(local
(defun op-dist-induct (x y n)
  (if (and (integerp n) (>= n 0))
      (if  (= n 0)
	  (list x y)
	(op-dist-induct (fl (/ x 2)) (fl (/ y 2)) (1- n)))
    ())))

(defthm OR-DIST-A
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp n) (>= n 0)
		  (< x (expt 2 n))
		  (< y (expt 2 n)))
	     (< (logior x y) (expt 2 n)))
  :rule-classes ()
  :hints (("Goal" :induct (op-dist-induct x y n))
	  ("Subgoal *1/2" :use ((:instance logior-def (i x) (j y))
				(:instance rem012)
				(:instance rem012 (x y))))))

(local
(defthm or-dist-b-1
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp n) (> n 0)
		  (< y (expt 2 n)))
	     (= (logior (* (expt 2 n) x) y)
		(+ (* 2 (logior (fl (* (expt 2 (1- n)) x))
				(fl (/ y 2))))
		   (logior (rem (* (expt 2 n) x) 2)
			   (rem y 2)))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance logior-def (i (* (expt 2 n) x)) (j y)))))))

(local
(defthm or-dist-b-2
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp n) (> n 0)
		  (< y (expt 2 n)))
	     (= (logior (* (expt 2 n) x) y)
		(+ (* 2 (logior (* (expt 2 (1- n)) x)
				(fl (/ y 2))))
		   (rem y 2))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance or-dist-b-1)
			(:instance fl-int (x (* (expt 2 (1- n)) x)))
			(:instance rem-2*i (i (* (expt 2 (1- n)) x))))))))

(local
(defthm or-dist-b-3
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp n) (> n 0)
		  (< y (expt 2 n))
		  (= (logior (* (expt 2 (1- n)) x)
			     (fl (/ y 2)))
		     (+ (* (expt 2 (1- n)) x)
			(fl (/ y 2)))))
	     (= (logior (* (expt 2 n) x) y)
		(+ (* (expt 2 n) x)
		   (* 2	(fl (/ y 2)))
		   (rem y 2))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance or-dist-b-2))))))

(local
(defthm or-dist-b-4
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp n) (> n 0)
		  (< y (expt 2 n))
		  (= (logior (* (expt 2 (1- n)) x)
			     (fl (/ y 2)))
		     (+ (* (expt 2 (1- n)) x)
			(fl (/ y 2)))))
	     (= (logior (* (expt 2 n) x) y)
		(+ (* (expt 2 n) x) y)))
  :rule-classes ()
  :hints (("Goal" :use ((:instance or-dist-b-3)
			(:instance rem-fl (m y) (n 2)))))))

(local
(defun or-dist-induct (y n)
  (if (and (integerp n) (>= n 0))
      (if (= n 0)
	  y
	(or-dist-induct (fl (/ y 2)) (1- n)))
    ())))

(defthm OR-DIST-B
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp n) (>= n 0)
		  (< y (expt 2 n)))
	     (= (logior (* (expt 2 n) x) y)
		(+ (* (expt 2 n) x) y)))
  :rule-classes ()
  :hints (("Goal" :induct (or-dist-induct y n))
	  ("Subgoal *1/2" :use ((:instance or-dist-b-4)))))

(local
(defthm or-dist-c-1
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp n) (> n 0))
	     (= (logior (* (expt 2 n) x)
			(* (expt 2 n) y))
		(+ (* 2 (logior (* (expt 2 (1- n)) x)
				(* (expt 2 (1- n)) y)))
		   (logior (rem (* (expt 2 n) x) 2)
			   (rem (* (expt 2 n) y) 2)))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance logior-def (i (* (expt 2 n) x)) (j (* (expt 2 n) y))))))))

(local
(defthm or-dist-c-2
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp n) (> n 0))
	     (= (logior (* (expt 2 n) x)
			(* (expt 2 n) y))
		(* 2 (logior (* (expt 2 (1- n)) x)
			     (* (expt 2 (1- n)) y)))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance or-dist-c-1)
			(:instance rem-2*i (i (* (expt 2 (1- n)) x)))
			(:instance rem-2*i (i (* (expt 2 (1- n)) y))))))))

(local
(defthm or-dist-c-3
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp n) (> n 0)
		  (= (logior (* (expt 2 (1- n)) x)
			     (* (expt 2 (1- n)) y))
		     (* (expt 2 (1- n)) (logior x y))))			     
	     (= (logior (* (expt 2 n) x)
			(* (expt 2 n) y))
		(* (expt 2 n)
		   (logior x y))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance or-dist-c-2))))))

(defthm OR-DIST-C
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp n) (>= n 0))
	     (= (logior (* (expt 2 n) x)
			(* (expt 2 n) y))
		(* (expt 2 n) (logior x y))))
  :rule-classes ()
  :hints (("Goal" :induct (induct-nat n))
	  ("Subgoal *1/1" :use ((:instance or-dist-c-3)))))

(local
(defthm or-dist-d-1
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp n) (> n 0))
	     (= (logior x y)
		(logior (logior (* (expt 2 n) (fl (/ x (expt 2 n))))
				(rem x (expt 2 n)))
			(logior (* (expt 2 n) (fl (/ y (expt 2 n))))
				(rem y (expt 2 n))))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance rem-fl (m x) (n (expt 2 n)))
			(:instance rem-fl (m y) (n (expt 2 n)))
			(:instance rem<n (m x) (n (expt 2 n)))
			(:instance rem<n (m y) (n (expt 2 n)))
			(:instance rem>=0 (m x) (n (expt 2 n)))
			(:instance rem>=0 (m y) (n (expt 2 n)))
			(:instance or-dist-b (x (fl (/ x (expt 2 n)))) (y (rem x (expt 2 n))))
			(:instance or-dist-b (x (fl (/ y (expt 2 n)))) (y (rem y (expt 2 n)))))))))

(local
(defthm or-dist-d-2
    (implies (and (integerp a) (>= a 0)
		  (integerp b) (>= b 0)
		  (integerp c) (>= c 0)
		  (integerp d) (>= d 0))
	     (= (logior (logior a b) (logior c d))
		(logior (logior a c) (logior b d))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance bit-basic-f (x (logior a b)) (y c) (z d))
			(:instance bit-basic-f (x a) (y b) (z c))
			(:instance bit-basic-d (x b) (y c))
			(:instance bit-basic-f (x a) (y c) (z b))
			(:instance bit-basic-f (x (logior a c)) (y b) (z d)))))))

(defthm FL-NONNEG
    (implies (and (integerp n)
		  (rationalp x)
		  (>= x 0))
	     (not (< (* (EXPT 2 N) (FL (* X (/ (EXPT 2 N))))) 0)))
  :hints (("Goal" :in-theory (disable expt-pos)
		  :use ((:instance expt-pos (x n))
			(:instance n<=fl-linear (x (* X (/ (EXPT 2 N)))) (n 0))))))

(local
(defthm or-dist-d-3
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp n) (> n 0))
	     (= (logior x y)
		(logior (logior (* (expt 2 n) (fl (/ x (expt 2 n))))
				(* (expt 2 n) (fl (/ y (expt 2 n)))))
			(logior (rem x (expt 2 n))
				(rem y (expt 2 n))))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance rem>=0 (m x) (n (expt 2 n)))
			(:instance rem>=0 (m y) (n (expt 2 n)))
			(:instance or-dist-d-1)
			(:instance or-dist-d-2
				   (a (* (expt 2 n) (fl (/ x (expt 2 n)))))
				   (b (rem x (expt 2 n)))
				   (c (* (expt 2 n) (fl (/ y (expt 2 n)))))
				   (d (rem y (expt 2 n)))))))))

(local
(defthm or-dist-d-4
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp n) (> n 0))
	     (= (logior x y)
		(+ (* (expt 2 n)
		      (logior (fl (/ x (expt 2 n)))
			      (fl (/ y (expt 2 n)))))
		   (logior (rem x (expt 2 n))
			   (rem y (expt 2 n))))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance or-dist-d-3)
			(:instance or-dist-c (x (fl (/ x (expt 2 n)))) (y (fl (/ y (expt 2 n)))))
			(:instance or-dist-b
				   (x (logior (fl (/ x (expt 2 n)))
					      (fl (/ y (expt 2 n)))))
				   (y (logior (rem x (expt 2 n))
					      (rem y (expt 2 n)))))
			(:instance rem<n (m x) (n (expt 2 n)))
			(:instance rem<n (m y) (n (expt 2 n)))
			(:instance or-dist-a (x (rem x (expt 2 n))) (y (rem y (expt 2 n))))
			(:instance rem>=0 (m x) (n (expt 2 n)))
			(:instance rem>=0 (m y) (n (expt 2 n))))))))

(local
(defthm or-dist-d-5
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp n) (> n 0))
	     (= (rem (logior x y) (expt 2 n))
		(rem (logior (rem x (expt 2 n)) (rem y (expt 2 n)))
		     (expt 2 n))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance or-dist-d-4)
			(:instance rem+ 
				   (m (logior (rem x (expt 2 n)) (rem y (expt 2 n))))
				   (n (expt 2 n))
				   (a (logior (fl (/ x (expt 2 n)))
					      (fl (/ y (expt 2 n))))))
			(:instance n<=fl-linear (x (/ x (expt 2 n))) (n 0))
			(:instance n<=fl-linear (x (/ y (expt 2 n))) (n 0))
			(:instance logior-nat (i (fl (/ x (expt 2 n)))) (j (fl (/ y (expt 2 n)))))
			(:instance logior-nat (i (rem x (expt 2 n))) (j (rem y (expt 2 n))))
			(:instance rem>=0 (m x) (n (expt 2 n)))
			(:instance rem>=0 (m y) (n (expt 2 n))))))))

(defthm OR-DIST-D
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp n) (> n 0))
	     (= (rem (logior x y) (expt 2 n))
		(logior (rem x (expt 2 n)) (rem y (expt 2 n)))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance or-dist-d-5)
			(:instance rem< 
				   (m (logior (rem x (expt 2 n)) (rem y (expt 2 n))))
				   (n (expt 2 n)))
			(:instance rem>=0 (m x) (n (expt 2 n)))
			(:instance rem>=0 (m y) (n (expt 2 n)))
			(:instance logior-nat (i (rem x (expt 2 n))) (j (rem y (expt 2 n))))
			(:instance rem<n (m x) (n (expt 2 n)))
			(:instance rem<n (m y) (n (expt 2 n)))
			(:instance or-dist-a (x (rem x (expt 2 n))) (y (rem y (expt 2 n))))))))

(local
(defthm and-dist-a-1
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0))
	     (<= (logand (rem x 2) (rem y 2)) (rem x 2)))
  :rule-classes ()
  :hints (("Goal" :use ((:instance rem012)
			(:instance rem012 (x y)))))))

(defthm AND-DIST-A
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0))
	     (<= (logand x y) x))
  :rule-classes ()
  :hints (("Goal" :induct (log-induct x y))
	  ("Subgoal *1/2" :use ((:instance and-dist-a-1)
				(:instance rem-fl (m (logand x y)) (n 2))
				(:instance rem-fl (m x) (n 2))))))

(local
(defthm and-dist-b-1
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp n) (> n 0))
	     (= (logand (* (expt 2 n) x) y)
		(* 2 (logand (* (expt 2 (1- n)) x)
			     (fl (/ y 2))))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance logand-def (x (* (expt 2 n) x)))
			(:instance rem-2*i (i (* (expt 2 (1- n)) x))))))))

(local
(defthm and-dist-b-2
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp n) (> n 0)
		  (= (logand (* (expt 2 (1- n)) x) (fl (/ y 2))) 
		     (* (expt 2 (1- n)) (logand x (fl (/ (fl (/ y 2)) (expt 2 (1- n))))))))
	     (= (logand (* (expt 2 n) x) y)
		(* 2 
		   (* (expt 2 (1- n))
		      (logand x
			      (fl (/ (fl (/ y 2)) (expt 2 (1- n)))))))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance and-dist-b-1))))))

(local
(defthm and-dist-b-3
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp n) (> n 0)
		  (= (logand (* (expt 2 (1- n)) x) (fl (/ y 2))) 
		     (* (expt 2 (1- n)) (logand x (fl (/ (fl (/ y 2)) (expt 2 (1- n))))))))
	     (= (logand (* (expt 2 n) x) y)
		(* 2 
		   (* (expt 2 (1- n))
		      (logand x
			      (fl (/ y (expt 2 n))))))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance and-dist-b-2)
			(:instance fl/int-rewrite (x (/ y 2)) (n (expt 2 (1- n)))))))))

(defthm AND-DIST-B
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp n) (>= n 0))
	     (= (logand (* (expt 2 n) x) y)
		(* (expt 2 n) (logand x (fl (/ y (expt 2 n)))))))
  :rule-classes ()
  :hints (("Goal" :induct (or-dist-induct y n))
	  ("Subgoal *1/2" :use ((:instance and-dist-b-3)))))

(local
(defthm and-dist-c-1
    (implies (and (integerp x) (>= x 0)
		  (integerp n) (>= n 0))
	     (= x (logior (* (expt 2 n) (fl (/ x (expt 2 n))))
			  (rem x (expt 2 n)))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance rem-fl (m x) (n (expt 2 n)))
			(:instance or-dist-b (x (fl (/ x (expt 2 n)))) (y (rem x (expt 2 n))))
			(:instance rem<n (m x) (n (expt 2 n)))
			(:instance rem>=0 (m x) (n (expt 2 n))))))))

(local
(defthm and-dist-c-2
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp n) (>= n 0))
	     (= (logand x y)
		(logior (logand (* (expt 2 n) (fl (/ x (expt 2 n))))
				y)
			(logand (rem x (expt 2 n))
				y))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance and-dist-c-1)
			(:instance rem>=0 (m x) (n (expt 2 n)))
			(:instance bit-basic-h 
				   (x y)
				   (y (* (expt 2 n) (fl (/ x (expt 2 n)))))
				   (z (rem x (expt 2 n))))
			(:instance bit-basic-c (x (* (expt 2 n) (fl (/ x (expt 2 n))))))
			(:instance bit-basic-c (x (rem x (expt 2 n))))
			(:instance bit-basic-c
				   (x (logior (* (expt 2 n) (fl (/ x (expt 2 n))))
					      (rem x (expt 2 n))))))))))

(local
 (defthm and-dist-c-3
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp n) (>= n 0))
	     (= (logand x y)
		(logior (* (expt 2 n) 
			   (logand (fl (/ x (expt 2 n)))
				   (fl (/ y (expt 2 n)))))
			(logand (rem x (expt 2 n))
				y))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance and-dist-c-2)
			(:instance and-dist-b (x (fl (/ x (expt 2 n))))))))))

(local
(defthm and-dist-c-4
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp n) (>= n 0))
	     (= (logand x y)
		(+ (* (expt 2 n) 
		      (logand (fl (/ x (expt 2 n)))
			      (fl (/ y (expt 2 n)))))
		   (logand (rem x (expt 2 n))
			   y))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance and-dist-c-3)
			(:instance or-dist-b
				   (x (logand (fl (/ x (expt 2 n)))
					      (fl (/ y (expt 2 n)))))
				   (y (logand (rem x (expt 2 n))
					      y)))
			(:instance rem>=0 (m x) (n (expt 2 n)))			
			(:instance rem<n (m x) (n (expt 2 n)))
			(:instance and-dist-a (x (rem x (expt 2 n)))))))))

(defthm AND-DIST-C
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp n) (>= n 0))
	     (= (rem (logand x y) (expt 2 n))
		(logand (rem x (expt 2 n)) y)))
  :rule-classes ()
  :hints (("Goal" :use ((:instance and-dist-c-4)
			(:instance rem+ 
				   (m (logand (rem x (expt 2 n)) y)) 
				   (n (expt 2 n))
				   (a (logand (fl (/ x (expt 2 n))) (fl (/ y (expt 2 n))))))
			(:instance rem>=0 (m x) (n (expt 2 n)))			
			(:instance rem<n (m x) (n (expt 2 n)))
			(:instance and-dist-a (x (rem x (expt 2 n))))
			(:instance rem< (m (logand (rem x (expt 2 n)) y)) (n (expt 2 n)))))))

(defthm AND-DIST-D
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp n) (>= n 0)
		  (< x (expt 2 n)))
	     (= (logand x y)
		(logand x (rem y (expt 2 n)))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance and-dist-c (x y) (y x))
			(:instance bit-basic-c)
			(:instance bit-basic-c (y (rem y (expt 2 n))))
			(:instance and-dist-a)
			(:instance rem< (m (logand x y)) (n (expt 2 n)))))))

(defthm BIT-DIST-A
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp n) (>= n 0))
	     (= (bitn (logand x y) n)
		(logand (bitn x n) (bitn y n))))
  :rule-classes ()
  :hints (("Goal" :induct (op-dist-induct x y n))
	  ("Subgoal *1/1" :use ((:instance logand-rem)
				(:instance bitn-alt-0)
				(:instance bitn-alt-0 (x y))
				(:instance bitn-alt-0 (x (logand x y)))))
	  ("Subgoal *1/2" :use ((:instance bitn-alt-pos (k n))
				(:instance bitn-alt-pos (k n) (x y))
				(:instance bitn-alt-pos (k n) (x (logand x y)))
				(:instance logand-fl)))))

(defthm BIT-DIST-B
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp n) (>= n 0))
	     (= (bitn (logior x y) n)
		(logior (bitn x n) (bitn y n))))
  :rule-classes ()
  :hints (("Goal" :induct (op-dist-induct x y n))
	  ("Subgoal *1/1" :use ((:instance logior-rem)
				(:instance bitn-alt-0)
				(:instance bitn-alt-0 (x y))
				(:instance bitn-alt-0 (x (logior x y)))))
	  ("Subgoal *1/2" :use ((:instance bitn-alt-pos (k n))
				(:instance bitn-alt-pos (k n) (x y))
				(:instance bitn-alt-pos (k n) (x (logior x y)))
				(:instance logior-fl)))))

(defthm AND-BITS-A
    (implies (and (integerp x) (>= x 0)
		  (integerp k) (>= k 0))
	     (= (logand x (expt 2 k))
		(* (expt 2 k) (bitn x k))))
  :rule-classes ()
  :hints (("Goal" :induct (or-dist-induct x k))
	  ("Subgoal *1/1" :use ((:instance logand-def (y 1))
				(:instance rem012)
				(:instance bitn-alt-0)))
	  ("Subgoal *1/2" :use ((:instance logand-def (y (expt 2 k)))
				(:instance rem-2*i (i (expt 2 (1- k))))
				(:instance fl-int (x (expt 2 (1- k))))
				(:instance bitn-alt-pos)))))

(defthm AND-BITS-B
    (implies (and (integerp x) (>= x 0)
		  (integerp k) (>= k 0))
	     (= (logior x (expt 2 k))
		(+ x
		   (* (expt 2 k) 
		      (- 1 (bitn x k))))))
  :rule-classes ()
  :hints (("Goal" :induct (or-dist-induct x k))
	  ("Subgoal *1/1" :use ((:instance logior-def (i x) (j 1))
				(:instance rem012)
				(:instance rem-fl (m x) (n 2))
				(:instance bitn-alt-0)))
	  ("Subgoal *1/2" :use ((:instance logior-def (i x) (j (expt 2 k)))
				(:instance rem-2*i (i (expt 2 (1- k))))
				(:instance rem-fl (m x) (n 2))
				(:instance fl-int (x (expt 2 (1- k))))
				(:instance bitn-alt-pos)))))

(local
 (defthm fl-2**n-1/2
    (implies (and (integerp n) (> n 0))
	     (= (fl (/ (1- (expt 2 n)) 2))
		(1- (expt 2 (1- n)))))
  :rule-classes ()))

(local
(defthm rem-2**n-1/2
    (implies (and (integerp n) (> n 0))
	     (= (rem (1- (expt 2 n)) 2)
		1))
  :rule-classes ()
  :hints (("Goal" :use ((:instance rem-2*i+1 (i (1- (expt 2 (1- n)))))
			(:instance rem012 (x (1- (expt 2 n)))))))))

(local
(defthm and-bits-c-<-0-1
    (implies (and (integerp x) (>= x 0)
		  (integerp n) (> n 0)
		  (< x (expt 2 n)))
	     (= (logand x (- (expt 2 n) 1))
		x))
  :rule-classes ()
  :hints (("Goal" :induct (or-dist-induct x n))
	  ("Subgoal *1/2" :use ((:instance logand-def (y (1- (expt 2 n))))
				(:instance fl-2**n-1/2)
				(:instance rem-fl (m x) (n 2))
				(:instance rem012)
				(:instance rem-2**n-1/2))))))

(local
(defthm and-bits-c-<-0
    (implies (and (integerp x) (>= x 0)
		  (integerp n) (> n 0)
		  (< x (expt 2 n)))
	     (= (logand x (- (expt 2 n) 1))
		(bits x (1- n) 0)))
  :rule-classes ()
  :hints (("Goal" :use ((:instance and-bits-c-<-0-1)
			(:instance rem< (m x) (n (expt 2 n))))))))

(local
(defthm and-bits-c-<-pos-1
    (implies (and (integerp x) (>= x 0)
		  (integerp n) (>= n 0)
		  (< x (expt 2 n))
		  (integerp k) (> k 0)
		  (< k n))
	     (= (logand x (- (expt 2 n) (expt 2 k)))
		(* 2 (logand (fl (/ x 2)) (- (expt 2 (1- n)) (expt 2 (1- k)))))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance logand-def (y (- (expt 2 n) (expt 2 k))))
			(:instance expt-monotone (n k) (m n))
			(:instance rem-2*i (i (- (expt 2 (1- n)) (expt 2 (1- k))))))))))

(local
(defthm and-bits-c-<-pos-2
    (implies (and (integerp x) (>= x 0)
		  (integerp n) (>= n 0)
		  (< x (expt 2 n))
		  (integerp k) (> k 0)
		  (< k n)
		  (= (logand (fl (/ x 2)) (- (expt 2 (1- n)) (expt 2 (1- k))))
		     (* (expt 2 (1- k)) (bits (fl (/ x 2)) (- n 2) (1- k)))))
	     (= (logand x (- (expt 2 n) (expt 2 k)))
		(* (expt 2 k) (bits (fl (/ x 2)) (- n 2) (1- k)))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance and-bits-c-<-pos-1))))))

(local
(defthm and-bits-c-<-pos-3
    (implies (and (integerp x) (>= x 0)
		  (integerp n) (>= n 0)
		  (< x (expt 2 n))
		  (integerp k) (> k 0)
		  (< k n)
		  (= (logand (fl (/ x 2)) (- (expt 2 (1- n)) (expt 2 (1- k))))
		     (* (expt 2 (1- k)) (bits (fl (/ x 2)) (- n 2) (1- k)))))
	     (= (logand x (- (expt 2 n) (expt 2 k)))
		(* (expt 2 k) (fl (/ (rem (fl (/ x 2)) (expt 2 (1- n))) (expt 2 (1- k)))))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance and-bits-c-<-pos-2))))))

(local
(defthm and-bits-c-<-pos-4
    (implies (and (integerp x) (>= x 0)
		  (integerp n) (>= n 0)
		  (< x (expt 2 n)))
	     (= (rem (fl (/ x 2)) (expt 2 (1- n)))
		(fl (/ x 2))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance rem< (m (fl (/ x 2))) (n (expt 2 (1- n))))
			(:instance fl-def-linear (x (/ x 2))))
           :in-theory (disable rem-equal)))))

(local
(defthm and-bits-c-<-pos-5
    (implies (and (integerp x) (>= x 0)
		  (integerp n) (>= n 0)
		  (< x (expt 2 n))
		  (integerp k) (> k 0)
		  (< k n)
		  (= (logand (fl (/ x 2)) (- (expt 2 (1- n)) (expt 2 (1- k))))
		     (* (expt 2 (1- k)) (bits (fl (/ x 2)) (- n 2) (1- k)))))
	     (= (logand x (- (expt 2 n) (expt 2 k)))
		(* (expt 2 k) (fl (/ (fl (/ x 2)) (expt 2 (1- k)))))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance and-bits-c-<-pos-3)
			(:instance and-bits-c-<-pos-4))))))

(local
(defthm and-bits-c-<-pos-6
    (implies (and (integerp x) (>= x 0)
		  (integerp n) (>= n 0)
		  (< x (expt 2 n))
		  (integerp k) (> k 0)
		  (< k n)
		  (= (logand (fl (/ x 2)) (- (expt 2 (1- n)) (expt 2 (1- k))))
		     (* (expt 2 (1- k)) (bits (fl (/ x 2)) (- n 2) (1- k)))))
	     (= (logand x (- (expt 2 n) (expt 2 k)))
		(* (expt 2 k) (fl (/ x (expt 2 k))))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance and-bits-c-<-pos-5)
			(:instance fl/int-rewrite (x (/ x 2)) (n (expt 2 (1- k)))))))))

(local
(defthm and-bits-c-<-pos-7
    (implies (and (integerp x) (>= x 0)
		  (integerp n) (>= n 0)
		  (< x (expt 2 n))
		  (integerp k) (> k 0)
		  (< k n))
	     (= (fl (/ x (expt 2 k)))
		(bits x (1- n) k)))
  :rule-classes ()
  :hints (("Goal" :use ((:instance rem< (m x) (n (expt 2 n))))))))

(local
(defthm and-bits-c-<-pos
    (implies (and (integerp x) (>= x 0)
		  (integerp n) (>= n 0)
		  (< x (expt 2 n))
		  (integerp k) (> k 0)
		  (< k n)
		  (= (logand (fl (/ x 2)) (- (expt 2 (1- n)) (expt 2 (1- k))))
		     (* (expt 2 (1- k)) (bits (fl (/ x 2)) (- n 2) (1- k)))))
	     (= (logand x (- (expt 2 n) (expt 2 k)))
		(* (expt 2 k) (bits x (1- n) k))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance and-bits-c-<-pos-6)
			(:instance and-bits-c-<-pos-7))))))

(local
(defun and-bits-induct (x n k)
  (if (and (integerp k) (>= k 0))
      (if (= k 0)
	  (list x n k)
	(and-bits-induct (fl (/ x 2)) (1- n) (1- k)))
    ())))

(local
(defthm and-bits-c-<
    (implies (and (integerp x) (>= x 0)
		  (integerp n) (>= n 0)
		  (< x (expt 2 n))
		  (integerp k) (>= k 0)
		  (< k n))
	     (= (logand x (- (expt 2 n) (expt 2 k)))
		(* (expt 2 k) (bits x (1- n) k))))
  :rule-classes ()
  :hints (("Goal" :induct (and-bits-induct x n k))
	  ("Subgoal *1/1" :use ((:instance and-bits-c-<-0)))
	  ("Subgoal *1/2" :use ((:instance and-bits-c-<-pos))))))

(local
(defthm and-bits-c-1
    (implies (and (integerp x) (>= x 0)
		  (integerp n) (>= n 0)
		  (integerp k) (>= k 0)
		  (< k n))
	     (= (logand x (- (expt 2 n) (expt 2 k)))
		(logand (rem x (expt 2 n)) (- (expt 2 n) (expt 2 k)))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance and-dist-d (x (- (expt 2 n) (expt 2 k))) (y x))
			(:instance expt-monotone (n k) (m n))
			(:instance bit-basic-c (y (- (expt 2 n) (expt 2 k))))
			(:instance bit-basic-c (x (rem x (expt 2 n))) (y (- (expt 2 n) (expt 2 k)))))))))

(local
(defthm and-bits-c-2
    (implies (and (integerp x) (>= x 0)
		  (integerp n) (>= n 0)
		  (integerp k) (>= k 0)
		  (< k n))
	     (= (logand x (- (expt 2 n) (expt 2 k)))
		(* (expt 2 k) (bits (rem x (expt 2 n)) (1- n) k))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance and-bits-c-1)
			(:instance and-bits-c-< (x (rem x (expt 2 n))))
			(:instance rem<n (m x) (n (expt 2 n)))
			(:instance rem>=0 (m x) (n (expt 2 n))))))))

(defthm AND-BITS-C
    (implies (and (integerp x) (>= x 0)
		  (integerp n) (>= n 0)
		  (integerp k) (>= k 0)
		  (< k n))
	     (= (logand x (- (expt 2 n) (expt 2 k)))
		(* (expt 2 k) (bits x (1- n) k))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance and-bits-c-2)
			(:instance rem-bits (y (rem x (expt 2 n))) (i n) (j k))
			(:instance rem-rem (a n) (b n))))))



(defthm logand-expt-3
    (implies (and (natp x)
		  (natp n)
		  (natp k)
		  (< k n))
	     (= (logand x (- (expt 2 n) (expt 2 k)))
		(* (expt 2 k) (bits x (1- n) k))))
  :rule-classes ()
  :hints (("Goal" :use (and-bits-c))))


;from logxor-lemmas.lisp


(defthm COMP1-COMP1
    (implies (and (integerp n)
		  (integerp x))
	     (= (comp1 (comp1 x n) n)
		x))
  :rule-classes ())

;integerp-expt was here

(local (defthm fl-comp1-1
    (implies (and (integerp n) (>= n k)
		  (integerp k) (>= k 0)
		  (integerp x) (>= x 0) (< x (expt 2 n)))
	     (= (/ (comp1 x n) (expt 2 k))
		(+ (expt 2 (- n k))
		   (/ (- -1 x) (expt 2 k)))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable a10)
		  :use ((:instance expo+ (m k) (n (- n k)))
			(:instance integerp-expt-type (n (- n k)))
			(:instance integerp-expt-type (n k))
			(:instance expt-pos (x k)))))))

(local (defthm fl=
    (implies (equal x y)
	     (equal (fl x) (fl y)))
  :rule-classes ()))

(local (defthm fl-comp1-2
    (implies (and (integerp n) (>= n k)
		  (integerp k) (>= k 0)
		  (integerp x) (>= x 0) (< x (expt 2 n)))
	     (= (fl (/ (comp1 x n) (expt 2 k)))
		(fl (+ (expt 2 (- n k))
		       (/ (- -1 x) (expt 2 k))))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable a10)
		  :use ((:instance fl-comp1-1)
			(:instance fl= 
				   (x (/ (comp1 x n) (expt 2 k))) 
				   (y (+ (expt 2 (- n k))
					 (/ (- -1 x) (expt 2 k))))))))))

(local (defthm fl-comp1-3
    (implies (and (integerp n) (>= n k)
		  (integerp k) (>= k 0)
		  (integerp x) (>= x 0) (< x (expt 2 n)))
	     (= (fl (/ (comp1 x n) (expt 2 k)))
		(+ (expt 2 (- n k))
		   (fl (/ (- -1 x) (expt 2 k))))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable a10)
		  :use ((:instance fl-comp1-2)
			(:instance fl+int-rewrite (n (expt 2 (- n k))) (x (/ (- (- x) 1) (expt 2 k))))
			(:instance integerp-expt-type (n (- n k)))
			(:instance integerp-expt-type (n k))
			(:instance expt-pos (x k)))))))

(defthm FL-COMP1
    (implies (and (integerp n) (>= n k)
		  (integerp k) (>= k 0)
		  (integerp x) (>= x 0) (< x (expt 2 n)))
	     (= (fl (/ (comp1 x n) (expt 2 k)))
		(comp1 (fl (/ x (expt 2 k))) (- n k))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable a10)
		  :use ((:instance fl-comp1-3)
			(:instance integerp-expt-type (n (- n k)))
			(:instance integerp-expt-type (n k))
			(:instance expt-pos (x k))
			(:instance floor-m+1 (m x) (n (expt 2 k)))))))

(local
(defthm rem=rem-1
    (implies (and (integerp a) (>= a 0)
		  (integerp b) (>= b 0)
		  (integerp n) (> n 0)
		  (= (rem a n) (rem b n)))
	     (= (- a (* n (fl (/ a n))))
		(- b (* n (fl (/ b n))))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance rem-fl (m a))
			(:instance rem-fl (m b)))))))

(local
(defthm rem=rem-2
    (implies (and (integerp a) (>= a 0)
		  (integerp b) (>= b 0)
		  (integerp n) (> n 0)
		  (= (rem a n) (rem b n)))
	     (= (- a b) (* n (- (fl (/ a n)) (fl (/ b n))))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance rem=rem-1))))))

(local
(defthm hack-m10
    (implies (and (rationalp a) (rationalp b) (rationalp c) (> b 0) (= a (* b c)))
	     (= (/ a b) c))
  :rule-classes ()))

(local
(defthm rem=rem-3
    (implies (and (integerp a) (>= a 0)
		  (integerp b) (>= b 0)
		  (integerp n) (> n 0)
		  (= (rem a n) (rem b n)))
	     (= (/ (- a b) n) (- (fl (/ a n)) (fl (/ b n)))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance rem=rem-2)
			(:instance hack-m10 (a (- a b)) (b n) (c (- (fl (/ a n)) (fl (/ b n))))))))))

(local (defthm rem=rem
    (implies (and (integerp a) (>= a 0)
		  (integerp b) (>= b 0)
		  (integerp n) (> n 0)
		  (= (rem a n) (rem b n)))
	     (integerp (/ (- a b) n)))
  :rule-classes ()
  :hints (("Goal" :use ((:instance rem=rem-3))))))

(local (defthm rem-comp1-1
    (implies (and (integerp n)
		  (> n 0)
		  (integerp x))
	     (not (integerp (+ (expt 2 (1- n)) (- x) -1/2))))
  :rule-classes ()))

(local (defthm rem-comp1-2
    (implies (and (integerp n)
		  (> n 0)
		  (integerp x))
	     (not (integerp (/ (- (comp1 x n) x) 2))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance expo+ (m (1- n)) (n 1))
			(:instance rem-comp1-1))))))

(defthm REM-COMP1
    (implies (and (integerp n)
		  (> n 0)
		  (integerp x)
		  (>= x 0)
		  (< x (expt 2 n)))
	     (not (= (rem (comp1 x n) 2)
		     (rem x 2))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance rem-comp1-2)
			(:instance rem=rem (a (comp1 x n)) (b x) (n 2))))))

(local (defthm bitn-comp1-1
    (implies (= x y)
	     (= (rem x 2) (rem y 2)))
  :rule-classes ()))

(local (defthm bitn-comp1-2
    (implies (and (integerp n) (>= n k)
		  (integerp k) (>= k 0)
		  (integerp x) (>= x 0) (< x (expt 2 n)))
	     (= (rem (fl (/ (comp1 x n) (expt 2 k))) 2)
		(rem (comp1 (fl (/ x (expt 2 k))) (- n k)) 2)))
  :rule-classes ()
  :hints (("Goal" :use ((:instance fl-comp1)
			(:instance bitn-comp1-1
				   (x (fl (/ (comp1 x n) (expt 2 k))))
				   (y (comp1 (fl (/ x (expt 2 k))) (- n k)))))))))

(local (defthm bitn-comp1-3
    (implies (and (integerp n)
		  (> n 0)
		  (integerp x)
		  (>= x 0)
		  (< x (expt 2 n))
		  (integerp k)
		  (>= k 0)
		  (< k n))
	     (< (/ x (expt 2 k)) (expt 2 (- n k))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable a2 a9 a8 *-strongly-monotonic)
		  :use ((:instance *-strongly-monotonic
				   (x (/ (expt 2 k)))
				   (y x)
				   (y+ (expt 2 n)))
			(:instance expo+ (m (- n k)) (n k))
			(:instance expt-pos (x k)))))))

(local (defthm bitn-comp1-4
    (implies (and (integerp n)
		  (> n 0)
		  (integerp x)
		  (>= x 0)
		  (< x (expt 2 n))
		  (integerp k)
		  (>= k 0)
		  (< k n))
	     (not (= (rem (fl (/ (comp1 x n) (expt 2 k))) 2)
		     (rem (fl (/ x (expt 2 k))) 2))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable comp1)
		  :use ((:instance rem-comp1 (x (fl (/ x (expt 2 k)))) (n (- n k)))
			(:instance bitn-comp1-3)
			(:instance fl-def-linear (x (/ x (expt 2 k))))
			(:instance expt-pos (x k))
			(:instance bitn-comp1-2))))))

(defthm BITN-COMP1
    (implies (and (integerp n)
		  (> n 0)
		  (integerp x)
		  (>= x 0)
		  (< x (expt 2 n))
		  (integerp k)
		  (>= k 0)
		  (< k n))
	     (not (= (bitn (comp1 x n) k)
		     (bitn x k))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance bitn-comp1-4)
			(:instance bitn-def)
			(:instance bitn-def (x (comp1 x n)))))))

(local (defthm logxor-rewrite-1
    (implies (and (integerp n) (> n 1)
		  (integerp x) (>= x 0) (< x (expt 2 n))
		  (integerp y) (>= y 0) (< y (expt 2 n)))
	     (= (logior (logand (fl (/ x 2)) (comp1 (fl (/ y 2)) (1- n)))
			(logand (fl (/ y 2)) (comp1 (fl (/ x 2)) (1- n))))
		(logior (logand (fl (/ x 2)) (fl (/ (comp1 y n) 2)))
			(logand (fl (/ y 2)) (fl (/ (comp1 x n) 2))))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable comp1)
		  :use ((:instance fl-comp1 (k 1))
			(:instance fl-comp1 (k 1) (x y)))))))

(local (defthm integerp-comp1
    (implies (and (integerp n)
		  (integerp x)
		  (> n 0))
	     (integerp (comp1 x n)))
  :rule-classes (:type-prescription)))

(local (defthm comp1-nat
    (implies (and (integerp n) (> n 0)
		  (integerp x) (< x (expt 2 n)))
	     (not (< (comp1 x n) 0)))))

(local (defthm logxor-rewrite-2
    (implies (and (integerp n) (> n 1)
		  (integerp x) (>= x 0) (< x (expt 2 n))
		  (integerp y) (>= y 0) (< y (expt 2 n)))
	     (= (logior (logand (fl (/ x 2)) (fl (/ (comp1 y n) 2)))
			(logand (fl (/ y 2)) (fl (/ (comp1 x n) 2))))
		(logior (fl (/ (logand x (comp1 y n)) 2))
			(fl (/ (logand y (comp1 x n)) 2)))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable logand-fl-rewrite comp1)
		  :use ((:instance logand-fl (y (comp1 y n)))
			(:instance logand-fl (x y) (y (comp1 x n))))))))

(local (defthm logxor-rewrite-3
    (implies (and (integerp n) (> n 1)
		  (integerp x) (>= x 0) (< x (expt 2 n))
		  (integerp y) (>= y 0) (< y (expt 2 n)))
	     (= (logior (logand (fl (/ x 2)) (comp1 (fl (/ y 2)) (1- n)))
			(logand (fl (/ y 2)) (comp1 (fl (/ x 2)) (1- n))))
		(logior (fl (/ (logand x (comp1 y n)) 2))
			(fl (/ (logand y (comp1 x n)) 2)))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance logxor-rewrite-1)
			(:instance logxor-rewrite-2))))))

(local (defthm logxor-rewrite-4
    (implies (and (integerp n) (> n 1)
		  (integerp x) (>= x 0) (< x (expt 2 n))
		  (integerp y) (>= y 0) (< y (expt 2 n)))
	     (= (logior (fl (/ (logand x (comp1 y n)) 2))
			(fl (/ (logand y (comp1 x n)) 2)))
		(fl (/ (logior (logand x (comp1 y n))
			       (logand y (comp1 x n)))
		       2))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance logior-fl 
				   (i (logand x (comp1 y n)))
				   (j (logand y (comp1 x n)))))))))

(local (defthm logxor-rewrite-5
    (implies (and (integerp n) (> n 1)
		  (integerp x) (>= x 0) (< x (expt 2 n))
		  (integerp y) (>= y 0) (< y (expt 2 n)))
	     (= (logior (logand (fl (/ x 2)) (comp1 (fl (/ y 2)) (1- n)))
			(logand (fl (/ y 2)) (comp1 (fl (/ x 2)) (1- n))))
		(fl (/ (logior (logand x (comp1 y n))
			       (logand y (comp1 x n)))
		       2))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance logxor-rewrite-3)
			(:instance logxor-rewrite-4))))))

(local (defthm logxor-rewrite-6
    (implies (and (integerp n) (> n 1)
		  (integerp x) (>= x 0) (< x (expt 2 n))
		  (integerp y) (>= y 0) (< y (expt 2 n))
		  (= (logxor (fl (/ x 2)) (fl (/ y 2)))
		     (logior (logand (fl (/ x 2)) (comp1 (fl (/ y 2)) (1- n)))
			     (logand (fl (/ y 2)) (comp1 (fl (/ x 2)) (1- n))))))
	     (= (fl (/ (logxor x y) 2))
		(fl (/ (logior (logand x (comp1 y n))
			       (logand y (comp1 x n)))
		       2))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance logxor-rewrite-5)
			(:instance logxor-fl (i x) (j y)))))))

(local (defthm logxor-rewrite-7
    (implies (and (integerp n)
		  (> n 0)
		  (integerp x)
		  (>= x 0)
		  (< x (expt 2 n)))
	     (= (rem (comp1 x n) 2)
		(comp1 (rem x 2) 1)))
  :rule-classes ()
  :hints (("Goal" :use ((:instance bitn-comp1-4 (k 0))
			(:instance rem012)
			(:instance rem012 (x (comp1 x n))))))))

(local (defthm logxor-rewrite-8
    (implies (and (integerp n) (> n 1)
		  (integerp x) (>= x 0) (< x (expt 2 n))
		  (integerp y) (>= y 0) (< y (expt 2 n)))
	     (= (rem (logxor x y) 2)
		(logior (logand (rem x 2) (comp1 (rem y 2) 1))
			(logand (rem y 2) (comp1 (rem x 2) 1)))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable logxor)
		  :use ((:instance logxor-rem (i x) (j y))
			(:instance rem012)
			(:instance rem012 (x y)))))))

(local (defthm logxor-rewrite-9
    (implies (and (integerp n) (> n 1)
		  (integerp x) (>= x 0) (< x (expt 2 n))
		  (integerp y) (>= y 0) (< y (expt 2 n)))
	     (= (rem (logxor x y) 2)
		(logior (logand (rem x 2) (rem (comp1 y n) 2))
			(logand (rem y 2) (rem (comp1 x n) 2)))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable comp1 logxor)
		  :use ((:instance logxor-rewrite-8)
			(:instance logxor-rewrite-7)
			(:instance logxor-rewrite-7 (x y)))))))

(local (defthm logxor-rewrite-10
    (implies (and (integerp n) (> n 1)
		  (integerp x) (>= x 0) (< x (expt 2 n))
		  (integerp y) (>= y 0) (< y (expt 2 n)))
	     (= (rem (logxor x y) 2)
		(logior (rem (logand x (comp1 y n)) 2)
			(rem (logand y (comp1 x n)) 2))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable logxor)
		  :use ((:instance logxor-rewrite-9)
			(:instance logand-rem (y (comp1 y n)))
			(:instance logand-rem (x y) (y (comp1 x n))))))))

(local (defthm logxor-rewrite-11
    (implies (and (integerp n) (> n 1)
		  (integerp x) (>= x 0) (< x (expt 2 n))
		  (integerp y) (>= y 0) (< y (expt 2 n)))
	     (= (rem (logxor x y) 2)
		(rem (logior (logand x (comp1 y n))
			     (logand y (comp1 x n)))
		     2)))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable logxor)
		  :use ((:instance logxor-rewrite-10)
			(:instance logior-rem (i (logand x (comp1 y n))) (j (logand y (comp1 x n)))))))))

(local (defthm logxor-rewrite-12
    (implies (and (integerp n) (> n 1)
		  (integerp x) (>= x 0) (< x (expt 2 n))
		  (integerp y) (>= y 0) (< y (expt 2 n))
		  (= (logxor (fl (/ x 2)) (fl (/ y 2)))
		     (logior (logand (fl (/ x 2)) (comp1 (fl (/ y 2)) (1- n)))
			     (logand (fl (/ y 2)) (comp1 (fl (/ x 2)) (1- n))))))
	     (= (logxor x y)
		(logior (logand x (comp1 y n))
			(logand y (comp1 x n)))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable logxor)
		  :use ((:instance logxor-rewrite-6)
			(:instance logxor-rewrite-11)
			(:instance rem-fl
				   (m (logxor x y))
				   (n 2))
			(:instance logxor-nat (i x) (j y))
			(:instance rem-fl
				   (m (logior (logand x (comp1 y n))
					      (logand y (comp1 x n))))
				   (n 2)))))))

(defthm LOGXOR-0
    (implies (integerp y)
	     (equal (logxor 0 y) y))
  :hints (("Goal" :in-theory (enable logior logand))))

(defthm LOGXOR-0-2
    (implies (integerp x)
	     (equal (logxor x 0) x))
  :hints (("Goal" :in-theory (enable logior logand))))

(local
(defun logxor-induct (x y n)
  (if (and (integerp n) (>= n 1))
      (if (= n 1)
	  (list x y)
	(logxor-induct (fl (/ x 2)) (fl (/ y 2)) (1- n)))
    ())))

(local (defthm x01
    (implies (and (integerp x)
		  (>= x 0)
		  (< x 2))
	     (or (= x 0) (= x 1)))
  :rule-classes ()))

(defthm LOGXOR-REWRITE
    (implies (and (integerp n) (>= n 1)
		  (integerp x) (>= x 0) (< x (expt 2 n))
		  (integerp y) (>= y 0) (< y (expt 2 n)))
	     (= (logxor x y)
		(logior (logand x (comp1 y n))
			(logand y (comp1 x n)))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable comp1 logxor)
		  :induct (logxor-induct x y n))
	  ("Subgoal *1/2" :use (logxor-rewrite-12))
	  ("Subgoal *1/1" :in-theory (enable comp1)
			  :use ((:instance x01)
				(:instance x01 (x y))
				(:instance fl-comp1 (x 0) (k 1))))))

(local
(defun log-induct-3 (x y z)
  (if (and (integerp x) (>= x 0)
	   (integerp y) (>= y 0)
	   (integerp z) (>= z 0))
      (if (or (= x 0) (= y 0) (= z 0))
	  ()
	(log-induct-3 (fl (/ x 2)) (fl (/ y 2)) (fl (/ z 2))))
    ())))

(local (defthm logxor-fl-rewrite
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0))
	     (equal (fl (* 1/2 (logxor x y)))
		    (logxor (fl (/ x 2)) (fl (/ y 2)))))
  :hints (("Goal" :use ((:instance logxor-fl (i x) (j y)))))))

(local (defthm logxor-rem-rewrite
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0))
	     (equal (rem (logxor x y) 2)
		    (logxor (rem x 2) (rem y 2))))
  :hints (("Goal" :use ((:instance logxor-rem (i x) (j y)))))))

(local (defthm logxor-nat-rewrite
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0))
	     (>= (logxor x y) 0))
  :hints (("Goal" :use ((:instance logxor-nat (i x) (j y)))))))

(local (defthm fl-rem-equal
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (= (fl (/ x 2)) (fl (/ y 2)))
		  (= (rem x 2) (rem y 2)))
	     (= x y))
  :rule-classes ()
  :hints (("Goal" :use ((:instance rem-fl (m x) (n 2))
			(:instance rem-fl (m y) (n 2)))))))

(local
(defthm logxor-assoc-1
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp z) (>= z 0))
	     (EQUAL (LOGXOR (LOGXOR (REM X 2) (REM Y 2))
			    (REM Z 2))
		    (LOGXOR (REM X 2)
			    (LOGXOR (REM Y 2) (REM Z 2)))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance rem012)
			(:instance rem012 (x y))
			(:instance rem012 (x z)))))))

(defthm LOGXOR-ASSOC
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp z) (>= z 0))
	     (= (logxor (logxor x y) z)
		(logxor x (logxor y z))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable logxor)
		  :induct (log-induct-3 x y z))
	  ("Subgoal *1/2.1" :use ((:instance logxor-assoc-1)
				  (:instance fl-rem-equal
					      (x (logxor (logxor x y) z))
					      (y (logxor x (logxor y z))))))))

(local
(defthm logxor-x-x-1
    (implies (and (integerp x) (>= x 0))
	     (equal (logxor (rem x 2) (rem x 2)) 0))
  :rule-classes ()
  :hints (("Goal" :use ((:instance rem012))))))

(local
(defun log-induct (x y)
  (if (and (integerp x) (>= x 0)
	   (integerp y) (>= y 0))
      (if (or (= x 0) (= y 0))
	  ()
	(log-induct (fl (/ x 2)) (fl (/ y 2))))
    ())))

(defthm LOGXOR-X-X
    (implies (and (integerp x) (>= x 0))
	     (equal (logxor x x) 0))
  :hints (("Goal" :in-theory (disable logxor)
		  :induct (log-induct x x))
	  ("Subgoal *1/2.1" :use ((:instance logxor-x-x-1)
				  (:instance fl-rem-equal (x 0) (y (logxor x x)))))))

(local
(defun op-dist-induct (x y n)
  (if (and (integerp n) (>= n 0))
      (if  (= n 0)
	  (list x y)
	(op-dist-induct (fl (/ x 2)) (fl (/ y 2)) (1- n)))
    ())))

(in-theory (disable logxor))

(defthm BITN-LOGXOR
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp n) (>= n 0))
	     (= (bitn (logxor x y) n)
		(logxor (bitn x n) (bitn y n))))
  :rule-classes ()
  :hints (("Goal" :induct (op-dist-induct x y n))
	  ("Subgoal *1/1" :use ((:instance logxor-rem (i x) (j y))
				(:instance bitn-alt-0)
				(:instance bitn-alt-0 (x y))
				(:instance bitn-alt-0 (x (logxor x y)))))
	  ("Subgoal *1/2" :use ((:instance bitn-alt-pos (k n))
				(:instance bitn-alt-pos (k n) (x y))
				(:instance bitn-alt-pos (k n) (x (logxor x y)))
				(:instance logxor-fl (i x) (j y))))))

(defthm LOGXOR<2**N
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp n) (>= n 0)
		  (< x (expt 2 n))
		  (< y (expt 2 n)))
	     (< (logxor x y) (expt 2 n)))
  :rule-classes ()
  :hints (("Goal" :induct (op-dist-induct x y n))
	  ("Subgoal *1/2" :use ((:instance logxor-def)
				(:instance rem012)
				(:instance rem012 (x y))))))

(defthm COMP1-BNDS
    (implies (and (integerp n) (> n 0)
		  (integerp x) (>= x 0) (< x (expt 2 n)))
	     (and (< (comp1 x n) (expt 2 n))
		  (>=  (comp1 x n) 0)))
  :rule-classes ())
