/***************************************************************************
 *   Copyright (C) 2003 by Sbastien Laot                                 *
 *   sebastien.laout@tuxfamily.org                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <stdlib.h>

#include <kuniqueapplication.h>
#include <kcmdlineargs.h>
#include <kaboutdata.h>
#include <kiconloader.h>
#include <qpixmap.h>
#include <klocale.h>
#include <kglobalaccel.h>
#include <kmessagebox.h>
#include <kstandarddirs.h>

#include "container.h"
#include "settings.h"
#include "clickcursorfeedback.h"
#include "global.h"
#include "debugwindow.h"
#include "itemdrag.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

static const char description[] = I18N_NOOP("A set of baskets to keep a full range of data on hand.");

static KCmdLineOptions options[] =
{
	{ "d", 0, 0 },
	{ "debug", I18N_NOOP("Show the debug window"), 0 },
	{ 0, 0, 0 }
};

/* Thanks to JuK for this Application class */
#if KDE_IS_VERSION( 3, 1, 90 )
typedef KUniqueApplication Application; // KDE 3.2 and later already re-show the main window
#else
class Application : public KUniqueApplication
{
  public:
	Application() : KUniqueApplication(true, true, false) {}
	virtual ~Application() {}
	virtual int newInstance() {
		if (Global::mainContainer)
			Global::mainContainer->setActive(true);
		return KUniqueApplication::newInstance();
	}
};
#endif

int main(int argc, char *argv[])
{
	/* Application */
	KAboutData aboutData( "basket", I18N_NOOP("BasKet"),
	                      VERSION, description, KAboutData::License_GPL_V2,
	                      "(c) 2003-2005, Sébastien Laoût", 0,
	                      "http://basket.kde.org/",
	                      "slaout@linux62.org"                              );
	aboutData.addAuthor( "Sébastien Laoût", I18N_NOOP("Author, maintenor"), "slaout@linux62.org" );
	aboutData.addAuthor( "Marco Martin",      I18N_NOOP("Icon"),              "m4rt@libero.it"     );

	KCmdLineArgs::init(argc, argv, &aboutData);
	KCmdLineArgs::addCmdLineOptions(options);

	KUniqueApplication::addCmdLineOptions();
	KUniqueApplication app;

	/* Settings */
	Settings::loadConfig();

	/* Debug window */
	if ( KCmdLineArgs::parsedArgs()->isSet("debug") ) {
		new DebugWindow();
		Global::debugWindow->show();
	}

	/* ClickCursorFeedback */
	Global::clickCursorFeedback = new ClickCursorFeedback();

	/* Main container */
	Global::mainContainer = new Container();
	app.setMainWidget(Global::mainContainer);
	Global::mainContainer->tabPlacementChanged(Settings::tabsOnTop());
	QObject::connect( Global::clickCursorFeedback, SIGNAL(shiftChanged(bool)),
	                  Global::mainContainer,       SLOT(feedbackShiftCursor(bool)) );

	/* Self-test of the presence of basketui.rc (the only requiered file after basket executable) */
	if (Global::mainContainer->popupMenu("basket") == 0L) {
		Global::mainContainer->hide();
		KStandardDirs stdDirs;
		KMessageBox::error( Global::mainContainer, i18n(
			"<p><b>The basketui.rc file seems to not exist or is too old.</b><br>"
			"BasKet cannot run without it and will stop.</p>"
			"<p>Please check your installation of BasKet.<br>"
			"If you haven't root access to install the program system wide, you can copy the file basketui.rc "
			"in %1 directory.").arg(stdDirs.saveLocation("data", "basket/")),
			i18n("Ressource not Found") );
		return 1;
	}

	/* System tray icon */
	Global::tray = new ContainerSystemTray(Global::mainContainer);
	if (Settings::useSystray())
		Global::tray->show();

	if ( Settings::useSystray() && app.isRestored())
		Global::mainContainer->setShown( !Settings::startDocked() );
	else
		Global::mainContainer->show();

	// If the main window is hidden when session is saved, Container::queryClose()
	//  isn't called and the last value would be kept
	Settings::setStartDocked(true);

	/* Global shortcuts */
	Global::globalAccel = new KGlobalAccel(Global::mainContainer);
	KGlobalAccel *globalAccel = Global::globalAccel; // Better for the following lines
	globalAccel->insert( "global_show_hide_main_window", i18n("Show/hide main window"),
	                     i18n("Allow you to show main Window if it is hidden, and to hide it if it is shown."),
	                     Qt::CTRL+Qt::SHIFT+Qt::Key_W, Qt::CTRL+Qt::SHIFT+Qt::Key_W,
	                     Global::mainContainer, SLOT(changeActive()),            true, true );
	globalAccel->insert( "global_paste", i18n("Paste clipboard content in current basket"),
	                     i18n("Allow you to paste clipboard content in the current basket without have to open main window."),
	                     Qt::CTRL+Qt::SHIFT+Qt::Key_V, Qt::CTRL+Qt::SHIFT+Qt::Key_V,
	                     Global::mainContainer, SLOT(pasteInCurrentBasket()),    true, true );
	globalAccel->insert( "global_recall_stack", i18n("Recall current stack item to current window"),
	                     i18n("Allow you to paste the current stack item into the active window without have to take the mouse and drag it."),
	                     Qt::CTRL+Qt::ALT+Qt::SHIFT+Qt::Key_V, Qt::CTRL+Qt::ALT+Qt::SHIFT+Qt::Key_V,
	                     Global::mainContainer, SLOT(insertStackItemInCurrentWindow()),true, true );
	globalAccel->insert( "global_rotate_stack", i18n("Rotate items in current stack"),
	                     i18n("Allow you to skip one or more items in a round stack basket."),
	                     Qt::CTRL+Qt::ALT+Qt::SHIFT+Qt::Key_R, Qt::CTRL+Qt::ALT+Qt::SHIFT+Qt::Key_R,
	                     Global::mainContainer, SLOT(rotateCurrentStack()),      true, true );
	globalAccel->insert( "global_clear_stack", i18n("Clear all items in the current stack"),
	                     i18n("Remove all items of the current stack basket."),
	                     "", "",
	                     Global::mainContainer, SLOT(clearCurrentStack()),       true, true );
	globalAccel->insert( "global_show_current_basket", i18n("Show current basket and, if any, the item of the stack"),
	                     i18n("Allow you to know which is the current basket, and in the case of a stack basket, "
	                          "know which will be the next item that will be dragged / recalled."),
	                     "", "",
	                     Global::mainContainer, SLOT(showPassiveContent()),      true, true );
	globalAccel->insert( "global_paste_selection", i18n("Paste selection in current basket"),
	                     i18n("Allow you to paste clipboard selection in the current basket without have to open main window."),
	                     Qt::CTRL+Qt::SHIFT+Qt::Key_S, Qt::CTRL+Qt::SHIFT+Qt::Key_S,
	                     Global::mainContainer, SLOT(pasteSelInCurrentBasket()), true, true );
	globalAccel->insert( "global_new_basket", i18n("Create a new basket"),
	                     i18n("Allow you to create a new basket without have to open main window (you then can use the other global shortcuts to add an item, paste clipboard or paste selection in this new basket)."),
	                     "", "",
	                     Global::mainContainer, SLOT(askNewBasket()),       true, true );
	globalAccel->insert( "global_previous_basket", i18n("Go to previous basket"),
	                     i18n("Allow you to change current basket to the previous one without have to open main window."),
	                     "", "",
	                     Global::mainContainer, SLOT(goToPreviousBasket()), true, true );
	globalAccel->insert( "global_next_basket", i18n("Go to next basket"),
	                     i18n("Allow you to change current basket to the next one without have to open main window."),
	                     "", "",
	                     Global::mainContainer, SLOT(goToNextBasket()),     true, true );


	globalAccel->insert( "global_item_add_text", i18n("Insert text item"),
	                     i18n("Add a text item to the current basket without having to open main window."),
	                     Qt::CTRL+Qt::SHIFT+Qt::Key_T, Qt::CTRL+Qt::SHIFT+Qt::Key_T,
	                     Global::mainContainer, SLOT(addItemText()),        true, true );
	globalAccel->insert( "global_item_add_html", i18n("Insert rich text item"),
	                     i18n("Add a rich text item to the current basket without having to open main window."),
	                     Qt::CTRL+Qt::SHIFT+Qt::Key_H, Qt::CTRL+Qt::SHIFT+Qt::Key_H,
	                     Global::mainContainer, SLOT(addItemHtml()),        true, true );
	globalAccel->insert( "global_item_add_image", i18n("Insert image item"),
	                     i18n("Add a image item to the current basket without having to open main window."),
	                     "", "",
	                     Global::mainContainer, SLOT(addItemImage()),       true, true );
	globalAccel->insert( "global_item_add_link", i18n("Insert link item"),
	                     i18n("Add a link item to the current basket without having to open main window."),
	                     Qt::CTRL+Qt::SHIFT+Qt::Key_Y, Qt::CTRL+Qt::SHIFT+Qt::Key_Y,
	                     Global::mainContainer, SLOT(addItemLink()),        true, true );
	globalAccel->insert( "global_item_add_color", i18n("Insert color item"),
	                     i18n("Add a color item to the current basket without having to open main window."),
	                     "", "",
	                     Global::mainContainer, SLOT(addItemColor()),       true, true );
	globalAccel->insert( "global_item_pick_color", i18n("Pick color from screen"),
	                     i18n("Add a color item picked from one pixel on screen to the current basket without "
	                          "having to open main window."),
	                     "", "",
	                     Global::mainContainer, SLOT(slotColorFromScreenGlobal()),true, true );
	globalAccel->readSettings();
	globalAccel->updateConnections();

	/* Go */
	ItemDrag::createAndEmptyCuttingTmpDir(); // If last exec hasn't done it: clean the temporary directory we will use
	int result = app.exec();
	Settings::setMainWindowPosition(Global::mainContainer->pos());
	Settings::setMainWindowSize(Global::mainContainer->size());
	Settings::saveConfig();
	delete Global::mainContainer; // We do it explicitly here, because the DesktopColorPicker need to be deleted to not crash!
	ItemDrag::createAndEmptyCuttingTmpDir(); // Clean the temporary directory we used
	return result;
}
