/* cmd-resolved.c
 *
 ****************************************************************
 * Copyright (C) 2004 Canonical Ltd.
 * Originally written by James Blackwell
 *
 * See the file "COPYING" for further information about
 * the copyright and warranty status of this work.
 */


#include "config-options.h"
#include "hackerlab/cmd/main.h"
#include "libarch/project-tree.h"
#include "libarch/proj-tree-lint.h"
#include "commands/resolved.h"
#include "commands/version.h"
#include "commands/cmdutils.h"
#include "commands/cmd.h"



static t_uchar * usage = "[options] [package]";

#define OPTS(OP) \
 OP (opt_help_msg, "h", "help", 0, \
     "Display a help message and exit.") \
 OP (opt_long_help, "H", 0, 0, \
    "Display a verbose help message and exit.") \
 OP (opt_version, "V", "version", 0, \
    "Display a release identifier string and exit.") \
 OP (opt_dir, "d", "dir DIR", 1, \
    "Change to DIR first.") \
 OP (opt_all, 0, "all", 0, \
    "Mark all problems as resolved") \
 OP (opt_rejects, 0, "rejects", 0, \
    "Mark rejects as resolved") \
 OP (opt_quiet, "q", "quiet", 0, \
    "Suppress progress information")


t_uchar arch_cmd_resolved_help[] = ("Tell bazaar one or more tree problems have "
                                    "been resolved\n"
                                    "\n"
                                    "This command is used to tell bazaar that one "
                                    "or more tree problems has been solved\n");

enum options
{
OPTS (OPT_ENUM)
};

static struct opt_desc opts[] =
{
  OPTS (OPT_DESC)
    {-1, 0, 0, 0, 0}
};



int
arch_cmd_resolved (t_uchar * program_name, int argc, char * argv[])
{
  int o;
  struct opt_parsed * option;
  t_uchar * dir = 0;
  int quiet = 0;

  int fixed_all = 0;
  int fixed_rejects = 0;

  dir = str_save (0, ".");

  safe_buffer_fd (1, 0, O_WRONLY, 0);

  option = 0;

  while (1)
    {
      if ((argc > 1) && !str_cmp ("--", argv[1]))
        break;

      o = opt_standard (lim_use_must_malloc, &option, opts,
                        &argc, argv, program_name, usage,
                        libarch_version_string, arch_cmd_resolved_help,
                        opt_help_msg, opt_long_help, opt_version);
      if (o == opt_none)
        break;
      switch (o)
        {
        default:
          safe_printfmt (2, "unhandled option `%s'\n", option->opt_string);
          panic ("internal error parsing arguments");

        usage_error:
          opt_usage (2, argv[0], program_name, usage, 1);
          exit (1);

          /* bogus_arg: */
          safe_printfmt (2, "ill-formed argument for `%s' (`%s')\n",
                         option->opt_string, option->arg_string);
          goto usage_error;


        case opt_dir:
            {
              lim_free (0, dir);
              dir = str_save (0, option->arg_string);
              break;
            }

        case opt_quiet:
            {
              quiet = 1;
              break;
            }
        
        case opt_all :
            {
              fixed_all = 1;
              break;
            }

        case opt_rejects :
            {
              fixed_rejects = 1;
              break;
            }
        }
    }

    {


      /* I'm starting to see these same five lines repeated so 
       * many times that I'm starting to think we should make this
       * an inline function
       */
      t_uchar * tree_root = 0;
      if (dir)
        tree_root = arch_tree_root (0, dir, 0);
      else
        tree_root = arch_tree_root (0, ".", 0);

      if (fixed_rejects || fixed_all)
        {
          arch_tree_clear_rejects(tree_root);
        }
      else
	{
	  safe_printfmt (2, "Sorry, baz resolved currently cannot mark individual file conflicts as resolved.  Please run resolved --all once all conflicts are resolved.\n");
	}

      lim_free (0, tree_root);
    }


  return 0;
}
