#!/usr/bin/perl -w
package main ;

########################### Canvas Stuff #############################
=head1 NAME - CanvasBloksi

=head1 DESCRIPTION

Gnome canvas objects for bloksi

=head1 SYNOPSIS

=head1 C<$piece = canvas_piece_new ($root, $blok, $name, [$w,$h],$connect)>

Returns a C<Gnome::CanvasGroup> attached to C<$root> (a
C<Gnome::CanvasGroup>) representing a piece of the puzzle. It has
extra keys :

=over 4

=item name :  string : the name (aka label) of the piece.

=item blok : \Bloksi : the Bloksi object the piece belongs to.

=back

The size of each cell of the piece is C<$w> by C<$h> pixels.

If C<$connect> is true (default), then the piece is connected to an
event handler.

=cut
use Data::Dumper;
# Return a CanvasGroup representing a piece
# - $iden is a string "$width,$height,$silhouette", as found in the
#   output of piece_mask()
# - $w,$h is the size of each square.
#
# As a side effect, when $blok->{(fg)?img}->{$name} is defined,
# $blok->{img}->{$name} is set to a Gtk::Gdk::ImlibImage representing
# this piece.
#
# When $blok->{fgimg} is defined, it is important that
# canvas_piece_new be called *first* with the 'initial' state of the
# puzzle, and then in the 'target' state. 
sub canvas_piece_new
{
    my ($root, $blok, $name, $wh,$connect) = @_ ;

    $connect = 1 unless defined $connect ;

    my $m = $blok->{piece}->{$name} =  piece_mask($blok,$name) ;

    my $iden = $m->{iden} ;
    my $pos = $m->{pos} ;
    my ($w,$h)       = @$wh ; 
    my ($xpos,$ypos) = ($pos->[0]*$w,$pos->[1]*$h) ;

    my $piece = $root->new( $root, "Gnome::CanvasGroup") ;
    ## $piece->{ipos} = [$xpos,$ypos] ;

    $piece->{name} = $name ;
    $piece->{blok} = $blok ;

				# Get piece size and silhouette
    my ($pw,$ph,$sil) = $iden =~ /(\d+),(\d+),(.*)/g 
	or warn "canvas_piece: \$iden='$iden' won't do";
    
    ## Fix 2001/07/09
    my (@cols) = $sil =~ /(.{$ph})/g;
    my (@rows) = map {$_} split "", shift @cols;
    while (@cols){
      my $i = 0;
      map {$rows[$i++] .= $_} split "", shift @cols;
    }
    $sil = join "",@rows;
    my ($i,$j);
				# "Margin" width for the pieces
    my ($bdx,$bdy) = ( int(0.1*$w)+1,int(0.1*$h)+1 ) ;
    my @joinleft = (-1) x $ph ;

				# Default color is green
    my @col = (0x00c000ff,0x004000ff,0x008000ff) ;	
    @col = colorval( @{$blok->{color}->{$name}} )
	if defined $blok->{color}->{$name} ;
    my $img ;

    # printf("0x%08x 0x%08x 0x%08x\n",@col) ;
    ## sayif 0,  "",map( {sprintf("0x%08x ",$_)} @col ),"\n";
    my @shape ;
    @shape[ 0..$pw+1 ] = map {[ (0)x($ph+2) ]} 0..$pw+1 ;
    $i = 0 ;
    map {$shape[1+($i % $pw)]->[int($i++/$pw)+1] = ($_ ne ' ')?1:0} split "",$sil;
    my @rect = ();
				# HERE : Clean that code!
    if( defined($img = $blok->{img}->{$name}) ){
				# Load image only once
	if( !defined($blok->{gdk_img}->{$name}) ){	
	    ## print "Loading image '$img' for piece '$name'\n" ;	
	    
	    # From now on, this piece will use this image.
	    $blok->{gdk_img}->{$name} = $img =
	      Gtk::Gdk::ImlibImage->load_image( "$Bloksi::datapath/$img" ) ; 
	}
    }
    if( defined( $blok->{gdk_img}->{$name} ) ) {
	# print "Re-Using  image for piece '$name'\n";
	$img = $blok->{gdk_img}->{$name} ;
    }
    if( !defined($img) && defined($img = $blok->{fgimg}) ){

				# Load image only once
	if( !defined($blok->{gdk_fgimg}) ){
	    # print "Loading foreground image '$img' for piece '$name'\n" ;
	    $blok->{gdk_fgimg} = $img = 
	      Gtk::Gdk::ImlibImage->load_image( "$Bloksi::datapath/$img" ) ;
	} else {
	    $img = $blok->{gdk_fgimg}
	}
	if( ! ref($img) ){
	    warn "Can't load '$blok->{fgimg}'" ;
	    undef $img
	} else {
	    ## image should be cropped when puzzle is in initial position.
	    my $mm = piece_mask($blok,$name,$blok->{initial}) ;
	    my $xx = int($img->rgb_width *$mm->{pos}->[0]/$blok->{size}->[0]);
	    my $yy = int($img->rgb_height*$mm->{pos}->[1]/$blok->{size}->[1]);
				# +1 needed?
	    my $ww = int($img->rgb_width *$pw/$blok->{size}->[0]); 
	    my $hh = int($img->rgb_height*$ph/$blok->{size}->[1]);
	    print "Clipping image for piece '$name'\n";
	    print "Croping ",$img->rgb_width," $pos->[0]/$blok->{size}->[0]\n";
	    print "  yields $xx,$yy,$ww,$hh\n";
	    $img = $img->crop_and_clone_image($xx,$yy,$ww,$hh);

	    # From now on, this piece will use this image.
	    $blok->{gdk_img}->{$name} = $img ; 

	}
    }
 
    if( defined $img ) 
    { 
	my $ww = $img->rgb_width ;
	my $hh = $img->rgb_height ;

	foreach $i ( 1..$pw+1 ){
	    my $xx = $xpos + ($i-1)*$w ;
	    foreach $j (1..$ph+1){
		my $yy = $ypos + ($j-1)*$h ;

		if( $shape[$i]->[$j] ){
		    my $im2 = $img->crop_and_clone_image(
			int( $ww*($i-1)/$pw ), # +1 needed??
			int( $hh*($j-1)/$ph ),
			int( $ww*    $i/$pw )-int( $ww*($i-1)/$pw ),
			int( $hh*    $j/$ph )-int( $hh*($j-1)/$ph ));
		    push @rect, new Gnome::CanvasItem 
			$piece, "Gnome::CanvasImage",
			'image' => $im2,
			'x'     => $xx+$w/2 , 'y'      => $yy+$h/2 ,
			'width' => $w       , 'height' => $h ;
		}
	    }
	}
##    } elsif($poly) {
      } else {
	foreach $i ( 0..$pw ){
	    my @x = map {$xpos+$_} 
	    (($i-1)*$w,($i-1)*$w+$bdx,$i*$w-$bdx,$i*$w,$i*$w+$bdx);
	    foreach $j (0..$ph){
		my @y = map {$ypos+$_} 
		(($j-1)*$h,($j-1)*$h+$bdy,$j*$h-$bdy,$j*$h,$j*$h+$bdy);
		my @z = map {($x[$_],$y[$_])} 0..3 ;
		if( $shape[$i]->[$j] ){
		    # Main rectangle
		    push @rect, new Gnome::CanvasItem
			$piece,"Gnome::CanvasRect",
			x1 => $x[1], x2 => $x[2]-1, 
			y1 => $y[1], y2 => $y[2]-1,
			"fill_color_rgba" => $col[2];

		    # Index in @col and in @x (or @y)
		    # Right ##############################
		    my ($ci,$xi) = $shape[$i+1]->[$j] ? (2,4) : (1,3) ;
		    
		    push @rect, new Gnome::CanvasItem
			$piece,"Gnome::CanvasRect",
			x1 => $x[2]       , x2 => $x[$xi]-1, 
			y1 => $y[1], y2 => $y[2],
			"fill_color_rgba" => $col[$ci];

		    # Bottom #############################
		    ($ci,$xi) = $shape[$i]->[$j+1] ? (2,4) : (1,3) ;
		    push @rect, new Gnome::CanvasItem
			$piece,"Gnome::CanvasRect",
			x1 => $x[1], x2 => $x[2]-1, 
			y1 => $y[2], y2 => $y[$xi]-1,
			"fill_color_rgba" => $col[$ci];
		    
		    # Little triangles from hell    
		    #         constituents | corner| neighborhood map
		    #           x y x y x y| dx dy | 00 01 10 11
		    my @trg = ([0,0,1,1,1,0, -1,-1,  0, 0, 0, 0],
			       [0,0,1,1,0,1, -1,-1,  0, 0, 0, 0],
			       [2,1,3,0,3,1,  1,-1,  1, 0, 1, 0],
			       [2,1,3,0,2,0,  1,-1,  0, 0, 1, 1],
			       [2,2,3,3,3,2,  1, 1,  1, 1, 1, 1],
			       [2,2,3,3,2,3,  1, 1,  1, 1, 1, 1],
			       [0,3,1,2,0,2, -1, 1,  0, 1, 0, 1],
			       [0,3,1,2,1,3, -1, 1,  1, 1, 0, 0], );

		    ## foreach (@shape){print join ", ",@{$_},"\n"} ;

		    foreach (@trg){
			my @a = @$_ ;
			my $k = 1 ;
			my @b = map { $k=1-$k ; 2*$_+ $k } @a[0..5] ;
			
			# Local map of occupied squares
			my @d = ($shape[$i+$a[6]]->[$j+$a[7]] , # Diag
				 $shape[$i+$a[6]]->[$j      ] , # Horiz
				 $shape[$i      ]->[$j+$a[7]]); # Vert
			
			my $c = $d[0] && $d[1] && $d[2] ? 
			    2 : $a[8+$d[1]+2*$d[2]] ;
			
			
			push @rect, new Gnome::CanvasItem
			    $piece,"Gnome::CanvasPolygon",
			    points => [@z[@b]],
			    fill_color_rgba => $col[$c]  ;
		    }
		    # Top ################################
		} else {
		    if( $shape[$i]->[$j+1] ) {

			push @rect, new Gnome::CanvasItem
			    $piece,"Gnome::CanvasRect",
			    x1 => $x[1], x2 => $x[2], 
			    y1 => $y[3], y2 => $y[4],
			    "fill_color_rgba" => $col[0];
		    }    
		    # Left ###############################
		    if( $shape[$i+1]->[$j] ) {

			push @rect, new Gnome::CanvasItem
			    $piece,"Gnome::CanvasRect",
			    x1 => $x[3]       , x2 => $x[4], 
			    y1 => $y[1], y2 => $y[2],
			    "fill_color_rgba" => $col[0];
		    }
		}
	    }
	}
    }

    # If I connect $piece, it receives many events
    #	    $piece->signal_connect("event", 
    #				   sub {print STDERR "Event $piece\n";
    #					piece_event(@_)});
    map { signal_connect $_  "event",
	  sub { piece_event($piece,$_[1]); return(1)} 
    } @rect if @rect && $connect ;

    $piece->show ;
    return $piece;
}				# End canvas_piece_new


# ($rgba_light,$rgba_dark,$rgba) = colorval($r,$g,$b)
sub colorval
{ my ($r,$g,$b) = map {$_ & 0xff} @_ ;
  my ($R,$G,$B) = map { int(0xff - ((0xff-(0xff & $_))/2)) } @_ ;
  (hex(sprintf("%02x%02x%02xff",$R,$G,$B)),
   hex(sprintf("%02x%02x%02xff",$r/2,$g/2,$b/2)),
   hex(sprintf("%02x%02x%02xff",$r,$g,$b)))
}

=head1 C<1 = piece_event($piece,$event)>

Actions to be taken when an event comes to the canvas group
representing a piece. Returns 1.

=over 4

=item button_pressed (if first button)

Set C<$piece->{pressed} = 1> and C<$piece->{ix}> to the position of
the cursor at the moment.

Also sets C<$piece->{mobile}->{$dir}> to the list of pieces that must
be moved to push $piece in the considered direction.

=item button_release (if first button)

Set C<$piece->{pressed} = 0> 

=item motion_notify (if C<$piece->{pressed}>)

Set C<$piece->{pressed} = 0> 

=back

=cut

sub piece_event
{
    my ($piece,$event) = @_ ;

    
    if($event->{type} eq "button_press" and 
       $event->{button} == 1) {
	warn "\tHEY!!!\n\n\tbutton_press when pressed == 1!\n\n\n" if $piece->{pressed};
	if( $piece->{pressed} ){
	    release_callback( $piece ) ;
	    $piece->{pressed} = 0; 
	}
	my $arrow = new Gtk::Gdk::Cursor 52 ;
	# print join ",",keys(%$event),"\n";
	# $piece->grab( ['button-release-mask', 'button-motion-mask'],
	#	      $arrow , $event->{time})
	#    or warn "piece_event : Can't grab cursor\n";
	
	## print join " ,  ",sort( keys(%Gtk::Gdk::Event::)),"\n";
	$piece->{x}  = [ @{$event}{qw/x y/} ] ;
	$piece->{ix} = [ @{$event}{qw/x y/} ] ;

	$piece->{pressed} = 1 ;
	sayif 0,  "PRESSED $piece  $event->{time}\n";

    } elsif($event->{type} eq "button_release" and 
	    $event->{button} == 1) 
    {
	# $piece->ungrab( $event->{time} );
	warn "\tHEY!!!\n\n\tbutton_release when pressed == 0!\n\n\n" unless $piece->{pressed};
	if( $piece->{pressed} ){
	    release_callback( $piece ) ;
	    $piece->{pressed} = 0; 
	    sayif 0,  "RELEASED $piece $event->{time}\n";
	}
    } elsif($event->{type} eq "motion_notify" and 
	    $piece->{pressed} ) 
    {
	my $dx = $event->{x} - $piece->{x}->[0];
	my $dy = $event->{y} - $piece->{x}->[1];
	motion_callback( $piece, $dx,$dy );

    } else {
	# print "Bop\n";
    }
    return 1 ;
}				# End of piece_event()

## piece_mobility ($piece) 
##
## For all $dir in left right down up none,
## 
## sets $piece->{mobile}->{$dir} to a ref to the list of
##    all pieces (including $piece) that are pushed if $piece is moved
##    in direction $dir.  This list is empty if $piece can't be pushed
##    in direction $dir.
##
sub piece_mobility
{
    my $piece = shift ;
    my ($p,$g) = @{$piece}{qw/name blok/};
    ## sayif 0,  "Mobility of $piece, name=$p, $piece->{name}, blok=$g\n";

    foreach (qw(left right down up none)){
	
	## sayif 0,  "Mobility of piece '$p' towards '$_' : ", 0+@{$_} ,"\n";
	$piece->{mobile}->{$_} = [ may_move( $g, $p, $_ ) ]; ;
    }
}

=head1 C<($dx,$dy) = start_motion($piece,$dx,$dy)>

Must be called when "motion_notify" is first received, before the
piece has started to move on the canvas. This function decides in what
direction (C<$piece->{dir}>) the piece will move. C<$piece->{moving}>
is set to ref to list of moving pieces.

=cut

sub start_motion
{
    my ($piece,$dx,$dy) = @_ ;
    my $dirx = $dx ? ( $dx<0 ? "left" : "right" ) : "none" ;
    my $diry = $dy ? ( $dy<0 ? "up"   : "down"  ) : "none" ;

    piece_mobility($piece);	# Set $piece->{mobile}

    $dx = 0 unless @{$piece->{mobile}->{$dirx}} ;
    $dy = 0 unless @{$piece->{mobile}->{$diry}} ;

				# Move only in one direction
    if( abs($dx) > abs($dy) ){
	$dy = 0; $piece->{dir} = $dirx ;
    } elsif( abs($dy) ) { 
	$dx = 0; $piece->{dir} = $diry ;
    } else {
	$dx = $dy = 0 ; $piece->{dir} = "none" ;
    }

    $piece->{moving} = [@{$piece->{mobile}->{ $piece->{dir} }}] ;

    return ($dx,$dy);
}

# motion_dumb ($piece, $dx, $dy)
#
# move pieces on canvas
sub motion_dumb
{
    my $piece = shift ;
    $piece->{x}->[0] += $_[0] ;
    $piece->{x}->[1] += $_[1] ;
    my $blok = $piece->{blok} ;
    my @moving = @{$piece->{moving}} ;
    ## sayif 0,  "Gonna move in direction $_[0],$_[1] :",join ",",@moving,"\n";
    foreach ( @moving ){
	if( defined $blok->{canvasp}->{$_} ){
	    $blok->{canvasp}->{$_}->move( @_ ) ;
	} else {
	    print STDERR "\e[41mWHOAA!!! No canvasitem for '$_'\e[0m\n";
	}
    }
}

sub motion_callback
{
    my ($piece,$dx,$dy) = @_ ;

    return unless $dx or $dy ;
    
    my @d0 = ($dx,$dy);
    ## print "Initial motion : ($dx,$dy) " ; 
				# If piece has not moved yet
    if( $piece->{x}->[0] == $piece->{ix}->[0] &&
	$piece->{x}->[1] == $piece->{ix}->[1] )
    {
	($dx, $dy) = start_motion( $piece, $dx, $dy ) ;
    }
    ## print "Becomes : '$piece->{dir}' ($dx,$dy)\n" ; 

    my ($x,$ix,$blok) = @{$piece}{qw/x ix blok/};

    # @d : motion
    # $i : index of motion direction 0|1 for horiz|vertical
    # $s : sign of motion 1|-1 for inc|decreasing coords
    # $m : maximum motion before changing position (size of block)
    my @d = ($dx,$dy);
    my $i = ($piece->{dir} =~ /up|down/)?  1 : 0 ;
    my $s = ($piece->{dir} =~ /up|left/)? -1 : 1 ;
    my $m = $blok->{blocksize}->[$i] ;	
    
    $d[ ($i+1)%2 ] = 0 ;
				# Check if piece will really move
    return unless $piece->{dir} ne "none" && $d[$i] ;
    
				# ####################################
				# Pushing motion #####################

    warn "YOW!!! $s, $d[$i], '$dir'\n" unless $s*$d[$i] ;

    if( $s * $d[$i] > 0 )
    {
	## print "Pushing  motion i=$i, s=$s\n";

				# Check bounds of motion
	if( abs( $d[$i]+$x->[$i] - $ix->[$i] ) >= $m )
	{
				# Piece is pushed to new position
	    $blok->move( $piece->{name}, $piece->{dir} );
	    $blok->register($piece->{moving},$piece->{dir});
	    ## show_status( $piece->{parent} );
	    my @dd = (0,0) ;
	    $dd[$i] = $s*$m - $x->[$i] + $ix->[$i] ;
	    motion_dumb( $piece, @dd );

	    $d[$i] -= $dd[$i] ;
	    $d[ ($i+1)%2 ] = $d0[ ($i+1)%2 ] ;

	    $piece->{x}->[$i] = $piece->{ix}->[$i] += $s * $m ;

				# Re-do motion from new position
	    # sayif 0,  "Re-calling motion\n";
	    motion_callback( $piece, @d );

	} else {		# Piece is in between positions : push
	    motion_dumb( $piece, @d );
	}
				# ####################################
    } else {			# Piece is pulled back ###############
	
				# More than halfway : push all except
				# $piece (assume convexity : no stuck
				# pieces). 
	sayif 0,  "Pulling back\n" ;
	if( abs($x->[$i]-$ix->[$i]) > $m/2 ){

				# Motion to new position
	    my @dd = (0,0) ;
	    $dd[$i] = $ix->[$i] - $x->[$i] + $s * $m ;

				# Select dropped pieces and draw them
	    my @tmp = grep {!/$piece->{name}/} @{$piece->{moving}} ;

	    map { $blok->{canvasp}->{$_}->move(@dd) ;
		  ## sayif 0,  "Moved '$blok->{canvasp}->{$_}->{name}' by ($dd[0],$dd[1])\n";
	      } @tmp ;
	    ## sayif 0,  "tmp ",join(", ",@tmp),"\n" ;

	    ## sayif 0,  "all pieces ",join(", ",@{$piece->{moving}}),"\n";
	    
				# Update $blok
	    $blok->move_dumb( \@tmp, $piece->{dir} );
	    $blok->register( \@tmp, $piece->{dir} ) ;
	    ## show_status( $piece->{parent} );
	    ## sayif 0,  "Did not move '$piece->{name}'\n";

	    $piece->{moving} = [$piece->{name}] ;
	} 
				# Move backwards $piece (and
				# eventually attached pieces)

				# Check if I'm back to old pos
	if( ($d[$i]+$x->[$i] - $ix->[$i])*($x->[$i] - $ix->[$i]) <= 0 ){
	    
	    my @dd = (0,0);
	    $dd[$i] = $ix->[$i] - $x->[$i] ;
	    motion_dumb( $piece, @dd );

	    $d[$i] -= ($x->[$i] - $ix->[$i]) ;
	    $d[ ($i+1)%2 ] = $d0[ ($i+1)%2 ] ;

				# Re-do motion from new position
	    sayif 0,  "I'm back : still to go : ($d[0],$d[1])\n" ;
	    warn "  failed : $piece->{x}->[$i] != $piece->{ix}->[$i]"
		unless $piece->{x}->[$i] == $piece->{ix}->[$i] ;
	    motion_callback( $piece, @d ) if $d[0] || $d[1] ;

	} else {		# Piece is in between positions : push
	    motion_dumb( $piece, @d );
	}
    }
    show_status() ;
}				# End of motion_callback #############

sub release_callback
{
    my $piece = shift ;
    return unless $piece->{dir} ;

    sayif 0,  "release_callback : '$piece->{dir}'\n";

    if( $piece->{dir} ne "none" ){
	my ($x,$ix,$blok) = @{$piece}{qw/x ix blok/};

	# $i : index of motion direction 0|1 for horiz|vertical
	# $s : sign of motion 1|-1 for inc|decreasing coords
	# $m : maximum motion before changing position (size of block)
	my $i = ($piece->{dir} =~ /up|down/)?  1 : 0 ;
	my $s = ($piece->{dir} =~ /up|left/)? -1 : 1 ;
	my $m = $blok->{blocksize}->[$i] ;	

	my @dd = (0,0) ;

	# Check if I'm more than halfway to new pos
	if( abs( $x->[$i] - $ix->[$i] ) >= $m/2 )
	{	    
	    $dd[$i] = $ix->[$i] - $x->[$i] + $s * $m ; 
	    
	    # Update $blok
	    # $blok->move_dumb( $piece->{moving}, $piece->{dir} );
	    # $blok->register( $piece->{moving}, $piece->{dir} );
	    ## show_status( $piece->{parent} );
	    
	} else {
	    $dd[$i] = $ix->[$i] - $x->[$i] ;
	}

	# Move pieces on canvas
	map( { ## sayif 0,  "Release : move '$_' = $blok->{canvasp}->{$_} in ($dd[0],$dd[1]):\n";
	       print STDERR "No canvasitem for '$_'\n" unless defined $blok->{canvasp}->{$_};
	       $blok->{canvasp}->{$_}->move(@dd) if defined $blok->{canvasp}->{$_};}
	     @{$piece->{moving}} ) 
	    if $dd[$i] ;

	if( abs( $x->[$i] - $ix->[$i] ) >= $m/2 )
	{	    
	    # Update $blok
	    $blok->move_dumb( $piece->{moving}, $piece->{dir} );
	    $blok->register( $piece->{moving}, $piece->{dir} );
	}

    }
    show_status() ;
    delete @{$piece}{qw/moving mobile dir x ix/} ;

}				# End of release_callback()

# canvas_unredo ($blok,$do_func)
#
# From gui, make or remake as many times as are needed so that
# movement count changes. $do_func is "remake" or "unmake";

sub canvas_unredo
{
  my ($blok,$do_func) = @_ ;
  # To bad Perl uses 'redo' ...
  my $action = $do_func eq "remake" ? "redo" : "undo" ;
  
  unless( @{$blok->{$do_func}} ){
    show_status("Can't $action");
    return ;
  }
  
  my $nm = $blok->{$do_func}->[0]->{nmoves} ;
  my $move = 0 ;
  
    while( $nm == $blok->{$do_func}->[0]->{nmoves} && 
	   ($move = $blok->$do_func()) ){
      
      foreach $dir (@{$move->{dir}}) {
		
		my $i = $dir =~ /up|down/ ? 1 : 0 ;
		my @d = (0,0) ;
		$d[$i] = ( $dir =~ /up|left/? -1 : 1 ) *
		  ($do_func eq "unmake" ? -1 : 1) *
			$blok->{blocksize}->[$i] ;
		
		foreach ( @{$move->{moved}} ){
		  $blok->{canvasp}->{$_}->move( @d ) ;
		}
	  }
      last unless @{$blok->{$do_func}} ;
    }
  
  ## FIXME : If
  ## $blok->{$do_func}->[0]->{nmoves} == 0 and ($do_func eq "remake") or
  ## $blok->{$do_func}->[0]->{nmoves} == last and ($do_func eq "unmake")
  ## Canvas should be redrawn in initial/final position
  if (@{$blok->{remake}} xor @{$blok->{unmake}}) {
	print "Should redraw\n";
  } else {
	## print "All ok\n";
  }
  
  show_status($move? "" : "Can't $action") ;
}

=head2 $group = bloksi_root( $blok, $parent )

Returns a group of pieces. 

Each piece's mask, as returned by L<piece_mask()> is stored in
C<$blok->{piece}->{$p}>.

Each piece's canvas item, mask, as returned by L<canvas_piece()> is
stored in C<$blok->{canvasp}->{$p}>.

=cut

sub bloksi_root
{
  ## check_for_poly() ; ## Assume polygons are available
    
    my ($filename,$parent) = @_ ;

    my $canvas = new Gnome::Canvas ;
    
    $canvas->set_scroll_region(0,0,200,200);
    $canvas->set_usize(200,200);

    $canvas->show;
    $canvas->{parent} = $parent if defined $parent ;

    my $croot = $canvas->root() ;

    $canvas->{filename} = $filename ;

    my $blok = $canvas->{blok} = Bloksi::load( $filename ) ;
	## print "s --------\n$blok->{s}\n";
	## print "target ---\n$blok->{target}\n";
    return undef unless defined $blok ;

				# Get size of canvas
    my ($cx1,$cy1,$cx2,$cy2) = $canvas->get_scroll_region() ;
    sayif 0,  "Scroll region is ($cx1,$cx2,$cy1,$cy2)\n" ;

				# size of each square
    my $size = [ int( ($cx2-$cx1)/$blok->{size}->[0]),
	       int( ($cy2-$cy1)/$blok->{size}->[1]) ] ;
    $blok->{blocksize} = $size ;
    sayif 0,  "\$size=[$size->[0],$size->[1]]\n";

				# Load eventual bgimage
    if( defined( $blok->{bgimg} ) ){
	my $bgg = $croot->new($croot, "Gnome::CanvasGroup") ;
	print "bgimage is '$blok->{bgimg}'\n";
	my $bg = load_image Gtk::Gdk::ImlibImage
	    "$Bloksi::datapath/$blok->{bgimg}" ;
	if( 0 && defined $bg ){
	    $blok->{gdk_bgimage} = $bg ;
	    new Gnome::CanvasItem 
		$bgg, "Gnome::CanvasImage",
		'image' => $bg ,
		'x'     => $cx1      , 'y'      => $cx2 ,
		'width' => int( $cx2-$cx1), 'height' => int($cy2-$cy1) ;
	    $bgg->show ;
	  } else {
	    warn "Couldn't open bgimage '$blok->{bgimg}'\n" ;
	  }
  }

    
    foreach (piece_list($blok->{s})){	# Loop over pieces
	$blok->{canvasp}->{$_} = 
	    canvas_piece_new( $croot, $blok, $_ ,$size );
    }

								# Now, draw the target position

    # Cruft : will pass $blok to funcs that use $blok->{s}
    my $tmp = $blok->{s} ;
    $blok->{s} = $blok->{target} ; 
    
    my $curtain = $croot->new( $croot, "Gnome::CanvasGroup") ;
    ## my $root2 = $curtain->root ;
    my $blank = new Gnome::CanvasItem
	$curtain,"Gnome::CanvasRect",
	x1 => $cx1, x2 => $cx2,
	y1 => $cy1, y2 => $cy2,
	fill_color_rgba => 0x808080ff ;
    $curtain->show ;
    $blank->show ;
    $blank->raise_to_top ;
    
    foreach (piece_list($blok->{s})){	# Loop over pieces
	$blok->{targetp}->{$_} = 
	    canvas_piece_new( $curtain, $blok, $_ ,$size, 0 );
    }
    
    $curtain->raise_to_top();
    $curtain->hide ;
    $blok->{s} = $tmp ;
    $canvas->{curtain} = $curtain ;

    return $canvas ;
}				# End bloksi_root

	    				# Reallocate pieces 
sub loop_detected
{
    my ($g,$moving) = @_ ;
    sayif 0,  "---$g---\n";
    my %pieces = %{$g->{canvasp}} ;
    my $newpieces = {} ;
    my %newnames = () ;
    my @a = split "",$g->{s} ;
    my @b = split "",$g->{unmake}->[0]->{s} ;
    my ($a,$b) ;
    foreach $a (@a){
	$b = shift @b ;
	next if( $a !~ /[0-9A-Za-z]/ ||
		 defined($newnames{$a}) );
	sayif 0,  "Piece '$a' is transformed into '$b'\n" unless $a eq $b ;
	$newpieces->{$b} = $pieces{$a} ;
	$newnames{$a} = $b ;
    }
    
    while( ($a,$b) = each(%{$g->{canvasp}}) ){
	$b->{name} = $newnames{$a} ;
	sayif 0,  "Renamed '$b->{name}' to '$a'\n" ;
    }
    $g->{canvasp} = $newpieces ;

    foreach (piece_list($g->{s})){
	print STDERR "\e[41mAfter renaming, '$_' has no canvas item any more\e[0m\n"
	    unless $g->{canvasp}->{$_} ;
    }
    ## print "keys :",join ",",keys(%{$g->{canvasp}}),"\n";
    foreach (0..$#$moving) { $moving->[$_] = $newnames{$moving->[$_]}}
    # while( ($a,$b) = each(%newname) ){
    # $g->{canvasp}->{$a}->{name} = $b ;
    # }
}
1;
