(**************************************************************************)
(*                   Cameleon                                             *)
(*                                                                        *)
(*      Copyright (C) 2002 Institut National de Recherche en Informatique et   *)
(*      en Automatique. All rights reserved.                              *)
(*                                                                        *)
(*      This program is free software; you can redistribute it and/or modify  *)
(*      it under the terms of the GNU General Public License as published by  *)
(*      the Free Software Foundation; either version 2 of the License, or  *)
(*      any later version.                                                *)
(*                                                                        *)
(*      This program is distributed in the hope that it will be useful,   *)
(*      but WITHOUT ANY WARRANTY; without even the implied warranty of    *)
(*      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     *)
(*      GNU General Public License for more details.                      *)
(*                                                                        *)
(*      You should have received a copy of the GNU General Public License  *)
(*      along with this program; if not, write to the Free Software       *)
(*      Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA          *)
(*      02111-1307  USA                                                   *)
(*                                                                        *)
(*      Contact: Maxence.Guesdon@inria.fr                                *)
(**************************************************************************)

type command = string

let rc_dir = Cam_messages.rc_dir

let display_message s = Cam_gui.display_message s 

class type embedded_app =
  object
    (** The box, to pack it in something. *)
    method box : GPack.box

    (** The method to call to close the app.*)
    method close : unit

    (** To perform some initializations with a window. *)
    method init_window : GWindow.window -> unit
  end

let add_embedded_app com f_app = Cam_global.add_embedded_app com f_app

let add_command = Cam_global.add_command

let available_commands = Cam_global.available_commands

let add_config_box = Cam_global.add_config_box

let selected_dir () = !Cam_global.selected_dir ()

type cvs_status = Ocamlcvs.Types.cvs_status =
  | Up_to_date (** The file is identical with the latest revision in the 
                  repository for the branch in use. *)
  | Locally_modified (** You have edited the file, and not yet committed
                        your changes. *)
  | Locally_added (** You have added the file with add, and not yet
                     committed your changes. *)
  | Locally_removed (** You have removed the file with remove, and not
                       yet committed your changes. *)
  | Needs_checkout (** Someone else has committed a newer revision to the
                      repository. The name is slightly misleading; 
                      you will ordinarily use update rather than checkout
                      to get that newer revision. *)
  | Needs_Patch (** Like Needs Checkout, but the CVS server will send a
                   patch rather than the entire file. 
                   Sending a patch or sending an entire file accomplishes
                   the same thing. *)
  | Needs_Merge (** Someone else has committed a newer revision to the 
                   repository, and you have also made modifications to the 
                   file. *)
  | Conflicts_on_merge 
                (** This is like Locally Modified, except that a previous
                   update command gave a conflict. If you have not already
                   done so, you need to resolve the conflict.*)
  | Unknown (** CVS doesn't know anything about this file. For example, 
                   you have created a new file and have not run add. *)


type editor = Cam_types.editor = 
    Efuns
  | Dbforge
  | Emacs
  | XEmacs
  | Report
  | Zoggy
  | Custom_editor of string

type file_type = Cam_types.file_type = 
    {
      mutable ft_regexp_string : string ;
      mutable ft_regexp : Str.regexp ;
      mutable ft_name : string ;
      mutable ft_color : string option;
      mutable ft_edit : editor ;
      mutable ft_templates : string list ;
      mutable ft_binary : bool
    } 

type file = Cam_types.file = 
    {
      f_name : string ; (** relative or absolute file name *)
      f_abs_name : string ; (** absolute *)
      f_date : float ; (** date of last update of the structure *)
      f_date_string : string ; (** cvs date string *)
      f_work_rev : string ; (** cvs working revision *)
      f_rep_rev : string ; (** cvs repository revision *)
      f_status : cvs_status ; (** cvs status *)
      mutable f_type : file_type ; (** type of file *)
    } 

let selected_files () = !Cam_global.selected_files ()

let eval = Cam_misc.execute_command

let background_execute = Cam_exec.background_execute

let background_execute_with_subst = Cam_exec.background_execute_with_subst

class type editor_app = 
  object
    method close : unit
    method reload : unit
    method save : unit
    method changed : bool
  end

let add_editor s f =
  Cam_global.custom_editors := !Cam_global.custom_editors @ [s, f]

let close_file file = Cam_data.data#close_file file


module Options = Options


module View = 
  struct
    class type view = 
      object
	method coerce : GObj.widget
	method display_dir : string option -> unit
      end
	  
    let add_view view label = Cam_global.add_view view label
  end
