(**************************************************************************)
(*                   Cameleon                                             *)
(*                                                                        *)
(*      Copyright (C) 2002 Institut National de Recherche en Informatique et   *)
(*      en Automatique. All rights reserved.                              *)
(*                                                                        *)
(*      This program is free software; you can redistribute it and/or modify  *)
(*      it under the terms of the GNU General Public License as published by  *)
(*      the Free Software Foundation; either version 2 of the License, or  *)
(*      any later version.                                                *)
(*                                                                        *)
(*      This program is distributed in the hope that it will be useful,   *)
(*      but WITHOUT ANY WARRANTY; without even the implied warranty of    *)
(*      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     *)
(*      GNU General Public License for more details.                      *)
(*                                                                        *)
(*      You should have received a copy of the GNU General Public License  *)
(*      along with this program; if not, write to the Free Software       *)
(*      Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA          *)
(*      02111-1307  USA                                                   *)
(*                                                                        *)
(*      Contact: Maxence.Guesdon@inria.fr                                *)
(**************************************************************************)

(** Global data. *)

open Chat_types
open Chat_proto

module M = Chat_messages


let conf = Chat_args.parse () 

let version = Chat_messages.software_version

let localhost = Unix.gethostname () 

let source () =
  (version, conf#id, (localhost, conf#port))


(** {2 Hosts ips} *)

(** Table of (hostname, ip) *)
let hosts = Hashtbl.create 13

let ip_of_host host =
  try
    Hashtbl.find hosts host
  with
    Not_found ->
      let s = 
	try Unix.string_of_inet_addr (Unix.gethostbyname host).Unix.h_addr_list.(0)
	with Not_found -> host
      in
      Hashtbl.add hosts host s;
      s

(** {2 Connections} *)

let (connections : (string * int, Unix.file_descr * string) Hashtbl.t) = Hashtbl.create 13

let connections_list () = 
  Hashtbl.fold (fun _ (sock,_) acc -> sock :: acc) connections []

let socket_info_list () =
  Hashtbl.fold (fun (h,p) (sock,id) acc -> (sock,(id,h,p)) :: acc) connections []

let find_connection (host,port) =
  Hashtbl.find connections (ip_of_host host,port)

let remove_connection (h,p) =
  Hashtbl.remove connections (ip_of_host h,p)

let add_connection adr sock id =
  (
   try 
     let (s,_) = find_connection adr in
     if s <> sock then Unix.close s;
     remove_connection adr 
   with
     Not_found -> ()
  );
  let (h,p) = adr in
  Hashtbl.add connections (ip_of_host h,p) (sock,id)

let is_connected adr = 
  try ignore(find_connection adr); true
  with Not_found -> false

let check_connections () = 
  let check_one (host,port) (sock,_) =
    try ignore (Unix.select [sock] [] [] 0.0)
    with
      Unix.Unix_error (Unix.EWOULDBLOCK,_,_) 
    | Unix.Unix_error (Unix.EAGAIN,_,_)
    | Unix.Unix_error (Unix.EINTR,_,_) -> ()
    | Unix.Unix_error (Unix.EBADF,_,_) ->
	remove_connection (host,port)
  in
  Hashtbl.iter check_one connections

let safe_close fd =
  try Unix.close fd
  with _ -> ()

(** {2 Rooms} *)


let rooms = ref ([] : (id * ((id * host * port) list) * bool) list)



(** {2 Misc} *)

let pred (i,h,p) (i2,h2,p2) = (ip_of_host h,p) = (ip_of_host h2,p2) 
