/*
 * Cheops Network User Interface
 *
 * Copyright (C) 1999, Adtran, Inc.
 * 
 * Distributed under the terms of the GNU GPL
 *
 */

#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <arpa/inet.h>
#include <string.h>
#include <netdb.h>
typedef unsigned int in_addr_t;
#include <asn1.h>
#include <mib.h>
#include <parse.h>
#include <snmp_api.h>
#include <snmp_client.h>
#include <snmp.h>
#include <snmp_impl.h>
#include <sys/time.h>
#include <gtk/gtk.h>
#include "config.h"
#include "cheops.h"

#define MAX_SNMP_RETRANS 3

struct snmp_request {
	struct net_object *no;
	struct net_page *np;
	char **src;
	char **dest;
	int cnt;
	int retrans;
	void *callback_magic;
	void (*callback)(void *);
};

struct snmp_io_callback {
	struct snmp_io_callback *next;
	int fd;
	int id;
	int seen;
};

struct snmp_io_callback *cbs=NULL;

int have_snmp_timeout=0;

void cheops_snmp_init()
{
	snmp_set_save_descriptions(1);
	init_mib();
}


static void snmp_session_init(struct snmp_session *s, 
		       char *hostname, char *community)
{
	memset(s, 0, sizeof(struct snmp_session));
	s->remote_port = SNMP_DEFAULT_REMPORT;
	s->timeout = SNMP_DEFAULT_TIMEOUT; /* In microseconds */
	s->retries = SNMP_DEFAULT_RETRIES;
	s->authenticator = NULL;
	s->peername = hostname;
	s->version = SNMP_VERSION_1; /* Could be 2c or 2p as well */
	s->community=community;
	s->community_len = strlen(community);
	
}

void register_snmp_timeout();

#ifdef SNMP
static int snmp_auto_close(void *data)
{
	struct net_object *no = (struct net_object *)data;
	if (no->ss && (no->snmp_reqcnt < 1)) {
#if 0
		printf("Autoclosing %s\n", no->hostname);
#endif
#if 0
		/* This disrupts monitoring somehow */
		snmp_close(no->ss);
#endif
		no->ss = NULL;
		/* Should we g_free no->snmp ? */
		g_free(no->snmp);
		no->snmp = NULL;
		no->snmp_reqcnt = 0;
	}
	return FALSE;
}

static int snmp_async_callback(int op, struct snmp_session *s, int reqid, 
			        struct snmp_pdu *response, void *magic)
{
	struct snmp_request *sr;
	static char search[256];
	struct variable_list *vars;
	struct snmp_pdu *pdu;
	int x;
	
	sr = (struct snmp_request *)(magic);
	if (!valid_no(sr->np, sr->no)) {
		/* Make sure this is valid */
		g_free(sr->dest);
		g_free(sr);
		return 1;
	}
	if (op == RECEIVED_MESSAGE) {
		if (response->errstat == SNMP_ERR_NOERROR) {
			vars = response->variables;
			while(vars) {
				if (sr->dest) {
					int x=0;
					sprint_objid(search, vars->name, vars->name_length);
					for (x=0;x<sr->cnt;x++)
						if (!strcmp(search, sr->src[x])) break;
					if (x<sr->cnt)
						sprint_value(sr->dest[x], vars->name, vars->name_length, vars);
				}
				vars=vars->next_variable;
			}
			if (sr->callback) 
				sr->callback(sr->callback_magic);
		} else
		if (response->errstat == SNMP_ERR_NOSUCHNAME) {
			vars = response->variables;
			x=1;
			while(vars && x != response->errindex) {
				vars=vars->next_variable;
				x++;
			}
			if (option_snmp_verbose) {
				fprintf(stderr, "snmp warning: peer %s doesnt know: ", sr->no->hostname);
				fprint_objid(stderr, vars->name, vars->name_length);
				fprintf(stderr, "\n");
			}
			if ((pdu = snmp_fix_pdu(response, SNMP_MSG_GET)) != NULL) {
				snmp_send((struct snmp_session *)sr->no->ss, pdu);
				register_snmp_timeout();
				return 1;
			}
		} else 
			if (option_snmp_verbose)
			fprintf(stderr, "snmp warning: error in packet from %s, reason %s\n", 
				sr->no->hostname,
				snmp_errstring(response->errstat));
	} else 
	if (op == TIMED_OUT) { 
		/* Connection will be closed automagically */
		if (sr->retrans++ < MAX_SNMP_RETRANS)
			return 0;
	} else {
		if (option_snmp_verbose)
			snmp_perror("snmp_get");
	}
	sr->no->snmp_reqcnt--;
	if (sr->no->snmp_reqcnt < 0)
		fprintf(stderr, "Request count < 0 on '%s'??\n", sr->no->hostname);
	if (sr->no->snmp_reqcnt < 1)
		gtk_timeout_add(1000, snmp_auto_close, sr->no);
	g_free(sr->dest);
	g_free(sr);
	return 1;
}

static void our_snmp_callback(gpointer data, int source, GdkInputCondition cond)
{
	fd_set fds;
	struct timeval tv;
	tv.tv_sec = 0;
	tv.tv_usec = 0;
	FD_ZERO(&fds);
	FD_SET(source, &fds);
	if (select(source + 1, &fds, NULL, NULL, &tv) > 0)
		snmp_read(&fds);
}

void register_snmp_timeout();

int our_snmp_timeout()
{
	/* Equvalent of the select expiring */
	snmp_timeout();
	have_snmp_timeout=0;
	register_snmp_timeout();
	return FALSE;
}

void register_snmp_timeout()
{
	fd_set fds;
	int fd_cnt;
	int ms;
	int block;
	struct timeval tv;
	
	/* Equivalent of a select timeout */
	
	if (have_snmp_timeout)
		gtk_timeout_remove(have_snmp_timeout);
	have_snmp_timeout=0;
	do {
		FD_ZERO(&fds);
		timerclear (&tv);
		fd_cnt=0;
		block=1;
		snmp_select_info(&fd_cnt, &fds, &tv, &block);
		ms = tv.tv_sec * 1000;
		ms += tv.tv_usec / 1000;
		if (ms)
			have_snmp_timeout = gtk_timeout_add(ms, our_snmp_timeout, NULL);
		else {
			snmp_timeout();
/*
			if (select(fd_cnt, &fds, NULL, NULL, &tv) > 0) {
				snmp_read(&fds); 
			} else {
				snmp_timeout();
			} */
		}
	} while (!block && !ms);
	
}

void register_snmp_callback()
{
	/* This is a big hack to figure out the fd's
	   from the snmp library 
	   
	   FIXME:  Not only is this procedure tacky, but
	   it is the MOST expensive in the entire  program,
	   with an average of 1250 us/call (gprof)
	   
	   */
	int fd = 0;
	int block;
	struct snmp_io_callback *cb, *cb2, *prev=NULL;
	struct timeval tv;
	fd_set fds;
	int fd_cnt=0;

	timerclear (&tv);
	FD_ZERO(&fds);
	snmp_select_info(&fd_cnt, &fds, &tv, &block);
	cb = cbs;
	while(cb) {
		cb->seen = 0;
		cb=cb->next;
	}
	for(fd = 0; fd < fd_cnt; fd++) {
		if (FD_ISSET(fd, &fds)) {
			/* SNMP thinks this is an important fd, so should we */
			cb = cbs;
			while(cb) {
				if (cb->fd == fd)
					break;
				cb=cb->next;
			}
			if (!cb) {
				/* Make a new entry for this feller */
				cb = g_new0(struct snmp_io_callback, 1);
				cb->fd = fd;
				cb->seen = 1;
				cb->id = gdk_input_add(fd, GDK_INPUT_READ, our_snmp_callback, NULL);
				cb->next = cbs;
				cbs = cb;
			} else 	
				cb->seen = 1;

		}	
	}
	cb = cbs;
	while(cb) {
		/* Look for expired callbacks */
		if (!cb->seen) {
			gdk_input_remove(cb->id);
			if (prev) {
				cb2 = cb;
				prev->next = cb->next;
				cb = cb->next;
				g_free(cb2);
			} else {
				cbs = cb->next;
				cb2 = cb;
				cb = cbs;
				g_free(cb2);
			}
		} else {
			prev=cb;
			cb=cb->next;
		}
	}
}

void do_snmp_close(void *data)
{
	snmp_close((struct snmp_session *)data);
	register_snmp_callback();
}

char *snmp_get_async(struct net_object *no, struct net_page *np, char *community, char **objid, int cnt, 
		     char **answers, void (*callback)(void *), void *callback_magic)
{
	/* Modeled after snmp_get...
	   Unfortuantely, the snmp routines don't have an g_snprintf equivalent
	   This version is asynchronous */
	   
	struct snmp_session snmp;
	struct snmp_pdu *pdu;
	struct in_addr ia;
	oid name[256];
	char hn[256];
	int namelen;
	int x;
	struct snmp_request *sr;
	
	sr = g_new0(struct snmp_request, 1);
	if (!no->ss) {
		ia.s_addr = no->ip_addr;
		strncpy(hn, inet_ntoa(ia), sizeof(hn));
		snmp_session_init(&snmp, hn, community);
		snmp.callback = NULL;
		snmp.callback_magic = NULL;
		no->ss = snmp_open(&snmp);
		if (!no->ss) {
			printf("Unable to open session\n");
			return NULL;
		}
		register_snmp_callback();
	}
#if 1	/* change to #if 0 to test checking */
	sr->no = no;
	sr->np = np;
#endif
	sr->retrans = 0;
	sr->cnt=cnt;
	sr->src = objid;
	sr->dest = answers;
	sr->callback = callback;
	sr->callback_magic = callback_magic;
	pdu = snmp_pdu_create(SNMP_MSG_GET);
	for (x=0;x<cnt;x++) {
		if (answers)
			strcpy(answers[x], "");
		namelen = sizeof(name);
		if (read_objid(objid[x], name, &namelen)) {
			snmp_add_null_var(pdu, name, namelen);
		} else {
			fprintf(stderr, "snmp warning: unknown identifier '%s'\n", objid[x]);
			return NULL;
		}
	}
	if (!snmp_async_send((struct snmp_session *)no->ss, pdu, snmp_async_callback, sr)) {
		return NULL;
	}
	no->snmp_reqcnt++;
	register_snmp_timeout();
	return "";
}

#endif

int snmp_walk(struct net_object *no, char *community,
	       void (*callback)(char *, char *, void *), void *data, int *cancel)
{
	/* Modeled after snmp_walk */
	static oid objid_mib[] = {1, 3, 6, 1, 2, 1};
	struct snmp_session session, *ss;
	struct snmp_pdu *pdu, *response;
	struct variable_list *vars;
	oid name[256];
	int namelen;
	int rootlen;
	int running;
	int status;
	int res=0;
	char hn[256];
	char obj[1024];
	char val[1024];
	struct in_addr ia;
	
	ia.s_addr = no->ip_addr;
	strncpy(hn, inet_ntoa(ia), sizeof(hn));
	snmp_session_init(&session, hn, community);
	snmp_synch_setup(&session);
	
	rootlen = sizeof(objid_mib)/sizeof(oid);
	
	
	ss = snmp_open(&session);
	if (!ss)
		return -1;
	running = 1;
	memmove(name, objid_mib, rootlen * sizeof(oid));
	namelen = rootlen;
	while(running) {
		pdu = snmp_pdu_create(SNMP_MSG_GETNEXT);
		snmp_add_null_var(pdu, name, namelen);
		
		status = snmp_synch_response(ss, pdu, &response);
		if (status == STAT_SUCCESS) {
			if (response->errstat == SNMP_ERR_NOERROR) {
				vars=response->variables;
		      		while(vars) {
					if (memcmp(objid_mib, vars->name, rootlen * sizeof(oid))!=0) {
						running = 0;
						break;
					}
#if 0
					print_variable(vars->name, vars->name_length, vars);
#endif
					sprint_objid(obj, vars->name, vars->name_length);
					sprint_value(val, vars->name, vars->name_length, vars);
					if (*cancel) {
						running = 0;
						res = 0;
						break;
					}
					callback(obj, val, data);
					switch(vars->type) {
					case SNMP_ENDOFMIBVIEW:
					case SNMP_NOSUCHOBJECT:
					case SNMP_NOSUCHINSTANCE:
						running=0;
						res=-1;
						break;
					default:
						namelen = vars->name_length;
						memmove(name, vars->name, namelen*sizeof(oid));
					}
					if (!running)	
						break;
		      			vars=vars->next_variable;
				}
			} else {
				/* Error */
				running = 0;
			}
		} else if (status == STAT_TIMEOUT) {
#if 0
			fprintf(stderr, "snmp warning: timeout from '%s'\n", no->hostname);
#endif
			running = 0;
		} else {
			snmp_perror("snmp warning");
			running = 0;
		}
		if (response)
			snmp_free_pdu(response);
	}
	snmp_close(ss);
	return res;
}

char *snmp_get(struct net_object *no, char *community, char **objid, int cnt, char **answers)
{
	/* Modeled after snmp_get...
	   Unfortuantely, the snmp routines don't have an g_snprintf equivalent */
	struct snmp_session session, *ss;
	struct snmp_pdu *pdu;
	struct snmp_pdu *response;
	struct variable_list *vars;
	struct in_addr ia;
	oid name[256];
	char hn[256];
	static char buf[1024];
	static char search[256];
	int namelen;
	int status;
	int x;
	
	ia.s_addr = no->ip_addr;
	strncpy(hn, inet_ntoa(ia), sizeof(hn));
	snmp_session_init(&session, hn, community);
	snmp_synch_setup(&session);
	ss = snmp_open(&session);
	if (!ss) {
		printf("Unable to open session\n");
		return NULL;
	}
	pdu = snmp_pdu_create(SNMP_MSG_GET);
	for (x=0;x<cnt;x++) {
		if (answers)
			strcpy(answers[x], "");
		namelen = sizeof(name);
		if (read_objid(objid[x], name, &namelen)) {
			snmp_add_null_var(pdu, name, namelen);
		} else {
			fprintf(stderr, "snmp warning: unknown identifier '%s'\n", objid[x]);
			return NULL;
		}
	}
	strcpy(buf, "");
retry:
	status = snmp_synch_response(ss, pdu, &response);
	if (status == STAT_SUCCESS) {
		if (response->errstat == SNMP_ERR_NOERROR) {
			vars = response->variables;
			while(vars) {
#if 0
				print_variable(vars->name, vars->name_length, vars);
#endif
				if (answers) {
					int x=0;
					sprint_objid(search, vars->name, vars->name_length);
					for (x=0;x<cnt;x++)
						if (!strcmp(search, objid[x])) break;
					if (x<cnt)
						sprint_value(answers[x], vars->name, vars->name_length, vars);
				} else {
					sprint_objid(search, vars->name, vars->name_length);
					if (!strcmp(search, objid[0]))
						sprint_value(buf, vars->name, vars->name_length, vars);
				}
				vars=vars->next_variable;
			}					
		} else
		if (response->errstat == SNMP_ERR_NOSUCHNAME) {
			vars = response->variables;
			x=1;
			while(vars && x != response->errindex) {
				vars=vars->next_variable;
				x++;
			}
			fprintf(stderr, "snmp warning: peer doesnt know: ");
			fprint_objid(stderr, vars->name, vars->name_length);
			fprintf(stderr, "\n");
			if ((pdu = snmp_fix_pdu(response, SNMP_MSG_GET)) != NULL)
				goto retry;
		} else 
			fprintf(stderr, "snmp warning: error in packet, reason %s\n", 
				snmp_errstring(response->errstat));
	} else 
	if (status == STAT_TIMEOUT) { 
		fprintf(stderr, "snmp warning: timeout from '%s'\n", no->hostname);
		snmp_close(ss);
		return NULL;
	} else {
		snmp_perror("snmp_get");
		snmp_close(ss);
		fprintf(stderr, "snmp warning: general error on snmp\n");
		return NULL;
	}
	if (response)
		snmp_free_pdu(response);
	snmp_close(ss);
	return buf;
}

#ifdef NEED_SNMP_ADD_VAR
/*
 * Add a variable with the requested name to the end of the list of
 * variables for this pdu. (snmp-set.c)
 */
void
snmp_add_var(
    struct snmp_pdu *pdu,
    oid *name,
    int name_length,
    int type,
    char *value)
{
    struct variable_list *vars;
    char buf[2048];

    if (pdu->variables == NULL){
      pdu->variables = vars =
            (struct variable_list *)malloc(sizeof(struct variable_list));
    } else {
      for(vars = pdu->variables;
            vars->next_variable;
            vars = vars->next_variable)
        ;

      vars->next_variable =
            (struct variable_list *)malloc(sizeof(struct variable_list));
      vars = vars->next_variable;
    }

    vars->next_variable = NULL;
    vars->name = (oid *)malloc(name_length * sizeof(oid));
    memmove(vars->name, name, name_length * sizeof(oid));
    vars->name_length = name_length;
    vars->type = type;
    switch(type){
    case ASN_INTEGER:
      vars->val.integer = (long *)malloc(sizeof(long));
      *(vars->val.integer) = atol(value);
      vars->val_len = sizeof(long);
      break;

    case ASN_UNSIGNED:
      vars->type = ASN_UNSIGNED;
      vars->val.integer = (long *)malloc(sizeof(long));
      sscanf(value, "%lu", vars->val.integer);
      vars->val_len = sizeof(long);
      break;

    case ASN_TIMETICKS:
      vars->val.integer = (long *)malloc(sizeof(long));
      sscanf(value, "%lu", vars->val.integer);
      vars->val_len = sizeof(long);
      break;

    case ASN_IPADDRESS:
      vars->val.integer = (long *)malloc(sizeof(long));
      *(vars->val.integer) = inet_addr(value);
      vars->val_len = sizeof(long);
      break;

    case ASN_OBJECT_ID:
      vars->val_len = MAX_NAME_LEN;
      read_objid(value, (oid *)buf, &vars->val_len);
      vars->val_len *= sizeof(oid);
      vars->val.objid = (oid *)malloc(vars->val_len);
            memmove(vars->val.objid, buf, vars->val_len);
      break;

    case ASN_OCTET_STR:
      vars->type = ASN_OCTET_STR;
      strcpy(buf, value);
      vars->val_len = strlen(buf);
      if (vars->val_len < 0) {
        fprintf (stderr, "snmp_add_var: Bad value: %s\n", value);
        vars->val_len = 0;
      }
      vars->val.string = (u_char *)malloc(vars->val_len);
      memmove(vars->val.string, buf, vars->val_len);
      break;

    case ASN_NULL:
      vars->val_len = 0;
      vars->val.string = NULL;
      break;
  }
}
#endif

char *snmp_set(struct net_object *no, char *community, char **objid, char **values, int *type, int cnt, char **answers)
{
	/* Modeled after snmp_get...
	   Unfortuantely, the snmp routines don't have an g_snprintf equivalent */
	struct snmp_session session, *ss;
	struct snmp_pdu *pdu;
	struct snmp_pdu *response;
	struct variable_list *vars;
	struct in_addr ia;
	oid name[256];
	char hn[256];
	static char buf[1024];
	static char search[256];
	int namelen;
	int status;
	int x;
	
	ia.s_addr = no->ip_addr;
	strncpy(hn, inet_ntoa(ia), sizeof(hn));
	snmp_session_init(&session, hn, community);
	snmp_synch_setup(&session);
	ss = snmp_open(&session);
	if (!ss) {
		printf("Unable to open session\n");
		return NULL;
	}
	pdu = snmp_pdu_create(SNMP_MSG_SET);
	for (x=0;x<cnt;x++) {
		if (answers)
			strcpy(answers[x], "");
		namelen = sizeof(name);
		if (read_objid(objid[x], name, &namelen)) {
			snmp_add_var(pdu, name, namelen, type[x], values[x]);
		} else {
			fprintf(stderr, "snmp warning: unknown identifier '%s'\n", objid[x]);
			return NULL;
		}
	}
	strcpy(buf, "");
retry:
	status = snmp_synch_response(ss, pdu, &response);
	if (status == STAT_SUCCESS) {
		if (response->errstat == SNMP_ERR_NOERROR) {
			vars = response->variables;
			while(vars) {
#if 0
				print_variable(vars->name, vars->name_length, vars);
#endif
				if (answers) {
					int x=0;
					sprint_objid(search, vars->name, vars->name_length);
					for (x=0;x<cnt;x++)
						if (!strcmp(search, objid[x])) break;
					if (x<cnt)
						sprint_value(answers[x], vars->name, vars->name_length, vars);
				} else {
					sprint_objid(search, vars->name, vars->name_length);
					if (!strcmp(search, objid[0]))
						sprint_value(buf, vars->name, vars->name_length, vars);
				}
				vars=vars->next_variable;
			}					
		} else
		if (response->errstat == SNMP_ERR_NOSUCHNAME) {
			vars = response->variables;
			x=1;
			while(vars && x != response->errindex) {
				vars=vars->next_variable;
				x++;
			}
			fprintf(stderr, "snmp warning: peer doesnt know: ");
			fprint_objid(stderr, vars->name, vars->name_length);
			fprintf(stderr, "\n");
			if ((pdu = snmp_fix_pdu(response, SNMP_MSG_SET)) != NULL)
				goto retry;
		} else 
			fprintf(stderr, "snmp warning: error in packet, reason %s\n", 
				snmp_errstring(response->errstat));
	} else 
	if (status == STAT_TIMEOUT) { 
		fprintf(stderr, "snmp warning: timeout from '%s'\n", no->hostname);
		snmp_close(ss);
		return NULL;
	} else {
		snmp_perror("snmp_get");
		snmp_close(ss);
		fprintf(stderr, "snmp warning: general error on snmp\n");
		return NULL;
	}
	if (response)
		snmp_free_pdu(response);
	snmp_close(ss);
	return buf;
}

#ifndef SNMP
int main(int argc, char *argv[])
{
	struct net_object no;
	char *res[5];
	char *req[5];
	int x;
	req[0] = "system.sysName.0";
	req[1] = "system.sysDescr.0";
	req[2] = "system.sysObjectID.0";
	req[3] = "system.sysContact.0";
	req[4] = "system.sysUpTime.0";
	inet_aton("127.0.0.1", &no.ip_addr);
	strcpy(no.hostname, "localhost");
	for (x=0;x<5;x++)
		res[x] = malloc(200);
	snmp_get(&no, "public", req, 5, res);
	for (x=0;x<5;x++) 
		printf("Result for %s is %s\n", req[x], res[x]);
	snmp_walk(&no, "public");
	return 0;
}
#endif
