/**************************************************************************/
/*                                                                        */
/*  cpbk - a mirroring utility for backing up your files                  */
/*  Copyright (C) 1998 Kevin Lindsay <klindsay@mkintraweb.com>            */
/*  Copyright (C) 2001 Yuuki NINOMIYA <gm@debian.or.jp>                   */
/*                                                                        */
/*  This program is free software; you can redistribute it and/or modify  */
/*  it under the terms of the GNU General Public License as published by  */
/*  the Free Software Foundation; either version 2, or (at your option)   */
/*  any later version.                                                    */
/*                                                                        */
/*  This program is distributed in the hope that it will be useful,       */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of        */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         */
/*  GNU General Public License for more details.                          */
/*                                                                        */
/*  You should have received a copy of the GNU General Public License     */
/*  along with this program; if not, write to the                         */
/*  Free Software Foundation, Inc., 59 Temple Place - Suite 330,          */
/*  Boston, MA 02111-1307, USA.                                           */
/*                                                                        */
/**************************************************************************/

/* $Id: trashbin.c,v 1.9 2001/03/30 08:48:59 gm Exp $ */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <errno.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <utime.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "intl.h"
#include "strlib.h"
#include "variable.h"
#include "proto.h"


static void check_trashbin_dir(void);
static int mkdir_parents_trash(const char *dest_path_dir, const char *trash_path_dir);


/* --- PUBLIC FUNCTIONS --- */

void prepare_trashbin_dir(void)
{
	char *temp;

	if (config.trashbin_dir == NULL) {
		return;
	}

	if (!config.simulate) {
		check_trashbin_dir();
	}

	temp = str_dup(config.trashbin_dir);
	config.trashbin_dir = canonicalize_path(temp, _("trash bin"));
}


void trash_it(Action act, const char *path_file)
{
	char *trash_path_file;
	struct stat status;
	Order order;

	if (config.simulate || config.trashbin_dir == NULL) {
		return;
	}

	if (strstr(path_file, config.dest_dir) == NULL) {
		err_printf(CAN_PROCEED, NULL, _("Path name `%s' doesn't contain destination path.\n"));
		internal_error(__FILE__, __LINE__);
	}

	trash_path_file = str_concat(config.trashbin_dir, "/", path_file + strlen(config.dest_dir) + 1, NULL);

	mkdir_parents_trash(path_file, trash_path_file);
	if (act == RMDIR) {
		return;
	}

	if (lstat(path_file, &status) == -1) {
		err_printf(CAN_PROCEED, "lstat", _("Cannot get file status: %s\n"), path_file);
		return;
	}
	order.status = status;

	if (link(path_file, trash_path_file) == -1) {
		copy_file(&order, path_file, trash_path_file);
	}
}


/* --- PRIVATE FUNCTIONS --- */

static void check_trashbin_dir(void)
{
	struct stat status;

	if (lstat(config.trashbin_dir, &status) == -1) {
		if (errno == ENOENT) {
			if (config.verbose) {
				printf(_("Trash bin directory `%s' does not exist. Making it.\n"), config.trashbin_dir);
			}
			if (mkdir_parents(config.trashbin_dir, 0700) == -1) {
				err_printf(ALWAYS_EXIT, NULL, _("Cannot make trash bin directory `%s'.\n"), config.trashbin_dir);
			}
		} else {
			err_printf(ALWAYS_EXIT, "lstat", _("Cannot get file status: %s\n"), config.trashbin_dir);
		}
	} else if (!S_ISDIR(status.st_mode)) {
		err_printf(ALWAYS_EXIT, NULL, _("Trash bin directory `%s' already exists, but it is not a valid directory.\n"), config.trashbin_dir);
	}
	if (config.verbose) {
		printf(_("`%s' will be used as the trash bin.\n"), config.trashbin_dir);
	}
}


static int mkdir_parents_trash(const char *dest_path_dir, const char *trash_path_dir)
{
	char *dest_temp;
	char *trash_temp;
	char *dest_dirptr;
	char *trash_dirptr;
	struct stat status;
        struct utimbuf newtime;

	if (access(trash_path_dir, F_OK) == 0) {
		return (0);
	}

	dest_temp = str_dup(dest_path_dir);
	trash_temp = str_dup(trash_path_dir);
	
	dest_dirptr = dest_temp + strlen(config.dest_dir) + 1;
	trash_dirptr = trash_temp + strlen(config.trashbin_dir) + 1;

	while (trash_dirptr != NULL) {
		if ((trash_dirptr = strchr(trash_dirptr, '/')) != NULL) {
			*trash_dirptr = '\0';
		}
		if ((dest_dirptr = strchr(dest_dirptr, '/')) != NULL) {
			*dest_dirptr = '\0';
		}
		if (access(trash_temp, F_OK) == -1) {
		        if (lstat(dest_temp, &status) == -1) {
				err_printf(CAN_PROCEED, "lstat", _("Cannot get file status: %s\n"), dest_temp);
				return (-1);
			}
			if (!S_ISDIR(status.st_mode)) {
				return (0);
			}
			if (mkdir(trash_temp, status.st_mode) == -1) {
				err_printf(CAN_PROCEED, "mkdir", NULL);
				return (-1);
			}
			if (am_i_root()) {
				if (lchown(trash_temp, status.st_uid, status.st_gid) == -1) {
					err_printf(CAN_PROCEED, "lchown", _("Cannot change ownership of directory `%s'.\n"), trash_temp);
				}
			}

			/* Set time on directory to same as source directory */
			newtime.actime = status.st_atime;
			newtime.modtime = status.st_mtime;

			if (utime(trash_temp, &newtime) == -1) {
				err_printf(CAN_PROCEED, "utime", _("Cannot change access and modification times of an inode: %s.\n"), trash_temp);
			}
		}
		if (trash_dirptr != NULL) {
			*trash_dirptr = '/';
			trash_dirptr++;
		}
		if (dest_dirptr != NULL) {
			*dest_dirptr = '/';
			dest_dirptr++;
		}
	}
	return (0);
}
