/* MacOSXPlayer.m - this file is part of Cynthiune
 *
 * Copyright (C) 2002-2004  Wolfgang Sourdeau
 *
 * Author: Wolfgang Sourdeau <wolfgang@contre.com>
 *
 * This file is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This file is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#import <Foundation/Foundation.h>

#import <CoreAudio/CoreAudio.h>
#import <CoreAudio/AudioHardware.h>
#import <AudioUnit/AudioUnit.h>
#import <AudioUnit/AudioUnitProperties.h>
#import <AudioToolbox/DefaultAudioOutput.h>
#import <AudioToolbox/AudioConverter.h>

#import <CynthiuneBundle.h>
#import <Output.h>

#import "MacOSXPlayer.h"

#define LOCALIZED(X) _b ([MacOSXPlayer class], X)

static OSStatus
inputCallback (AudioConverterRef inAudioConverter, UInt32* outDataSize,
	       void** outData, void* selfRef)
{
  PlayerRef *self;

  self = selfRef;

  memcpy (self->localBuffer, self->buffer, self->dataSize);

  *outDataSize = self->dataSize;
  *outData = self->localBuffer;

  memset (self->buffer, 0, self->dataSize);

  [self->parentPlayer
       performSelectorOnMainThread: @selector (chunkFinishedPlaying)
       withObject: nil
       waitUntilDone: NO];

  return noErr;
}

static OSStatus
converterRenderer (void* selfRef, AudioUnitRenderActionFlags inActionFlags, 
                   const AudioTimeStamp *inTimeStamp, UInt32 inBusNumber, 
                   AudioBuffer *ioData)
{
  UInt32 size;
  PlayerRef *self;

  self = selfRef;

  size = ioData->mDataByteSize;
  AudioConverterFillBuffer (self->converter, inputCallback, self,
			    &size, ioData->mData);

  return noErr;
}

@implementation MacOSXPlayer : NSObject

+ (NSArray *) bundleClasses
{
  return [NSArray arrayWithObject: [self class]];
}

- (id) init
{
  if ((self = [super init]))
    {
      parentPlayer = nil;
      buffer = NULL;
      localBuffer = NULL;
      bufferSize = 0;
      dataSize = 0;
      bytes = 0;
      converter = NULL;
      inputFormat.mFormatID = kAudioFormatLinearPCM;
      inputFormat.mFormatFlags = (kLinearPCMFormatFlagIsSignedInteger 
                                  | kLinearPCMFormatFlagIsPacked);
      inputFormat.mFramesPerPacket = 1;
      inputFormat.mBitsPerChannel = 16;
    }

  return self;
}

- (void) setParentPlayer: (id) aPlayer;
{
  parentPlayer = aPlayer;
}

- (BOOL) _modifyConverter
{
  UInt32 aStreamSize;

  aStreamSize = sizeof (AudioStreamBasicDescription);
  AudioUnitGetProperty (outputUnit,
                        kAudioUnitProperty_StreamFormat,
                        kAudioUnitScope_Output,
                        0,
                        &outputFormat,
                        &aStreamSize);
  if (converter)
    AudioConverterDispose (converter);

  inputFormat.mSampleRate = rate;
  inputFormat.mBytesPerPacket = channels * 2;
  inputFormat.mBytesPerFrame = channels * 2;
  inputFormat.mChannelsPerFrame = channels;

  return (AudioConverterNew (&inputFormat, &outputFormat, &converter)
          == noErr);
}

- (BOOL) _audioInit
{
  UInt32 aStreamSize;
  struct AudioUnitInputCallback input;

  input.inputProc = converterRenderer;
  input.inputProcRefCon = self;

  aStreamSize = sizeof (AudioStreamBasicDescription);

  return (OpenDefaultAudioOutput (&outputUnit) == noErr
          && AudioUnitInitialize (outputUnit) == noErr
          && AudioUnitSetProperty (outputUnit, 
                                   kAudioUnitProperty_SetInputCallback, 
                                   kAudioUnitScope_Input,
                                   0,
                                   &input, sizeof (input)) == noErr
          && AudioUnitGetProperty (outputUnit,
                                   kAudioUnitProperty_StreamFormat,
                                   kAudioUnitScope_Output,
                                   0,
                                   &outputFormat,
                                   &aStreamSize) == noErr);
}

- (BOOL) openDevice
{
  isOpen = ([self _audioInit]
            && [self _modifyConverter]
            && AudioOutputUnitStart (outputUnit) == noErr);

  return isOpen;
}

- (void) closeDevice
{
  AudioOutputUnitStop (outputUnit);	
  CloseComponent (outputUnit);
  if (buffer)
    {
      free (buffer);
      buffer = NULL;
      bufferSize = 0;
    }
  if (localBuffer)
    {
      free (localBuffer);
      localBuffer = NULL;
    }
  dataSize = 0;
  isOpen = NO;
}

- (BOOL) prepareDeviceWithChannels: (unsigned int) numberOfChannels
                           andRate: (unsigned long) sampleRate
{
  BOOL result;

  channels = numberOfChannels;
  rate = sampleRate;

  if (isOpen)
    {
      AudioOutputUnitStop (outputUnit);
      result = ([self _modifyConverter]
                && (AudioOutputUnitStart (outputUnit) == noErr));
    }
  else
    result = YES;

  return result;
}

- (void) playChunk: (NSData *) chunk
{
  dataSize = [chunk length];

  if (bufferSize < dataSize)
    {
      if (buffer)
        free (buffer);
      buffer = malloc (dataSize);
      if (localBuffer)
        free (localBuffer);
      localBuffer = malloc (dataSize);
      bufferSize = dataSize;
    }

  [chunk getBytes: buffer];
}

- (unsigned int) outputBytes
{
  return bytes;
}

@end
