/*
 * proxy_slip.c - Proxy interface specific code in diald.
 *		  The proxy interface is used to monitor packets
 *		  when the physical link is down.
 *
 * Copyright (c) 1994, 1995, 1996 Eric Schenk.
 * Copyright (c) 1999 Mike Jagdis.
 * All rights reserved. Please see the file LICENSE which should be
 * distributed with this software for terms of use.
 */

#include "diald.h"


static int proxy_slip_send(proxy_t *proxy,
	unsigned short wprot, unsigned char *p, size_t len);
static int proxy_slip_recv(proxy_t *proxy, unsigned char *p, size_t len);
static void proxy_slip_start(proxy_t *proxy);
static void proxy_slip_stop(proxy_t *proxy);
static void proxy_slip_close(proxy_t *proxy);
static void proxy_slip_release(proxy_t *proxy);
int proxy_slip_init(proxy_t *proxy, char *proxydev);


static FILE *proxy_mfp;		/* also have an fp. Hackery for recv_packet. */
static int proxy_sfd;		/* slave pty fd */

/*
 * SLIP PACKET READING AND WRITING CODE FROM RFC 1055 by J. Romkey.
 *
 */

/* SLIP special character codes */
#define END             0300    /* indicates end of packet */
#define ESC             0333    /* indicates byte stuffing */
#define ESC_END         0334    /* ESC ESC_END means END data byte */
#define ESC_ESC         0335    /* ESC ESC_ESC means ESC data byte */

/* SEND_PACKET: sends a packet of length "len", starting at
 * location "p".
 */
static int
proxy_slip_send(proxy_t *proxy,
	unsigned short wprot, unsigned char *p, size_t len)
{
    /* send an initial END character to flush out any data that may
     * have accumulated in the receiver due to line noise
     */
    putc(END,proxy_mfp);

    /* for each byte in the packet, send the appropriate character
     * sequence
     */
    while(len--) {
	switch(*p) {
	/* if it's the same code as an END character, we send a
	 * special two character code so as not to make the
	 * receiver think we sent an END
	 */
	case END:
	    putc(ESC,proxy_mfp);
	    putc(ESC_END,proxy_mfp);
	    break;

	/* if it's the same code as an ESC character,
	 * we send a special two character code so as not
	 * to make the receiver think we sent an ESC
	 */
	case ESC:
	    putc(ESC,proxy_mfp);
	    putc(ESC_ESC,proxy_mfp);
	    break;
	/* otherwise, we just send the character
	 */
	default:
	    putc(*p,proxy_mfp);
	}

	p++;
    }

    /* tell the receiver that we're done sending the packet
     */
    putc(END,proxy_mfp);

    return 0;
}


/*
 * RECV_PACKET: receives a packet into the buffer located at "p".
 *      If more than len bytes are received, the packet will
 *      be truncated.
 *      Returns the number of bytes stored in the buffer.
 */

static int
proxy_slip_recv(proxy_t *proxy, unsigned char *p, size_t len)
{
    int c;
    int received = 0;

    *(unsigned short *)p = htons(ETH_P_IP);
    p += sizeof(unsigned short);
    len -= sizeof(unsigned short);

    /* sit in a loop reading bytes until we put together
    * a whole packet.
    * Make sure not to copy them into the packet if we
    * run out of room.
    */
    while(1) {
       /* get a character to process
	*/
       c = getc(proxy_mfp);

       /* handle bytestuffing if necessary
	*/
       switch(c) {

       /* if it's an END character then we're done with
	* the packet
	*/
       case END:
	       /* a minor optimization: if there is no
		* data in the packet, ignore it. This is
		* meant to avoid bothering IP with all
		* the empty packets generated by the
		* duplicate END characters which are in
		* turn sent to try to detect line noise.
		*/
	       if(received)
		       return received;
	       else
		       break;

       /* if it's the same code as an ESC character, wait
	* and get another character and then figure out
	* what to store in the packet based on that.
	*/
       case ESC:
	       c = getc(proxy_mfp);

	       /* if "c" is not one of these two, then we
		* have a protocol violation.  The best bet
		* seems to be to leave the byte alone and
		* just stuff it into the packet
		*/
	       switch(c) {
	       case ESC_END:
		       c = END;
		       break;
	       case ESC_ESC:
		       c = ESC;
		       break;
		       }

       /* here we fall into the default handler and let
	* it store the character for us
	*/
       default:
	       if(received < len)
		       p[received++] = c;
	       }
     }
}


static void
proxy_slip_start(proxy_t *proxy)
{
    iface_start("proxy", proxy->iftype, proxy->ifunit,
	orig_local_ip, orig_remote_ip, orig_broadcast_ip, metric+1);
}


static void
proxy_slip_stop(proxy_t *proxy)
{
    iface_stop("proxy", proxy->iftype, proxy->ifunit,
	orig_local_ip, orig_remote_ip, orig_broadcast_ip, metric+1);
}


static void
proxy_slip_close(proxy_t *proxy)
{
    close(proxy->fd);
    close(proxy_sfd);
}


static void
proxy_slip_release(proxy_t *proxy)
{
    proxy_slip_stop(proxy);
    iface_down("proxy", proxy->iftype, proxy->ifunit,
	orig_local_ip, orig_remote_ip, orig_broadcast_ip, metric+1);

    /* clear the line discipline */
    if (ioctl(proxy_sfd, TIOCSETD, &orig_disc) < 0)
	mon_syslog(LOG_ERR,"Can't set line discipline: %m");

    proxy_slip_close(proxy);
}


/*
 * Set the pty to an 8 bit clean mode and change it to the
 * desired SLIP line disciple, and run ifconfig to configure the
 * device as up.
 * The SLIP configuration is essentially what slattach
 * does, but we do it here so we know what interface (sl*)
 * gets opened as a result. (slattach doesn't return this)
 */

int
proxy_slip_init(proxy_t *proxy, char *proxydev)
{
    int d, unit, disc, sencap = 0;

    get_pty(&d, &proxy_sfd);
    if (d < 0)
	return -1;
    fcntl(d, F_SETFD, FD_CLOEXEC);
    proxy_mfp = fdopen(d, "r+");

    /* change proxy_sfd to 8 bit clean line, 38400 speed */
    set_up_tty(proxy_sfd,1, 38400);

    if (ioctl(proxy_sfd, TIOCGETD, &orig_disc) < 0) {
	mon_syslog(LOG_ERR,"Can't get line discipline on proxy device: %m");
	goto close_and_fail;
    }

    /* change line disciple to SLIP and set the SLIP encapsulation */
    disc = N_SLIP;
    if ((unit = ioctl(proxy_sfd, TIOCSETD, &disc)) < 0) {
	if (errno == ENFILE) {
	   mon_syslog(LOG_ERR,"No free slip device available for proxy.");
	   goto close_and_fail;
	} else if (errno == EEXIST) {
	    mon_syslog(LOG_ERR,"Proxy device already in slip mode!?");
	} else if (errno == EINVAL) {
	    mon_syslog(LOG_ERR,"SLIP not supported by kernel, can't build proxy.");
	    goto close_and_fail;
	} else {
	   mon_syslog(LOG_ERR,"Can't set line discipline: %m");
	   goto close_and_fail;
	}
    }

    if (ioctl(proxy_sfd, SIOCSIFENCAP, &sencap) < 0) {
	mon_syslog(LOG_ERR,"Can't set encapsulation: %m");
	goto close_and_fail;
    }

    /* verify that it worked */
    if (ioctl(proxy_sfd, TIOCGETD, &disc) < 0) {
	mon_syslog(LOG_ERR,"Can't get line discipline: %m");
	goto close_and_fail;
    }
    if (ioctl(proxy_sfd, SIOCGIFENCAP, &sencap) < 0) {
	mon_syslog(LOG_ERR,"Can't get encapsulation: %m");
	goto close_and_fail;
    }

    if (disc != N_SLIP || sencap != 0) {
        mon_syslog(LOG_ERR,"Couldn't set up the proxy link correctly!");
	goto close_and_fail;
    }

    if (debug&DEBUG_VERBOSE)
        mon_syslog(LOG_INFO,"Proxy device established on interface sl%d",
	    unit);

    strcpy(proxy->iftype, "sl");
    proxy->ifunit = unit;
    proxy->send = proxy_slip_send;
    proxy->recv = proxy_slip_recv;
    proxy->init = proxy_slip_init;
    proxy->start = proxy_slip_start;
    proxy->stop = proxy_slip_stop;
    proxy->close = proxy_slip_close;
    proxy->release = proxy_slip_release;
    proxy->fd = d;
    return d;

close_and_fail:
    close(d);
    close(proxy_sfd);
    return -1;
}
