/* http.c  -  HTTP protocol handler
 * Copyright (C) 1999, 2001, 2002, 2003, 2004 Free Software Foundation, Inc.
 *
 * This file is part of GnuPG.
 *
 * GnuPG is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * GnuPG is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <config.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>

#include <unistd.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <time.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>

#include "dirmngr.h"
#include "misc.h"
#include "http.h"

#define HTTP_PROXY_ENV           "http_proxy"
#define HTTP_PROXY_ENV_PRINTABLE "$http_proxy"

#ifdef _WIN32   /* Most of the Windows code has been removed, though. */
#define sock_close(a)  closesocket(a)
#else
#define sock_close(a)  close(a)
#endif

#define MAX_LINELEN 20000  /* Max. length of a HTTP header line. */
#define VALID_URI_CHARS "abcdefghijklmnopqrstuvwxyz"   \
                        "ABCDEFGHIJKLMNOPQRSTUVWXYZ"   \
                        "01234567890@"                 \
                        "!\"#$%&'()*+,-./:;<=>?[\\]^_{|}~"

#ifndef EAGAIN
#define EAGAIN  EWOULDBLOCK
#endif

static gpg_error_t do_parse_uri (parsed_uri_t uri, int only_local_part);
static int remove_escapes (unsigned char *string);
static int insert_escapes (unsigned char *buffer, const unsigned char *string,
                           const unsigned char *special);
static uri_tuple_t parse_tuple (unsigned char *string);
static gpg_error_t send_request (http_t hd, const char *proxy);
static unsigned char *build_rel_path (parsed_uri_t uri);
static gpg_error_t parse_response (http_t hd);

static int connect_server (const char *server, unsigned short port);
static gpg_error_t write_server (int sock, const char *data, size_t length);


gpg_error_t
http_open (http_t hd, http_req_t reqtype, const char *url, unsigned int flags,
           const char *proxy)
{
  gpg_error_t err;

  if (!(reqtype == HTTP_REQ_GET || reqtype == HTTP_REQ_POST))
    return gpg_error (GPG_ERR_INV_ARG);

  /* Initialize the handle. */
  memset (hd, 0, sizeof *hd);
  hd->sock = -1;
  hd->initialized = 1;
  hd->req_type = reqtype;
  hd->flags = flags;

  err = http_parse_uri (&hd->uri, url);
  if (!err)
    {
      err = send_request (hd, proxy);
      if (!err)
        {
          hd->fp_write = fdopen (hd->sock, "w");
          if (hd->fp_write)
            return 0;
          err = gpg_error_from_errno (errno);
        }
    }

  if (!hd->fp_read && !hd->fp_write && hd->sock != -1)
    sock_close (hd->sock);
  if (hd->fp_read)
    fclose (hd->fp_read);
  if (hd->fp_write)
    fclose (hd->fp_write);
  http_release_parsed_uri (hd->uri);
  hd->initialized = 0;

  return err;
}


void
http_start_data (http_t hd)
{
  fflush (hd->fp_write);
  if (!hd->in_data)
    {
      write_server (hd->sock, "\r\n", 2);
      hd->in_data = 1;
    }
}


gpg_error_t
http_wait_response (http_t hd, unsigned int *ret_status)
{
  gpg_error_t err;

  http_start_data (hd);  /* Make sure that we are in the data. */

  /* We dup the socket, to cope with thenfact that fclose closes the
     underlying socket. */
  hd->sock = dup (hd->sock); 
  if (hd->sock == -1)
    return gpg_error_from_errno (errno);
  fclose (hd->fp_write);
  hd->fp_write = NULL;

  if (!(hd->flags & HTTP_FLAG_NO_SHUTDOWN))
    shutdown (hd->sock, 1);
  hd->in_data = 0;

  hd->fp_read = fdopen (hd->sock, "r");
  if (!hd->fp_read)
    return gpg_error_from_errno (errno);

  err = parse_response (hd);
  if (!err && ret_status)
    *ret_status = hd->status_code;

  return err;
}


/* Convenience function to send a request and wait for the response.
   Closes the handle on error.  If PROXY is not NULL, this value will
   be used as a HTTP proxy and any enabled $http_proxy gets
   ignored. */
gpg_error_t
http_open_document (http_t hd, const char *document, unsigned int flags,
                    const char *proxy)
{
  gpg_error_t err;

  err = http_open (hd, HTTP_REQ_GET, document, flags, proxy);
  if (err)
    return err;

  err = http_wait_response (hd, NULL);
  if (err)
    http_close (hd, 0);

  return err;
}


void
http_close (http_t hd, int keep_read_stream)
{
  if (!hd || !hd->initialized)
    return;
  if (!hd->fp_read && !hd->fp_write && hd->sock != -1)
    sock_close (hd->sock);
  if (hd->fp_read && !keep_read_stream)
    fclose (hd->fp_read);
  if (hd->fp_write)
    fclose (hd->fp_write);
  http_release_parsed_uri (hd->uri);
  xfree (hd->buffer);
  hd->initialized = 0;
}



/*
 * Parse an URI and put the result into the newly allocated RET_URI.
 * The caller must always use release_parsed_uri() to releases the
 * resources (even on error).
 */
gpg_error_t
http_parse_uri (parsed_uri_t *ret_uri, const char *uri)
{
  *ret_uri = xcalloc (1, sizeof **ret_uri + strlen (uri));
  strcpy ((*ret_uri)->buffer, uri);
  return do_parse_uri (*ret_uri, 0);
}

void
http_release_parsed_uri (parsed_uri_t uri)
{
  if (uri)
    {
      uri_tuple_t r, r2;

      for (r = uri->query; r; r = r2)
        {
          r2 = r->next;
          xfree (r);
        }
      xfree (uri);
    }
}


static gpg_error_t
do_parse_uri (parsed_uri_t uri, int only_local_part)
{
  uri_tuple_t *tail;
  char *p, *p2, *p3;
  int n;

  p = uri->buffer;
  n = strlen (uri->buffer);
  /* Initialize all fields to an empty string or an empty list. */
  uri->scheme = uri->host = uri->path = p + n;
  uri->port = 0;
  uri->params = uri->query = NULL;

  /* A quick validity check. */
  if (strspn (p, VALID_URI_CHARS) != n)
    return gpg_error (GPG_ERR_BAD_URI);  /* Invalid characters found. */

  if (!only_local_part)
    {
      /* Find the scheme */
      if (!(p2 = strchr (p, ':')) || p2 == p)
        return gpg_error (GPG_ERR_BAD_URI);  /* No scheme. */
      *p2++ = 0;
      strlwr (p);
      uri->scheme = p;
      uri->port = 80;
      if (!strcmp (uri->scheme, "http"))
        ;
      else
        return gpg_error (GPG_ERR_INV_URI);  /* Unsupported scheme */

      p = p2;

      /* Find the hostname */
      if (*p != '/')
        return gpg_error (GPG_ERR_INV_URI); /* Does not start with a slash. */

      p++;
      if (*p == '/')
        {                       /* There seems to be a hostname. */
          p++;
          if ((p2 = strchr (p, '/')))
            *p2++ = 0;
          strlwr (p);
          uri->host = p;
          if ((p3 = strchr (p, ':')))
            {
              *p3++ = 0;
              uri->port = atoi (p3);
            }

          uri->host = p;
          if ((n = remove_escapes (uri->host)) < 0)
            return gpg_error (GPG_ERR_BAD_URI);
          if (n != strlen (p))
            return gpg_error (GPG_ERR_BAD_URI); /* Hostname with a Nul in it. */
          p = p2 ? p2 : NULL;
        }
    } /* End global URI part */

  /* Parse the pathname part */
  if (!p || !*p)   /* We don't have a path. */
    return 0;      /* Okay. */

  /* TODO: Here we have to check params. */

  /* Do we have a query part? */
  if ((p2 = strchr (p, '?')))
    *p2++ = 0;

  uri->path = p;
  if ((n = remove_escapes (p)) < 0)
    return gpg_error (GPG_ERR_BAD_URI);
  if (n != strlen (p))
    return gpg_error (GPG_ERR_BAD_URI); /* Path with a Nul in it. */
  p = p2 ? p2 : NULL;

  if (!p || !*p)  /* We don't have a query string. */
    return 0;     /* Okay. */

  /* Now parse the query string. */
  tail = &uri->query;
  for (;;)
    {
      uri_tuple_t elem;

      if ((p2 = strchr (p, '&')))
        *p2++ = 0;
      if (!(elem = parse_tuple (p)))
        return gpg_error (GPG_ERR_BAD_URI);
      *tail = elem;
      tail = &elem->next;

      if (!p2)
        break; /* Ready. */
      p = p2;
    }

  return 0;
}


/****************
 * Remove all %xx escapes; this is done inplace.
 * Returns: New length of the string.
 */
static int
remove_escapes (unsigned char *string)
{
  int n = 0;
  unsigned char *p, *s;

  for (p = s = string; *s; s++)
    {
      if (*s == '%')
        {
          if (s[1] && s[2] && isxdigit (s[1]) && isxdigit (s[2]))
            {
              s++;
              *p = *s >= '0' && *s <= '9' ? *s - '0' :
                *s >= 'A' && *s <= 'F' ? *s - 'A' + 10 : *s - 'a' + 10;
              *p <<= 4;
              s++;
              *p |= *s >= '0' && *s <= '9' ? *s - '0' :
                *s >= 'A' && *s <= 'F' ? *s - 'A' + 10 : *s - 'a' + 10;
              p++;
              n++;
            }
          else
            {
              *p++ = *s++;
              if (*s)
                *p++ = *s++;
              if (*s)
                *p++ = *s++;
              if (*s)
                *p = 0;
              return -1;   /* Bad URI. */
            }
        }
      else
        {
          *p++ = *s;
          n++;
        }
    }
  *p = 0;  /* Always keep a string terminator. */
  return n;
}


static int
insert_escapes (unsigned char *buffer, const unsigned char *string,
                const unsigned char *special)
{
  int n = 0;

  for (; *string; string++)
    {
      if (strchr (VALID_URI_CHARS, *string) && !strchr (special, *string))
        {
          if (buffer)
            *buffer++ = *string;
          n++;
        }
      else
        {
          if (buffer)
            {
              sprintf (buffer, "%%%02X", *string);
              buffer += 3;
            }
          n += 3;
        }
    }
  return n;
}


static uri_tuple_t
parse_tuple (unsigned char *string)
{
  unsigned char *p = string;
  unsigned char *p2;
  int n;
  uri_tuple_t tuple;

  if ((p2 = strchr (p, '=')))
    *p2++ = 0;
  if ((n = remove_escapes (p)) < 0)
    return NULL;                /* Bad URI. */
  if (n != strlen (p))
    return NULL;                /* Name with a Nul in it. */
  tuple = xtrycalloc (1, sizeof *tuple);
  if (!tuple)
    return NULL; /* Out of core. */
  tuple->name = p;
  if (!p2) /* We have only the name, so we assume an empty value string. */
    {
      tuple->value = p + strlen (p);
      tuple->valuelen = 0;
    }
  else /* name and value */
    {
      if ((n = remove_escapes (p2)) < 0)
        {
          xfree (tuple);
          return NULL; /* Bad URI. */
        }
      tuple->value = p2;
      tuple->valuelen = n;
    }
  return tuple;
}


/*
 * Send a HTTP request to the server
 * Returns 0 if the request was successful
 */
static gpg_error_t
send_request (http_t hd, const char *proxy)
{
  const unsigned char *server;
  unsigned char *request, *p;
  unsigned short port;
  const char *http_proxy = NULL;
  gpg_error_t err;
  int save_errno;

  server = *hd->uri->host ? hd->uri->host : "localhost";
  port = hd->uri->port ? hd->uri->port : 80;

  if (proxy
      || ( (hd->flags & HTTP_FLAG_TRY_PROXY)
           && (http_proxy = getenv (HTTP_PROXY_ENV))))
    {
      parsed_uri_t uri;
      
      if (proxy)
        http_proxy = proxy;

      err = http_parse_uri (&uri, http_proxy);
      if (err)
        {
          log_error (_("invalid HTTP proxy (%s): %s\n"),
                     http_proxy, gpg_strerror (err));
          http_release_parsed_uri (uri);
          return gpg_error (GPG_ERR_CONFIGURATION);
        }
      hd->sock = connect_server (*uri->host ? uri->host : "localhost",
                                 uri->port ? uri->port : 80);
      save_errno = errno;
      http_release_parsed_uri (uri);

    }
  else
    {
      hd->sock = connect_server (server, port);
      save_errno = errno;
    }

  if (hd->sock == -1)
    return gpg_error_from_errno (save_errno);

  p = build_rel_path (hd->uri);
  if (!p)
    return gpg_error_from_errno (errno);
  request = xtrymalloc (strlen (server) * 2 + strlen (p) + 50);
  if (!request)
    {
      err = gpg_error_from_errno (errno);
      xfree (p);
      return err;
    }

  if (http_proxy)
    {
      sprintf (request, "%s http://%s:%hu%s%s HTTP/1.0\r\n",
               hd->req_type == HTTP_REQ_GET ? "GET" :
               hd->req_type == HTTP_REQ_HEAD ? "HEAD" :
               hd->req_type == HTTP_REQ_POST ? "POST" : "OOPS",
               server, port, *p == '/' ? "" : "/", p);
    }
  else
    {
      sprintf (request, "%s %s%s HTTP/1.0\r\nHost: %s\r\n",
               hd->req_type == HTTP_REQ_GET ? "GET" :
               hd->req_type == HTTP_REQ_HEAD ? "HEAD" :
               hd->req_type == HTTP_REQ_POST ? "POST" : "OOPS",
               *p == '/' ? "" : "/", p, server);
    }
  xfree (p);

  err = write_server (hd->sock, request, strlen (request));
  xfree (request);

  return err;
}


/*
 * Build the relative path from the parsed URI.  Minimal
 * implementation.  May return NULL in case of memory failure; errno
 * is then set accordingly.
 */
static byte *
build_rel_path (parsed_uri_t uri)
{
  uri_tuple_t r;
  unsigned char *rel_path, *p;
  int n;

  /* Count the needed space. */
  n = insert_escapes (NULL, uri->path, "%;?&");
  /* TODO: build params. */
  for (r = uri->query; r; r = r->next)
    {
      n++;                      /* '?'/'&' */
      n += insert_escapes (NULL, r->name, "%;?&=");
      n++;                      /* '=' */
      n += insert_escapes (NULL, r->value, "%;?&=");
    }
  n++;

  /* Now allocate and copy. */
  p = rel_path = xtrymalloc (n);
  if (!p)
    return NULL;
  n = insert_escapes (p, uri->path, "%;?&");
  p += n;
  /* TODO: add params. */
  for (r = uri->query; r; r = r->next)
    {
      *p++ = r == uri->query ? '?' : '&';
      n = insert_escapes (p, r->name, "%;?&=");
      p += n;
      *p++ = '=';
      /* TODO: Use valuelen. */
      n = insert_escapes (p, r->value, "%;?&=");
      p += n;
    }
  *p = 0;
  return rel_path;
}



/*
   Same as fgets() but if the buffer is too short a larger one will be
   allocated up to some limit *MAX_LENGTH.  A line is considered a
   byte stream ending in a LF.  Returns the length of the line. EOF is
   indicated by a line of length zero. The last LF may be missing due
   to an EOF.  If MAX_LENGTH is zero on return, the line has been
   truncated.  If the returned buffer is NULL, not enough memory was
   enable to increase it, the return value will also be 0 and some
   bytes might have been lost which should be no problem becuase
   out-of-memory is pretty fatal for most applications.

   If a line has been truncated, the file pointer is internally moved
   forward to the end of the line.

   Note: The returned buffer is allocated with enough extra space to
   append a CR,LF,Nul
 */
static size_t
read_line (FILE *fp, unsigned char **addr_of_buffer, size_t *length_of_buffer,
           size_t *max_length)
{
  int c;
  char  *buffer = *addr_of_buffer;
  size_t length = *length_of_buffer;
  size_t nbytes = 0;
  size_t maxlen = *max_length;
  char *p;

  if (!buffer)
    { /* Must allocate a new buffer. */
      length = 256;
      buffer = xtrymalloc (length);
      *addr_of_buffer = buffer;
      if (!buffer)
        {
          *length_of_buffer = *max_length = 0;
          return 0;
        }
      *length_of_buffer = length;
    }

  length -= 3; /* Reserve 3 bytes (cr,lf,eol). */
  p = buffer;
  while  ((c = getc (fp)) != EOF)
    {
      if (nbytes == length)
        { /* Increase the buffer. */
          if (length > maxlen ) /* This is out limit */
            {
              /* Skip the rest of the line. */
              while (c != '\n' && (c=getc (fp)) != EOF)
                ;
              *p++ = '\n'; /* Always append a LF (we reserved some space). */
              nbytes++;
              *max_length = 0; /* Indicate truncation */
              break; /* the while loop. */
            }
          length += 3; /* Adjust for the reserved bytes. */
          length += length < 1024? 256 : 1024;
          *addr_of_buffer = xtryrealloc (buffer, length);
          if (!*addr_of_buffer)
            {
              int save_errno = errno;
              xfree (buffer); 
              *length_of_buffer = *max_length = 0;
              errno = save_errno;
              return 0;
            }
          buffer = *addr_of_buffer;
          *length_of_buffer = length;
          length -= 3; /* And readjust for the reservation. */
          p = buffer + nbytes;
	}
      *p++ = c;
      nbytes++;
      if (c == '\n')
        break;
    }
  *p = 0; /* Make sure the line is a string. */

  return nbytes;
}

/*
 * Parse the response from a server.
 * Returns: Errorcode and sets some files in the handle
 */
static gpg_error_t
parse_response (http_t hd)
{
  unsigned char *line, *p, *p2;
  unsigned maxlen, len;

  /* Wait for the status line. */
  do
    {
      maxlen = MAX_LINELEN;
      len = read_line (hd->fp_read, &hd->buffer, &hd->buffer_size, &maxlen);
      line = hd->buffer;
      if (!line)
        return gpg_error_from_errno (errno); /* Out of core. */
      if (!maxlen)
        return gpg_error (GPG_ERR_TRUNCATED);  /* Line has been truncated. */
      if (!len)
        return gpg_error (GPG_ERR_EOF);
    }
  while (!*line);
  
  if ((p = strchr (line, '/')))
    *p++ = 0;
  if (!p || strcmp (line, "HTTP"))
    return 0;  /* Assume http 0.9. */

  if ((p2 = strpbrk (p, " \t")))
    {
      *p2++ = 0;
      p2 += strspn (p2, " \t");
    }
  if (!p2)
    return 0; /* Also assume http 0.9. */
  p = p2;
  /* TODO: Add HTTP version number check. */
  if ((p2 = strpbrk (p, " \t")))
    *p2++ = 0;
  if (!isdigit (p[0]) || !isdigit (p[1]) || !isdigit (p[2]) || p[3])
    {
      /* Malformed HTTP status code - assume http 0.9. */
      hd->is_http_0_9 = 1;
      hd->status_code = 200;
      return 0;
    }
  hd->status_code = atoi (p);

  /* Skip all the header lines and wait for the empty line. */
  do
    {
      maxlen = MAX_LINELEN;
      len = read_line (hd->fp_read, &hd->buffer,
                       &hd->buffer_size, &maxlen);
      line = hd->buffer;
      if (!line)
        return gpg_error_from_errno (errno); /* Out of core. */
      /* Note, that we can silently ignore truncated lines. */
      if (!len)
        return gpg_error (GPG_ERR_EOF);
      /* Trim line endings of empty lines. */
      if ((*line == '\r' && line[1] == '\n') || *line == '\n')
        *line = 0;
    }
  while (len && *line);

  return 0;
}

#if 0
static int
start_server ()
{
  struct sockaddr_in mya;
  struct sockaddr_in peer;
  int fd, client;
  fd_set rfds;
  int addrlen;
  int i;

  if ((fd = socket (AF_INET, SOCK_STREAM, 0)) == -1)
    {
      log_error ("socket() failed: %s\n", strerror (errno));
      return -1;
    }
  i = 1;
  if (setsockopt (fd, SOL_SOCKET, SO_REUSEADDR, (byte *) & i, sizeof (i)))
    log_info ("setsockopt(SO_REUSEADDR) failed: %s\n", strerror (errno));

  mya.sin_family = AF_INET;
  memset (&mya.sin_addr, 0, sizeof (mya.sin_addr));
  mya.sin_port = htons (11371);

  if (bind (fd, (struct sockaddr *) &mya, sizeof (mya)))
    {
      log_error ("bind to port 11371 failed: %s\n", strerror (errno));
      sock_close (fd);
      return -1;
    }

  if (listen (fd, 5))
    {
      log_error ("listen failed: %s\n", strerror (errno));
      sock_close (fd);
      return -1;
    }

  for (;;)
    {
      FD_ZERO (&rfds);
      FD_SET (fd, &rfds);

      if (select (fd + 1, &rfds, NULL, NULL, NULL) <= 0)
        continue;               /* ignore any errors */

      if (!FD_ISSET (fd, &rfds))
        continue;

      addrlen = sizeof peer;
      client = accept (fd, (struct sockaddr *) &peer, &addrlen);
      if (client == -1)
        continue;               /* oops */

      log_info ("connect from %s\n", inet_ntoa (peer.sin_addr));

      fflush (stdout);
      fflush (stderr);
      if (!fork ())
        {
          int c;
          FILE *fp;

          fp = fdopen (client, "r");
          while ((c = getc (fp)) != EOF)
            putchar (c);
          fclose (fp);
          exit (0);
        }
      sock_close (client);
    }


  return 0;
}
#endif

/* Actually connect to a server.  Returns the file descripto or -1 on
   error.  errno is set on error. */
static int
connect_server (const char *server, unsigned short port)
{
  int sock = -1;
  int connected = 0;
  int last_errno = 0;

#ifdef HAVE_GETADDRINFO
  struct addrinfo hints, *res, *ai;
  char portstr[20];
  
  sprintf (portstr, "%hu", port);
  memset (&hints, 0, sizeof (hints));
  hints.ai_socktype = SOCK_STREAM;
  if (getaddrinfo (server, portstr, &hints, &res))
    {
      log_error (_("error resolving `%s': host not found\n"), server);
      errno = ENOENT; /* fixme: Is there a better one? */
      return -1;
    }
    
  for (ai = res; ai && !connected; ai = ai->ai_next)
    {
      if (sock != -1)
        sock_close (sock);
      sock = socket (ai->ai_family, ai->ai_socktype, ai->ai_protocol);
      if (sock == -1)
        {
          int save_errno = errno;
          log_error (_("error creating socket: %s\n"), strerror (errno));
          freeaddrinfo (res);
          errno = save_errno;
          return -1;
        }
      
      if (connect (sock, ai->ai_addr, ai->ai_addrlen))
        last_errno = errno;
      else
        connected = 1;
    }
  freeaddrinfo (res);
#else /* !HAVE_GETADDRINFO */
  int i;
  struct hostent *host = NULL;
  struct sockaddr_in addr;

  /* Note, that this code is not thread-safe. */

  memset (&addr, 0, sizeof (addr));
  host = gethostbyname (server); 
  if (!host)
    {
      log_error (_("error resolving `%s': host not found\n"), server);
      errno = ENOENT; /* fixme: Is there a better one? */
      return -1;
    }

  sock = socket (host->h_addrtype, SOCK_STREAM, 0);
  if (sock == -1)
    {
      log_error (_("error creating socket: %s\n"), strerror (errno));
      return -1;
    }

  addr.sin_family = host->h_addrtype;
  addr.sin_port = htons (port);

  /* Try all A records until one responds. */
  for (i=0; host->h_addr_list[i]; i++)
    {
      memcpy (&addr.sin_addr, host->h_addr_list[i], host->h_length);
      if (connect (sock, (struct sockaddr *) &addr, sizeof (addr)))
        last_errno = errno;
      else
        {
          connected = 1;
          break;
        }
    }
  
#endif /* !HAVE_GETADDRINFO */
  if (!connected)
    {
      log_error ("%s: %s\n", server, strerror (last_errno));
      if (sock != -1)
        sock_close (sock);
      errno = last_errno;
      return -1;
    }
  return sock;
}


static gpg_error_t
write_server (int sock, const char *data, size_t length)
{
  int nleft;

  nleft = length;
  while (nleft > 0)
    {
      int nwritten = write (sock, data, nleft);
      if (nwritten == -1)
        {
          if (errno == EINTR)
            continue;
          if (errno == EAGAIN)
            {
              struct timeval tv;

              tv.tv_sec = 0;
              tv.tv_usec = 50000;
              select (0, NULL, NULL, NULL, &tv);
              continue;
            }
          log_info (_("network write failed: %s\n"), strerror (errno));
          return gpg_error_from_errno (errno);
        }
      nleft -= nwritten;
      data += nwritten;
    }
  
  return 0;
}

/**** Test code ****/
#ifdef TEST

int
main (int argc, char **argv)
{
  int rc;
  parsed_uri_t uri;
  uri_tuple_t r;
  struct http_context_s hd;
  int c;

  log_set_prefix ("http-test", 1|4); 
  if (argc == 1)
    {
      /*start_server ();*/
      return 0;
    }

  if (argc != 2)
    {
      fprintf (stderr, "usage: http-test uri\n");
      return 1;
    }
  argc--;
  argv++;

  rc = http_parse_uri (&uri, *argv);
  if (rc)
    {
      log_error ("`%s': %s\n", *argv, gpg_strerror (rc));
      http_release_parsed_uri (uri);
      return 1;
    }

  printf ("Scheme: %s\n", uri->scheme);
  printf ("Host  : %s\n", uri->host);
  printf ("Port  : %u\n", uri->port);
  printf ("Path  : %s\n", uri->path);
  for (r = uri->params; r; r = r->next)
    {
      printf ("Params: %s=%s", r->name, r->value);
      if (strlen (r->value) != r->valuelen)
        printf (" [real length=%d]", (int) r->valuelen);
      putchar ('\n');
    }
  for (r = uri->query; r; r = r->next)
    {
      printf ("Query : %s=%s", r->name, r->value);
      if (strlen (r->value) != r->valuelen)
        printf (" [real length=%d]", (int) r->valuelen);
      putchar ('\n');
    }
  http_release_parsed_uri (uri);
  uri = NULL;

  rc = http_open_document (&hd, *argv, 0);
  if (rc)
    {
      log_error ("can't get `%s': %s\n", *argv, gpg_strerror (rc));
      return 1;
    }
  log_info ("open_http_document succeeded; status=%u\n", hd.status_code);
  while ((c = getc (hd.fp_read)) != EOF)
    putchar (c);
  http_close (&hd, 0);
  return 0;
}
#endif /*TEST*/
