import java.io.*;
import java.util.zip.Inflater;
import java.util.zip.DataFormatException;


/**
 * This class is used to parse information out of PNG images
 *
 * @version $Id: PNGParser.java,v 1.11 2002/09/22 23:13:29 blsecres Exp $
 * @author Ben Secrest &lt;blsecres@users.sourceforge.net&gt;
 */
public class PNGParser implements FileParser {
    /** Search string for basic text comments */
    private final static byte[] findText = {'t', 'E', 'X', 't'};

    /** Search string for compressed text comments */
    private final static byte[] findZText = {'z', 'T', 'X', 't'};

    /* * Search string for internationalized/UTF-8 text comments */
    // TODO private static byte[] findIText = new byte[]{'i', 'T', 'X', 't'};

    /**
     * Size for input buffers.  Taken from /usr/include/stdio.h#BUFSIZ
     */
    private final static int inputBufferSize = 8196;

    /** Size for output buffer. */
    private final static int outputBufferSize = 32;

    /** The default logging level for this module */
    private final static int LOGLEVEL = 9;

    /** File extensions for PNG files */
    private final static String[] extensions = {"png"};

    /** Mime types for PNG files */
    private final static String[] mimeTypes = {"image/png"};

    /** Type of file the parser works with */
    private final static String fileType = "PNG";

    /** PNG file magic signature */
    private final static byte[][] magic = {{(byte) 0x89, 0x50, 0x4e, 0x47,
	0x0d, 0x0a, 0x1a, 0x0a}};

    /** PNG headers are always at the beginning of the file */
    private final static boolean magicOffset = false;

    /** PNG headers are case sensitive */
    private final static boolean magicCase = true;

    /** PNG FileMagic structure */
    private final static FileMagic pngMagic = new FileMagic(magic, magicOffset,
	    magicCase);

    // track current text section being read
    private final static byte NO_TAG	= 0x00;
    private final static byte TITLE	= 0x01;
    private final static byte AUTHOR	= 0x02;
    private final static byte DESCR	= 0x03;

    /** Determines if this parser will search for title */
    private boolean wantTitle;

    /** Determines if this parser will search for author */
    private boolean wantAuthor;

    /** Determines if this parser will search for description */
    private boolean wantDescription;

    /** Determines if the parser will provide file type */
    private boolean wantFileType;

    /** Determines if the parser will provide parser information */
    private boolean wantParser;

    /** The logging object for this module */
    private IGLog log;


    /**
     * Construct a new PNGParser object with the given parsing options.
     * @param logObj The object to use for logging data
     * @param extract The set of desired fields to extract
     */
    public PNGParser() {
	log = null;

	wantTitle = wantAuthor = wantDescription = wantFileType = wantParser
	    = false;
    }


    /**
     * Parse the given file, filling in found information
     * @param IGFile The file to open and structure to fill in
     * @throws IOException if an error occurs reading file
     * @throws FileNotFoundException if the file given to be parsed does not
     *	exist
     */
    public void parse(IGFile file) throws IOException, FileNotFoundException {
	if (LOGLEVEL >= IGLog.PROCEDURE)
	    log.add(IGLog.PROCEDURE, "PNGParser.parse(IGFile)");

	try {
	    parse(file, new FileInputStream(file.getLocation()));
	} catch (StreamResetException sre) {
	    throw new IOException("Stream Reset Exception shouldn't happen");
	}
    }


    /**
     * Parse an already opened stream
     * @param file The IGFile structure to fill in
     * @param stream The stream to read data from
     * @throws IOException if an error occurs reading data
     */
    public void parse(IGFile file, InputStream stream)
	    throws IOException, StreamResetException {
	if (log == null)
	    // FIXME
	    return;

	if (LOGLEVEL >= IGLog.PROCEDURE)
	    log.add(IGLog.PROCEDURE, "PNGParser.parse(IGFile, Reader)");

	if (LOGLEVEL >= IGLog.FILE)
	    log.addResource(IGLog.FILE, "PROCESS_FILE",
		    new String[]{file.getLocation()});

	/*
	 * reset attribute strings
	 * if not searching for an item, assign it a value so the search can
	 * kick out when all items != null
	 */
	String title = (wantTitle ? null : "");
	String author = (wantAuthor ? null : "");
	String description = (wantDescription ? null : "");

	boolean foundTag = false;	// found a text tag
	boolean isCompressed = false;	// tag data is compressed
	boolean readText = false;	// reading tag data
	boolean isSearching = true;	// found all tags
	byte lastValue = 0;		// last value of previous buffer
	byte curTag = NO_TAG;		// current tag being read
	byte[] buffer = new byte[inputBufferSize]; // buffer to read image into
	byte[] curText = null;		// buffer to store read data
	int bytesRead;			// number of bytes read
	int textLength = -1;		// length of text data
	int findTextPosition = 0;	// position in text tag search
	int findZTextPosition = 0;	// position in compressed tag search
	String curKey = "";		// current key

	if (LOGLEVEL >= IGLog.SECTION)
	    log.addResource(IGLog.SECTION, "FP_BEGIN_PARSE",
		    new String[]{fileType});


	while (isSearching && (bytesRead = stream.read(buffer)) != -1) {
	    /*
	    if (LOGLEVEL >= IGLog.DEBUG)
		log.add(IGLog.DEBUG, "bytesRead = " + bytesRead);
	    */

	    for (int i = 0; isSearching && i < bytesRead; i++) {
		// search through the current buffer
		/*
		if (LOGLEVEL >= IGLog.DEBUG)
		    if (Character.isLetter((char) buffer[i])) {
			log.add(IGLog.DEBUG, "current char = " +
				new Character((char) buffer[i]));
		    } else {
			log.add(IGLog.DEBUG, "current char = " + buffer[i]);
		    }
		 */

		if (foundTag) {
		    // reading a text chunk's key/tag
		    if (buffer[i] == 0x00) {
			// a null character separates the key from the data
			if (title == null && curKey.equalsIgnoreCase("title"))
			    curTag = TITLE;
			else if (author == null &&
				 curKey.equalsIgnoreCase("author"))
			    curTag = AUTHOR;
			else if (description == null
				&& (curKey.equalsIgnoreCase("description")
				    || curKey.equalsIgnoreCase("comment")))
			    curTag = DESCR;

			/*
			 * the text length given before the chunk leader
			 * accounts for the key length and the null separator
			 * remove this ammount from the remaining text length
			 */
			textLength -= curKey.length() + 1;
			curText = new byte[textLength];
			foundTag = false;
			readText = true;
			curKey = "";
		    } else {
			curKey += new Character((char) buffer[i]);
		    }
		} else if (readText) {
		    // reading text data
		    if (textLength == 0) {
			// when textLength == 0, all text data has been read
			switch (curTag) {
			case TITLE :
			    title = (isCompressed ? decompress(curText)
						  : new String(curText));
			    if (LOGLEVEL >= IGLog.PROGRESS)
				log.addResource(IGLog.PROGRESS,
					"FP_FOUND_TITLE", new String[]{title});
			    break;
			case AUTHOR :
			    author = (isCompressed ? decompress(curText)
						   : new String(curText));
			    if (LOGLEVEL >= IGLog.PROGRESS)
				log.addResource(IGLog.PROGRESS,
					"FP_FOUND_AUTHOR",
					new String[]{author});
			    break;
			case DESCR :
			    description = (isCompressed ? decompress(curText) :
					   new String(curText));
			    if (LOGLEVEL >= IGLog.PROGRESS)
				log.addResource(IGLog.PROGRESS, "FP_FOUND_DESC",
					new String[]{description});
			    break;
			}
			curKey = "";
			curTag = NO_TAG;
			readText = false;
			isCompressed = false;
		    } else {
			/*
			 * store bytes in buffer while there's still data to be
			 * read
			 */
			curText[curText.length - textLength] = buffer[i];
			textLength--;
		    }
		} else if (findZText[findZTextPosition] == buffer[i]) {
		    /*
		     * if found the first character of a text chunk leader,
		     * store the length value that is stored before it
		     */
		    if (findZTextPosition == 0)
			textLength = (i - 1 != -1 ? buffer[i - 1] : lastValue);

		    findZTextPosition++;
		    findTextPosition = 0;
		} else if (findText[findTextPosition] == buffer[i]) {
		    if (findTextPosition == 0)
			textLength = (i - 1 != -1 ? buffer[i - 1] : lastValue);

		    findTextPosition++;
		    findZTextPosition = 0;
		} else {
		    // no matches, keep searches at zero
		    findTextPosition = 0;
		    findZTextPosition = 0;
		}

		if (findTextPosition == findText.length) {
		    // found a tEXt chunk
		    foundTag = true;
		    isCompressed = false;
		    findTextPosition = 0;
		} else if (findZTextPosition == findZText.length) {
		    // found a zTXt chunk
		    foundTag = true;
		    isCompressed = true;
		    findZTextPosition = 0;
		}
		
		// see if everything desired has been found
		if (title != null && author != null && description != null)
		    isSearching = false;
	    }

	    /*
	     * keep track of the the last value in the buffer so that in the
	     * unlikely event that a tEXt or zTXt tags starts at the begging of
	     * a new buffer the data length won't be lost
	     */
	    lastValue = buffer[bytesRead - 1];
	}

	stream.close();

	if (wantTitle)
	    file.put(IGKey.TITLE, title);
	if (wantAuthor)
	    file.put(IGKey.AUTHOR, author);
	if (wantDescription)
	    file.put(IGKey.DESCRIPTION, description);
	if (wantFileType)
	    file.put(IGKey.FILE_TYPE, new String(fileType));
	if (wantParser)
	    file.put(IGKey.PARSER, getClass().getName());

	if (LOGLEVEL >= IGLog.SECTION)
	    log.addResource(IGLog.SECTION, "FP_FINISH_PARSE",
		    new String[]{fileType});
    }


    /**
     * Set the desired attributes to extract
     * @param wanted A set of bits describing preferences
     */
    public void setWantedItems(IGKeySet wanted) {
	wantTitle = wanted.wants(IGKey.TITLE);
	wantAuthor = wanted.wants(IGKey.AUTHOR);
	wantDescription = wanted.wants(IGKey.DESCRIPTION);
	wantFileType = wanted.wants(IGKey.FILE_TYPE);
	wantParser = wanted.wants(IGKey.PARSER);
    }


    /**
     * Set the logger to use with this parser
     * @param logObj The object to use for logging data
     */
    public void setLog(IGLog logObj) {
	log = logObj;
    }


    /**
     * Decompress a gzip'd string.
     * @param gzData The compressed string
     * @return The decompressed string
     */
    private static String decompress(byte[] gzData) {
	Inflater inflater = new Inflater(true);

	inflater.reset();
  
	/*
	 * The data from a zTXt chunk starts with a single PNG compression
	 * method byte, followed by a single `deflate' comression method byte, a
	 * `deflate' flag, the compressed data, and a four byte checksum,
	 * compensate for this extra data when loading the Inflater
	 */
	inflater.setInput(gzData, 3, gzData.length - 7);

	byte[] output = new byte[outputBufferSize];
	int outputBytes = 0, bufferPosition = 0;

	while (! inflater.finished()) {
	    try {
		outputBytes = inflater.inflate(output, bufferPosition,
			output.length - bufferPosition);
	    } catch (DataFormatException dfe) {
		System.err.println("Bad compressed image text chunk.");
		return "";
	    }

	    if (! inflater.finished()) {
		/*
		 * buffer was filled and more data remains, double buffer
		 * size, copy data to new buffer and continue reading
		 */
		byte[] tmp = new byte[output.length * 2];
		for (int i = 0; i < output.length; i++)
		    tmp[i] = output[i];
		output = tmp;
	    }
	    bufferPosition += outputBytes;
	}

	inflater.end();

	// return a String version of the decompressed data
	return new String(output, 0, bufferPosition);
    }


    /**
     * Supply extensions this parser can handle
     * @return String array of file extensions
     */
    public String[] getExtensions() {
	return extensions;
    }


    /**
     * Supply mime types this parser can handle
     * @return String array of mime types
     */
    public String[] getMimeTypes() {
	return mimeTypes;
    }


    /**
     * Supply file magic for files this parser can handle
     * @return Array of byte arrays containing magic signature
     */
    public FileMagic getMagic() {
	return pngMagic;
    }
}
