/* Copyright (C) 2003 2004 Chris Vine

This program is distributed under the General Public Licence, version 2.
For particulars of this and relevant disclaimers see the file
COPYING distributed with the source files.

*/

#ifndef SOCKET_SERVER_H
#define SOCKET_SERVER_H

#include "prog_defs.h"

#include <string>
#include <list>
#include <utility>

#include <netinet/in.h>

#include <glibmm/thread.h>
#include <glibmm/dispatcher.h>

#include <sigc++/sigc++.h>

#include "sem_sync.h"
#include "shared_ptr.h"

typedef std::list<std::pair<std::string, unsigned int> > FilenamesList;


class Socket_server: public sigc::trackable {

  bool server_running;
  bool other_sock_client_address;
  bool closing_socket;
  int serve_sock_fd;
  unsigned int count;
  std::string working_dir;
  Glib::Mutex working_dir_mutex;

  Sem_sync close_sem;

  // make filenames_mutex mutable so that we can use it in
  // the const member function Socket_server::get_filenames()
  // we can use a Shared_ptr rather than a Shared_lock_ptr
  // object here - filenames_mutex will do the locking of the
  // contained object and we do not actually copy filenames_s
  // in more than one thread (it is only copied in the main (GUI)
  // thread)
  Shared_ptr<FilenamesList> filenames_s;
  mutable Glib::Mutex filenames_mutex;

  Glib::Mutex socket_mutex;
  Glib::Mutex stdout_mutex;
  Glib::Mutex fax_to_send_mutex;

  Glib::Dispatcher stdout_notify;
  Glib::Cond stdout_cond;
  Glib::Cond fax_to_send_cond;

  std::string stdout_text;
  void write_stdout_from_thread(const char*);
  void write_stdout_dispatcher_slot(void);

  std::string port;

  std::pair<std::string, unsigned int> fax_to_send;

  void socket_thread(void);
  bool accept_on_client(void);
  void read_socket(int, int, const char*);
  void read_queued_faxes(void);
  void save_queued_faxes(void);
  bool is_valid_peer(const sockaddr_in&);

  // this is a concrete class not to be copied or assigned
  Socket_server(const Socket_server&);
  void operator=(const Socket_server&);
public:

  // Glib::Dispatcher is a thread safe signal
  // the main GUI thread should connect to it to be
  // notified when the server has received a fax which
  // is to be sent by efax-gtk or a fax has been
  // removed from the list using Socket_server::remove_file()
  Glib::Dispatcher filelist_changed_notify;
  // this signal will also be emitted if the reason for the
  // change was a fax received from the socket for sending
  // (the fax particulars can be obtained with get_fax_to_send())
  Glib::Dispatcher fax_to_send_notify;

  sigc::signal1<void, const char*> stdout_message;

  void add_file(const std::string&);
  int remove_file(const std::string&);
  std::pair<Shared_ptr<FilenamesList>, Shared_ptr<Glib::Mutex::Lock> > get_filenames(void) const;
  std::pair<std::string, unsigned int> get_fax_to_send(void);  
  bool is_server_running(void) const {return server_running;}
  std::string get_port(void) const {return port;}
  bool get_other_sock_client_address(void) const {return other_sock_client_address;}
  int get_count(void) const {return count;}
  void start(const std::string& port, bool other_sock_client_address);
  void stop(void);
  Socket_server(void);
  ~Socket_server(void) {if (server_running) stop();}
};

#endif
