/*
 * etPan! -- a mail user agent
 *
 * Copyright (C) 2001, 2002 - DINH Viet Hoa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the libEtPan! project nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * $Id: etpan-folder-params.c,v 1.9 2004/03/18 01:32:02 hoa Exp $
 */

#include "etpan-folder-params.h"

#include <stdlib.h>

#include "etpan-errors.h"

struct etpan_folder_param {
  int info_fetched;
  uint32_t num;
  uint32_t recent;
  uint32_t unseen;
  int opened;
  int visibility;
  int folder_type;
};

static inline struct etpan_folder_param * param_new(void)
{
  struct etpan_folder_param * param;

  param = malloc(sizeof(struct etpan_folder_param));
  if (param == NULL)
    return param;
  
  param->info_fetched = 0;
  param->num = 0;
  param->recent = 0;
  param->unseen = 0;
  param->opened = 1;
  param->visibility = 1;
  param->folder_type = ETPAN_FOLDER_TYPE_NONE;

  return param;
}

static inline void param_free(struct etpan_folder_param * param)
{
  free(param);
}

/*
  create the folder states
*/

int etpan_folder_params_add(struct etpan_folder_params * params,
    struct mailfolder * folder)
{
  chashdatum key;
  chashdatum data;
  struct etpan_folder_param * param;
  int r;

  key.data = &folder;
  key.len = sizeof(folder);
  r = chash_get(params->hash, &key, &data);
  if (r == 0)
    return NO_ERROR;

  param = param_new();
  if (param == NULL)
    goto err;

  r = carray_add(params->tab, folder, NULL);
  if (r < 0) {
    param_free(param);
    goto err;
  }
  
  data.data = param;
  data.len = 0;

  r = chash_set(params->hash, &key, &data, NULL);
  if (r < 0)
    goto err;

  return NO_ERROR;

 err:
  return ERROR_MEMORY;
}


/*
  add recursively parameters for folder, don't add parameters for 
  folders that already have it.
*/

int etpan_folder_params_add_recursive(struct etpan_folder_params * params,
    struct mailfolder * root)
{
  unsigned int i;
  int r;

  r = etpan_folder_params_add(params, root);
  if (r != NO_ERROR)
    goto err;

  for(i = 0 ; i < carray_count(root->fld_children) ; i ++) {
    struct mailfolder *  child;

    child = carray_get(root->fld_children, i);

    r = etpan_folder_params_add_recursive(params, child);
    if (r != NO_ERROR)
      goto err;
  }

  return NO_ERROR;

 err:
  return ERROR_MEMORY;
}


/* initializes structures */

struct etpan_folder_params * etpan_folder_params_new(void)
{
  struct etpan_folder_params * params;

  params = malloc(sizeof(struct etpan_folder_params));
  if (params == NULL)
    goto err;

  params->hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (params->hash == NULL)
    goto free;
  params->tab = carray_new(128);
  if (params->tab == NULL)
    goto free_hash;

  return params;

 free_hash:
  chash_free(params->hash);
 free:
  free(params);
 err:
  return NULL;
}

void etpan_folder_params_free(struct etpan_folder_params * params)
{
  etpan_folder_params_clear(params);
  carray_free(params->tab);
  chash_free(params->hash);
  free(params);
}


/*
  destroys the folder states
*/

void etpan_folder_params_clear(struct etpan_folder_params * params)
{
  unsigned int i;
  int r;
  chashdatum key;
  chashdatum data;

  for(i = 0 ; i < carray_count(params->tab) ; i ++) {
    struct etpan_folder_param * param;
    struct mailfolder * folder;

    folder = carray_get(params->tab, i);

    key.data = &folder;
    key.len = sizeof(folder);
    r = chash_get(params->hash, &key, &data);
    if (r == 0) {
      param = data.data;
      param_free(param);
    }
  }
  chash_clear(params->hash);
  carray_set_size(params->tab, 0);
}


/* stat folder */

int etpan_folder_get_stat(struct etpan_folder_params * folder_params,
    struct mailfolder * folder, uint32_t * num,
    uint32_t * recent, uint32_t * unseen)
{
  struct etpan_folder_param * param;
  chashdatum key;
  chashdatum data;

  key.data = &folder;
  key.len = sizeof(folder);
  chash_get(folder_params->hash, &key, &data);

  param = data.data;
  if (!param->info_fetched)
    return 0;
  
  * num = param->num;
  * recent = param->recent;
  * unseen = param->unseen;
  
  return 1;
}

void etpan_folder_set_stat(struct etpan_folder_params * folder_params,
    struct mailfolder * folder, int fetched, uint32_t num,
    uint32_t recent, uint32_t unseen)
{
  struct etpan_folder_param * param;
  chashdatum key;
  chashdatum data;

  key.data = &folder;
  key.len = sizeof(folder);
  chash_get(folder_params->hash, &key, &data);

  param = data.data;

  param->info_fetched = fetched;
  param->num = num;
  param->recent = recent;
  param->unseen = unseen;
}

void etpan_folder_set_visibility(struct etpan_folder_params * folder_params,
    struct mailfolder * folder, int visible)
{
  struct etpan_folder_param * param;
  chashdatum key;
  chashdatum data;

  key.data = &folder;
  key.len = sizeof(folder);
  chash_get(folder_params->hash, &key, &data);

  param = data.data;

  param->visibility = visible;
}

int etpan_folder_is_visible(struct etpan_folder_params * folder_params,
    struct mailfolder * folder)
{
  struct etpan_folder_param * param;
  chashdatum key;
  chashdatum data;

  key.data = &folder;
  key.len = sizeof(folder);
  chash_get(folder_params->hash, &key, &data);

  param = data.data;

  return param->visibility;
}

static void update_visibility(struct etpan_folder_params * params,
    struct mailfolder * folder, int visible)
{
  unsigned int i;

  for(i = 0 ; i < carray_count(folder->fld_children) ; i ++) {
    chashdatum key;
    chashdatum data;
    struct mailfolder * child;
    struct etpan_folder_param * child_param;
    
    child = carray_get(folder->fld_children, i);
    
    key.data = &child;
    key.len = sizeof(child);
    chash_get(params->hash, &key, &data);
    
    child_param = data.data;
    
    if (child_param->visibility != visible) {
      child_param->visibility = visible;
      
      if (!visible)
        update_visibility(params, child, 0);
      else
        update_visibility(params, child, child_param->opened);
    }
  }
}

int etpan_folder_is_opened(struct etpan_folder_params * folder_params,
    struct mailfolder * folder)
{
  struct etpan_folder_param * param;
  chashdatum key;
  chashdatum data;

  key.data = &folder;
  key.len = sizeof(folder);
  chash_get(folder_params->hash, &key, &data);

  param = data.data;

  return param->opened;
}

void etpan_folder_set_opened(struct etpan_folder_params * folder_params,
    struct mailfolder * folder, int opened)
{
  struct etpan_folder_param * param;
  chashdatum key;
  chashdatum data;

  key.data = &folder;
  key.len = sizeof(folder);
  chash_get(folder_params->hash, &key, &data);

  param = data.data;

  param->opened = opened;

  update_visibility(folder_params, folder, param->opened);
}

void etpan_folder_change_opened(struct etpan_folder_params * folder_params,
    struct mailfolder * folder)
{
  struct etpan_folder_param * param;
  chashdatum key;
  chashdatum data;

  key.data = &folder;
  key.len = sizeof(folder);
  chash_get(folder_params->hash, &key, &data);

  param = data.data;

  param->opened = !param->opened;

  update_visibility(folder_params, folder, param->opened);
}

void etpan_folder_set_folder_type(struct etpan_folder_params *
    folder_params,
    struct mailfolder * folder, int folder_type)
{
  struct etpan_folder_param * param;
  chashdatum key;
  chashdatum data;

  key.data = &folder;
  key.len = sizeof(folder);
  chash_get(folder_params->hash, &key, &data);

  param = data.data;

  param->folder_type = folder_type;
}


int etpan_folder_get_folder_type(struct etpan_folder_params *
    folder_params,
    struct mailfolder * folder)
{
  struct etpan_folder_param * param;
  chashdatum key;
  chashdatum data;

  key.data = &folder;
  key.len = sizeof(folder);
  chash_get(folder_params->hash, &key, &data);

  param = data.data;

  return param->folder_type;
}
