/*
 * FISG - Fast IRC Statistic Generator
 * Programmed and designed by Matti 'ccr' Hamalainen
 * (C) Copyright 2003 Tecnic Software productions (TNSP)
 *
 * Please read file 'COPYING' for information on license and distribution.
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <errno.h>
#include <ctype.h>
#include <time.h>
#include <assert.h>
#include "fisg.h"
#include "th_util.h"
#include "th_args.h"
#include "th_config.h"
#include "in_formats.h"
#include "out_formats.h"


/*
 * Misc globals
 */
int	setOutputFormat = 0;
int	setCurrInputFormat = 0;

int	nverbosity = 2;
int	nsourceFileNames = 0, nconfigFileNames = 0;
int	sourceFileFormats[SET_MAX_INFILES];
char	*sourceFileNames[SET_MAX_INFILES],
	*destFileName = NULL,
	*userFileName = NULL;
char	*configFileNames[SET_MAX_INFILES];

char	*progName = NULL;


/*
 * Options and help
 */
t_arg argList[] = {
	{ "?",	"help",		"Show this help", 0 },
	{ "o",	"output",	"Specify output file (default stdout)", 1 },
	{ "u",	"user-file",	"Users file (default: no user-file)", 1 },
	{ "f",	"format",	"Specify input format (logfile format)", 1 },
	{ "of", "output-format","Specify output format", 1 },
	{ "F",	"show-formats",	"Show list of predefined log- and output-formats", 0 },
	{ "c",	"config",	"Specify configuration file (default: none)", 1 },
	{ "q",	"quiet",	"Use multiple times for more silence.", 0 }
};

const int argListN = (sizeof(argList) / sizeof(t_arg));


void FERR(const char *pcFormat, ...)
{
 va_list ap;
 va_start(ap, pcFormat);
 fprintf(stderr, RA_NAME ": ");
 vfprintf(stderr, pcFormat, ap);
 va_end(ap);
}


void NICKDEBUG(const char *pcFormat, ...)
{
/*
 va_list ap;
 va_start(ap, pcFormat);
 fprintf(stderr, "NICKDEBUG: ");
 vfprintf(stderr, pcFormat, ap);
 va_end(ap);
*/
}


void showHelp()
{
 fprintf(stderr, "\n" RA_NAME " " RA_VERSION " " RA_COPYRIGHT "\n");
 fprintf(stderr, "This software is licensed under GNU General Public License version 2\n");
 fprintf(stderr, "Usage: %s [options] [source#1] [source#2...]\n", progName);

 th_showHelp(argList, argListN);
}


void handleOpt(const int optN, char *optArg, char *currArg)
{
 int i;
 BOOL isFound;
 
 switch (optN) {
 case 0: showHelp(); exit(0); break;
 
 case 1:
 	/* Specify output filename */
 	if (optArg)
 		destFileName = optArg;
 		else {
 		FERR("No output filename specified!\n");
 		exit(2);
 		}
	break;

 case 2:
 	/* Specify user-file filename */
 	if (optArg)
 		userFileName = optArg;
 		else {
 		FERR("No user-file filename specified!\n");
 		exit(2);
 		}
	break;

 case 3:
 	/* Specify input format */
 	if (optArg)
 		{
 		/* Go through the list */
 		isFound = FALSE;
 		i = 0;
 		while ((i < nInputFormats) && (!isFound))
 			{
 			if (strcmp(optArg, inputFormats[i].ifName) == 0)
 				isFound = TRUE;
 				else
 				i++;
 			}

 		
 		/* Check */
 		if (!isFound)
 			{
 			FERR("Invalid input (log-file) format '%s'\n", optArg);
 			exit(2);
 			}
 		
 		setCurrInputFormat = i;
 		} else {
 		FERR("No input (log-file) format specified!\n");
 		exit(2);
 		}
	break;

 case 4:
 	/* Specify output format */
 	if (optArg)
 		{
 		/* Go through the list */
 		isFound = FALSE;
 		i = 0;
 		while ((i < nOutputFormats) && (!isFound))
 			{
 			if (strcmp(optArg, outputFormats[i].ofName) == 0)
 				isFound = TRUE;
 				else
 				i++;
 			}

 		/* Check */
 		if (!isFound)
 			{
 			FERR("Invalid output format '%s'\n", optArg);
 			exit(2);
 			}

 		setOutputFormat = i;
 		} else {
 		FERR("No output format specified!\n");
 		exit(2);
 		}
 	break;

 case 5:
 	/* Show list of input and output formats */
 	fprintf(stderr, RA_NAME ": Available pre-defined INPUT (log) formats:\n");
 	for (i = 0; i < nInputFormats; i++)
 		{
 		fprintf(stderr, "  %-8s	- %s %s\n",
 			inputFormats[i].ifName,
 			inputFormats[i].ifDescription,
 			(i == 0) ? "(default)" : "");
 		}
 
 	fprintf(stderr, "\n" RA_NAME ": Available OUTPUT formats:\n");
 	for (i = 0; i < nOutputFormats; i++)
 		{
 		fprintf(stderr, "  %-8s	- %s %s\n",
 			outputFormats[i].ofName,
 			outputFormats[i].ofDescription,
 			(i == setOutputFormat) ? "(default)" : "");
 		}
 	
 	fprintf(stderr, "\n");
 	exit(0);
 	break;

 case 6:
 	/* Specify configuration filename */
 	if (optArg)
 		{
		if (nconfigFileNames < SET_MAX_INFILES)
			{
			configFileNames[nconfigFileNames] = optArg;
			nconfigFileNames++;
			}
 		} else {
 		FERR("No configuration filename specified!\n");
 		exit(2);
 		}
	break;

 case 7:
 	/* Quiet -- lessen verbosity */
 	nverbosity--;
 	break;
 	
 default:
 	/* Error */
 	FERR("Unknown argument '%s'.\n", currArg);
 	break;
 }
} 	


void handleFile(char *currArg)
{
 /* Was not option argument */
 if (nsourceFileNames < SET_MAX_INFILES)
	{
	sourceFileNames[nsourceFileNames] = currArg;
	sourceFileFormats[nsourceFileNames] = setCurrInputFormat;
	nsourceFileNames++;
	}
}


/*
 * Parsers
 */
int parse_int(char *inLine, int iLen, int *linePos)
{
 int iResult = 0;

 while (th_isdigit(inLine[*linePos]) && (iLen--))
 	{
 	iResult *= 10;
 	iResult += (inLine[(*linePos)++] - '0');
 	}

 return iResult;
}


t_user_entry *parse_newuser(t_stats *pStats, char *newNick)
{
 t_user_entry *tmpUser;
 t_nick_entry *tmpNick;
 
 /* Check if nick matches existing user record */
 tmpUser = user_search(pStats->nickList, newNick);
 if (tmpUser == NULL)
	{
	/* No, we need to create a new one */
	tmpUser = user_new(newNick);
	tmpNick = nick_new(newNick);

	tmpNick->user = tmpUser;
	if (nick_insert(pStats->nickList, tmpNick) != 0)
		{
		/* Failed, due to hash */
		FERR("nick_insert() failed, hash: '%s'\n", newNick);
		user_free(tmpUser);
		nick_free(tmpNick);
		return NULL;
		}

	user_insert(&pStats->userList, tmpUser);
	}

 return tmpUser;
}


int parse_generic(char *inLine, char *fmt, t_lineinfo *lineInfo, t_stats *pStats)
{
 int linePos, i;
 BOOL isOK, isEnd, tmpNick1S = FALSE, tmpNick2S = FALSE;
 t_user_entry *tmpUser;
 char	tmpStr[SET_MAX_NICKLEN + 1] = "",
 	tmpNick1[SET_MAX_NICKLEN + 1],
 	tmpNick2[SET_MAX_NICKLEN + 1],
 	tmpDest, c;

 /* Initialize */
 linePos = 0;
 tmpUser = NULL;
 isOK = TRUE;
 
 /* Parse the line via format-string */
 while (*fmt && isOK)
 {
 if (*fmt == '%')
	{
	switch (*(++fmt)) {
	/* Generic matching */
	case '?':
		/* Match anything */
		fmt++;
		if (inLine[linePos])
			linePos++;
			else
			isOK = FALSE;
		break;

	case '*':
		/* Match anything until next char */
		fmt++;
		while (inLine[linePos] && (inLine[linePos] != *fmt)) linePos++;
		break;
		
	case '@':
		/* Match irssi style optional '@|+| ' */
		fmt++;
		if (!inLine[linePos]) isOK = FALSE;
		if ((inLine[linePos] == '@') ||
			(inLine[linePos] == '+') ||
			th_isspace(inLine[linePos]))
			linePos++;
		break;

	/* Timestamps */
	case 'H': lineInfo->tHours = parse_int(inLine, 2, &linePos); fmt++; break;
	case 'M': lineInfo->tMinutes = parse_int(inLine, 2, &linePos); fmt++; break;	
	case 'S': lineInfo->tSeconds = parse_int(inLine, 2, &linePos); fmt++; break;

	case 'Y': lineInfo->dYear = parse_int(inLine, 4, &linePos); fmt++; break;
		
	case 'y':
		/* 2-digit year */
		i = parse_int(inLine, 2, &linePos);
		if (i < 70)
			i += 2000;
			else
			i += 1900;

		lineInfo->dYear = i;
		fmt++;
		break;

	case 'd': lineInfo->dDay = parse_int(inLine, 2, &linePos); fmt++; break;
	case 'j': lineInfo->dMonth = parse_int(inLine, 2, &linePos); fmt++; break;


	/* Special matches */
	case 'n':
	case 'N':
		/* Nick */
		tmpDest = *fmt;
		fmt++;

		/* Find the start of the nick */
		th_findnext(inLine, &linePos);

		/* Get the nick to temp buffer */
		i = 0; isEnd = FALSE;

		c = inLine[linePos];
		if (!th_isalpha(c) && !th_isspecial(c)) isOK = FALSE;
		
		while (isOK && !isEnd)
			{
			c = inLine[linePos];
			if (!c || (c == *fmt) || th_isspace(c) || (i >= SET_MAX_NICKLEN))
				isEnd = TRUE;
				else
				{
				if (th_isalpha(c) || th_isdigit(c) || th_isspecial(c) || (c == '-'))
					tmpStr[i++] = inLine[linePos++];
					else
					isOK = FALSE;
				}
			}
			
		tmpStr[i++] = 0;
		
		while (inLine[linePos] && th_isspace(inLine[linePos]) && (inLine[linePos] != *fmt)) linePos++;
		if (inLine[linePos] != *fmt) isOK = FALSE;

		/* Find user or add new */
		if (isOK && (i > 0))
			switch (tmpDest) {
			case 'n': tmpNick1S = TRUE; strcpy(tmpNick1, tmpStr); break;
			case 'N': tmpNick2S = TRUE; strcpy(tmpNick2, tmpStr); break;
			}
		break;

	case 'm':
		/* Mode */
		fmt++;
		while (inLine[linePos] && (inLine[linePos] != *fmt)) linePos++;
		break;

	case 'c':
		/* Channel */
		fmt++;
		while (inLine[linePos] && (inLine[linePos] != *fmt)) linePos++;
		break;

	case 't':
		/* Text */
		fmt++;
		i = 0;
		while (inLine[linePos] && (inLine[linePos] != *fmt) && (i < SET_MAX_BUF))
			lineInfo->pText[i++] = inLine[linePos++];

		lineInfo->pText[i++] = 0;
		break;

	/* Error */
	default:
		FERR("Syntax error in format-string '%s'\n", fmt);
		return -1;
	}
	} else {
	/* Check matches */
	if (*fmt != inLine[linePos])
		isOK = FALSE;

	fmt++;
	linePos++;
	}

 } /* while(*fmt) */


 if (isOK)
 	{
 	if (tmpNick1S)
 		lineInfo->pUser = parse_newuser(pStats, tmpNick1);

	if (tmpNick2S)
		lineInfo->pUser2 = parse_newuser(pStats, tmpNick2);
 	}

 return !isOK;
}


t_user_entry *parse_public(char *infLine, char *fmt, t_stats *pStats)
{
 t_lineinfo lineInfo;
 t_int	nWords, nQuestions, nYelling;
 int linePos;
 BOOL isWord;

 /* Try to parse the line */
 if (parse_generic(infLine, fmt, &lineInfo, pStats))
 	return NULL;

 /* If the text is empty, we don't need to analyze it */
 if (!lineInfo.pText[0])
 	return lineInfo.pUser;
 
 /* Detect URLs */
 if (strstr(lineInfo.pText, "http://") != NULL)
 	{
	/* Add the URL in list */
	lineInfo.pUser->nURLs++;
	}

 /* Statisticize the actual public message-line */
 linePos = 0;
 isWord = FALSE;

 nQuestions = nYelling = nWords = 0;

 while (lineInfo.pText[linePos])
	{
	if (isWord && th_isspace(lineInfo.pText[linePos]))
		{
		nWords++;
		isWord = FALSE;
		} else
	if ((!isWord) && !th_isspace(lineInfo.pText[linePos]))
		isWord = TRUE;

	if (th_isupper(lineInfo.pText[linePos]))
		lineInfo.pUser->nCaps++;

	switch (lineInfo.pText[linePos]) {
	case '=':
	case ':':
	case ';':
		switch (lineInfo.pText[linePos + 1]) {
		case ')': /* :) */
		case 'D': /* :D */
		case 'P': /* :P */
		case '>': /* :> */
		case ']': /* :] */
			lineInfo.pUser->nHappiness++;
			break;
	
		case '(': /* :( */
		case '[': /* :[ */
		case '/': /* :/ */
		case 'I': /* :I */
			lineInfo.pUser->nHappiness--;
			break;
		}
		break;

	case '(':
	case '<':
		switch (lineInfo.pText[linePos + 1]) {
		case ':':
		case ';':
			lineInfo.pUser->nHappiness++;
			break;
		}
		break;
			
	case ')':
	case '>':
		switch (lineInfo.pText[linePos + 1]) {
		case ':':
		case ';':
			lineInfo.pUser->nHappiness--;
			break;
		}
		break;
			
	case '!':
		nYelling++;
		break;
	
	case '?':
		nQuestions++;
		break;
	}

	lineInfo.pUser->nChars++;
	linePos++;
	}

 /* Add to user's stats */
 if (nYelling) lineInfo.pUser->nYelling++;
 if (nQuestions) lineInfo.pUser->nQuestions++;
 
 lineInfo.pUser->nWords += nWords;
 lineInfo.pUser->nPublics++;

 if ((lineInfo.tHours >= 0) && (lineInfo.tHours < SET_HOURS_DAY))
	{
	lineInfo.pUser->nWordsPerHour[lineInfo.tHours] += nWords;
	lineInfo.pUser->nPublicsPerHour[lineInfo.tHours]++;

	if (lineInfo.pUser->nWords >=
		(lineInfo.pUser->nWordsPerHour[lineInfo.tHours] /
		(lineInfo.pUser->nPublicsPerHour[lineInfo.tHours]+1)))
		{
		if ((!lineInfo.pUser->sComment) || (random() < (RAND_MAX / 3)))
		if (strlen(lineInfo.pText) > SET_MIN_COMMENT_LEN)
			th_strcalloc(&lineInfo.pUser->sComment, lineInfo.pText);
		}
	}

 /* Done, ok. */ 
 return lineInfo.pUser;
}


t_user_entry *parse_misc(char *infLine, char *fmt, t_stats *pStats)
{
 t_lineinfo lineInfo;

 /* Try to parse the line */
 if (parse_generic(infLine, fmt, &lineInfo, pStats))
 	return NULL;

 /* Done, ok. */ 
 return lineInfo.pUser;
}


int parse_kick(char *infLine, char *fmt, t_stats *pStats)
{
 t_lineinfo lineInfo;

 /* Try to parse the line */
 if (parse_generic(infLine, fmt, &lineInfo, pStats))
 	return -1;

 /* Add to user's stats */
 lineInfo.pUser->nGotKicked++;
 lineInfo.pUser2->nKicks++;
 
 /* Done, ok. */ 
 return 0;
}


int parse_nickchange(char *infLine, char *fmt, t_stats *pStats, BOOL autoFollow, BOOL autoHeur)
{
 t_lineinfo lineInfo;
 int i;
 
 /* Try to parse the line */
 if (parse_generic(infLine, fmt, &lineInfo, pStats))
 	return -1;

 /* Let's see if we can autofollow the nick-changes */
 if (autoFollow && (lineInfo.pUser != lineInfo.pUser2))
 {
NICKDEBUG("['%s' -> '%s'] -- ", lineInfo.pUser->userHandle, lineInfo.pUser2->userHandle);
 if (lineInfo.pUser->isManaged && !lineInfo.pUser2->isManaged)
	{
NICKDEBUG("'%s' is alias to '%s'\n", lineInfo.pUser2->userHandle, lineInfo.pUser->userHandle);
	nick_change(pStats->nickList, lineInfo.pUser2, lineInfo.pUser);
	user_delete(&pStats->userList, lineInfo.pUser2);
	user_free(lineInfo.pUser2);
	lineInfo.pUser->nNickChanges++;
	} else
 if (!lineInfo.pUser->isManaged && lineInfo.pUser2->isManaged)
 	{
NICKDEBUG("'%s' is alias to '%s'\n", lineInfo.pUser->userHandle, lineInfo.pUser2->userHandle);

	nick_change(pStats->nickList, lineInfo.pUser, lineInfo.pUser2);
	user_delete(&pStats->userList, lineInfo.pUser);
	user_free(lineInfo.pUser);
	lineInfo.pUser2->nNickChanges++;
 	} else
 if (autoHeur)
 	{
 	/*
 	 * Let's try to determine the "real" user with simple heuristics
 	 */
NICKDEBUG("guessing... %i, %i",autoHeur,autoHeur);

	i = 0;
 	if (strlen(lineInfo.pUser->userHandle) < strlen(lineInfo.pUser2->userHandle))
		i--;
		else
		i++;

	if (th_strmatch(lineInfo.pUser2->userHandle, lineInfo.pUser->userHandle))
		i--;
		
	if (th_strmatch(lineInfo.pUser->userHandle, lineInfo.pUser2->userHandle))
		i++;
		
	if (th_strmatch(lineInfo.pUser2->userHandle, "*^*") || th_strmatch(lineInfo.pUser2->userHandle, "*_*"))
		i -= 2;

	if (th_strmatch(lineInfo.pUser->userHandle, "*^*") || th_strmatch(lineInfo.pUser->userHandle, "*_*"))
		i += 2;

	if (i <= 0)
		{
NICKDEBUG("'%s' is alias to '%s'\n", lineInfo.pUser2->userHandle, lineInfo.pUser->userHandle);

		nick_change(pStats->nickList, lineInfo.pUser2, lineInfo.pUser);
		user_delete(&pStats->userList, lineInfo.pUser2);
		user_free(lineInfo.pUser2);
		lineInfo.pUser->nNickChanges++;
		lineInfo.pUser->isManaged = TRUE;
		} else {
NICKDEBUG("'%s' is alias to '%s'\n", lineInfo.pUser->userHandle, lineInfo.pUser2->userHandle);

		nick_change(pStats->nickList, lineInfo.pUser, lineInfo.pUser2);
		user_delete(&pStats->userList, lineInfo.pUser);
		user_free(lineInfo.pUser);
		lineInfo.pUser2->nNickChanges++;
		lineInfo.pUser2->isManaged = TRUE;
		}
 	}
 } else {
 /* Update the stats */
 lineInfo.pUser->nNickChanges++;
 lineInfo.pUser2->nNickChanges++; 
 }
 
 /* Done, ok. */ 
 return 0;
}


/*
 * A generic logfile parser
 */
int parse_log(FILE *inFile, t_stats *pStats, t_logformat *logFmt, t_config *pCfg)
{
 BOOL autoFollow, autoHeur;
 char inLine[SET_MAX_BUF + 1];
 int lineNum, linePos;
 t_user_entry *tmpUser;
 
 /* Get configuration options */
 autoFollow = th_config_get_bool(pCfg, CFG_GEN_AUTO_FOLLOW_NICKS, TRUE);
 autoHeur = th_config_get_bool(pCfg, CFG_GEN_AUTO_FOLLOW_HEURISTIC, FALSE);

 /* Initial stats */
 pStats->nLogFiles++;

 /* Read and parse the data */
 lineNum = 0;
 while (fgets(inLine, sizeof(inLine), inFile) != NULL)
 {
 linePos = strlen(inLine) - 1;
 if (inLine[linePos] == '\n') inLine[linePos--] = 0;
 if (inLine[linePos] == '\r') inLine[linePos--] = 0;
 pStats->nChars += linePos;
 pStats->nLines++;
 lineNum++;
 linePos = 0;

 /* Check if the line is OK and what type it is */ 
 if (inLine[0])
	{
	if (!parse_public(inLine, logFmt->fmtPublic, pStats))
	if (!parse_public(inLine, logFmt->fmtNotice, pStats))
		{
		if ((tmpUser = parse_public(inLine, logFmt->fmtAction, pStats)))
			tmpUser->nActions++;
			else
		if ((tmpUser = parse_public(inLine, logFmt->fmtNotice, pStats)))
			tmpUser->nNotices++;
			else
		if ((tmpUser = parse_misc(inLine, logFmt->fmtJoin, pStats)))
			tmpUser->nJoins++;
			else
		if (parse_kick(inLine, logFmt->fmtKick, pStats))
		if (parse_nickchange(inLine, logFmt->fmtNickChange, pStats, autoFollow, autoHeur))
			{
			}
		}
 	}
 
 } /* if (fgets()) */

 return 0;
}


/*
 * Userfile parser
 */
void parse_pfield(char *inLine, char *tmpBuf, int bufSize, int *linePos)
{
 int i = 0;

 while (inLine[*linePos] && (inLine[*linePos] != ':') && (i < bufSize))
	tmpBuf[i++] = inLine[(*linePos)++];

 tmpBuf[i++] = 0;
}


int parse_userfile(char *usrFilename, t_stats *pStats, t_config *pCfg)
{
 char inLine[SET_MAX_BUF + 1], tmpStr[SET_MAX_BUF + 1];
 FILE *inFile;
 int i, lineNum, linePos;
 t_user_entry *tmpUser = NULL;
 t_nick_entry *tmpNick = NULL;
 BOOL ignoredUser;

 assert(pStats);
 
 /* Get configuration options */

 /* Try to open the file */
 if ((inFile = fopen(usrFilename, "ra")) == NULL)
	return -1;
 
 /* Read and parse the data */
 lineNum = 0;
 while (fgets(inLine, sizeof(inLine), inFile) != NULL)
 {
 lineNum++;
 linePos = 0;

 /* Check if the line is OK and what type it is */ 
 if ((strlen(inLine) > 1) && (inLine[0] != '#'))
	{
	/* Check if it's ignored user */
	if (inLine[linePos] == '!')
		{
		linePos++;
		ignoredUser = TRUE;
		} else
		ignoredUser = FALSE;
	
	/* Get the handle */
	parse_pfield(inLine, tmpStr, SET_MAX_BUF, &linePos);

	/* Check if next field exists */
	if (inLine[linePos] != ':')
		{
		FERR("Error in userfile '%s', line %i - missing fields.\n",
			usrFilename, lineNum);
		} else {
		/* Allocate a new user and nick records */
		tmpUser = user_new(tmpStr);
		tmpUser->isIgnored = ignoredUser;
		tmpUser->isManaged = TRUE;
		user_insert(&pStats->userList, tmpUser);
		
		/* Get alias nicks */
		linePos++;
		while (inLine[linePos] && (inLine[linePos] != ':'))
			{
			/* Get one nick */
			th_findnext(inLine, &linePos);

			i = 0;
			while (inLine[linePos] && (inLine[linePos] != ':') &&
				(!th_isspace(inLine[linePos])) && (i < SET_MAX_BUF))
				tmpStr[i++] = inLine[linePos++];
	
			tmpStr[i++] = 0;

			th_findnext(inLine, &linePos);

			/* Add to user */
			tmpNick = nick_new(tmpStr);
			tmpNick->user = tmpUser;
			
			if (nick_insert(pStats->nickList, tmpNick) != 0)
			FERR("nick_insert() failed, hash: '%s'\n", tmpStr);
			}
		
		/* Get image path */
		if (inLine[linePos] == ':')
			{
			linePos++;
			if ((inLine[linePos] != ':') && !th_isspace(inLine[linePos]))
				{
				/* Set to user record */
				parse_pfield(inLine, tmpStr, SET_MAX_BUF, &linePos);
				
				th_strcalloc(&tmpUser->picPath, tmpStr);
				}
			}
		}
 	}
 
 } /* if (fgets()) */
 
 return 0;
}


/*
 * Allocates and initializes a new stats structure
 */
t_stats *stats_new(void)
{
 t_stats *pResult;
 
 /* Allocate memory for new node */
 pResult = (t_stats *) calloc(1, sizeof(t_stats));
 if (pResult == NULL) return NULL;
 
 /* Initialize fields */

 /* Return result */
 return pResult;
}


/*
 * Deallocates stats structure
 */
void stats_free(t_stats *pStats)
{
 t_user_entry *pUser, *nUser;
 t_nick_entry *pNick, *nNick;
 int i;
 assert(pStats);
 
 /* Free userlist */
 pUser = pStats->userList;
 while (pUser)
 	{
 	nUser = pUser->pNext;
 	user_free(pUser);
 	pUser = nUser;
 	}

 /* Free nicklist */
 for (i = 0; i < SET_HASH_MAXINDEX; i++)
	{
	/* Find from linked list */
	pNick = pStats->nickList[i];
	while (pNick)
		{
		nNick = pNick->pNext;
		nick_free(pNick);
		pNick = nNick;
		}
	}
 
 /* Free user index */
 free(pStats->userIndex);
  
 /* Free stats structure */
 free(pStats);
}


/*
 * Compare function for qsort() that compares 2 nodes for nTotalScore
 */
int stats_index_cmp(const void *pNode1, const void *pNode2)
{
 t_user_entry *pUser1, *pUser2;

 pUser1 = * (t_user_entry **) pNode1;
 pUser2 = * (t_user_entry **) pNode2;

 if (pUser1->nTotalScore > pUser2->nTotalScore)
 	return -1;
 	else
 if (pUser1->nTotalScore < pUser2->nTotalScore)
 	return 1;
 	else
 	return 0;
}


/*
 * Create a userIndex for given stats from userList
 */
int stats_index(t_stats *pStats)
{
 t_user_entry *pCurr;
 long int i;
 assert(pStats);
 
 /* Create sorted index */
 pCurr = pStats->userList;
 pStats->nUsers = pStats->nIgnored = 0;
 while (pCurr)
 	{
 	if (pCurr->isIgnored)
		pStats->nIgnored++;
		else
		pStats->nUsers++;
		
 	pCurr = pCurr->pNext;
 	}

 /* Check number of nodes */
 if (pStats->nUsers > 0)
	{
	/* Allocate memory for index-table */
	pStats->userIndex = (t_user_entry **) malloc(sizeof(t_user_entry *) * pStats->nUsers);
	if (pStats->userIndex == NULL) return -6;

	/* Get node-pointers to table */
	i = 0;
	pCurr = pStats->userList;
	while (pCurr)
		{
		/* Index only, if user is unignored */
		if (!pCurr->isIgnored)
			pStats->userIndex[i++] = pCurr;

		pCurr = pCurr->pNext;
		}
	}

 return 0;
}


/*
 * Compute stats from userlist
 */
int stats_compute_rank(t_stats *pStats, t_config *pCfg)
{
 t_user_entry *pCurr;
 int i;
 t_float iTotalActivity, iMaxActivity;
 BOOL usePisgScoring;
 
 /* Get configuration options */
 usePisgScoring = th_config_get_bool(pCfg, CFG_GEN_USE_PISG_SCORING, FALSE);

 /* Go through the userlist */
 pCurr = pStats->userList;
 while (pCurr)
 	{
 	/* Check if user is ignored */
 	if (!pCurr->isIgnored)
 	{
	/* Activity */
	iTotalActivity = 0;
	for (i = 0; i < SET_HOURS_DAY; i++)
		{
		if (usePisgScoring)
			pCurr->activityPercentPerHour[i] = pCurr->nPublicsPerHour[i];
			else
			pCurr->activityPercentPerHour[i] =
			((t_float) pCurr->nWordsPerHour[i]) * ((t_float) pCurr->nPublicsPerHour[i]);

 		iTotalActivity += pCurr->activityPercentPerHour[i];

		pStats->activityPercentPerHour[i] += pCurr->activityPercentPerHour[i];
 		}

	/* Compute activity-% for each hour */
	for (i = 0; i < SET_HOURS_DAY; i++)
		{
		if (iTotalActivity > 0)
			{
			pCurr->activityPercentPerHour[i] =
			(pCurr->activityPercentPerHour[i] / iTotalActivity) * 100.0f;
			} else
			pCurr->activityPercentPerHour[i] = 0.0f;
		}

	/* Compute W/P and C/W */
	if (pCurr->nPublics > 0)
		pCurr->nWordsPerPublic = ((t_float) pCurr->nWords / (t_float) pCurr->nPublics);
		else
		pCurr->nWordsPerPublic = 0;

	if (pCurr->nWords > 0)
		pCurr->nCharsPerWord = ((t_float) pCurr->nChars / (t_float) pCurr->nWords);
		else
		pCurr->nCharsPerWord = 0;
	
	/*
	 * Compute total score
	 */
	if (usePisgScoring)
		pCurr->nTotalScore = pCurr->nPublics;
		else
		pCurr->nTotalScore = (pCurr->nWordsPerPublic + pCurr->nCharsPerWord) * pCurr->nPublics;

	} /* isIgnored? */

	/* Next node */
 	pCurr = pCurr->pNext;
 	}

 /*
  * Compute total activity %
  */
 pStats->activityPeak = -1;
 iMaxActivity = -1;
 iTotalActivity = 0.0f;
 for (i = 0; i < SET_HOURS_DAY; i++)
	{
	iTotalActivity += pStats->activityPercentPerHour[i];
	if (pStats->activityPercentPerHour[i] > iMaxActivity)
		{
		iMaxActivity = pStats->activityPercentPerHour[i];
		pStats->activityPeak = i;
		}
	}

 if (iTotalActivity > 0) { 
 for (i = 0; i < SET_HOURS_DAY; i++)
	{
	pStats->activityPercentPerHour[i] = (pStats->activityPercentPerHour[i] / iTotalActivity) * 100.0f;
	}
 }

 return 0;
}


int stats_compute_rest(t_stats *pStats, t_config *pCfg)
{
 t_user_entry *pCurr;
 t_int statMax, nUser;
 
 /* Get configuration options */
 if (th_config_get_bool(pCfg, CFG_GEN_STAT_ONLY_LISTED, FALSE))
 	{
	statMax = th_config_get_int(pCfg, CFG_GEN_SHOWMAX, CFG_GEN_SHOWMAX_DEF);
	if (statMax > pStats->nUsers)
		statMax = pStats->nUsers;
	} else
	statMax = pStats->nUsers;
 
 /* Initialize */
 nUser = 0;
 while (pStats->userIndex[nUser]->isIgnored) nUser++;
 
 pStats->mostStupid = pStats->mostLoud = pStats->mostActions =
 pStats->mostModes = pStats->mostKicks = pStats->mostKicked =
 pStats->mostCaps = pStats->mostHappy = pStats->mostSad =
 pStats->mostURLs = pStats->mostJoins = pStats->userIndex[nUser];

 /* Go through the userlist */
 for (nUser = 0; nUser < statMax; nUser++)
 	{
 	pCurr = pStats->userIndex[nUser];

 	/* Check if user is ignored */
 	if (!pCurr->isIgnored)
 	{
	/* More stupid */
	if (pCurr->nQuestions > pStats->mostStupid->nQuestions)
		pStats->mostStupid = pCurr;

	/* More loud? */
	if (pCurr->nYelling > pStats->mostLoud->nYelling)
		pStats->mostLoud = pCurr;
		
	/* More actions? */
	if (pCurr->nActions > pStats->mostActions->nActions)
		pStats->mostActions = pCurr;

	/* More kicks? */
	if (pCurr->nKicks > pStats->mostKicks->nKicks)
		pStats->mostKicks = pCurr;

	/* More kicked? */
	if (pCurr->nGotKicked > pStats->mostKicked->nGotKicked)
		pStats->mostKicked = pCurr;

	/* More caps per chars? */
	if (pCurr->nCaps > 0)
		pCurr->nCapsPercent = ((t_float) pCurr->nCaps / (t_float) pCurr->nChars) * 100.0f;

	if (pCurr->nCapsPercent > pStats->mostCaps->nCapsPercent)
		pStats->mostCaps = pCurr;
	
	/* More happy? */
	if (pCurr->nHappiness > pStats->mostHappy->nHappiness)
		pStats->mostHappy = pCurr;

	/* More sad? */
	if (pCurr->nHappiness < pStats->mostSad->nHappiness)
		pStats->mostSad = pCurr;
		
	/* More URLs pasted? */
	if (pCurr->nURLs > pStats->mostURLs->nURLs)
		pStats->mostURLs = pCurr;

	/* More Joins? */
	if (pCurr->nJoins > pStats->mostJoins->nJoins)
		pStats->mostJoins = pCurr;

	} /* isIgnored? */

 	}

 return 0;
}


void output_userfile(FILE *f, t_stats *pStats)
{
 t_user_entry *pCurr;
 t_nick_entry *pNick;
 int i;

 fprintf(f, "# Userfile for " RA_NAME " (" RA_FULLNAME ")\n");
 
 pCurr = pStats->userList;
 while (pCurr)
 	{
 	if (pCurr->isIgnored)
 		fprintf(f, "!");
 	
 	fprintf(f, "%s:", pCurr->userHandle);
 
 	for (i = 0; i < SET_HASH_MAXINDEX; i++)
		{
 		/* Find from linked list */
		pNick = pStats->nickList[i];
		while (pNick)
			{
			if (pNick->user == pCurr)
			fprintf(f, "%s ", pNick->nick);
			pNick = pNick->pNext;
			}
		}

	fprintf(f, ":");

	if (pCurr->picPath)
		fprintf(f, "%s:", pCurr->picPath);
		else
	if (pCurr->linkURL)
		fprintf(f, ":");

	if (pCurr->linkURL)
		fprintf(f, "%s", pCurr->linkURL);
	fprintf(f, "\n");

	pCurr = pCurr->pNext;
	}
}




/*
 * NOTICE: Since this utility is designed to be "one pass/shot" program,
 * we don't free any memory used here. If you take this code, remember it
 * and add possible *free()'s where appropriate.
 */
int main(int argc, char *argv[])
{
 FILE *myFile;
 int iResult, i, j;
 t_config *myConfig = NULL;
 t_stats *myStats = NULL;
 time_t myTime1, myTime2;
 char tmpStr[1024] = "";

 /*
  * Initialize basics
  */
 progName = argv[0];

 time(&myTime1);
 srandom(myTime1);
 
 /*
  * Allocate stats
  */
 myStats = stats_new();
 if (myStats == NULL)
 	{
 	FERR("Could not allocate memory for statistics!\n");
 	return -11;
 	}
 
 /*
  * Parse arguments
  */
 th_processArgs(argc, argv,
 	argList, argListN,
 	handleOpt, handleFile);

 if (nsourceFileNames <= 0)
 	{
 	FERR("No input files specified!\n");
 	return 0;
 	}

 /*
  * Read and parse general configuration file
  */
 if (nconfigFileNames <= 0)
 	{
 	FERR("No configuration file(s) specified.\n");
 	}

 for (i = 0; i < nconfigFileNames; i++)
	{
	if (nverbosity >= 1)
	FERR("Configuration '%s'\n", configFileNames[i]);

	if (th_config_read(configFileNames[i], &myConfig) < 0)
	 	{
	 	FERR("Error reading configuration file.\n");
	 	return -12;
	 	}
	}

 /*
  * Parse the users file
  */
 if (userFileName)
	parse_userfile(userFileName, myStats, myConfig);
	else
	parse_userfile(th_config_get_str(myConfig, CFG_GEN_USER_FILE, NULL), myStats, myConfig);

 
 /*
  * Read the source-file(s)
  */
 if (nverbosity >= 1)
 FERR("Parsing %d sources. Please wait...\n", nsourceFileNames);

 for (i = 0; i < nsourceFileNames; i++)
	{
	/* Try to open the logfile */
	if ((myFile = fopen(sourceFileNames[i], "ra")) == NULL)
		{
		FERR("Error opening input file '%s' (%s)\n", sourceFileNames[i], strerror(errno));
		return -1;
		}

	/* Parse with selected parser */
	iResult = parse_log(myFile, myStats,
		&inputFormats[sourceFileFormats[i]],
		myConfig);

	if (nverbosity >= 2)
		{
		/* Show progress meter */
		for (j = 0; j < strlen(tmpStr); j++)
			fputc('\b', stderr);

		snprintf(tmpStr, sizeof(tmpStr),
			RA_NAME ": Processed %i%%",  (((i+1) * 100) / nsourceFileNames));

		fputs(tmpStr, stderr);
		}

	/* Close file, report errors */
	fclose(myFile);
	if (iResult < 0)
		{
		FERR("Error #%i reading file (%s)\n", iResult, strerror(errno));
		return 2;
		}
	}

 if (nverbosity >= 2)
	fprintf(stderr, "\n"); 

 /*
  * Calculate rank and stats for sorting
  */
 if (nverbosity >= 1)
 FERR("Computing statistics...\n");

 if (stats_compute_rank(myStats, myConfig) < 0)
 	{
 	FERR("Error while computing rankings!\n");
 	return -10;
 	}

 /*
  * Index the list for sorting
  */
 if (stats_index(myStats) < 0)
 	{
 	FERR("Error while indexing userlist!\n");
 	return -9;
 	}

 /* Sort the indexes by score */
 if (nverbosity >= 1)
 FERR("%li users (%li ignored), sorting...\n",
 	myStats->nUsers, myStats->nIgnored);

 qsort(myStats->userIndex, myStats->nUsers, sizeof(t_user_entry *), stats_index_cmp);

 /*
  * Compute rest of statistics
  */
 if (nverbosity >= 1)
 FERR("Computing more statistics...\n");
 if (stats_compute_rest(myStats, myConfig) < 0)
 	{
 	FERR("Error while computing statistics!\n");
 	return -10;
 	}


 /*
  * Check number of users
  */
 if (myStats->nUsers <= 0)
 	{
 	FERR("No users found? Check that you really specified correct log-format!\n");
 	return -10;
 	}

 /* Calculate time */
 time(&myTime2);
 myStats->nTimeElapsed = (myTime2 - myTime1);
 
 /* Open output-file */
 if (nverbosity >= 1)
 FERR("Using %s\n", outputFormats[setOutputFormat].ofDescription);

 if (destFileName == NULL) myFile = stdout; else
 if ((myFile = fopen(destFileName, "wa")) == NULL)
 	{
 	FERR("Error opening output file '%s' (%s)\n", destFileName, strerror(errno));
 	return -1;
 	}

 
 iResult = outputFormats[setOutputFormat].ofFunction(myFile, myStats, myConfig);

 fclose(myFile);

 /*
  * OK! Show final stats
  */
 if (nverbosity >= 1)
 FERR("%ld lines in %ld logfile(s), total of %1.2f MB\n",
 	myStats->nLines,
 	myStats->nLogFiles,
 	((t_float) myStats->nChars) / (1024.0f*1024.0f)
 	);
  
 if (iResult == 0)
 	{
	if (nverbosity >= 1)
	FERR("Done. Time elapsed %ld hours, %ld minutes and %ld seconds.\n",
 		(myStats->nTimeElapsed / (60*60)),
		(myStats->nTimeElapsed % (60*60)) / 60, 
		(myStats->nTimeElapsed % (60*60)) % 60
		);
	} else
	FERR("Error in output! Return code #%i.\n", iResult);


 /*
  * Free allocate memory
  */
 /*output_userfile(stdout, myStats);*/
 stats_free(myStats);
 th_config_free(myConfig);
 
 return 0;
}

