#import "InstallerDownload.h"

@implementation InstallerDownload

- (id)initWithClient:(id)newClient
{
	// Call the super
	[super init];
	
	// Specify the client
	client = newClient;
	
	// Intialize varaibles
	currentDownload = 0;
	complained = NO;
	isDownloading = NO;
	
	return self;
}

- (void)reset
{
	// Complete that required for reseting
	if (downloader) [downloader cancelLoadInBackground];
	if (downloader) { [downloader autorelease]; downloader = NULL; }
	if (downloads) { [downloads autorelease]; downloads = NULL; }
	currentDownload = 0;
	complained = NO;
	isDownloading = NO;
}

- (void)dealloc
{
	[self reset];
}

- (BOOL)commenceDownloads:(NSArray *)downloadObjects withCallback:(SEL)callback
{
	NSURL *tempURL;
	int i;
	
	// If we are busy return failure
	if (isDownloading)
		return NO;

	// Copy our parameters since we use them later and don't want them to change
	downloads = [downloadObjects retain];
	
	// Store the callbackSelector
	callbackSelector = callback;
	
	// Start downloading from the beginning
	currentDownload = 0;
	curDownloadSize = 0;
	
	// Determine the total size of all downloads
	totalDownloadSize = 0;
	for (i = 0; i < [downloads count]; i++) {
		totalDownloadSize += [[downloads objectAtIndex:i] downloadSize];
	}
	
	// Commence the downloading
	isDownloading = YES;
	for (currentDownload = 0; currentDownload < [downloads count]; currentDownload++) {
		tempURL = [[downloads objectAtIndex:currentDownload] downloadURL];
		if (tempURL != NULL) {
			downloader = [[NSURLHandle cachedHandleForURL:tempURL] retain];
			[downloader addClient:self];
			[downloader loadInBackground];
			return YES;
		}
	}
	
	// Otherwise we're done
	if ([client respondsToSelector:@selector(downloadProgress: value:)] == YES)
		[client downloadProgress:self value:progressSize];
	isDownloading = NO;
	[client performSelector:callbackSelector withObject:self];
	
	// Free the downloader
	if (downloads) { [downloads autorelease]; downloads = NULL; }
	
	return YES;
}

- (NSString *)currentDownloadName
{
	// If possible return name of the file currently being downloaded to
	if (isDownloading && currentDownload < [downloads count])
		return [[[downloads objectAtIndex:currentDownload] destinationPath] lastPathComponent];
	else 
		return NULL;
}

- (BOOL)isDownloading
{
	return isDownloading;
}

- (void)setProgressSize:(double)size
{
	progressSize = size;
}

- (void)URLHandleResourceDidBeginLoading:(NSURLHandle *)sender
{
}

- (void)URLHandleResourceDidCancelLoading:(NSURLHandle *)sender
{
}

- (void)URLHandleResourceDidFinishLoading:(NSURLHandle *)sender
{
	NSURL *tempURL;
	
	// Write the data to disk
	[[sender resourceData] writeToFile:[[downloads objectAtIndex:currentDownload] destinationPath] atomically:NO];
	
	// Increment the currentDownload, reset the curDownloadSize
	currentDownload++;
	curDownloadSize = 0;

	// Move to the next download	
	for (currentDownload = currentDownload; currentDownload < [downloads count]; currentDownload++) {
		tempURL = [[downloads objectAtIndex:currentDownload] downloadURL];
		if (tempURL != NULL) {
			downloader = [[NSURLHandle cachedHandleForURL:tempURL] retain];
			[downloader addClient:self];
			[downloader loadInBackground];
			return;
		}
	}
	
	// Otherwise we're done
	if ([client respondsToSelector:@selector(downloadProgress: value:)] == YES)
		[client downloadProgress:self value:progressSize];
	isDownloading = NO;
	[client performSelector:callbackSelector withObject:self];
	
	// Free the downloader
	if (downloader) { [downloader autorelease]; downloader = NULL; }
	if (downloads) { [downloads autorelease]; downloads = NULL; }
}

- (void)URLHandle:(NSURLHandle *)sender resourceDataDidBecomeAvailable:(NSData *)newBytes
{
	int i, sizeSoFar = 0;
	
	if ([client respondsToSelector:@selector(downloadProgress: value:)] == YES) {
		
		// Work out the new curDownloadSize
		if (curDownloadSize + [newBytes length] < [[downloads objectAtIndex:currentDownload] downloadSize])
			curDownloadSize += [newBytes length];
		else
			curDownloadSize = [[downloads objectAtIndex:currentDownload] downloadSize];
			
		// Work out the new sizeSoFar
		for (i = 0; i < currentDownload; i++)
			sizeSoFar += [[downloads objectAtIndex:i] downloadSize];
		sizeSoFar += curDownloadSize;
		
		// Inform the client
		[client downloadProgress:self value:((double)sizeSoFar / (double)totalDownloadSize) * progressSize];
		
	}
}

- (void)URLHandle:(NSURLHandle *)sender resourceDidFailLoadingWithReason:(NSString *)reason
{
	// Complain to client about failure
	if (complained == NO && [client respondsToSelector:@selector(downloadFailed: reason:)] == YES) {
		complained = YES;
		[client downloadFailed:self reason:reason];
	}
	
	// Free the downloader
	if (downloader) { [downloader autorelease]; downloader = NULL; }
	if (downloads) { [downloads autorelease]; downloads = NULL; }
}

@end
