//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: SurfaceMeshOfTriangles.hpp,v 1.8 2004/05/31 18:49:32 delpinux Exp $

#ifndef SURFACE_MESH_OF_TRIANGLES_HPP
#define SURFACE_MESH_OF_TRIANGLES_HPP

#include <Vector.hpp>
#include <Triangle.hpp>

#include <SurfaceMesh.hpp>

#include <sstream>
#include <fstream>

/**
 * @file   SurfaceMeshOfTriangles.hpp
 * @class  SurfaceMeshOfTriangles
 * @author Stephane Del Pino
 * @date   Sun Jun  9 14:31:05 2002
 * 
 * @brief This class describes suface meshes made of triangles
 * 
 * This class describes suface meshes made of triangles
 */

class SurfaceMeshOfTriangles
  : public SurfaceMesh
{
public:
  typedef Triangle ElementGeometry;

  typedef struct {} BorderMeshType; /**< BorderMeshType is not defined  */

  typedef Mesh::T_iterator<SurfaceMeshOfTriangles, Triangle> iterator;
  typedef Mesh::T_iterator<const SurfaceMeshOfTriangles, const Triangle> const_iterator;

private:
  ReferenceCounting<Vector<Triangle> > __cells;	/**< The set of cells */

public:
  //! Returns \p true if the point \a p is inside the mesh.
  inline bool inside(const real_t& x, const real_t& y, const real_t& z) const
  {
    fferr(0) << "Trying to find 3d point on a surface mesh\n";
    std::exit(1);
    return true;
  }

  //! Returns \p true if the point \a p is inside the mesh.
  inline bool inside(const TinyVector<3>& p) const
  {
    return this->inside(p[0], p[1], p[2]);
  }  

  void computesFictitiousCells() const
  {
    this->__computesFictitiousCells<Triangle>(*__cells);
  }

  /** 
   * Plots the mesh in the OpenDX format
   * 
   * @param os where to write
   * @param cr the "next line" string
   */
  void plotOpenDX(std::ostream& os, const std::string& cr) const
  {
    os << "object 1 class array type float rank 1 shape 3 items "
       << numberOfVertices() << " data follows\n";
    for (size_t i=0; i<numberOfVertices(); ++i) {
      const Vertex& V = vertex(i);
      os << V[0] << ' ' << V[1] << ' ' << V[2] << '\n';
    }
    os << "object 2 class array type int rank 1 shape 3 items "
       << (*__cells).size() << " data follows\n";
    for (size_t i=0; i<(*__cells).size(); ++i) {
      const Triangle& ST = dynamic_cast<const Triangle&>((*__cells)[i]);
      os << vertexNumber(ST(0)) << ' '
	 << vertexNumber(ST(1)) << ' '
	 << vertexNumber(ST(2)) << '\n';
    }
    os << "attribute \"element type\" string \"triangles\"\n";
    os << "attribute \"ref\" string \"positions\"\n";
    os << "attribute \"dep\" string \"connections\"\n";
    os << "object \"Mesh\" class field\n";
    os << "component \"positions\" value 1\n";
    os << "component \"connections\" value 2\n";
    os << "end\n";
  }

  /** 
   * Access to the ith cell
   * 
   * @param i 
   * 
   * @return the ith cell
   */
  Triangle& cell(size_t i)
  {
    return (*__cells)[i];
  }

  /** 
   * RO-Access to the ith cell
   * 
   * @param i 
   * 
   * @return the ith cell
   */
  const Triangle& cell(size_t i) const
  {
    return (*__cells)[i];
  }

  /** 
   * Reserves storage for \a n surface elements.
   * @note the content is lost!
   * @param n number of elements
   */
  inline void setNumberOfCells(const int n)
  {
    __cells = new Vector<Triangle>(n);
  }

  /** 
   * 
   * Returns the number of cells
   * 
   * @return number of cells
   */
  inline const size_t& numberOfCells() const
  {
    return (*__cells).size();
  }

  size_t cellNumber(const Triangle& t) const
  {
    return (*__cells).number(t);
  }

  SurfaceMeshOfTriangles(const size_t theNumberOfCells)
    : SurfaceMesh(Mesh::surfaceMeshTriangles),
      __cells(new Vector<Triangle>(theNumberOfCells))
  {
    ;
  }

  SurfaceMeshOfTriangles(ReferenceCounting<VerticesSet> vertices,
			 ReferenceCounting<Vector<Triangle> > triangles)
    : SurfaceMesh(Mesh::surfaceMeshTriangles,
		  vertices),
      __cells(triangles)
  {
    ;
  }

  /** 
   * Default constructor
   * 
   */
  SurfaceMeshOfTriangles()
    : SurfaceMesh(Mesh::surfaceMeshTriangles)
  {
    ;
  }

  /** 
   * Destructor
   * 
   */
  ~SurfaceMeshOfTriangles()
  {
    ;
  }
};

#endif // SURFACE_MESH_OF_TRIANGLES_HPP

