/* 

                          Firewall Builder

                 Copyright (C) 2003 NetCitadel, LLC

  Author:  Vadim Kurland     vadim@fwbuilder.org

  $Id: FWBTree.cpp,v 1.19 2004/08/24 04:41:21 vkurland Exp $

  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include "config.h"
#include "global.h"

#include <qobject.h>
#include <qmessagebox.h>

#include "FWBTree.h"

#include "fwbuilder/FWObjectDatabase.h"

#include "fwbuilder/Library.h"
#include "fwbuilder/Firewall.h"
#include "fwbuilder/Host.h"
#include "fwbuilder/Network.h"
#include "fwbuilder/IPv4.h"
#include "fwbuilder/AddressRange.h"
#include "fwbuilder/ObjectGroup.h"
#include "fwbuilder/Interface.h"
#include "fwbuilder/CustomService.h"
#include "fwbuilder/IPService.h"
#include "fwbuilder/ICMPService.h"
#include "fwbuilder/TCPService.h"
#include "fwbuilder/UDPService.h"
#include "fwbuilder/ServiceGroup.h"
#include "fwbuilder/Interval.h"
#include "fwbuilder/IntervalGroup.h"

#include <iostream>

using namespace std;
using namespace libfwbuilder;

FWBTree *FWBTree::standardObjectTreeFormat=NULL;

const char* systemObjects[] = {
    "Objects",
    "Objects/Addresses",
    "Objects/Address Ranges",
    "Objects/Groups",
    "Objects/Hosts",
    "Objects/Networks",

    "Services",
    "Services/Custom",
    "Services/Groups",
    "Services/IP",
    "Services/ICMP",
    "Services/TCP",
    "Services/UDP",

    "Firewalls",

    "Time",

    NULL
};

map<string,bool> standardIDs;

FWBTree::FWBTree()
{
    assert(standardObjectTreeFormat==NULL);
    standardObjectTreeFormat=this;

    systemGroupPaths[Library::TYPENAME]       = "";

    systemGroupPaths[IPv4::TYPENAME]          = "Objects/Addresses";
    systemGroupPaths[AddressRange::TYPENAME]  = "Objects/Address Ranges";
    systemGroupPaths[ObjectGroup::TYPENAME]   = "Objects/Groups";
    systemGroupPaths[Host::TYPENAME]          = "Objects/Hosts";
    systemGroupPaths[Network::TYPENAME]       = "Objects/Networks";

    systemGroupPaths[ServiceGroup::TYPENAME]  = "Services/Groups";
    systemGroupPaths[CustomService::TYPENAME] = "Services/Custom";
    systemGroupPaths[IPService::TYPENAME]     = "Services/IP";
    systemGroupPaths[ICMPService::TYPENAME]   = "Services/ICMP";
    systemGroupPaths[TCPService::TYPENAME]    = "Services/TCP";
    systemGroupPaths[UDPService::TYPENAME]    = "Services/UDP";

    systemGroupPaths[Firewall::TYPENAME]      = "Firewalls";

    systemGroupPaths[Interval::TYPENAME]      = "Time";



    systemGroupTypes[Firewall::TYPENAME]=      ObjectGroup::TYPENAME;
    systemGroupNames[Firewall::TYPENAME]=      "Firewalls"    ;

    systemGroupTypes[Host::TYPENAME]=          ObjectGroup::TYPENAME;
    systemGroupNames[Host::TYPENAME]=          "Hosts"          ;

    systemGroupTypes[Network::TYPENAME]=       ObjectGroup::TYPENAME;
    systemGroupNames[Network::TYPENAME]=       "Networks"       ;

    systemGroupTypes[IPv4::TYPENAME]=          ObjectGroup::TYPENAME;
    systemGroupNames[IPv4::TYPENAME]=          "Addresses"      ;

    systemGroupTypes[AddressRange::TYPENAME]=  ObjectGroup::TYPENAME;
    systemGroupNames[AddressRange::TYPENAME]=  "Address Ranges" ;

    systemGroupTypes[ObjectGroup::TYPENAME]=   ObjectGroup::TYPENAME;
    systemGroupNames[ObjectGroup::TYPENAME]=   "Groups"         ;

    systemGroupTypes[CustomService::TYPENAME]= ServiceGroup::TYPENAME;
    systemGroupNames[CustomService::TYPENAME]= "Custom";

    systemGroupTypes[IPService::TYPENAME]=     ServiceGroup::TYPENAME;
    systemGroupNames[IPService::TYPENAME]=     "IP"            ;

    systemGroupTypes[ICMPService::TYPENAME]=   ServiceGroup::TYPENAME;
    systemGroupNames[ICMPService::TYPENAME]=   "ICMP"          ;

    systemGroupTypes[TCPService::TYPENAME]=    ServiceGroup::TYPENAME;
    systemGroupNames[TCPService::TYPENAME]=    "TCP"           ;

    systemGroupTypes[UDPService::TYPENAME]=    ServiceGroup::TYPENAME;
    systemGroupNames[UDPService::TYPENAME]=    "UDP"           ;

    systemGroupTypes[ServiceGroup::TYPENAME]=  ServiceGroup::TYPENAME;
    systemGroupNames[ServiceGroup::TYPENAME]=  "Groups"        ;

    systemGroupTypes[Interval::TYPENAME]=      IntervalGroup::TYPENAME;
    systemGroupNames[Interval::TYPENAME]=      "Time"         ;

    systemGroupTypes[Interface::TYPENAME]=     "";
    systemGroupNames[Interface::TYPENAME]=     "";

    systemGroupTypes[Library::TYPENAME]=      FWObjectDatabase::TYPENAME;
    systemGroupNames[Library::TYPENAME]=      "FWObjectDatabase";

    standardIDs["syslib000"]=true;
    standardIDs["syslib001"]=true;

    standardIDs["sysid0"]   =true;
    standardIDs["sysid1"]   =true;
    standardIDs["sysid2"]   =true;
    standardIDs["sysid99"]  =true;

    standardIDs["stdid01"]  =true;
    standardIDs["stdid01_1"]=true;
    standardIDs["stdid02"]  =true;
    standardIDs["stdid02_1"]=true;
    standardIDs["stdid03"]  =true;
    standardIDs["stdid03_1"]=true;
    standardIDs["stdid04"]  =true;
    standardIDs["stdid04_1"]=true;
    standardIDs["stdid05"]  =true;
    standardIDs["stdid05_1"]=true;
    standardIDs["stdid06"]  =true;
    standardIDs["stdid06_1"]=true;
    standardIDs["stdid07"]  =true;
    standardIDs["stdid07_1"]=true;
    standardIDs["stdid08"]  =true;
    standardIDs["stdid08_1"]=true;
    standardIDs["stdid09"]  =true;
    standardIDs["stdid09_1"]=true;
    standardIDs["stdid10"]  =true;
    standardIDs["stdid10_1"]=true;
    standardIDs["stdid11"]  =true;
    standardIDs["stdid11_1"]=true;
    standardIDs["stdid12"]  =true;
    standardIDs["stdid12_1"]=true;
    standardIDs["stdid13"]  =true;
    standardIDs["stdid13_1"]=true;
    standardIDs["stdid14"]  =true;
    standardIDs["stdid14_1"]=true;
    standardIDs["stdid15"]  =true;
    standardIDs["stdid15_1"]=true;
    standardIDs["stdid16"]  =true;
    standardIDs["stdid16_1"]=true;

}

/**
 * returns true if object 'obj' is a system group.  System groups are
 * those that hold other objects. Unlike user-defined groups, system
 * groups always contain only objects themselves and never contain
 * references to objects. User-defined groups, on the other hand,
 * always contain only references to objects.
 *
 */
bool FWBTree::isSystem(FWObject *obj)
{
    if (Library::isA(obj))   return (obj->getId()==STANDARD_LIB);

    if (FWObjectDatabase::isA(obj)) return true;

    string path=obj->getPath(true);  // relative path

    for (const char **cptr=systemObjects; *cptr!=NULL; cptr++)
        if (path== *cptr) return true;


    return false;
}

bool FWBTree::isStandardId(FWObject *obj)
{
    return standardIDs[ obj->getId() ];
}

bool FWBTree::validateForInsertion(FWObject *target,FWObject *obj)
{
    if (fwbdebug) qDebug("FWBTree::validateForInsertion  target %s  obj %s",
                         target->getTypeName().c_str(),
                         obj->getTypeName().c_str());

    if (Host::isA(target)      && Interface::isA(obj))   return true;
    if (Firewall::isA(target)  && Interface::isA(obj))   return true;
    if (Interface::isA(target) && IPv4::isA(obj))        return true;
    if (Interface::isA(target) && physAddress::isA(obj)) return true;

    QString parentType = standardObjectTreeFormat->systemGroupTypes[obj->getTypeName().c_str()];
    QString parentName = standardObjectTreeFormat->systemGroupNames[obj->getTypeName().c_str()];

/* parentType or/and parentName are going to be empty if information
 * about object obj is missing in systemGroupTypes/Names tables
 */
    if (parentType.isEmpty() || parentName.isEmpty()) return false;

    if (target->getTypeName() == string(parentType.latin1()) &&
	target->getName()     == string(parentName.latin1()) )
        return true;

    QMessageBox::warning(
        0,"Firewall Builder", 
        QObject::tr("Impossible to insert object %1 (type %2) into %3\nbecause of incompatible type.")
        .arg(obj->getName().c_str())
	.arg(obj->getTypeName().c_str())
	.arg(target->getName().c_str()),
        "&Continue", QString::null, QString::null,
        0, 1 );


    return false;
}

void FWBTree::getStandardSlotForObject(const QString &objType,
                                       QString &parentType,
                                       QString &parentName)
{
    parentType = standardObjectTreeFormat->systemGroupTypes[objType];
    parentName = standardObjectTreeFormat->systemGroupNames[objType];
}

/**
 * this method finds standard system folder for an object of a given
 * type in a given library. This method implemented our standard tree
 * structure (the one that is created in the method createNewLibrary)
 */
FWObject* FWBTree::getStandardSlotForObject(FWObject* lib,const QString &objType)
{
    QString path = standardObjectTreeFormat->systemGroupPaths[objType];

    if (path.isEmpty()) return lib;

    QString level1 = path.section('/',0,0);
    QString level2 = path.section('/',1,1);

    FWObject::iterator i=std::find_if(lib->begin(),lib->end(),
                                      FWObjectNameEQPredicate(level1.ascii()));
    if (i==lib->end()) return NULL;
    FWObject *l1obj = *i;
    if (level2.isEmpty()) return l1obj;

    i=std::find_if(l1obj->begin(),l1obj->end(),
                   FWObjectNameEQPredicate(level2.ascii()));
    if (i==l1obj->end()) return NULL;
    return (*i);
}

FWObject* FWBTree::createNewLibrary(FWObjectDatabase *db)
{

    FWObject *nlib = db->create(Library::TYPENAME,true);
    db->add(nlib);
    nlib->setName( QObject::tr("New Library").latin1() );

    FWObject *o1 = db->create(ObjectGroup::TYPENAME,true);
    o1->setName("Objects");
    nlib->add(o1);

    FWObject *o2 = db->create(ObjectGroup::TYPENAME,true);
    o2->setName("Addresses");
    o1->add(o2);

    o2 = db->create(ObjectGroup::TYPENAME,true);
    o2->setName("Groups");
    o1->add(o2);

    o2 = db->create(ObjectGroup::TYPENAME,true);
    o2->setName("Hosts");
    o1->add(o2);

    o2 = db->create(ObjectGroup::TYPENAME,true);
    o2->setName("Networks");
    o1->add(o2);

    o2 = db->create(ObjectGroup::TYPENAME,true);
    o2->setName("Address Ranges");
    o1->add(o2);

    o1 = db->create(ServiceGroup::TYPENAME,true);
    o1->setName("Services");
    nlib->add(o1);

    o2 = db->create(ServiceGroup::TYPENAME,true);
    o2->setName("Groups");
    o1->add(o2);

    o2 = db->create(ServiceGroup::TYPENAME,true);
    o2->setName("ICMP");
    o1->add(o2);

    o2 = db->create(ServiceGroup::TYPENAME,true);
    o2->setName("IP");
    o1->add(o2);

    o2 = db->create(ServiceGroup::TYPENAME,true);
    o2->setName("TCP");
    o1->add(o2);

    o2 = db->create(ServiceGroup::TYPENAME,true);
    o2->setName("UDP");
    o1->add(o2);

    o2 = db->create(ServiceGroup::TYPENAME,true);
    o2->setName("Custom");
    o1->add(o2);

    o1 = db->create(ObjectGroup::TYPENAME,true);
    o1->setName("Firewalls");
    nlib->add(o1);

    o1 = db->create(IntervalGroup::TYPENAME,true);
    o1->setName("Time");
    nlib->add(o1);

    return nlib;
}
