/* 

                          Firewall Builder

                 Copyright (C) 2003 NetCitadel, LLC

  Author:  Vadim Kurland     vadim@fwbuilder.org

  $Id: utils.cpp,v 1.38 2004/08/02 01:56:21 vkurland Exp $

  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include "config.h"
#include "global.h"
#include "utils.h"

#include <qobject.h>
#include <qpopupmenu.h>
#include <qcombobox.h>
#include <qlistbox.h>
#include <qmessagebox.h>
#include <qpalette.h>
#include <qnamespace.h>
#include <qdns.h>
#include <qhostaddress.h>
#include <qapplication.h>

#include "fwbuilder/FWObject.h"
#include "fwbuilder/FWObjectDatabase.h"
#include "fwbuilder/Firewall.h"
#include "fwbuilder/Library.h"
#include "fwbuilder/Resources.h"

#if defined(_WIN32)
#  include <stdio.h>
#  include <sys/timeb.h>
#  include <sys/stat.h>
#  include <fcntl.h>
#  include <time.h>
#else
#  include <unistd.h>
#  include <string.h>
#  include <errno.h>
#endif

#include <iostream>
#include <algorithm>

using namespace std;
using namespace libfwbuilder;


class mapValEQPredicate {
    string descr;
    public:
    mapValEQPredicate(const string &d) { descr=d; }
    bool operator()(pair<string,string> _d) { return (descr == _d.second); }
};



void addPopupMenuItem(QObject *res,
                      QPopupMenu* menu,
                      const QString &resourceIconPath,
                      const char* itemName,
                      const char* member,
                      const QKeySequence &accel)
{
    string icn;
    QPixmap pm;

    icn = Resources::global_res->getResourceStr(resourceIconPath.latin1());
    if(icn!="")
    {
        pm = QPixmap::fromMimeSource( icn.c_str() );
        menu->insertItem( pm, itemName , res , member, accel );
    } else
        menu->insertItem( itemName , res , member, accel);
}

void fillLibraries(QComboBox *libs, libfwbuilder::FWObject *obj, bool rw)
{
    QString lib="";
    if ( ! FWObjectDatabase::isA(obj)) lib=obj->getLibraryName().c_str();
    libs->clear();
    list<FWObject*> ll = obj->getRoot()->getByType( Library::TYPENAME );
    ll.sort(FWObjectNameCmpPredicate());
    int n=0;
    int cn=0;
    string libicn;
    for (FWObject::iterator i=ll.begin(); i!=ll.end(); i++)
    {
        if (libicn.empty())
            libicn=Resources::global_res->getObjResourceStr(*i,"icon-tree").c_str();

        if ( (*i)->getId()==STANDARD_LIB ||
             (*i)->getId()==TEMPLATE_LIB ||
             (*i)->getId()==DELETED_LIB )    continue;

        if (rw && (*i)->isReadOnly()) continue;
        if (lib==(*i)->getName().c_str()) cn=n;
        libs->insertItem(QPixmap::fromMimeSource( libicn.c_str() ),
                         (*i)->getName().c_str() );
        n++;
    }
    libs->setCurrentItem(cn);
}


void fillLibraries(QListBox *libs, libfwbuilder::FWObject *obj, bool rw)
{
    QString lib="";
    if ( ! FWObjectDatabase::isA(obj)) lib=obj->getLibraryName().c_str();
    libs->clear();
    list<FWObject*> ll = obj->getRoot()->getByType( Library::TYPENAME );
    ll.sort(FWObjectNameCmpPredicate());
    int n=0;
    int cn=0;
    string libicn;
    for (FWObject::iterator i=ll.begin(); i!=ll.end(); i++)
    {
        if (libicn.empty())
            libicn=Resources::global_res->getObjResourceStr(*i,"icon-tree").c_str();

        if ( (*i)->getId()==STANDARD_LIB ||
             (*i)->getId()==TEMPLATE_LIB ||
             (*i)->getId()==DELETED_LIB )    continue;

        if (rw && (*i)->isReadOnly()) continue;
        if (lib==(*i)->getName().c_str()) cn=n;
        libs->insertItem(QPixmap::fromMimeSource( libicn.c_str() ),
                         (*i)->getName().c_str() );
        n++;
    }
    libs->setCurrentItem(cn);
}


bool isTreeReadWrite(QWidget *parent, FWObject *obj)
{
    if (obj->isReadOnly())
    {
        QMessageBox::warning(parent, "Firewall Builder",
 QObject::tr("Impossible to apply changes because object is located in read-only\npart of the tee or data file was opened read-only"),
 QObject::tr("&Continue"), 0, 0,
 0, 2 );

        return false;
    }
    return true;
}

/*
 * compare names as QString objects to catch non-ascii names
 */
bool validateName(QWidget *parent,
                  libfwbuilder::FWObject *obj,const QString &newname)
{
    FWObject *p = obj->getParent();
    for (FWObject::iterator i=p->begin(); i!=p->end(); ++i)
    {
        FWObject *o1= *i;
        if (QString(o1->getName().c_str())==newname && o1!=obj)
        {
            QMessageBox::warning(parent, "Firewall Builder",
 QObject::tr("Object with name '%1' already exists, please choose different name.").
                                 arg(o1->getName().c_str()),
                                 QObject::tr("&Continue editing"), NULL, NULL, 0, 2 );
            return false;
        }
    }
    return true;
}

QString quoteString(const QString &str)
{
    QString res;

    if (str.find(' ')!=-1) res="\"";
    res+=str;
    if (str.find(' ')!=-1) res+="\"";

    return res;
}

QString getUserName()
{
#ifdef _WIN32
/* also need to set env variable USER for rcs tools, but if the user name
 * contains spaces, replace them with underscores (like "John Smith")
 */
    QString uname=getenv("USERNAME");
    uname=uname.replace(' ','_');
#else
    QString uname=getenv("USER");
    if (uname.isEmpty()) uname=getenv("LOGNAME");
    if (uname.isEmpty()) uname=getlogin();
#endif
    return uname;
}


QString getFileDir(const QString &file)
{
    int sn1  = file.findRev('/',-1);
    int sn2  = file.findRev('\\',-1);
    int sn   = (sn1>=0)?sn1:sn2;
    QString dir = (sn<0) ? "./" : file.left( sn );

#ifdef _WIN32
/* on windows, if directory is in the root of the drive (like "c:"),
 * I must append "\" to it
 */
    if (dir.find(":")==(dir.length()-1)) dir=dir + "\\";
#endif
    return dir;
}

QMap<QString,QString> getAllPlatforms()
{
    QMap<QString,QString> res;

    map<string,string> platforms = Resources::getPlatforms();
    map<string,string>::iterator i;
    for (i=platforms.begin(); i!=platforms.end(); i++)
        res[ i->first.c_str() ] = i->second.c_str();

    return res;
}

QMap<QString,QString> getAllOS()
{
    QMap<QString,QString> res;

    map<string,string> OSs = Resources::getOS();
    map<string,string>::iterator i;
    for (i=OSs.begin(); i!=OSs.end(); i++)
        res[ i->first.c_str() ] = i->second.c_str();

    return res;
}

QString readPlatform(QComboBox *platform)
{
    map<string,string> platforms = Resources::getPlatforms();
    map<string,string>::iterator i1 = std::find_if( platforms.begin(), platforms.end(),
                         mapValEQPredicate(platform->currentText().latin1()));
    return (*i1).first.c_str();
}

void setPlatform(QComboBox *platform,const QString &pl)
{
    platform->clear();
    int cp=0;
    QMap<QString,QString> platforms = getAllPlatforms();
    QMap<QString,QString>::iterator i;
    for (i=platforms.begin(); i!=platforms.end(); i++,cp++)
    {
        platform->insertItem( i.data() );
        if ( pl == i.key() ) platform->setCurrentItem( cp );
    }
}

QString readHostOS(QComboBox *hostOS)
{
    map<string,string> OSs       = Resources::getOS();
    map<string,string>::iterator i2 = std::find_if( OSs.begin(), OSs.end(),
                           mapValEQPredicate(hostOS->currentText().latin1()));
    return (*i2).first.c_str();
}

void setHostOS(QComboBox *hostOS,const QString &os)
{
    hostOS->clear();

    int cp=0;
    QMap<QString,QString> OSs = getAllOS();
    QMap<QString,QString>::iterator i;
    for (i=OSs.begin(); i!=OSs.end(); i++,cp++)
    {
        hostOS->insertItem( i.data() );
        if ( os == i.key() ) hostOS->setCurrentItem( cp );
    }
}

/**
 * this method finds all firewalls in the tree and makes no
 * assumtions about tree structure
 */
void findFirewalls(FWObject *o, std::list<FWObject*> &fwlist)
{
    findByObjectType(o,Firewall::TYPENAME,fwlist);
}

void findHosts(FWObject *o, std::list<FWObject*> &fwlist)
{
    findByObjectType(o,Host::TYPENAME,fwlist);
}

void findByObjectType(FWObject *o,
                      const string &otype,
                      list<FWObject*> &fwlist)
{
    if (o->getId()==FWObjectDatabase::getDeletedObjectsId()) return;

    for (list<FWObject*>::iterator m=o->begin(); m!=o->end(); m++)
    {
        FWObject *o1=*m;
        if (FWReference::cast(o1)!=NULL) continue;
        if (o1->getTypeName()==otype) fwlist.push_back(o1);
        else                   findByObjectType(o1,otype,fwlist);
    }
}


void setDisabledPalette(QWidget *w)
{
    QColorGroup cg;
    QPalette    pal=w->palette();

    cg = pal.active();
    cg.setColor( QColorGroup::Text, Qt::black );
    pal.setActive( cg );

    cg = pal.inactive();
    cg.setColor( QColorGroup::Text, Qt::black );
    pal.setInactive( cg );

    cg = pal.disabled();
    cg.setColor( QColorGroup::Text, Qt::black );
    pal.setDisabled( cg );

    w->setPalette( pal );
}

QString getAddrByName(const QString &name)
{
    QDns a(name);
    QValueList<QHostAddress> alist = a.addresses();

    while (a.isWorking())
    {
        app->processEvents(200);
    }
    alist = a.addresses();
    if (alist.empty()) return "";
    return alist.front().toString();
}
