/**
 * File:          $RCSfile: image_defs.c,v $
 * Module:        Image definitions and general functions
 * Part of:       Gandalf Library
 *
 * Revision:      $Revision: 1.37 $
 * Last edited:   $Date: 2003/08/21 08:58:54 $
 * Author:        $Author: pm $
 * Copyright:     (c) 2000 Imagineer Software Limited
 */

/* This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdarg.h>
#include <gandalf/image/image_defs.h>
#include <gandalf/image/image_gl_uchar.h>
#include <gandalf/image/image_gl_short.h>
#include <gandalf/image/image_gl_ushort.h>
#include <gandalf/image/image_gl_int.h>
#include <gandalf/image/image_gl_uint.h>
#include <gandalf/image/image_gl_float.h>
#include <gandalf/image/image_gl_double.h>
#include <gandalf/image/image_bit.h>
#include <gandalf/image/image_pointer.h>
#include <gandalf/image/image_gla_uchar.h>
#include <gandalf/image/image_gla_short.h>
#include <gandalf/image/image_gla_ushort.h>
#include <gandalf/image/image_gla_int.h>
#include <gandalf/image/image_gla_uint.h>
#include <gandalf/image/image_gla_float.h>
#include <gandalf/image/image_gla_double.h>
#include <gandalf/image/image_rgb_uchar.h>
#include <gandalf/image/image_rgb_short.h>
#include <gandalf/image/image_rgb_ushort.h>
#include <gandalf/image/image_rgb_int.h>
#include <gandalf/image/image_rgb_uint.h>
#include <gandalf/image/image_rgb_float.h>
#include <gandalf/image/image_rgb_double.h>
#include <gandalf/image/image_rgba_uchar.h>
#include <gandalf/image/image_rgba_short.h>
#include <gandalf/image/image_rgba_ushort.h>
#include <gandalf/image/image_rgba_int.h>
#include <gandalf/image/image_rgba_uint.h>
#include <gandalf/image/image_rgba_float.h>
#include <gandalf/image/image_rgba_double.h>
#include <gandalf/image/image_vfield2D_float.h>
#include <gandalf/image/image_vfield2D_double.h>
#include <gandalf/image/image_vfield2D_short.h>
#include <gandalf/image/image_vfield2D_int.h>
#include <gandalf/image/image_vfield3D_float.h>
#include <gandalf/image/image_vfield3D_double.h>
#include <gandalf/image/image_vfield3D_short.h>
#include <gandalf/image/image_vfield3D_int.h>
#include <gandalf/image/image_extract.h>
#include <gandalf/common/bit_array.h>
#include <gandalf/common/misc_error.h>

/**
 * \addtogroup ImagePackage
 * \{
 */

/**
 * \addtogroup ImageAllocate
 * \{
 */

/**
 * \brief Form image.
 *
 * Forms and returns an image \a img with given \a format, \a type,
 * dimensions \a height and \a width. If the provided \a pix_data and/or
 * \a row_data pointers are \c NULL (in any combination), the pixel data array
 * and/or the row start pointer array are dynamically allocated.
 * If either are not \c NULL they must be big enough to hold the pixel/row
 * pointer array data. If necessary use gan_image_data_size() to compute the
 * necessary size of the \a pix_data array (the necessary size of \a row_data
 * is \a height pointers).
 *
 * The \a stride indicates the separation in memory between adjacent rows
 * of the image, and is measured in bytes. Without any padding of the image it
 * will correspond to \a width pixels, in which case you should use the
 * gan_image_form_data() macro instead.
 *
 * If you want to dynamically allocate the pixel data array and/or the
 * row pointer array with sizes greater than that necessary for the
 * given \a height and \a width, pass \a pix_data and/or \a row_data
 * as \c NULL and set \a pix_data_size and/or \a row_data_size appropriately.
 *
 * See also gan_image_alloc() and gan_image_form().
 */
Gan_Image *
 gan_image_form_gen ( Gan_Image *img,
                      Gan_ImageFormat format, Gan_Type type,
                      unsigned long height, unsigned long width,
                      unsigned long stride,
                      Gan_Bool alloc_pix_data,
                      void *pix_data, size_t pix_data_size,
                      void *row_data, size_t row_data_size )
{
   gan_heap_push();
   switch ( format )
   {
      case GAN_GREY_LEVEL_IMAGE:
      switch ( type )
      {
         case GAN_UCHAR:
           img = gan_image_form_gen_gl_uc ( img, height, width, stride,
                                            alloc_pix_data,
                                  (unsigned char *)  pix_data, pix_data_size,
                                  (unsigned char **) row_data, row_data_size );
           break;

         case GAN_SHORT:
           img = gan_image_form_gen_gl_s ( img, height, width, stride,
                                           alloc_pix_data,
                                          (short *)  pix_data, pix_data_size,
                                          (short **) row_data, row_data_size );
           break;

         case GAN_USHORT:
           img = gan_image_form_gen_gl_us ( img, height, width, stride,
                                            alloc_pix_data,
                                 (unsigned short *)  pix_data, pix_data_size,
                                 (unsigned short **) row_data, row_data_size );
           break;

         case GAN_INT:
           img = gan_image_form_gen_gl_i ( img, height, width, stride,
                                           alloc_pix_data,
                                           (int *)  pix_data, pix_data_size,
                                           (int **) row_data, row_data_size );
           break;

         case GAN_UINT:
           img = gan_image_form_gen_gl_ui ( img, height, width, stride,
                                            alloc_pix_data,
                                   (unsigned int *)  pix_data, pix_data_size,
                                   (unsigned int **) row_data, row_data_size );
           break;

         case GAN_FLOAT:
           img = gan_image_form_gen_gl_f ( img, height, width, stride,
                                           alloc_pix_data,
                                           (float *)  pix_data, pix_data_size,
                                           (float **) row_data, row_data_size);
           break;

         case GAN_DOUBLE:
           img = gan_image_form_gen_gl_d (img, height, width, stride,
                                          alloc_pix_data,
                                          (double *)  pix_data, pix_data_size,
                                          (double **) row_data, row_data_size);
           break;

         case GAN_BOOL:
           img = gan_image_form_gen_b ( img, height, width, stride,
                                        alloc_pix_data,
                                  (Gan_BitWord  *) pix_data,  pix_data_size,
                                  (Gan_BitWord **) row_data, row_data_size );
           break;

         case GAN_POINTER:
           img = gan_image_form_gen_p ( img, height, width, stride,
                                        alloc_pix_data,
                                        (void **)  pix_data, pix_data_size,
                                        (void ***) row_data, row_data_size);
           break;

         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_image_form_gen",
                              GAN_ERROR_ILLEGAL_TYPE, "" );
           return NULL;
      }
      break;

      case GAN_GREY_LEVEL_ALPHA_IMAGE:
      switch ( type )
      {
         case GAN_UCHAR:
         img = gan_image_form_gen_gla_uc ( img, height, width, stride,
                                           alloc_pix_data,
                                (Gan_GLAPixel_uc *)  pix_data, pix_data_size,
                                (Gan_GLAPixel_uc **) row_data, row_data_size );
         break;

         case GAN_SHORT:
         img = gan_image_form_gen_gla_s ( img, height, width, stride,
                                          alloc_pix_data,
                                (Gan_GLAPixel_s *)  pix_data, pix_data_size,
                                (Gan_GLAPixel_s **) row_data, row_data_size );
         break;

         case GAN_USHORT:
         img = gan_image_form_gen_gla_us ( img, height, width, stride,
                                           alloc_pix_data,
                                (Gan_GLAPixel_us *)  pix_data, pix_data_size,
                                (Gan_GLAPixel_us **) row_data, row_data_size );
         break;

         case GAN_INT:
         img = gan_image_form_gen_gla_i ( img, height, width, stride,
                                          alloc_pix_data,
                                (Gan_GLAPixel_i *)  pix_data, pix_data_size,
                                (Gan_GLAPixel_i **) row_data, row_data_size );
         break;

         case GAN_UINT:
         img = gan_image_form_gen_gla_ui ( img, height, width, stride,
                                           alloc_pix_data,
                                (Gan_GLAPixel_ui *)  pix_data, pix_data_size,
                                (Gan_GLAPixel_ui **) row_data, row_data_size );
         break;

         case GAN_FLOAT:
         img = gan_image_form_gen_gla_f ( img, height, width, stride,
                                          alloc_pix_data,
                                (Gan_GLAPixel_f *)  pix_data, pix_data_size,
                                (Gan_GLAPixel_f **) row_data, row_data_size );
         break;

         case GAN_DOUBLE:
         img = gan_image_form_gen_gla_d ( img, height, width, stride,
                                          alloc_pix_data,
                                (Gan_GLAPixel_d *)  pix_data, pix_data_size,
                                (Gan_GLAPixel_d **) row_data, row_data_size );
         break;

         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_image_form_gen",
                              GAN_ERROR_ILLEGAL_TYPE, "" );
           return NULL;
      }
      break;

      case GAN_RGB_COLOUR_IMAGE:
      switch ( type )
      {
         case GAN_UCHAR:
           img = gan_image_form_gen_rgb_uc ( img, height, width, stride,
                                             alloc_pix_data,
                                (Gan_RGBPixel_uc *)  pix_data, pix_data_size,
                                (Gan_RGBPixel_uc **) row_data, row_data_size );
           break;

         case GAN_SHORT:
           img = gan_image_form_gen_rgb_s ( img, height, width, stride,
                                            alloc_pix_data,
                                 (Gan_RGBPixel_s *)  pix_data, pix_data_size,
                                 (Gan_RGBPixel_s **) row_data, row_data_size );
           break;

         case GAN_USHORT:
           img = gan_image_form_gen_rgb_us ( img, height, width, stride,
                                             alloc_pix_data,
                                (Gan_RGBPixel_us *)  pix_data, pix_data_size,
                                (Gan_RGBPixel_us **) row_data, row_data_size );
           break;

         case GAN_INT:
           img = gan_image_form_gen_rgb_i ( img, height, width, stride,
                                            alloc_pix_data,
                                 (Gan_RGBPixel_i *)  pix_data, pix_data_size,
                                 (Gan_RGBPixel_i **) row_data, row_data_size );
           break;

         case GAN_UINT:
           img = gan_image_form_gen_rgb_ui ( img, height, width, stride,
                                             alloc_pix_data,
                                (Gan_RGBPixel_ui *)  pix_data, pix_data_size,
                                (Gan_RGBPixel_ui **) row_data, row_data_size );
           break;

         case GAN_FLOAT:
           img = gan_image_form_gen_rgb_f ( img, height, width, stride,
                                            alloc_pix_data,
                                 (Gan_RGBPixel_f *)  pix_data, pix_data_size,
                                 (Gan_RGBPixel_f **) row_data, row_data_size );
           break;

         case GAN_DOUBLE:
           img = gan_image_form_gen_rgb_d ( img, height, width, stride,
                                            alloc_pix_data,
                                 (Gan_RGBPixel_d *)  pix_data, pix_data_size,
                                 (Gan_RGBPixel_d **) row_data, row_data_size );
           break;

         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_image_form_gen",
                              GAN_ERROR_ILLEGAL_TYPE, "" );
           return NULL;
      }
      break;

      case GAN_RGB_COLOUR_ALPHA_IMAGE:
      switch ( type )
      {
         case GAN_UCHAR:
           img = gan_image_form_gen_rgba_uc ( img, height, width, stride,
                                              alloc_pix_data,
                               (Gan_RGBAPixel_uc *)  pix_data, pix_data_size,
                               (Gan_RGBAPixel_uc **) row_data, row_data_size );
           break;

         case GAN_SHORT:
           img = gan_image_form_gen_rgba_s ( img, height, width, stride,
                                             alloc_pix_data,
                                (Gan_RGBAPixel_s *)  pix_data, pix_data_size,
                                (Gan_RGBAPixel_s **) row_data, row_data_size );
           break;

         case GAN_USHORT:
           img = gan_image_form_gen_rgba_us ( img, height, width, stride,
                                              alloc_pix_data,
                               (Gan_RGBAPixel_us *)  pix_data, pix_data_size,
                               (Gan_RGBAPixel_us **) row_data, row_data_size );
           break;

         case GAN_INT:
           img = gan_image_form_gen_rgba_i ( img, height, width, stride,
                                             alloc_pix_data,
                                (Gan_RGBAPixel_i *)  pix_data, pix_data_size,
                                (Gan_RGBAPixel_i **) row_data, row_data_size );
           break;

         case GAN_UINT:
           img = gan_image_form_gen_rgba_ui ( img, height, width, stride,
                                              alloc_pix_data,
                               (Gan_RGBAPixel_ui *)  pix_data, pix_data_size,
                               (Gan_RGBAPixel_ui **) row_data, row_data_size );
           break;

         case GAN_FLOAT:
           img = gan_image_form_gen_rgba_f ( img, height, width, stride,
                                             alloc_pix_data,
                                (Gan_RGBAPixel_f *)  pix_data, pix_data_size,
                                (Gan_RGBAPixel_f **) row_data, row_data_size );
           break;

         case GAN_DOUBLE:
           img = gan_image_form_gen_rgba_d ( img, height, width, stride,
                                             alloc_pix_data,
                                (Gan_RGBAPixel_d *)  pix_data, pix_data_size,
                                (Gan_RGBAPixel_d **) row_data, row_data_size );
           break;

         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_image_form_gen",
                              GAN_ERROR_ILLEGAL_TYPE, "" );
           return NULL;
      }
      break;

      case GAN_VECTOR_FIELD_2D:
      switch ( type )
      {
         case GAN_FLOAT:
           img = gan_image_form_gen_vfield2D_f ( img, height, width, stride,
                                                 alloc_pix_data,
                                  (Gan_Vector2_f *)  pix_data, pix_data_size,
                                  (Gan_Vector2_f **) row_data, row_data_size );
           break;

         case GAN_DOUBLE:
           img = gan_image_form_gen_vfield2D_d ( img, height, width, stride,
                                                 alloc_pix_data,
                                    (Gan_Vector2 *)  pix_data, pix_data_size,
                                    (Gan_Vector2 **) row_data, row_data_size );
           break;

         case GAN_SHORT:
           img = gan_image_form_gen_vfield2D_s ( img, height, width, stride,
                                                 alloc_pix_data,
                                 (Gan_Vector2_s *)  pix_data, pix_data_size,
                                 (Gan_Vector2_s **) row_data, row_data_size );
           break;

         case GAN_INT:
           img = gan_image_form_gen_vfield2D_i ( img, height, width, stride,
                                                 alloc_pix_data,
                                 (Gan_Vector2_i *)  pix_data, pix_data_size,
                                 (Gan_Vector2_i **) row_data, row_data_size );
           break;

         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_image_form_gen",
                              GAN_ERROR_ILLEGAL_TYPE, "" );
           return NULL;
      }
      break;

      case GAN_VECTOR_FIELD_3D:
      switch ( type )
      {
         case GAN_FLOAT:
           img = gan_image_form_gen_vfield3D_f ( img, height, width, stride,
                                                 alloc_pix_data,
                                  (Gan_Vector3_f *)  pix_data, pix_data_size,
                                  (Gan_Vector3_f **) row_data, row_data_size );
           break;

         case GAN_DOUBLE:
           img = gan_image_form_gen_vfield3D_d ( img, height, width, stride,
                                                 alloc_pix_data,
                                    (Gan_Vector3 *)  pix_data, pix_data_size,
                                    (Gan_Vector3 **) row_data, row_data_size );
           break;

         case GAN_SHORT:
           img = gan_image_form_gen_vfield3D_s ( img, height, width, stride,
                                                 alloc_pix_data,
                                 (Gan_Vector3_s *)  pix_data, pix_data_size,
                                  (Gan_Vector3_s **) row_data, row_data_size );
           break;

         case GAN_INT:
           img = gan_image_form_gen_vfield3D_i ( img, height, width, stride,
                                                 alloc_pix_data,
                                 (Gan_Vector3_i *)  pix_data, pix_data_size,
                                  (Gan_Vector3_i **) row_data, row_data_size );
           break;

         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_image_form_gen",
                              GAN_ERROR_ILLEGAL_TYPE, "" );
           return NULL;
      }
      break;

      default:
        gan_err_flush_trace();
        gan_err_register ( "gan_image_form_gen",
                           GAN_ERROR_ILLEGAL_TYPE, "" );
        return NULL;
   }

   gan_heap_pop();
   return img;
}

/**
 * \}
 */

/**
 * \addtogroup ImageSet
 * \{
 */

/**
 * \brief Sets the function to free the image data buffer.
 * \param img The image
 * \param data_free_func The free function
 * \return #GAN_TRUE on success, or #GAN_FALSE on failure.
 *
 * For images that have been created using a user-supplied data buffer,
 * provide a function to use to free the user data buffer.
 */
Gan_Bool
 gan_image_set_data_free_func ( Gan_Image *img,
                                void (*data_free_func)(void *) )
{
   /* we can't provide a function to free the data when it was allocated
      internally using malloc() */
   gan_err_test_bool ( !img->pix_data_alloc, "gan_image_set_data_free_func",
                       GAN_ERROR_INCOMPATIBLE, "" );

   /* set free function */
   img->data_free_func = data_free_func;

   /* success */
   return GAN_TRUE;
}

/**
 * \}
 */

/**
 * \addtogroup ImageSize
 * \{
 */

/**
 * \brief Returns the size of an image pixel in bytes.
 * \param format The format of the image pixel
 * \param type The type of the image pixel
 * \return The size of the image pixel.
 *
 * Returns the size of an image pixel in bytes.
 */
size_t
 gan_image_pixel_size ( Gan_ImageFormat format, Gan_Type type )
{
   switch ( format )
   {
      case GAN_GREY_LEVEL_IMAGE:
      switch ( type )
      {
         case GAN_UCHAR:   return sizeof(unsigned char);
         case GAN_USHORT:  return sizeof(unsigned short);
         case GAN_SHORT:   return sizeof(short);
         case GAN_UINT:    return sizeof(unsigned int);
         case GAN_INT:     return sizeof(int);
         case GAN_FLOAT:   return sizeof(float);
         case GAN_DOUBLE:  return sizeof(double);
         case GAN_POINTER: return sizeof(void *);
         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_image_pixel_size", GAN_ERROR_ILLEGAL_TYPE,
                              "" );
           break;
      }
      break;

      case GAN_GREY_LEVEL_ALPHA_IMAGE:
      switch ( type )
      {
         case GAN_UCHAR:  return sizeof(Gan_GLAPixel_uc);
         case GAN_USHORT: return sizeof(Gan_GLAPixel_us);
         case GAN_SHORT:  return sizeof(Gan_GLAPixel_s);
         case GAN_UINT:   return sizeof(Gan_GLAPixel_ui);
         case GAN_INT:    return sizeof(Gan_GLAPixel_i);
         case GAN_FLOAT:  return sizeof(Gan_GLAPixel_f);
         case GAN_DOUBLE: return sizeof(Gan_GLAPixel_d);
         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_image_pixel_size", GAN_ERROR_ILLEGAL_TYPE,
                              "" );
           break;
      }
      break;

      case GAN_RGB_COLOUR_IMAGE:
      switch ( type )
      {
         case GAN_UCHAR:  return sizeof(Gan_RGBPixel_uc);
         case GAN_USHORT: return sizeof(Gan_RGBPixel_us);
         case GAN_SHORT:  return sizeof(Gan_RGBPixel_s);
         case GAN_UINT:   return sizeof(Gan_RGBPixel_ui);
         case GAN_INT:    return sizeof(Gan_RGBPixel_i);
         case GAN_FLOAT:  return sizeof(Gan_RGBPixel_f);
         case GAN_DOUBLE: return sizeof(Gan_RGBPixel_d);
         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_image_pixel_size", GAN_ERROR_ILLEGAL_TYPE,
                              "" );
           break;
      }
      break;

      case GAN_RGB_COLOUR_ALPHA_IMAGE:
      switch ( type )
      {
         case GAN_UCHAR:  return sizeof(Gan_RGBAPixel_uc);
         case GAN_USHORT: return sizeof(Gan_RGBAPixel_us);
         case GAN_SHORT:  return sizeof(Gan_RGBAPixel_s);
         case GAN_UINT:   return sizeof(Gan_RGBAPixel_ui);
         case GAN_INT:    return sizeof(Gan_RGBAPixel_i);
         case GAN_FLOAT:  return sizeof(Gan_RGBAPixel_f);
         case GAN_DOUBLE: return sizeof(Gan_RGBAPixel_d);
         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_image_pixel_size", GAN_ERROR_ILLEGAL_TYPE,
                              "" );
           break;
      }
      break;

      case GAN_VECTOR_FIELD_2D:
      switch ( type )
      {
         case GAN_FLOAT:  return sizeof(Gan_Vector2_f);
         case GAN_DOUBLE: return sizeof(Gan_Vector2);
         case GAN_SHORT:  return sizeof(Gan_Vector2_s);
         case GAN_INT:    return sizeof(Gan_Vector2_i);
         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_image_pixel_size", GAN_ERROR_ILLEGAL_TYPE,
                              "" );
           break;
      }
      break;

      case GAN_VECTOR_FIELD_3D:
      switch ( type )
      {
         case GAN_FLOAT:  return sizeof(Gan_Vector3_f);
         case GAN_DOUBLE: return sizeof(Gan_Vector3);
         case GAN_SHORT:  return sizeof(Gan_Vector3_s);
         case GAN_INT:    return sizeof(Gan_Vector3_i);
         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_image_pixel_size", GAN_ERROR_ILLEGAL_TYPE,
                              "" );
           break;
      }
      break;

      default:
        gan_err_flush_trace();
        gan_err_register ( "gan_image_pixel_size", GAN_ERROR_ILLEGAL_TYPE,
                           "" );
        break;
   }

   /* shouldn't get here */
   return 0;
}

/**
 * \brief Returns the minimum size of a row of image pixels in bytes.
 * \param format The format of the image pixel
 * \param type The type of the image pixel
 * \param width The number of image pixels on a row
 * \param alignment The byte alignment of a row of the image or zero
 * \return The number of bytes in a row.
 *
 * Returns the minimum size of a row of image pixels in bytes, given the
 * \a format, \a type and \a width of the image. The size is padded to a
 * multiple of the \a alignment value, unless \a alignment is passed as zero,
 * in which case no padding occurs.
 */
size_t
 gan_image_min_stride ( Gan_ImageFormat format, Gan_Type type,
                        unsigned long width, size_t alignment )
{
   switch ( format )
   {
      case GAN_GREY_LEVEL_IMAGE:
      switch ( type )
      {
         case GAN_UCHAR:   width *= sizeof(unsigned char);  break;
         case GAN_USHORT:  width *= sizeof(unsigned short); break;
         case GAN_SHORT:   width *= sizeof(short);          break;
         case GAN_UINT:    width *= sizeof(unsigned int);   break;
         case GAN_INT:     width *= sizeof(int);            break;
         case GAN_FLOAT:   width *= sizeof(float);          break;
         case GAN_DOUBLE:  width *= sizeof(double);         break;
         case GAN_POINTER: width *= sizeof(void *);         break;
         case GAN_BOOL:
           width = ((width + GAN_BITWORD_SIZE - 1)/GAN_BITWORD_SIZE)
                   *sizeof(Gan_BitWord);
           break;

         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_image_min_stride", GAN_ERROR_ILLEGAL_TYPE,
                              "" );
           break;
      }
      break;

      case GAN_GREY_LEVEL_ALPHA_IMAGE:
      switch ( type )
      {
         case GAN_UCHAR:  width *= sizeof(Gan_GLAPixel_uc); break;
         case GAN_USHORT: width *= sizeof(Gan_GLAPixel_us); break;
         case GAN_SHORT:  width *= sizeof(Gan_GLAPixel_s);  break;
         case GAN_UINT:   width *= sizeof(Gan_GLAPixel_ui); break;
         case GAN_INT:    width *= sizeof(Gan_GLAPixel_i);  break;
         case GAN_FLOAT:  width *= sizeof(Gan_GLAPixel_f);  break;
         case GAN_DOUBLE: width *= sizeof(Gan_GLAPixel_d);  break;
         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_image_min_stride", GAN_ERROR_ILLEGAL_TYPE,
                              "" );
           break;
      }
      break;

      case GAN_RGB_COLOUR_IMAGE:
      switch ( type )
      {
         case GAN_UCHAR:  width *= sizeof(Gan_RGBPixel_uc); break;
         case GAN_USHORT: width *= sizeof(Gan_RGBPixel_us); break;
         case GAN_SHORT:  width *= sizeof(Gan_RGBPixel_s);  break;
         case GAN_UINT:   width *= sizeof(Gan_RGBPixel_ui); break;
         case GAN_INT:    width *= sizeof(Gan_RGBPixel_i);  break;
         case GAN_FLOAT:  width *= sizeof(Gan_RGBPixel_f);  break;
         case GAN_DOUBLE: width *= sizeof(Gan_RGBPixel_d);  break;
         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_image_min_stride", GAN_ERROR_ILLEGAL_TYPE,
                              "" );
           break;
      }
      break;

      case GAN_RGB_COLOUR_ALPHA_IMAGE:
      switch ( type )
      {
         case GAN_UCHAR:  width *= sizeof(Gan_RGBAPixel_uc); break;
         case GAN_USHORT: width *= sizeof(Gan_RGBAPixel_us); break;
         case GAN_SHORT:  width *= sizeof(Gan_RGBAPixel_s);  break;
         case GAN_UINT:   width *= sizeof(Gan_RGBAPixel_ui); break;
         case GAN_INT:    width *= sizeof(Gan_RGBAPixel_i);  break;
         case GAN_FLOAT:  width *= sizeof(Gan_RGBAPixel_f);  break;
         case GAN_DOUBLE: width *= sizeof(Gan_RGBAPixel_d);  break;
         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_image_min_stride", GAN_ERROR_ILLEGAL_TYPE,
                              "" );
           break;
      }
      break;

      case GAN_VECTOR_FIELD_2D:
      switch ( type )
      {
         case GAN_FLOAT:  width *= sizeof(Gan_Vector2_f); break;
         case GAN_DOUBLE: width *= sizeof(Gan_Vector2);   break;
         case GAN_SHORT:  width *= sizeof(Gan_Vector2_s); break;
         case GAN_INT:    width *= sizeof(Gan_Vector2_i); break;
         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_image_min_stride", GAN_ERROR_ILLEGAL_TYPE,
                              "" );
           break;
      }
      break;

      case GAN_VECTOR_FIELD_3D:
      switch ( type )
      {
         case GAN_FLOAT:  width *= sizeof(Gan_Vector3_f); break;
         case GAN_DOUBLE: width *= sizeof(Gan_Vector3);   break;
         case GAN_SHORT:  width *= sizeof(Gan_Vector3_s); break;
         case GAN_INT:    width *= sizeof(Gan_Vector3_i); break;
         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_image_min_stride", GAN_ERROR_ILLEGAL_TYPE,
                              "" );
           break;
      }
      break;

      default:
        gan_err_flush_trace();
        gan_err_register ( "gan_image_min_stride", GAN_ERROR_ILLEGAL_TYPE,
                           "" );
        break;
   }

   /* if no alignment was specified, return without padding */
   if ( alignment == 0 ) return width;

   /* pad the size out to a multiple of the byte alignment */
   return ((width + alignment - 1)/alignment)*alignment;
}

/**
 * \}
 */

/**
 * \addtogroup ImageSet
 * \{
 */

/**
 * \brief Sets the format, type, dimensions and stride of an image.
 * \param img Pointer to the image
 * \param format The format to set the image
 * \param type The type to set the image
 * \param height The height to set the image
 * \param width The width to set the image
 * \param stride The stride of the image in bytes
 * \param alloc_pix_data Whether or not to allocate the pixel data
 * \return Pointer to the successfully modified image, or \c NULL on failure.
 *
 * Sets the \a format, \a type, \a height, \a width and \a stride of an image.
 */
Gan_Image *
 gan_image_set_format_type_dims_gen  ( Gan_Image *img,
                                       Gan_ImageFormat format, Gan_Type type,
                                       unsigned long height,
                                       unsigned long width,
                                       unsigned long stride,
                                       Gan_Bool alloc_pix_data )
{
   switch ( format )
   {
      case GAN_GREY_LEVEL_IMAGE:
      switch ( type )
      {
         case GAN_UCHAR:
           return gan_image_set_gen_gl_uc ( img, height, width, stride,
                                            alloc_pix_data );

         case GAN_USHORT:
           return gan_image_set_gen_gl_us ( img, height, width, stride,
                                            alloc_pix_data );

         case GAN_SHORT:
           return gan_image_set_gen_gl_s ( img, height, width, stride,
                                           alloc_pix_data );

         case GAN_UINT:
           return gan_image_set_gen_gl_ui ( img, height, width, stride,
                                            alloc_pix_data );

         case GAN_INT:
           return gan_image_set_gen_gl_i ( img, height, width, stride,
                                           alloc_pix_data );

         case GAN_FLOAT:
           return gan_image_set_gen_gl_f ( img, height, width, stride,
                                           alloc_pix_data );

         case GAN_DOUBLE:
           return gan_image_set_gen_gl_d ( img, height, width, stride,
                                           alloc_pix_data );

         case GAN_BOOL:
           return gan_image_set_gen_b ( img, height, width, stride,
                                        alloc_pix_data );

         case GAN_POINTER:
           return gan_image_set_gen_p ( img, height, width, stride,
                                        alloc_pix_data );

         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_image_set_format_type_dims",
                              GAN_ERROR_ILLEGAL_TYPE, "" );
           return NULL;
      }

      case GAN_GREY_LEVEL_ALPHA_IMAGE:
      switch ( type )
      {
         case GAN_UCHAR:
           return gan_image_set_gen_gla_uc ( img, height, width, stride,
                                             alloc_pix_data );

         case GAN_USHORT:
           return gan_image_set_gen_gla_us ( img, height, width, stride,
                                             alloc_pix_data );

         case GAN_SHORT:
           return gan_image_set_gen_gla_s ( img, height, width, stride,
                                            alloc_pix_data );

         case GAN_UINT:
           return gan_image_set_gen_gla_ui ( img, height, width, stride,
                                             alloc_pix_data );

         case GAN_INT:
           return gan_image_set_gen_gla_i ( img, height, width, stride,
                                            alloc_pix_data );

         case GAN_FLOAT:
           return gan_image_set_gen_gla_f ( img, height, width, stride,
                                            alloc_pix_data );

         case GAN_DOUBLE:
           return gan_image_set_gen_gla_d ( img, height, width, stride,
                                            alloc_pix_data );

         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_image_set_format_type_dims",
                              GAN_ERROR_ILLEGAL_TYPE, "" );
           return NULL;
      }

      case GAN_RGB_COLOUR_IMAGE:
      switch ( type )
      {
         case GAN_UCHAR:
           return gan_image_set_gen_rgb_uc ( img, height, width, stride,
                                             alloc_pix_data );

         case GAN_USHORT:
           return gan_image_set_gen_rgb_us ( img, height, width, stride,
                                             alloc_pix_data );

         case GAN_SHORT:
           return gan_image_set_gen_rgb_s ( img, height, width, stride,
                                            alloc_pix_data );

         case GAN_UINT:
           return gan_image_set_gen_rgb_ui ( img, height, width, stride,
                                             alloc_pix_data );

         case GAN_INT:
           return gan_image_set_gen_rgb_i ( img, height, width, stride,
                                            alloc_pix_data );

         case GAN_FLOAT:
           return gan_image_set_gen_rgb_f ( img, height, width, stride,
                                            alloc_pix_data );

         case GAN_DOUBLE:
           return gan_image_set_gen_rgb_d ( img, height, width, stride,
                                            alloc_pix_data );

         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_image_set_format_type_dims",
                              GAN_ERROR_ILLEGAL_TYPE, "" );
           return NULL;
      }

      case GAN_RGB_COLOUR_ALPHA_IMAGE:
      switch ( type )
      {
         case GAN_UCHAR:
           return gan_image_set_gen_rgba_uc ( img, height, width, stride,
                                              alloc_pix_data );

         case GAN_USHORT:
           return gan_image_set_gen_rgba_us ( img, height, width, stride,
                                              alloc_pix_data );

         case GAN_SHORT:
           return gan_image_set_gen_rgba_s ( img, height, width, stride,
                                             alloc_pix_data );

         case GAN_UINT:
           return gan_image_set_gen_rgba_ui ( img, height, width, stride,
                                              alloc_pix_data );

         case GAN_INT:
           return gan_image_set_gen_rgba_i ( img, height, width, stride,
                                             alloc_pix_data );

         case GAN_FLOAT:
           return gan_image_set_gen_rgba_f ( img, height, width, stride,
                                             alloc_pix_data );

         case GAN_DOUBLE:
           return gan_image_set_gen_rgba_d ( img, height, width, stride,
                                             alloc_pix_data );

         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_image_set_format_type_dims",
                              GAN_ERROR_ILLEGAL_TYPE, "" );
           return NULL;
      }

      case GAN_VECTOR_FIELD_2D:
      switch ( type )
      {
         case GAN_SHORT:
           return gan_image_set_gen_vfield2D_s ( img, height, width, stride,
                                                 alloc_pix_data );

         case GAN_INT:
           return gan_image_set_gen_vfield2D_i ( img, height, width, stride,
                                                 alloc_pix_data );

         case GAN_FLOAT:
           return gan_image_set_gen_vfield2D_f ( img, height, width, stride,
                                                 alloc_pix_data );

         case GAN_DOUBLE:
           return gan_image_set_gen_vfield2D_d ( img, height, width, stride,
                                                 alloc_pix_data );

         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_image_set_format_type_dims",
                              GAN_ERROR_ILLEGAL_TYPE, "" );
           return NULL;
      }

      case GAN_VECTOR_FIELD_3D:
      switch ( type )
      {
         case GAN_SHORT:
           return gan_image_set_gen_vfield3D_s ( img, height, width, stride,
                                                 alloc_pix_data );

         case GAN_INT:
           return gan_image_set_gen_vfield3D_i ( img, height, width, stride,
                                                 alloc_pix_data );

         case GAN_FLOAT:
           return gan_image_set_gen_vfield3D_f ( img, height, width, stride,
                                                 alloc_pix_data );

         case GAN_DOUBLE:
           return gan_image_set_gen_vfield3D_d ( img, height, width, stride,
                                                 alloc_pix_data );

         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_image_set_format_type_dims",
                              GAN_ERROR_ILLEGAL_TYPE, "" );
           return NULL;
      }

      default:
        gan_err_flush_trace();
        gan_err_register ( "gan_image_set_format_type_dims",
                           GAN_ERROR_ILLEGAL_TYPE, "" );
        return NULL;
   }

   /* shouldn't get here */
   return NULL;
}

/**
 * \}
 */

/**
 * \addtogroup ImageFill
 * \{
 */

/**
 * \brief Fill an image with a constant value.
 * \return #GAN_TRUE on success, #GAN_FALSE on failure.
 *
 * Fill image \a img with constant value \a pixel.
 */
Gan_Bool
 gan_image_fill_const ( Gan_Image *img, Gan_Pixel *pixel )
{
   gan_err_test_bool ( img->format == pixel->format &&
                       img->type == pixel->type, "gan_image_fill_const",
                       GAN_ERROR_INCOMPATIBLE, "" );
   switch ( img->format )
   {
      case GAN_GREY_LEVEL_IMAGE:
      switch ( img->type )
      {
         case GAN_UCHAR:
           return gan_image_fill_const_gl_uc ( img, pixel->data.gl.uc );

         case GAN_SHORT:
           return gan_image_fill_const_gl_s ( img, pixel->data.gl.s );

         case GAN_USHORT:
           return gan_image_fill_const_gl_us ( img, pixel->data.gl.us );

         case GAN_INT:
           return gan_image_fill_const_gl_i ( img, pixel->data.gl.i );

         case GAN_UINT:
           return gan_image_fill_const_gl_ui ( img, pixel->data.gl.ui );

         case GAN_FLOAT:
           return gan_image_fill_const_gl_f ( img, pixel->data.gl.f );

         case GAN_DOUBLE:
           return gan_image_fill_const_gl_d ( img, pixel->data.gl.d );

         case GAN_BOOL:
           return gan_image_fill_const_b ( img, pixel->data.gl.b );

         case GAN_POINTER:
           return gan_image_fill_const_p ( img, pixel->data.gl.p );

         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_image_fill_const", GAN_ERROR_ILLEGAL_TYPE,
                              "" );
           break;
      }
      break;

      case GAN_GREY_LEVEL_ALPHA_IMAGE:
      switch ( img->type )
      {
         case GAN_UCHAR:
           return gan_image_fill_const_gla_uc ( img, &pixel->data.gla.uc );

         case GAN_SHORT:
           return gan_image_fill_const_gla_s ( img, &pixel->data.gla.s );

         case GAN_USHORT:
           return gan_image_fill_const_gla_us ( img, &pixel->data.gla.us );

         case GAN_INT:
           return gan_image_fill_const_gla_i ( img, &pixel->data.gla.i );

         case GAN_UINT:
           return gan_image_fill_const_gla_ui ( img, &pixel->data.gla.ui );

         case GAN_FLOAT:
           return gan_image_fill_const_gla_f ( img, &pixel->data.gla.f );

         case GAN_DOUBLE:
           return gan_image_fill_const_gla_d ( img, &pixel->data.gla.d );

         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_image_fill_const", GAN_ERROR_ILLEGAL_TYPE,
                              "" );
           break;
      }
      break;

      case GAN_RGB_COLOUR_IMAGE:
      switch ( img->type )
      {
         case GAN_UCHAR:
           return gan_image_fill_const_rgb_uc ( img, &pixel->data.rgb.uc );

         case GAN_SHORT:
           return gan_image_fill_const_rgb_s ( img, &pixel->data.rgb.s );

         case GAN_USHORT:
           return gan_image_fill_const_rgb_us ( img, &pixel->data.rgb.us );

         case GAN_INT:
           return gan_image_fill_const_rgb_i ( img, &pixel->data.rgb.i );

         case GAN_UINT:
           return gan_image_fill_const_rgb_ui ( img, &pixel->data.rgb.ui );

         case GAN_FLOAT:
           return gan_image_fill_const_rgb_f ( img, &pixel->data.rgb.f );

         case GAN_DOUBLE:
           return gan_image_fill_const_rgb_d ( img, &pixel->data.rgb.d );

         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_image_fill_const", GAN_ERROR_ILLEGAL_TYPE,
                              "" );
           break;
      }
      break;

      case GAN_RGB_COLOUR_ALPHA_IMAGE:
      switch ( img->type )
      {
         case GAN_UCHAR:
           return gan_image_fill_const_rgba_uc ( img, &pixel->data.rgba.uc );

         case GAN_SHORT:
           return gan_image_fill_const_rgba_s ( img, &pixel->data.rgba.s );

         case GAN_USHORT:
           return gan_image_fill_const_rgba_us ( img, &pixel->data.rgba.us );

         case GAN_INT:
           return gan_image_fill_const_rgba_i ( img, &pixel->data.rgba.i );

         case GAN_UINT:
           return gan_image_fill_const_rgba_ui ( img, &pixel->data.rgba.ui );

         case GAN_FLOAT:
           return gan_image_fill_const_rgba_f ( img, &pixel->data.rgba.f );

         case GAN_DOUBLE:
           return gan_image_fill_const_rgba_d ( img, &pixel->data.rgba.d );

         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_image_fill_const", GAN_ERROR_ILLEGAL_TYPE,
                              "" );
           break;
      }
      break;

      case GAN_VECTOR_FIELD_2D:
      switch ( img->type )
      {
         case GAN_FLOAT:
           return gan_image_fill_const_vfield2D_f(img,&pixel->data.vfield2D.f);

         case GAN_DOUBLE:
           return gan_image_fill_const_vfield2D_d(img,&pixel->data.vfield2D.d);

         case GAN_SHORT:
           return gan_image_fill_const_vfield2D_s(img,&pixel->data.vfield2D.s);

         case GAN_INT:
           return gan_image_fill_const_vfield2D_i(img,&pixel->data.vfield2D.i);

         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_image_fill_const", GAN_ERROR_ILLEGAL_TYPE,
                              "" );
           break;
      }
      break;

      case GAN_VECTOR_FIELD_3D:
      switch ( img->type )
      {
         case GAN_FLOAT:
           return gan_image_fill_const_vfield3D_f(img,&pixel->data.vfield3D.f);

         case GAN_DOUBLE:
           return gan_image_fill_const_vfield3D_d(img,&pixel->data.vfield3D.d);

         case GAN_SHORT:
           return gan_image_fill_const_vfield3D_s(img,&pixel->data.vfield3D.s);

         case GAN_INT:
           return gan_image_fill_const_vfield3D_i(img,&pixel->data.vfield3D.i);

         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_image_fill_const", GAN_ERROR_ILLEGAL_TYPE,
                              "" );
           break;
      }
      break;

      default:
        gan_err_flush_trace();
        gan_err_register ( "gan_image_fill_const", GAN_ERROR_ILLEGAL_TYPE,
                           "" );
        break;
   }

   /* shouldn't get here */
   assert(0);
   return GAN_FALSE;
}

/**
 * \brief Fills an image with zero pixels.
 * \param img The image to fill
 * \return #GAN_TRUE on success, or #GAN_FALSE on failure.
 *
 * Fills image \a img with zero pixels.
 */
Gan_Bool
 gan_image_fill_zero ( Gan_Image *img )
{
   switch ( img->format )
   {
      case GAN_GREY_LEVEL_IMAGE:
      switch ( img->type )
      {
         case GAN_UCHAR:
           return gan_image_fill_const_gl_uc ( img, 0 );

         case GAN_SHORT:
           return gan_image_fill_const_gl_s ( img, 0 );

         case GAN_USHORT:
           return gan_image_fill_const_gl_us ( img, 0 );

         case GAN_INT:
           return gan_image_fill_const_gl_i ( img, 0 );

         case GAN_UINT:
           return gan_image_fill_const_gl_ui ( img, 0 );

         case GAN_FLOAT:
           return gan_image_fill_const_gl_f ( img, 0.0F );

         case GAN_DOUBLE:
           return gan_image_fill_const_gl_d ( img, 0.0 );

         case GAN_BOOL:
           return gan_image_fill_const_b ( img, GAN_FALSE );

         case GAN_POINTER:
           return gan_image_fill_const_p ( img, NULL );

         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_image_fill_zero", GAN_ERROR_ILLEGAL_TYPE,
                              "" );
           break;
      }
      break;

      case GAN_GREY_LEVEL_ALPHA_IMAGE:
      switch ( img->type )
      {
         case GAN_UCHAR:
         {
            Gan_GLAPixel_uc pix = { 0, 0 };

            return gan_image_fill_const_gla_uc ( img, &pix );
         }

         case GAN_SHORT:
         {
            Gan_GLAPixel_s pix = { 0, 0 };

            return gan_image_fill_const_gla_s ( img, &pix );
         }

         case GAN_USHORT:
         {
            Gan_GLAPixel_us pix = { 0, 0 };

            return gan_image_fill_const_gla_us ( img, &pix );
         }

         case GAN_INT:
         {
            Gan_GLAPixel_i pix = { 0, 0 };

            return gan_image_fill_const_gla_i ( img, &pix );
         }

         case GAN_UINT:
         {
            Gan_GLAPixel_ui pix = { 0, 0 };

            return gan_image_fill_const_gla_ui ( img, &pix );
         }

         case GAN_FLOAT:
         {
            Gan_GLAPixel_f pix = { 0.0F, 0.0F };

            return gan_image_fill_const_gla_f ( img, &pix );
         }

         case GAN_DOUBLE:
         {
            Gan_GLAPixel_d pix = { 0.0, 0.0 };

            return gan_image_fill_const_gla_d ( img, &pix );
         }

         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_image_fill_zero", GAN_ERROR_ILLEGAL_TYPE,
                              "" );
           break;
      }
      break;

      case GAN_RGB_COLOUR_IMAGE:
      switch ( img->type )
      {
         case GAN_UCHAR:
         {
            Gan_RGBPixel_uc pix = { 0, 0, 0 };

            return gan_image_fill_const_rgb_uc ( img, &pix );
         }

         case GAN_SHORT:
         {
            Gan_RGBPixel_s pix = { 0, 0, 0 };

            return gan_image_fill_const_rgb_s ( img, &pix );
         }

         case GAN_USHORT:
         {
            Gan_RGBPixel_us pix = { 0, 0, 0 };

            return gan_image_fill_const_rgb_us ( img, &pix );
         }

         case GAN_INT:
         {
            Gan_RGBPixel_i pix = { 0, 0, 0 };

            return gan_image_fill_const_rgb_i ( img, &pix );
         }

         case GAN_UINT:
         {
            Gan_RGBPixel_ui pix = { 0, 0, 0 };

            return gan_image_fill_const_rgb_ui ( img, &pix );
         }

         case GAN_FLOAT:
         {
            Gan_RGBPixel_f pix = { 0.0F, 0.0F, 0.0F };

            return gan_image_fill_const_rgb_f ( img, &pix );
         }

         case GAN_DOUBLE:
         {
            Gan_RGBPixel_d pix = { 0.0, 0.0, 0.0 };

            return gan_image_fill_const_rgb_d ( img, &pix );
         }

         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_image_fill_zero", GAN_ERROR_ILLEGAL_TYPE,
                              "" );
           break;
      }
      break;

      case GAN_RGB_COLOUR_ALPHA_IMAGE:
      switch ( img->type )
      {
         case GAN_UCHAR:
         {
            Gan_RGBAPixel_uc pix = { 0, 0, 0, 0 };

            return gan_image_fill_const_rgba_uc ( img, &pix );
         }

         case GAN_SHORT:
         {
            Gan_RGBAPixel_s pix = { 0, 0, 0, 0 };

            return gan_image_fill_const_rgba_s ( img, &pix );
         }

         case GAN_USHORT:
         {
            Gan_RGBAPixel_us pix = { 0, 0, 0, 0 };

            return gan_image_fill_const_rgba_us ( img, &pix );
         }

         case GAN_INT:
         {
            Gan_RGBAPixel_i pix = { 0, 0, 0, 0 };

            return gan_image_fill_const_rgba_i ( img, &pix );
         }

         case GAN_UINT:
         {
            Gan_RGBAPixel_ui pix = { 0, 0, 0, 0 };

            return gan_image_fill_const_rgba_ui ( img, &pix );
         }

         case GAN_FLOAT:
         {
            Gan_RGBAPixel_f pix = { 0.0F, 0.0F, 0.0F, 0.0F };

            return gan_image_fill_const_rgba_f ( img, &pix );
         }

         case GAN_DOUBLE:
         {
            Gan_RGBAPixel_d pix = { 0.0, 0.0, 0.0, 0.0 };

            return gan_image_fill_const_rgba_d ( img, &pix );
         }

         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_image_fill_zero", GAN_ERROR_ILLEGAL_TYPE,
                              "" );
           break;
      }
      break;

      case GAN_VECTOR_FIELD_2D:
      switch ( img->type )
      {
         case GAN_FLOAT:
         {
            Gan_Vector2_f vec = { 0.0F, 0.0F };

            return gan_image_fill_const_vfield2D_f ( img, &vec );
         }

         case GAN_DOUBLE:
         {
            Gan_Vector2 vec = { 0.0, 0.0 };

            return gan_image_fill_const_vfield2D_d ( img, &vec );
         }

         case GAN_SHORT:
         {
            Gan_Vector2_s vec = { 0, 0 };

            return gan_image_fill_const_vfield2D_s ( img, &vec );
         }

         case GAN_INT:
         {
            Gan_Vector2_i vec = { 0, 0 };

            return gan_image_fill_const_vfield2D_i ( img, &vec );
         }

         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_image_fill_zero", GAN_ERROR_ILLEGAL_TYPE,
                              "" );
           break;
      }
      break;

      case GAN_VECTOR_FIELD_3D:
      switch ( img->type )
      {
         case GAN_FLOAT:
         {
            Gan_Vector3_f vec = { 0.0F, 0.0F, 0.0F };

            return gan_image_fill_const_vfield3D_f ( img, &vec );
         }

         case GAN_DOUBLE:
         {
            Gan_Vector3 vec = { 0.0, 0.0, 0.0 };

            return gan_image_fill_const_vfield3D_d ( img, &vec );
         }

         case GAN_SHORT:
         {
            Gan_Vector3_s vec = { 0, 0, 0 };

            return gan_image_fill_const_vfield3D_s ( img, &vec );
         }

         case GAN_INT:
         {
            Gan_Vector3_i vec = { 0, 0, 0 };

            return gan_image_fill_const_vfield3D_i ( img, &vec );
         }

         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_image_fill_zero", GAN_ERROR_ILLEGAL_TYPE,
                              "" );
           break;
      }
      break;

      default:
        gan_err_flush_trace();
        gan_err_register ( "gan_image_fill_zero", GAN_ERROR_ILLEGAL_TYPE,
                           "" );
        break;
   }

   /* shouldn't get here */
   assert(0);
   return GAN_FALSE;
}

/**
 * \brief Fill a subwindow of an image.
 * \param img The image to fill
 * \param r0 The top row of the subwindow
 * \param c0 The left column of the subwindow
 * \param height The height of the subwindow
 * \param width The width of the subwindow
 * \param pixel The pixel value with which to fill the subwindow
 * \return #GAN_TRUE on success, or #GAN_FALSE on failure.
 *
 * Fills a subwindow of image \a img with value \a pixel, the subwindow being
 * defined by the \a r0, \a c0, \a height and \a width parameters.
 * A \c NULL \a pixel pointer indicates zero value.
 */
Gan_Bool
 gan_image_fill_const_window ( Gan_Image *img, unsigned r0, unsigned c0,
                               unsigned height, unsigned width,
                               Gan_Pixel *pixel )
{
   Gan_Image *sub_image;

   if ( img->type == GAN_BOOL )
   {
      unsigned i;

      gan_err_test_bool ( img->format == GAN_GREY_LEVEL_IMAGE &&
                          (pixel == NULL ||
                           (pixel->format == GAN_GREY_LEVEL_IMAGE &&
                            pixel->type == GAN_BOOL)),
                          "gan_image_fill_const_window",
                          GAN_ERROR_INCOMPATIBLE, "" );
      gan_err_test_bool ( r0+height <= img->height,
                          "gan_image_fill_const_window",
                          GAN_ERROR_INCOMPATIBLE, "" );
      if ( pixel == NULL || !pixel->data.gl.b )
      {
         for ( i = 0; i < height; i++ )
            if ( !gan_bit_array_fill_part ( &img->ba[r0+i],
                                            c0, width, GAN_FALSE ) )
            {
               gan_err_register ( "gan_image_fill_const_window",
                                  GAN_ERROR_INCOMPATIBLE, "" );
               return GAN_FALSE;
            }
      }
      else
      {
         for ( i = 0; i < height; i++ )
            if ( !gan_bit_array_fill_part ( &img->ba[r0+i],
                                            c0, width, GAN_TRUE ) )
            {
               gan_err_register ( "gan_image_fill_const_window",
                                  GAN_ERROR_INCOMPATIBLE, "" );
               return GAN_FALSE;
            }
      }
               
      /* success */
      return GAN_TRUE;
   }

   /* point sub-image into main image */
   sub_image = gan_image_extract_s ( img, r0, c0, height, width,
                                     img->format, img->type, GAN_FALSE );
   if ( sub_image == NULL )
   {
      gan_err_register ( "gan_image_fill_const_window", GAN_ERROR_FAILURE, "");
      return GAN_FALSE;
   }

   /* fill sub-image with constant value */
   if ( pixel == NULL ? !gan_image_fill_zero ( sub_image )
                      : !gan_image_fill_const ( sub_image, pixel ) )
   {
      gan_err_register ( "gan_image_fill_const_window", GAN_ERROR_FAILURE, "");
      return GAN_FALSE;
   }

   /* success */
   gan_image_free(sub_image);
   return GAN_TRUE;
}
 
/**
 * \}
 */

/**
 * \addtogroup ImageAccessPixel
 * \{
 */

/**
 * \brief Gets a pixel from an image.
 * \param img The image
 * \param row The row position of the pixel
 * \param col The column position of the pixel
 * \return The pixel at the given position
 *
 * Gets the pixel from image \a img at position \a row, \a col.
 * \sa gan_image_set_pix().
 */
Gan_Pixel
 gan_image_get_pix ( Gan_Image *img, unsigned row, unsigned col )
{
   Gan_Pixel pixel;

   /* set format and type of pixel */
   pixel.format = img->format;
   pixel.type   = img->type;
   switch ( img->format )
   {
      case GAN_GREY_LEVEL_IMAGE:
      switch ( img->type )
      {
         case GAN_UCHAR:
           pixel.data.gl.uc = gan_image_get_pix_gl_uc ( img, row, col );
           break;

         case GAN_SHORT:
           pixel.data.gl.s = gan_image_get_pix_gl_s ( img, row, col );
           break;

         case GAN_USHORT:
           pixel.data.gl.us = gan_image_get_pix_gl_us ( img, row, col );
           break;

         case GAN_INT:
           pixel.data.gl.i = gan_image_get_pix_gl_i ( img, row, col );
           break;

         case GAN_UINT:
           pixel.data.gl.ui = gan_image_get_pix_gl_ui ( img, row, col );
           break;

         case GAN_FLOAT:
           pixel.data.gl.f = gan_image_get_pix_gl_f ( img, row, col );
           break;

         case GAN_DOUBLE:
           pixel.data.gl.d = gan_image_get_pix_gl_d ( img, row, col );
           break;

         case GAN_BOOL:
           pixel.data.gl.b = gan_image_get_pix_b ( img, row, col );
           break;

         case GAN_POINTER:
           pixel.data.gl.p = gan_image_get_pix_p ( img, row, col );
           break;

         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_image_get_pix", GAN_ERROR_ILLEGAL_TYPE,
                              "" );
           break;
      }
      break;

      case GAN_GREY_LEVEL_ALPHA_IMAGE:
      switch ( img->type )
      {
         case GAN_UCHAR:
           pixel.data.gla.uc = gan_image_get_pix_gla_uc ( img, row, col );
           break;

         case GAN_SHORT:
           pixel.data.gla.s = gan_image_get_pix_gla_s ( img, row, col );
           break;

         case GAN_USHORT:
           pixel.data.gla.us = gan_image_get_pix_gla_us ( img, row, col );
           break;

         case GAN_INT:
           pixel.data.gla.i = gan_image_get_pix_gla_i ( img, row, col );
           break;

         case GAN_UINT:
           pixel.data.gla.ui = gan_image_get_pix_gla_ui ( img, row, col );
           break;

         case GAN_FLOAT:
            pixel.data.gla.f = gan_image_get_pix_gla_f ( img, row, col );
            break;

         case GAN_DOUBLE:
            pixel.data.gla.d = gan_image_get_pix_gla_d ( img, row, col );
            break;

         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_image_get_pix", GAN_ERROR_ILLEGAL_TYPE,
                              "" );
           break;
      }
      break;

      case GAN_RGB_COLOUR_IMAGE:
      switch ( img->type )
      {
         case GAN_UCHAR:
           pixel.data.rgb.uc = gan_image_get_pix_rgb_uc ( img, row, col );
           break;

         case GAN_SHORT:
           pixel.data.rgb.s = gan_image_get_pix_rgb_s ( img, row, col );
           break;

         case GAN_USHORT:
           pixel.data.rgb.us = gan_image_get_pix_rgb_us ( img, row, col );
           break;

         case GAN_INT:
           pixel.data.rgb.i = gan_image_get_pix_rgb_i ( img, row, col );
           break;

         case GAN_UINT:
           pixel.data.rgb.ui = gan_image_get_pix_rgb_ui ( img, row, col );
           break;

         case GAN_FLOAT:
           pixel.data.rgb.f = gan_image_get_pix_rgb_f ( img, row, col );
           break;

         case GAN_DOUBLE:
           pixel.data.rgb.d = gan_image_get_pix_rgb_d ( img, row, col );
           break;

         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_image_get_pix", GAN_ERROR_ILLEGAL_TYPE,
                              "" );
           break;
      }
      break;

      case GAN_RGB_COLOUR_ALPHA_IMAGE:
      switch ( img->type )
      {
         case GAN_UCHAR:
           pixel.data.rgba.uc = gan_image_get_pix_rgba_uc (img, row, col);
           break;

         case GAN_SHORT:
           pixel.data.rgba.s = gan_image_get_pix_rgba_s (img, row, col);
           break;

         case GAN_USHORT:
           pixel.data.rgba.us = gan_image_get_pix_rgba_us (img, row, col);
           break;

         case GAN_INT:
           pixel.data.rgba.i = gan_image_get_pix_rgba_i (img, row, col);
           break;

         case GAN_UINT:
           pixel.data.rgba.ui = gan_image_get_pix_rgba_ui (img, row, col);
           break;

         case GAN_FLOAT:
           pixel.data.rgba.f = gan_image_get_pix_rgba_f (img, row, col);
           break;

         case GAN_DOUBLE:
           pixel.data.rgba.d = gan_image_get_pix_rgba_d (img, row, col);
           break;

         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_image_get_pix", GAN_ERROR_ILLEGAL_TYPE,
                              "" );
           break;
      }
      break;

      case GAN_VECTOR_FIELD_2D:
      switch ( img->type )
      {
         case GAN_FLOAT:
           pixel.data.vfield2D.f = gan_image_get_pix_vfield2D_f(img, row, col);
           break;

         case GAN_DOUBLE:
           pixel.data.vfield2D.d = gan_image_get_pix_vfield2D_d(img, row, col);
           break;

         case GAN_SHORT:
           pixel.data.vfield2D.s = gan_image_get_pix_vfield2D_s(img, row, col);
           break;

         case GAN_INT:
           pixel.data.vfield2D.i = gan_image_get_pix_vfield2D_i(img, row, col);
           break;

         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_image_get_pix", GAN_ERROR_ILLEGAL_TYPE,
                              "" );
           break;
      }
      break;

      case GAN_VECTOR_FIELD_3D:
      switch ( img->type )
      {
         case GAN_FLOAT:
           pixel.data.vfield3D.f = gan_image_get_pix_vfield3D_f(img, row, col);
           break;

         case GAN_DOUBLE:
           pixel.data.vfield3D.d = gan_image_get_pix_vfield3D_d(img, row, col);
           break;

         case GAN_SHORT:
           pixel.data.vfield3D.s = gan_image_get_pix_vfield3D_s(img, row, col);
           break;

         case GAN_INT:
           pixel.data.vfield3D.i = gan_image_get_pix_vfield3D_i(img, row, col);
           break;

         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_image_get_pix", GAN_ERROR_ILLEGAL_TYPE,
                              "" );
           break;
      }
      break;

      default:
        gan_err_flush_trace();
        gan_err_register ( "gan_image_get_pix", GAN_ERROR_ILLEGAL_TYPE,
                           "" );
        break;
   }

   /* success */
   return pixel;
}

/**
 * \brief Gets a pointer to a pixel from an image.
 * \param img The image
 * \param row The row position of the pixel
 * \param col The column position of the pixel
 * \return A pointer to the pixel at the given position
 *
 * Gets the pixel pointer from image \a img at position \a row, \a col.
 * \sa gan_image_get_pix().
 */
void *
 gan_image_get_pixptr ( Gan_Image *img, unsigned row, unsigned col )
{
   void *result=NULL;

   switch ( img->format )
   {
      case GAN_GREY_LEVEL_IMAGE:
      switch ( img->type )
      {
         case GAN_UCHAR:
           result = gan_image_get_pixptr_gl_uc ( img, row, col );
           break;

         case GAN_SHORT:
           result = gan_image_get_pixptr_gl_s ( img, row, col );
           break;

         case GAN_USHORT:
           result = gan_image_get_pixptr_gl_us ( img, row, col );
           break;

         case GAN_INT:
           result = gan_image_get_pixptr_gl_i ( img, row, col );
           break;

         case GAN_UINT:
           result = gan_image_get_pixptr_gl_ui ( img, row, col );
           break;

         case GAN_FLOAT:
           result = gan_image_get_pixptr_gl_f ( img, row, col );
           break;

         case GAN_DOUBLE:
           result = gan_image_get_pixptr_gl_d ( img, row, col );
           break;

         case GAN_POINTER:
           result = gan_image_get_pixptr_p ( img, row, col );
           break;

         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_image_get_pixptr", GAN_ERROR_ILLEGAL_TYPE,
                              "" );
           break;
      }
      break;

      case GAN_GREY_LEVEL_ALPHA_IMAGE:
      switch ( img->type )
      {
         case GAN_UCHAR:
           result = gan_image_get_pixptr_gla_uc ( img, row, col );
           break;

         case GAN_SHORT:
           result = gan_image_get_pixptr_gla_s ( img, row, col );
           break;

         case GAN_USHORT:
           result = gan_image_get_pixptr_gla_us ( img, row, col );
           break;

         case GAN_INT:
           result = gan_image_get_pixptr_gla_i ( img, row, col );
           break;

         case GAN_UINT:
           result = gan_image_get_pixptr_gla_ui ( img, row, col );
           break;

         case GAN_FLOAT:
            result = gan_image_get_pixptr_gla_f ( img, row, col );
            break;

         case GAN_DOUBLE:
            result = gan_image_get_pixptr_gla_d ( img, row, col );
            break;

         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_image_get_pixptr", GAN_ERROR_ILLEGAL_TYPE,
                              "" );
           break;
      }
      break;

      case GAN_RGB_COLOUR_IMAGE:
      switch ( img->type )
      {
         case GAN_UCHAR:
           result = gan_image_get_pixptr_rgb_uc ( img, row, col );
           break;

         case GAN_SHORT:
           result = gan_image_get_pixptr_rgb_s ( img, row, col );
           break;

         case GAN_USHORT:
           result = gan_image_get_pixptr_rgb_us ( img, row, col );
           break;

         case GAN_INT:
           result = gan_image_get_pixptr_rgb_i ( img, row, col );
           break;

         case GAN_UINT:
           result = gan_image_get_pixptr_rgb_ui ( img, row, col );
           break;

         case GAN_FLOAT:
           result = gan_image_get_pixptr_rgb_f ( img, row, col );
           break;

         case GAN_DOUBLE:
           result = gan_image_get_pixptr_rgb_d ( img, row, col );
           break;

         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_image_get_pixptr", GAN_ERROR_ILLEGAL_TYPE,
                              "" );
           break;
      }
      break;

      case GAN_RGB_COLOUR_ALPHA_IMAGE:
      switch ( img->type )
      {
         case GAN_UCHAR:
           result = gan_image_get_pixptr_rgba_uc (img, row, col);
           break;

         case GAN_SHORT:
           result = gan_image_get_pixptr_rgba_s (img, row, col);
           break;

         case GAN_USHORT:
           result = gan_image_get_pixptr_rgba_us (img, row, col);
           break;

         case GAN_INT:
           result = gan_image_get_pixptr_rgba_i (img, row, col);
           break;

         case GAN_UINT:
           result = gan_image_get_pixptr_rgba_ui (img, row, col);
           break;

         case GAN_FLOAT:
           result = gan_image_get_pixptr_rgba_f (img, row, col);
           break;

         case GAN_DOUBLE:
           result = gan_image_get_pixptr_rgba_d (img, row, col);
           break;

         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_image_get_pixptr", GAN_ERROR_ILLEGAL_TYPE,
                              "" );
           break;
      }
      break;

      case GAN_VECTOR_FIELD_2D:
      switch ( img->type )
      {
         case GAN_FLOAT:
           result = gan_image_get_pixptr_vfield2D_f(img, row, col);
           break;

         case GAN_DOUBLE:
           result = gan_image_get_pixptr_vfield2D_d(img, row, col);
           break;

         case GAN_SHORT:
           result = gan_image_get_pixptr_vfield2D_s(img, row, col);
           break;

         case GAN_INT:
           result = gan_image_get_pixptr_vfield2D_i(img, row, col);
           break;

         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_image_get_pixptr", GAN_ERROR_ILLEGAL_TYPE,
                              "" );
           break;
      }
      break;

      case GAN_VECTOR_FIELD_3D:
      switch ( img->type )
      {
         case GAN_FLOAT:
           result = gan_image_get_pixptr_vfield3D_f(img, row, col);
           break;

         case GAN_DOUBLE:
           result = gan_image_get_pixptr_vfield3D_d(img, row, col);
           break;

         case GAN_SHORT:
           result = gan_image_get_pixptr_vfield3D_s(img, row, col);
           break;

         case GAN_INT:
           result = gan_image_get_pixptr_vfield3D_i(img, row, col);
           break;

         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_image_get_pixptr", GAN_ERROR_ILLEGAL_TYPE,
                              "" );
           break;
      }
      break;

      default:
        gan_err_flush_trace();
        gan_err_register ( "gan_image_get_pixptr", GAN_ERROR_ILLEGAL_TYPE,
                           "" );
        break;
   }

   if ( result == NULL )
   {
      gan_err_register ( "gan_image_get_pixptr", GAN_ERROR_FAILURE, "" );
      return NULL;
   }
      
   /* success */
   return result;
}

/**
 * \brief Gets a pixel from an image and compares it with zero.
 * \param img The image
 * \param row The row position of the pixel
 * \param col The column position of the pixel
 * \return #GAN_TRUE if the pixel is zero, #GAN_FALSE otherwise.
 *
 * Gets a pixel from an image and compares it with zero.
 * \sa gan_image_get_pix().
 */
Gan_Bool
 gan_image_get_pix_zero ( Gan_Image *img, unsigned row, unsigned col )
{
   switch ( img->format )
   {
      case GAN_GREY_LEVEL_IMAGE:
      switch ( img->type )
      {
         case GAN_UCHAR:
           return (Gan_Bool)
                  ( gan_image_get_pix_gl_uc ( img, row, col ) == 0 );

         case GAN_SHORT:
           return (Gan_Bool)
                  ( gan_image_get_pix_gl_s ( img, row, col ) == 0 );

         case GAN_USHORT:
           return (Gan_Bool)
                  ( gan_image_get_pix_gl_us ( img, row, col ) == 0 );

         case GAN_INT:
           return (Gan_Bool)
                  ( gan_image_get_pix_gl_i ( img, row, col ) == 0 );

         case GAN_UINT:
           return (Gan_Bool)
                  ( gan_image_get_pix_gl_ui ( img, row, col ) == 0 );

         case GAN_FLOAT:
           return (Gan_Bool)
                  ( gan_image_get_pix_gl_f ( img, row, col ) == 0.0F );

         case GAN_DOUBLE:
           return (Gan_Bool)
                  ( gan_image_get_pix_gl_d ( img, row, col ) == 0.0 );

         case GAN_BOOL:
           return (Gan_Bool)
                  ( gan_image_get_pix_b ( img, row, col ) == GAN_FALSE );

         case GAN_POINTER:
           return (Gan_Bool)
                  ( gan_image_get_pix_p ( img, row, col ) == NULL );

         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_image_get_pix_zero", GAN_ERROR_ILLEGAL_TYPE,
                              "" );
           break;
      }
      break;

      case GAN_GREY_LEVEL_ALPHA_IMAGE:
      switch ( img->type )
      {
         case GAN_UCHAR:
         {
            Gan_GLAPixel_uc pix = gan_image_get_pix_gla_uc ( img, row, col );

            return (Gan_Bool) ( pix.I == 0 && pix.A == 0 );
         }

         case GAN_SHORT:
         {
            Gan_GLAPixel_s pix = gan_image_get_pix_gla_s ( img, row, col );

            return (Gan_Bool) ( pix.I == 0 && pix.A == 0 );
         }

         case GAN_USHORT:
         {
            Gan_GLAPixel_us pix = gan_image_get_pix_gla_us ( img, row, col );

            return (Gan_Bool) ( pix.I == 0 && pix.A == 0 );
         }

         case GAN_INT:
         {
            Gan_GLAPixel_i pix = gan_image_get_pix_gla_i ( img, row, col );

            return (Gan_Bool) ( pix.I == 0 && pix.A == 0 );
         }

         case GAN_UINT:
         {
            Gan_GLAPixel_ui pix = gan_image_get_pix_gla_ui ( img, row, col );

            return (Gan_Bool) ( pix.I == 0 && pix.A == 0 );
         }

         case GAN_FLOAT:
         {
            Gan_GLAPixel_f pix = gan_image_get_pix_gla_f ( img, row, col );

            return (Gan_Bool) ( pix.I == 0.0F && pix.A == 0.0F );
         }

         case GAN_DOUBLE:
         {
            Gan_GLAPixel_d pix = gan_image_get_pix_gla_d ( img, row, col );

            return (Gan_Bool) ( pix.I == 0.0 && pix.A == 0.0 );
         }

         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_image_get_pix_zero", GAN_ERROR_ILLEGAL_TYPE,
                              "" );
           break;
      }
      break;

      case GAN_RGB_COLOUR_IMAGE:
      switch ( img->type )
      {
         case GAN_UCHAR:
         {
            Gan_RGBPixel_uc pix = gan_image_get_pix_rgb_uc ( img, row, col );

            return (Gan_Bool) ( pix.R == 0 && pix.G == 0 && pix.B == 0 );
         }

         case GAN_SHORT:
         {
            Gan_RGBPixel_s pix = gan_image_get_pix_rgb_s ( img, row, col );

            return (Gan_Bool) ( pix.R == 0 && pix.G == 0 && pix.B == 0 );
         }

         case GAN_USHORT:
         {
            Gan_RGBPixel_us pix = gan_image_get_pix_rgb_us ( img, row, col );

            return (Gan_Bool) ( pix.R == 0 && pix.G == 0 && pix.B == 0 );
         }

         case GAN_INT:
         {
            Gan_RGBPixel_i pix = gan_image_get_pix_rgb_i ( img, row, col );

            return (Gan_Bool) ( pix.R == 0 && pix.G == 0 && pix.B == 0 );
         }

         case GAN_UINT:
         {
            Gan_RGBPixel_ui pix = gan_image_get_pix_rgb_ui ( img, row, col );

            return (Gan_Bool) ( pix.R == 0 && pix.G == 0 && pix.B == 0 );
         }

         case GAN_FLOAT:
         {
            Gan_RGBPixel_f pix = gan_image_get_pix_rgb_f ( img, row, col );

            return (Gan_Bool) ( pix.R == 0.0F && pix.G == 0.0F &&
                                pix.B == 0.0F );
         }

         case GAN_DOUBLE:
         {
            Gan_RGBPixel_d pix = gan_image_get_pix_rgb_d ( img, row, col );

            return (Gan_Bool) ( pix.R == 0.0 && pix.G == 0.0 && pix.B == 0.0 );
         }

         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_image_get_pix_zero", GAN_ERROR_ILLEGAL_TYPE,
                              "" );
           break;
      }
      break;

      case GAN_RGB_COLOUR_ALPHA_IMAGE:
      switch ( img->type )
      {
         case GAN_UCHAR:
         {
            Gan_RGBAPixel_uc pix = gan_image_get_pix_rgba_uc ( img, row, col );

            return (Gan_Bool) ( pix.R == 0 && pix.G == 0 &&
                                pix.B == 0 && pix.A == 0 );
         }

         case GAN_SHORT:
         {
            Gan_RGBAPixel_s pix = gan_image_get_pix_rgba_s ( img, row, col );

            return (Gan_Bool) ( pix.R == 0 && pix.G == 0 &&
                                pix.B == 0 && pix.A == 0 );
         }

         case GAN_USHORT:
         {
            Gan_RGBAPixel_us pix = gan_image_get_pix_rgba_us ( img, row, col );

            return (Gan_Bool) ( pix.R == 0 && pix.G == 0 &&
                                pix.B == 0 && pix.A == 0 );
         }

         case GAN_INT:
         {
            Gan_RGBAPixel_i pix = gan_image_get_pix_rgba_i ( img, row, col );

            return (Gan_Bool) ( pix.R == 0 && pix.G == 0 &&
                                pix.B == 0 && pix.A == 0 );
         }

         case GAN_UINT:
         {
            Gan_RGBAPixel_ui pix = gan_image_get_pix_rgba_ui ( img, row, col );

            return (Gan_Bool) ( pix.R == 0 && pix.G == 0 &&
                                pix.B == 0 && pix.A == 0 );
         }

         case GAN_FLOAT:
         {
            Gan_RGBAPixel_f pix = gan_image_get_pix_rgba_f ( img, row, col );

            return (Gan_Bool) ( pix.R == 0.0F && pix.G == 0.0F &&
                                pix.B == 0.0F && pix.A == 0.0F );
         }

         case GAN_DOUBLE:
         {
            Gan_RGBAPixel_d pix = gan_image_get_pix_rgba_d ( img, row, col );

            return (Gan_Bool) ( pix.R == 0.0 && pix.G == 0.0 &&
                                pix.B == 0.0 && pix.A == 0 );
         }

         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_image_get_pix_zero", GAN_ERROR_ILLEGAL_TYPE,
                              "" );
           break;
      }
      break;

      case GAN_VECTOR_FIELD_2D:
      switch ( img->type )
      {
         case GAN_FLOAT:
         {
            Gan_Vector2_f pix = gan_image_get_pix_vfield2D_f ( img, row, col );

            return (Gan_Bool) ( pix.x == 0.0F && pix.y == 0.0F );
         }

         case GAN_DOUBLE:
         {
            Gan_Vector2 pix = gan_image_get_pix_vfield2D_d ( img, row, col );

            return (Gan_Bool) ( pix.x == 0.0 && pix.y == 0.0 );
         }

         case GAN_SHORT:
         {
            Gan_Vector2_s pix = gan_image_get_pix_vfield2D_s ( img, row, col );

            return (Gan_Bool) ( pix.x == 0 && pix.y == 0 );
         }

         case GAN_INT:
         {
            Gan_Vector2_i pix = gan_image_get_pix_vfield2D_i ( img, row, col );

            return (Gan_Bool) ( pix.x == 0 && pix.y == 0 );
         }

         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_image_get_pix_zero", GAN_ERROR_ILLEGAL_TYPE,
                              "" );
           break;
      }
      break;

      case GAN_VECTOR_FIELD_3D:
      switch ( img->type )
      {
         case GAN_FLOAT:
         {
            Gan_Vector3_f pix = gan_image_get_pix_vfield3D_f ( img, row, col );

            return (Gan_Bool) ( pix.x == 0.0F && pix.y == 0.0F &&
                                pix.z == 0.0F );
         }

         case GAN_DOUBLE:
         {
            Gan_Vector3 pix = gan_image_get_pix_vfield3D_d ( img, row, col );

            return (Gan_Bool) ( pix.x == 0.0 && pix.y == 0.0 && pix.z == 0.0 );
         }

         case GAN_SHORT:
         {
            Gan_Vector3_s pix = gan_image_get_pix_vfield3D_s ( img, row, col );

            return (Gan_Bool) ( pix.x == 0 && pix.y == 0 && pix.z == 0 );
         }

         case GAN_INT:
         {
            Gan_Vector3_i pix = gan_image_get_pix_vfield3D_i ( img, row, col );

            return (Gan_Bool) ( pix.x == 0 && pix.y == 0 && pix.z == 0 );
         }

         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_image_get_pix_zero", GAN_ERROR_ILLEGAL_TYPE,
                              "" );
           break;
      }
      break;

      default:
        gan_err_flush_trace();
        gan_err_register ( "gan_image_get_pix_zero", GAN_ERROR_ILLEGAL_TYPE,
                           "" );
        break;
   }

   /* shouldn't get here */
   assert(0);
   return GAN_FALSE;
}

/**
 * \brief Sets a pixel in an image.
 * \param img The image
 * \param row The row position of the pixel
 * \param col The column position of the pixel
 * \param pix The pixel to set
 * \return #GAN_TRUE on success, or #GAN_FALSE on failure.
 *
 * Sets the pixel an image \a img at position \a row, \a col to the given
 * value \a pix.
 *
 * \sa gan_image_get_pix().
 */
Gan_Bool
 gan_image_set_pix ( Gan_Image *img, unsigned row, unsigned col,
                     Gan_Pixel *pix )
{
   /* consistency check */
   gan_err_test_bool ( pix->format == img->format && pix->type == img->type,
                       "gan_image_set_pix", GAN_ERROR_INCOMPATIBLE, "" );

   switch ( img->format )
   {
      case GAN_GREY_LEVEL_IMAGE:
      switch ( img->type )
      {
         case GAN_UCHAR:
           return gan_image_set_pix_gl_uc ( img, row, col, pix->data.gl.uc );

         case GAN_SHORT:
           return gan_image_set_pix_gl_s ( img, row, col, pix->data.gl.s );

         case GAN_USHORT:
           return gan_image_set_pix_gl_us ( img, row, col, pix->data.gl.us );

         case GAN_INT:
           return gan_image_set_pix_gl_i ( img, row, col, pix->data.gl.i );

         case GAN_UINT:
           return gan_image_set_pix_gl_ui ( img, row, col, pix->data.gl.ui );

         case GAN_FLOAT:
           return gan_image_set_pix_gl_f ( img, row, col, pix->data.gl.f );

         case GAN_DOUBLE:
           return gan_image_set_pix_gl_d ( img, row, col, pix->data.gl.d );

         case GAN_BOOL:
           return gan_image_set_pix_b ( img, row, col, pix->data.gl.b );

         case GAN_POINTER:
           return gan_image_set_pix_p ( img, row, col, pix->data.gl.p );

         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_image_set_pix", GAN_ERROR_ILLEGAL_TYPE,
                              "" );
           break;
      }
      break;

      case GAN_GREY_LEVEL_ALPHA_IMAGE:
      switch ( img->type )
      {
         case GAN_UCHAR:
           return gan_image_set_pix_gla_uc ( img, row, col, &pix->data.gla.uc);

         case GAN_SHORT:
           return gan_image_set_pix_gla_s ( img, row, col, &pix->data.gla.s );

         case GAN_USHORT:
           return gan_image_set_pix_gla_us ( img, row, col, &pix->data.gla.us);

         case GAN_INT:
           return gan_image_set_pix_gla_i ( img, row, col, &pix->data.gla.i );

         case GAN_UINT:
           return gan_image_set_pix_gla_ui ( img, row, col, &pix->data.gla.ui);

         case GAN_FLOAT:
           return gan_image_set_pix_gla_f ( img, row, col, &pix->data.gla.f);

         case GAN_DOUBLE:
           return gan_image_set_pix_gla_d ( img, row, col, &pix->data.gla.d);

         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_image_set_pix", GAN_ERROR_ILLEGAL_TYPE,
                              "" );
           break;
      }
      break;

      case GAN_RGB_COLOUR_IMAGE:
      switch ( img->type )
      {
         case GAN_UCHAR:
           return gan_image_set_pix_rgb_uc ( img, row, col, &pix->data.rgb.uc);

         case GAN_SHORT:
           return gan_image_set_pix_rgb_s ( img, row, col, &pix->data.rgb.s );

         case GAN_USHORT:
           return gan_image_set_pix_rgb_us ( img, row, col, &pix->data.rgb.us);

         case GAN_INT:
           return gan_image_set_pix_rgb_i ( img, row, col, &pix->data.rgb.i );

         case GAN_UINT:
           return gan_image_set_pix_rgb_ui ( img, row, col, &pix->data.rgb.ui);

         case GAN_FLOAT:
           return gan_image_set_pix_rgb_f ( img, row, col, &pix->data.rgb.f );

         case GAN_DOUBLE:
           return gan_image_set_pix_rgb_d ( img, row, col, &pix->data.rgb.d );

         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_image_set_pix", GAN_ERROR_ILLEGAL_TYPE,
                              "" );
           break;
      }
      break;

      case GAN_RGB_COLOUR_ALPHA_IMAGE:
      switch ( img->type )
      {
         case GAN_UCHAR:
           return gan_image_set_pix_rgba_uc ( img, row, col,
                                              &pix->data.rgba.uc );

         case GAN_SHORT:
           return gan_image_set_pix_rgba_s ( img, row, col,
                                             &pix->data.rgba.s );

         case GAN_USHORT:
           return gan_image_set_pix_rgba_us ( img, row, col,
                                              &pix->data.rgba.us );

         case GAN_INT:
           return gan_image_set_pix_rgba_i ( img, row, col,
                                             &pix->data.rgba.i );

         case GAN_UINT:
           return gan_image_set_pix_rgba_ui ( img, row, col,
                                              &pix->data.rgba.ui );

         case GAN_FLOAT:
           return gan_image_set_pix_rgba_f ( img, row, col,
                                             &pix->data.rgba.f );

         case GAN_DOUBLE:
           return gan_image_set_pix_rgba_d ( img, row, col,
                                             &pix->data.rgba.d );

         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_image_set_pix", GAN_ERROR_ILLEGAL_TYPE,
                              "" );
           break;
      }
      break;

      case GAN_VECTOR_FIELD_2D:
      switch ( img->type )
      {
         case GAN_FLOAT:
           return gan_image_set_pix_vfield2D_f ( img, row, col,
                                                 &pix->data.vfield2D.f );

         case GAN_DOUBLE:
           return gan_image_set_pix_vfield2D_d ( img, row, col,
                                                 &pix->data.vfield2D.d );

         case GAN_SHORT:
           return gan_image_set_pix_vfield2D_s ( img, row, col,
                                                 &pix->data.vfield2D.s );

         case GAN_INT:
           return gan_image_set_pix_vfield2D_i ( img, row, col,
                                                 &pix->data.vfield2D.i );

         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_image_set_pix", GAN_ERROR_ILLEGAL_TYPE,
                              "" );
           break;
      }
      break;

      case GAN_VECTOR_FIELD_3D:
      switch ( img->type )
      {
         case GAN_FLOAT:
           return gan_image_set_pix_vfield3D_f ( img, row, col,
                                                 &pix->data.vfield3D.f );

         case GAN_DOUBLE:
           return gan_image_set_pix_vfield3D_d ( img, row, col,
                                                 &pix->data.vfield3D.d );

         case GAN_SHORT:
           return gan_image_set_pix_vfield3D_s ( img, row, col,
                                                 &pix->data.vfield3D.s );

         case GAN_INT:
           return gan_image_set_pix_vfield3D_i ( img, row, col,
                                                 &pix->data.vfield3D.i );

         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_image_set_pix", GAN_ERROR_ILLEGAL_TYPE,
                              "" );
           break;
      }
      break;

      default:
        gan_err_flush_trace();
        gan_err_register ( "gan_image_set_pix", GAN_ERROR_ILLEGAL_TYPE,
                           "" );
        break;
   }

   /* shouldn't get here */
   assert(0);
   return GAN_FALSE;
}

/**
 * \brief Sets a pixel in an image to zero.
 * \param img The image
 * \param row The row position of the pixel
 * \param col The column position of the pixel
 * \return #GAN_TRUE on success, or #GAN_FALSE on failure.
 *
 * Sets the pixel in image \a img at position \a row, \a col to zero.
 * \sa gan_image_set_pix().
 */
Gan_Bool
 gan_image_set_pix_zero ( Gan_Image *img, unsigned row, unsigned col )
{
   switch ( img->format )
   {
      case GAN_GREY_LEVEL_IMAGE:
      switch ( img->type )
      {
         case GAN_UCHAR:
           return gan_image_set_pix_gl_uc ( img, row, col, 0 );

         case GAN_SHORT:
           return gan_image_set_pix_gl_s ( img, row, col, 0 );

         case GAN_USHORT:
           return gan_image_set_pix_gl_us ( img, row, col, 0 );

         case GAN_INT:
           return gan_image_set_pix_gl_i ( img, row, col, 0 );

         case GAN_UINT:
           return gan_image_set_pix_gl_ui ( img, row, col, 0 );

         case GAN_FLOAT:
           return gan_image_set_pix_gl_f ( img, row, col, 0.0F );

         case GAN_DOUBLE:
           return gan_image_set_pix_gl_d ( img, row, col, 0.0 );

         case GAN_BOOL:
           return gan_image_set_pix_b ( img, row, col, GAN_FALSE );

         case GAN_POINTER:
           return gan_image_set_pix_p ( img, row, col, NULL );

         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_image_set_pix_zero", GAN_ERROR_ILLEGAL_TYPE,
                              "" );
           break;
      }
      break;

      case GAN_GREY_LEVEL_ALPHA_IMAGE:
      switch ( img->type )
      {
         case GAN_UCHAR:
         {
            Gan_GLAPixel_uc pix = {0,0};
            
            return gan_image_set_pix_gla_uc ( img, row, col, &pix );
         }

         case GAN_SHORT:
         {
            Gan_GLAPixel_s pix = {0,0};
            
            return gan_image_set_pix_gla_s ( img, row, col, &pix );
         }

         case GAN_USHORT:
         {
            Gan_GLAPixel_us pix = {0,0};
            
            return gan_image_set_pix_gla_us ( img, row, col, &pix );
         }

         case GAN_INT:
         {
            Gan_GLAPixel_i pix = {0,0};
            
            return gan_image_set_pix_gla_i ( img, row, col, &pix );
         }

         case GAN_UINT:
         {
            Gan_GLAPixel_ui pix = {0,0};
            
            return gan_image_set_pix_gla_ui ( img, row, col, &pix );
         }

         case GAN_FLOAT:
         {
            Gan_GLAPixel_f pix = {0.0F,0.0F};
            
            return gan_image_set_pix_gla_f ( img, row, col, &pix );
         }

         case GAN_DOUBLE:
         {
            Gan_GLAPixel_d pix = {0.0,0.0};
            
            return gan_image_set_pix_gla_d ( img, row, col, &pix );
         }

         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_image_set_pix_zero", GAN_ERROR_ILLEGAL_TYPE,
                              "" );
           break;
      }
      break;

      case GAN_RGB_COLOUR_IMAGE:
      switch ( img->type )
      {
         case GAN_UCHAR:
         {
            Gan_RGBPixel_uc pix = {0,0,0};
            
            return gan_image_set_pix_rgb_uc ( img, row, col, &pix );
         }

         case GAN_SHORT:
         {
            Gan_RGBPixel_s pix = {0,0,0};
            
            return gan_image_set_pix_rgb_s ( img, row, col, &pix );
         }

         case GAN_USHORT:
         {
            Gan_RGBPixel_us pix = {0,0,0};
            
            return gan_image_set_pix_rgb_us ( img, row, col, &pix );
         }

         case GAN_INT:
         {
            Gan_RGBPixel_i pix = {0,0,0};
            
            return gan_image_set_pix_rgb_i ( img, row, col, &pix );
         }

         case GAN_UINT:
         {
            Gan_RGBPixel_ui pix = {0,0,0};
            
            return gan_image_set_pix_rgb_ui ( img, row, col, &pix );
         }

         case GAN_FLOAT:
         {
            Gan_RGBPixel_f pix = {0.0F,0.0F,0.0F};
            
            return gan_image_set_pix_rgb_f ( img, row, col, &pix );
         }

         case GAN_DOUBLE:
         {
            Gan_RGBPixel_d pix = {0.0,0.0,0.0};
            
            return gan_image_set_pix_rgb_d ( img, row, col, &pix );
         }

         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_image_set_pix_zero", GAN_ERROR_ILLEGAL_TYPE,
                              "" );
           break;
      }
      break;

      case GAN_RGB_COLOUR_ALPHA_IMAGE:
      switch ( img->type )
      {
         case GAN_UCHAR:
         {
            Gan_RGBAPixel_uc pix = {0,0,0,0};
            
            return gan_image_set_pix_rgba_uc ( img, row, col, &pix );
         }

         case GAN_SHORT:
         {
            Gan_RGBAPixel_s pix = {0,0,0,0};
            
            return gan_image_set_pix_rgba_s ( img, row, col, &pix );
         }

         case GAN_USHORT:
         {
            Gan_RGBAPixel_us pix = {0,0,0,0};
            
            return gan_image_set_pix_rgba_us ( img, row, col, &pix );
         }

         case GAN_INT:
         {
            Gan_RGBAPixel_i pix = {0,0,0,0};
            
            return gan_image_set_pix_rgba_i ( img, row, col, &pix );
         }

         case GAN_UINT:
         {
            Gan_RGBAPixel_ui pix = {0,0,0,0};
            
            return gan_image_set_pix_rgba_ui ( img, row, col, &pix );
         }

         case GAN_FLOAT:
         {
            Gan_RGBAPixel_f pix = {0.0F,0.0F,0.0F,0.0F};
            
            return gan_image_set_pix_rgba_f ( img, row, col, &pix );
         }

         case GAN_DOUBLE:
         {
            Gan_RGBAPixel_d pix = {0.0,0.0,0.0,0.0};
            
            return gan_image_set_pix_rgba_d ( img, row, col, &pix );
         }

         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_image_set_pix_zero", GAN_ERROR_ILLEGAL_TYPE,
                              "" );
           break;
      }
      break;

      case GAN_VECTOR_FIELD_2D:
      switch ( img->type )
      {
         case GAN_FLOAT:
         {
            Gan_Vector2_f pix = {0.0F,0.0F};
            
            return gan_image_set_pix_vfield2D_f ( img, row, col, &pix );
         }

         case GAN_DOUBLE:
         {
            Gan_Vector2 pix = {0.0,0.0};
            
            return gan_image_set_pix_vfield2D_d ( img, row, col, &pix );
         }

         case GAN_SHORT:
         {
            Gan_Vector2_s pix = {0,0};
            
            return gan_image_set_pix_vfield2D_s ( img, row, col, &pix );
         }

         case GAN_INT:
         {
            Gan_Vector2_i pix = {0,0};
            
            return gan_image_set_pix_vfield2D_i ( img, row, col, &pix );
         }

         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_image_set_pix_zero", GAN_ERROR_ILLEGAL_TYPE,
                              "" );
           break;
      }
      break;

      case GAN_VECTOR_FIELD_3D:
      switch ( img->type )
      {
         case GAN_FLOAT:
         {
            Gan_Vector3_f pix = {0.0F,0.0F,0.0F};
            
            return gan_image_set_pix_vfield3D_f ( img, row, col, &pix );
         }

         case GAN_DOUBLE:
         {
            Gan_Vector3 pix = {0.0,0.0,0.0};
            
            return gan_image_set_pix_vfield3D_d ( img, row, col, &pix );
         }

         case GAN_SHORT:
         {
            Gan_Vector3_s pix = {0,0,0};
            
            return gan_image_set_pix_vfield3D_s ( img, row, col, &pix );
         }

         case GAN_INT:
         {
            Gan_Vector3_i pix = {0,0,0};
            
            return gan_image_set_pix_vfield3D_i ( img, row, col, &pix );
         }

         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_image_set_pix_zero", GAN_ERROR_ILLEGAL_TYPE,
                              "" );
           break;
      }
      break;

      default:
        gan_err_flush_trace();
        gan_err_register ( "gan_image_set_pix_zero", GAN_ERROR_ILLEGAL_TYPE,
                           "" );
        break;
   }

   /* shouldn't get here */
   assert(0);
   return GAN_FALSE;
}

/**
 * \}
 */

/**
 * \addtogroup ImageGet
 * \{
 */

/**
 * \brief Computes bounding box of non-zero pixels in grey-level image.
 * \return #GAN_TRUE on success, or #GAN_FALSE on failure.
 *
 * Computes image window \a subwin which covers all pixels in given grey-level
 * \a image which are non-zero. If \a image is zero everywhere the fields of
 * the \a subwin structure will be set to zero.
 */
Gan_Bool
 gan_image_get_active_subwindow ( Gan_Image *image,
                                  Gan_ImageWindow *subwin )
{
   switch ( image->format )
   {
      case GAN_GREY_LEVEL_IMAGE:
      switch ( image->type )
      {
         case GAN_UCHAR:
           return gan_image_get_active_subwindow_gl_uc ( image, subwin );

         case GAN_SHORT:
           return gan_image_get_active_subwindow_gl_s ( image, subwin );

         case GAN_USHORT:
           return gan_image_get_active_subwindow_gl_us ( image, subwin );

         case GAN_INT:
           return gan_image_get_active_subwindow_gl_i ( image, subwin );

         case GAN_UINT:
           return gan_image_get_active_subwindow_gl_ui ( image, subwin );

         case GAN_FLOAT:
           return gan_image_get_active_subwindow_gl_f ( image, subwin );

         case GAN_DOUBLE:
           return gan_image_get_active_subwindow_gl_d ( image, subwin );

         case GAN_BOOL:
           return gan_image_get_active_subwindow_b ( image, GAN_BIT_ALIGNMENT,
                                                    subwin );

         case GAN_POINTER:
           return gan_image_get_active_subwindow_p ( image, subwin );

         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_image_get_active_subregion",
                              GAN_ERROR_ILLEGAL_TYPE, "" );
           break;
      }
      break;

      default:
        gan_err_flush_trace();
        gan_err_register ( "gan_image_get_active_subwindow",
                           GAN_ERROR_ILLEGAL_TYPE, "" );
        break;
   }

   /* shouldn't get here */
   assert(0);
   return GAN_FALSE;
}

/**
 * \}
 */

/**
 * \addtogroup ImageFill
 * \{
 */

/**
 * \brief Clear image to zero except in specified rectangular region.
 * \return #GAN_TRUE on success, or #GAN_FALSE on failure.
 *
 * Clear \a image to zero except in specified rectangular region.
 */
Gan_Bool
 gan_image_mask_window ( Gan_Image *image,
                         unsigned r0,     unsigned c0,
                         unsigned height, unsigned width )
{
   switch ( image->format )
   {
      case GAN_GREY_LEVEL_IMAGE:
      switch ( image->type )
      {
         case GAN_UCHAR:
           return gan_image_mask_window_gl_uc ( image, r0, c0, height, width );

         case GAN_SHORT:
           return gan_image_mask_window_gl_s ( image, r0, c0, height, width );

         case GAN_USHORT:
           return gan_image_mask_window_gl_us ( image, r0, c0, height, width );

         case GAN_INT:
           return gan_image_mask_window_gl_i ( image, r0, c0, height, width );

         case GAN_UINT:
           return gan_image_mask_window_gl_ui ( image, r0, c0, height, width );

         case GAN_FLOAT:
           return gan_image_mask_window_gl_f ( image, r0, c0, height, width );

         case GAN_DOUBLE:
           return gan_image_mask_window_gl_d ( image, r0, c0, height, width );

         case GAN_BOOL:
           return gan_image_mask_window_b ( image, r0, c0, height, width );

         case GAN_POINTER:
           return gan_image_mask_window_p ( image, r0, c0, height, width );

         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_image_mask_window", GAN_ERROR_ILLEGAL_TYPE,
                              "" );
           break;
      }
      break;

      default:
        gan_err_flush_trace();
        gan_err_register ( "gan_image_mask_window", GAN_ERROR_ILLEGAL_TYPE,
                           "" );
        break;
   }

   /* shouldn't get here */
   assert(0);
   return GAN_FALSE;
}

/**
 * \}
 */

/**
 * \addtogroup ImageAllocate
 * \{
 */

/**
 * \brief Free a \c NULL-terminated variable argument list of images.
 * \return No value.
 *
 * Free a \c NULL-terminated variable argument list of images, starting
 * with image \a img.
 */
void
 gan_image_free_va ( Gan_Image *img, ... )
{
   va_list ap;

   va_start ( ap, img );
   while ( img != NULL )
   {
      /* free next image */
      gan_image_free ( img );

      /* get next image in list */
      img = va_arg ( ap, Gan_Image * );
   }

   va_end ( ap );
}

/**
 * \}
 */

/**
 * \}
 */
