/**
 * File:          $RCSfile: image_extract.h,v $
 * Module:        Extract/convert part of an image
 * Part of:       Gandalf Library
 *
 * Revision:      $Revision: 1.17 $
 * Last edited:   $Date: 2003/01/31 18:56:45 $
 * Author:        $Author: pm $
 * Copyright:     (c) 2000 Imagineer Software Limited
 */

/* This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef _GAN_IMAGE_EXTRACT_H
#define _GAN_IMAGE_EXTRACT_H

#include <gandalf/image/image_defs.h>

#ifdef __cplusplus
extern "C" {
#endif

/**
 * \addtogroup ImagePackage
 * \{
 */

/**
 * \addtogroup ImageExtract
 * \{
 */

Gan_Image *gan_image_extract_q ( Gan_Image *source,
                                 int      r0,     int      c0,
                                 unsigned height, unsigned width,
                                 Gan_ImageFormat format, Gan_Type type,
                                 Gan_Bool copy_pixels,
                                 Gan_Image *dest );

/**
 * \brief Macro: Extracts part of an image.
 * \param source The input image
 * \param r0 Vertical offset of start of region
 * \param c0 Horizontal offset of start of region
 * \param height Height of region in pixels
 * \param width Width of region in pixels
 * \param format The format of the output image
 * \param type The type of the output image
 * \param copy_pixels Whether to copy the pixels from the source image
 * \return Pointer to the extracted/converted image, or \c NULL on failure.
 *
 * Extracts a sub-part of an image, converting the image to a different format
 * and type if desired. This is a macro call to gan_image_extract_q().
 *
 * \sa gan_image_extract_q().
 */
#ifdef GAN_GENERATE_DOCUMENTATION
Gan_Image *gan_image_extract_s ( Gan_Image *source,
                                 int      r0,     int      c0,
                                 unsigned height, unsigned width,
                                 Gan_ImageFormat format, Gan_Type type,
                                 Gan_Bool copy_pixels );
#else
#define gan_image_extract_s(source,r0,c0,height,width,format,type,copy_pixels)\
    gan_image_extract_q(source,r0,c0,height,width,format,type,copy_pixels,NULL)
#endif

/**
 * \brief Macro: Converts an image from one format/type to another.
 * \param source The input image
 * \param format The format of the output image
 * \param type The type of the output image
 * \param dest Destination image
 * \return Pointer to the converted image, or \c NULL on failure.
 *
 * Applies the relevant pixel-by-pixel conversion to convert an image from one
 * format/type to another. This is a macro call to gan_image_extract_q().
 *
 * \sa gan_image_convert_s().
 */
#ifdef GAN_GENERATE_DOCUMENTATION
Gan_Image *gan_image_convert_q ( Gan_Image *source,
                                 Gan_ImageFormat format, Gan_Type type,
                                 Gan_Image *dest );
#else
#define gan_image_convert_q(source,format,type,dest)\
           gan_image_extract_q(source,0,0,(source)->height,(source)->width,format,type,GAN_TRUE,dest)
#endif

/**
 * \brief Macro: Converts an image from one format/type to another.
 * \param source The input image
 * \param format The format of the output image
 * \param type The type of the output image
 * \return Pointer to the converted image, or \c NULL on failure.
 *
 * Applies the relevant pixel-by-pixel conversion to convert an image from one
 * format/type to another. This is a macro call to gan_image_convert_q().
 *
 * \sa gan_image_convert_q().
 */
#ifdef GAN_GENERATE_DOCUMENTATION
Gan_Image *gan_image_convert_s ( Gan_Image *source,
                                 Gan_ImageFormat format, Gan_Type type );
#else
#define gan_image_convert_s(source,format,type)\
           gan_image_convert_q(source,format,type,NULL)
#endif

/**
 * \}
 */

/**
 * \}
 */

#ifdef __cplusplus
}
#endif

#endif /* #ifndef _GAN_IMAGE_EXTRACT_H */
