/**
 * File:          $RCSfile: ppm_io.c,v $
 * Module:        PPM format image file I/O functions
 * Part of:       Gandalf Library
 *
 * Revision:      $Revision: 1.17 $
 * Last edited:   $Date: 2003/01/31 18:56:57 $
 * Author:        $Author: pm $
 * Copyright:     (c) 2000 Imagineer Software Limited
 */

/* This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <string.h>
#include <gandalf/image/io/ppm_io.h>
#include <gandalf/image/image_rgb_uchar.h>
#include <gandalf/common/misc_error.h>

/**
 * \addtogroup ImagePackage
 * \{
 */

/**
 * \addtogroup ImageIO
 * \{
 */

static Gan_Bool read_line ( FILE *infile, char *s )
{
  char *news, *sptr;
  static char line[80]="";

  if ((strlen (line) == 0) || ((sptr = strtok (NULL, " \n\t")) == NULL))
  {
     while ((news = fgets (line, 80, infile)) != NULL)
        if ( (line[0] != '#') && ((sptr = strtok (line, " \n\t")) != NULL) )
           break;

     if (news == NULL)
     {
        gan_err_flush_trace();
        gan_err_register ( "read_line", GAN_ERROR_CORRUPTED_FILE,
                           "reading line");
        return GAN_FALSE;
     }
  }

  strcpy ( s, sptr );
  return GAN_TRUE;
}

/**
 * \brief Reads a RGB colour image file in PPM format from a stream.
 * \param infile The file stream to be read
 * \param image The image structure to read the image data into or \c NULL
 * \return Pointer to image structure, or \c NULL on failure.
 *
 * Reads the PPM image from the given file stream \a infile into the given
 * \a image. If \a image is \c NULL a new image is dynamically allocated,
 * otherwise the already allocated \a image structure is reused.
 *
 * \sa gan_write_ppm_image_stream().
 */
Gan_Image *
 gan_read_ppm_image_stream ( FILE *infile, Gan_Image *image )
{
   char s[80]="";
   int width, height;
   Gan_Bool result=GAN_TRUE;

   fgets ( s, 4, infile );
   if ( strcmp ( s, "P6\n" ) != 0 )
   {
      gan_err_flush_trace();
      gan_err_register ( "gan_read_ppm_image", GAN_ERROR_CORRUPTED_FILE,
                         "corrupted PPM file header (no P6)" );
      return NULL;
   }

   result = (Gan_Bool)(result & read_line ( infile, s )); width  = atoi(s);
   result = (Gan_Bool)(result & read_line ( infile, s )); height = atoi(s);
   result = (Gan_Bool)(result & read_line ( infile, s )); /*maxval = atoi(s);*/

   /* check whether any of the header lines was corrupted */
   if ( !result )
   {
      gan_err_flush_trace();
      gan_err_register ( "gan_read_ppm_image", GAN_ERROR_CORRUPTED_FILE,
                         "corrupted PPM file header" );
      return NULL;
   }

   if ( image == NULL )
      image = gan_image_alloc_rgb_uc ( height, width );
   else
      image = gan_image_set_rgb_uc ( image, height, width );

   if ( image == NULL )
   {
      gan_err_register ( "gan_read_ppm_image", GAN_ERROR_FAILURE,
                         "setting up image" );
      return NULL;
   }
   
   /* if the image has zero size then we have finished */
   if ( width == 0 || height == 0 ) return image;

   if ( fread ( image->row_data.rgb.uc[0], sizeof(Gan_RGBPixel_uc),
                height*width, infile ) != (size_t)(height*width) )
   {
      gan_err_flush_trace();
      gan_err_register ( "gan_read_ppm_image", GAN_ERROR_TRUNCATED_FILE,
                         "truncated PPM file" );
      return NULL;
   }

   /* success */
   return image;
}

/**
 * \brief Reads a RGB colour image file in PPM format.
 * \param filename The name of the image file
 * \param image The image structure to read the image data into or \c NULL
 * \return Pointer to image structure, or \c NULL on failure.
 *
 * Reads the PPM image with the in the file \a filename into the given
 * \a image. If \a image is \c NULL a new image is dynamically allocated;
 * otherwise the already allocated \a image structure is reused.
 *
 * \sa gan_write_ppm_image().
 */
Gan_Image *
 gan_read_ppm_image ( const char *filename, Gan_Image *image )
{
   FILE *infile;

   /* attempt to open file */
   infile = fopen ( filename, "rb" );
   if ( infile == NULL )
   {
      gan_err_flush_trace();
      gan_err_register ( "gan_read_ppm_image", GAN_ERROR_OPENING_FILE,
                         filename );
      return NULL;
   }

   return gan_read_ppm_image_stream ( infile, image );
}

/**
 * \brief Writes a RGB colour image to a file stream in PPM format.
 * \param outfile The file stream
 * \param image The image structure to write to the file
 * \return #GAN_TRUE on successful write operation, or #GAN_FALSE on failure.
 *
 * Writes the \a image into the file stream \a outfile in PPM format.
 *
 * \sa gan_read_ppm_image_stream().
 */
Gan_Bool
 gan_write_ppm_image_stream ( FILE *outfile, Gan_Image *image )
{
   gan_err_test_bool ( image->format == GAN_RGB_COLOUR_IMAGE &&
                       image->type == GAN_UCHAR, "gan_write_ppm_image_stream",
                       GAN_ERROR_INCOMPATIBLE, "image format/type" );

   fprintf ( outfile, "P6\n#Constructed by Gandalf\n%d %d\n255\n",
             (int)image->width, (int)image->height );

   /* if the image has zero size then we have finished */
   if ( image->width == 0 || image->height == 0 ) return GAN_TRUE;

   /* write image data to stream */
   if ( image->stride == image->width*gan_image_pixel_size(image->format,
                                                           image->type) )
      /* we can write it in one go */
      fwrite ( image->row_data.rgb.uc[0], sizeof(Gan_RGBPixel_uc),
               image->height*image->width, outfile );
   else
   {
      /* write one line of image data at a time */
      unsigned long r;

      for ( r=0; r < image->height; r++ )
         fwrite ( image->row_data.rgb.uc[r], sizeof(Gan_RGBPixel_uc),
                  image->width, outfile );
   }

   /* success */
   return GAN_TRUE;
}

/**
 * \brief Writes a RGB colour image file in PPM format.
 * \param filename The name of the image file
 * \param image The image structure to write to the file
 * \return #GAN_TRUE on successful write operation, #GAN_FALSE on failure.
 *
 * Writes the \a image into PPM file \a filename.
 *
 * \sa gan_read_ppm_image().
 */
Gan_Bool
 gan_write_ppm_image ( const char *filename, Gan_Image *image )
{
   FILE *outfile;
   Gan_Bool result;

   /* attempt to open file */
   outfile = fopen ( filename, "wb" );
   if ( outfile == NULL )
   {
      gan_err_flush_trace();
      gan_err_register ( "gan_write_ppm_image", GAN_ERROR_OPENING_FILE,
                         filename );
      return GAN_FALSE;
   }

   result = gan_write_ppm_image_stream ( outfile, image );
   fclose(outfile);
   return result;
}

/**
 * \}
 */

/**
 * \}
 */
