# vim:sw=4:et
#
# Weak Method, from Python Cookbook, pg 191.
# Credit: Joseph A. Knapka

import weakref

class _weak_callable:
    def __init__(self, obj, func):
        self.im_self = obj
        self.im_func = func

    def __call__(self, *args, **kwargs):
        if self.im_self is None:
            return self.im_func(*args, **kwargs)
        else:
            return self.im_func(self.im_self, *args, **kwargs)


class WeakMethod:
    """Wraps a function or, more importantly, a bound method in a way
    that it allows the bound method's object to be GC'ed, while providing
    the same interface as a normal weak reference.
    """

    def __init__(self, fn):
        try:
            self._obj = weakref.ref(fn.im_self)
            self._meth = fn.im_func
        except AttributeError:
            # It's not a bound method
            self._obj = None
            self._meth = fn

    def __call__(self):
        # object is dead?
        o = self._obj
        if o is not None and o() is None:
            return None
        return _weak_callable(o and o(), self._meth)

if __name__ == '__main__':
    class C:
        def f(self): print 'Hello'
        def __del__(self): print 'C dying'

    c = C()
    cf = c.f
    del c # c is not dead yet
    del cf
    # "C dying"
    c = C()
    cf = WeakMethod(c.f)
    cf()()
    # "Hello"
    del c
    # "C dying"
    print cf()
    # "None"
    assert cf() is None
