/*
Copyright (C) 2003 by Sean David Fleming

sean@power.curtin.edu.au

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <math.h>
#include <gdk/gdk.h>
#include <gtk/gtkgl.h>
#include <GL/gl.h>
#include <GL/glu.h>

#include "gdis.h"
#include "coords.h"
#include "edit.h"
#include "graph.h"
#include "matrix.h"
#include "molsurf.h"
#include "morph.h"
#include "spatial.h"
#include "opengl.h"
#include "render.h"
#include "select.h"
#include "surface.h"
#include "numeric.h"
#include "measure.h"
#include "interface.h"
#include "gl_varray.h"

/* externals */
extern struct sysenv_pak sysenv;
extern struct elem_pak elements[];

#define DRAW_PICTURE 0

/* transformation/projection matrices */
GLint viewport[4];
GLdouble mvmatrix[16], projmatrix[16];
gdouble gl_acm[9];
gdouble halo_fade[16];
gint halo_segments=16;
gpointer gl_font;
gint gl_fontsize=10;
gint font_offset=-1;

/* all rendering parameters are handled by the same dialog */
/* as serves the povray rendering */

/*******************************************/
/* get the canvas associated with a widget */
/*******************************************/
struct canvas_pak *find_canvas(GtkWidget *w)
{
GSList *list;
struct canvas_pak *canvas;

for (list=sysenv.canvas_list ; list ; list=g_slist_next(list))
  {
  canvas = (struct canvas_pak *) list->data;
  if (canvas->glarea == w)
    return(canvas);
  }
return(NULL);
}

/*******************/
/* configure event */
/*******************/
#define DEBUG_GL_CONFIG_EVENT 0
gint gl_configure_event(GtkWidget *w, GdkEventConfigure *event)
{
gint i, x=0, y=0, size;
static gint n=0;
GdkGLContext *glcontext;
GdkGLDrawable *gldrawable;
struct canvas_pak *canvas;
struct model_pak *data;

n++;
#if DEBUG_GL_CONFIG_EVENT
printf("gl_configure() : %d\n", n);
#endif

/*
printf ("wxh = %dx%d\n", sysenv.display_box->allocation.width,
                         sysenv.display_box->allocation.height);
*/

/* find canvas associated with this glarea */
canvas = find_canvas(w);
g_assert(canvas != NULL);

/* NEW - allow a rectangular (non-distorted) canvas */
x = 0;
y = 0;
if (w->allocation.width > w->allocation.height)
  {
/* too wide */
  size = w->allocation.height;
  }
else
  {
/* too high (or equal) */
  size = w->allocation.width;
  }

/* store the canvas parameters */
canvas->x = 0;
canvas->y = 0;
canvas->width = w->allocation.width;
canvas->height = w->allocation.height;
canvas->size = size;

/* TODO - can't remove the sysenv stuff (still in use elsewhere) */
/* TODO - convert to using canvas-> stuff instead */
sysenv.x = 0;
sysenv.y = 0;
sysenv.width = w->allocation.width;
sysenv.height = w->allocation.height;
sysenv.size = size;

#if DEBUG_GL_CONFIG_EVENT
printf("Relative canvas origin: (%d,%d)\n",sysenv.x,sysenv.y);
printf("     Canvas dimensions:  %dx%d\n",sysenv.width,sysenv.height);
#endif

/* OpenGL window setup */
glcontext = gtk_widget_get_gl_context(w);
gldrawable = gtk_widget_get_gl_drawable(w);
if (!gdk_gl_drawable_gl_begin(gldrawable, glcontext))
  return(FALSE);

#if DEBUG_GL_CONFIG_EVENT
g_print("\n");
g_print("GL_RENDERER   = %s\n", (gchar *) glGetString(GL_RENDERER));
g_print("GL_VERSION    = %s\n", (gchar *) glGetString(GL_VERSION));
g_print("GL_VENDOR     = %s\n", (gchar *) glGetString(GL_VENDOR));
/* this is large */
/*
g_print("GL_EXTENSIONS = %s\n", (gchar *) glGetString(GL_EXTENSIONS));
*/
g_print("\n");
#endif

/* NEW - allow a rectangular (non-distorted) canvas */
glViewport(0, 0, sysenv.width, sysenv.height);
gdk_gl_drawable_gl_end(gldrawable);

/* update coords */
for (i=0 ; i<sysenv.num_displayed ; i++)
  {
  data = model_ptr(sysenv.displayed[i], RECALL);
  if (data)
    init_objs(REDO_COORDS, data);
  }

/* new screen size to be saved as default */
sysenv.write_gdisrc = TRUE;

return(TRUE);
}

/*****************/
/* expose event */
/****************/
#define DEBUG_GL_EXPOSE 0
gint gl_expose_event(GtkWidget *w, GdkEventExpose *event)
{
static gint n=0;
struct canvas_pak *canvas;

canvas = find_canvas(w);
g_assert(canvas != NULL);

n++;

#if DEBUG_GL_EXPOSE
printf("gl_expose_event(%p) : %d\n", w, n);
printf("canvas: %p, glarea: %p, model: %p\n", canvas, canvas->glarea, canvas->model);
printf("canvas: %d,%d x %d,%d\n", canvas->x, canvas->y, canvas->width, canvas->height);
#endif

gl_draw(canvas, canvas->model);

return(TRUE);
}

/***************************************************/
/* setup the visual for subsequent canvas creation */
/***************************************************/
GdkGLConfig *glconfig;
gint gl_init_visual(void)
{
/* attempt to get best visual */
/* order: stereo, double buffered, depth buffered */
glconfig = gdk_gl_config_new_by_mode(GDK_GL_MODE_RGB |
                                     GDK_GL_MODE_DEPTH |
                                     GDK_GL_MODE_DOUBLE |
                                     GDK_GL_STEREO);
/* windowed stereo possible? */
if (glconfig)
  sysenv.stereo_windowed = TRUE;
else
  {
  glconfig = gdk_gl_config_new_by_mode(GDK_GL_MODE_RGB |
                                       GDK_GL_MODE_DEPTH |
                                       GDK_GL_MODE_DOUBLE);
  sysenv.stereo_windowed = FALSE;
  }
if (!glconfig)
  {
  printf("WARNING: cannot create a double-buffered visual.\n");
  glconfig = gdk_gl_config_new_by_mode(GDK_GL_MODE_RGB | GDK_GL_MODE_DEPTH);
  if (!glconfig)
    {
    printf("ERROR: no appropriate visual could be acquired.\n");
    return(1);
    }
  }
return(0);
}

/*******************************************************/
/* create an OpenGL canvas for packing in a GTK widget */
/*******************************************************/
#define DEBUG_NEW_CANVAS 0
struct canvas_pak *gl_new_canvas(gint width, gint height)
{
struct canvas_pak *canvas;

g_assert(glconfig != NULL);

canvas = g_malloc(sizeof(struct canvas_pak));
sysenv.canvas_list = g_slist_prepend(sysenv.canvas_list, canvas);

#if DEBUG_NEW_CANVAS
printf("creating canvas: %p\n", canvas);
#endif

/* create an OpenGL capable drawing area */
canvas->active = FALSE;
canvas->model = NULL;
canvas->glarea = gtk_drawing_area_new();
gtk_widget_set_gl_capability(canvas->glarea,
                             glconfig, NULL, TRUE, GDK_GL_RGBA_TYPE);
gtk_widget_set_size_request(canvas->glarea, width, height);

/* init signals */
g_signal_connect(GTK_OBJECT(canvas->glarea), "expose_event",
                (GtkSignalFunc) gl_expose_event, NULL);
g_signal_connect(GTK_OBJECT(canvas->glarea), "configure_event",
                (GtkSignalFunc) gl_configure_event, NULL);
g_signal_connect(GTK_OBJECT(canvas->glarea), "motion_notify_event",
                (GtkSignalFunc) motion_notify_event, NULL);
g_signal_connect(GTK_OBJECT(canvas->glarea), "button_press_event",
                (GtkSignalFunc) button_press_event, NULL);
g_signal_connect(GTK_OBJECT(canvas->glarea), "button_release_event",
                (GtkSignalFunc) button_release_event, NULL);

gtk_widget_set_events(GTK_WIDGET(canvas->glarea), GDK_EXPOSURE_MASK
                                                | GDK_LEAVE_NOTIFY_MASK
                                                | GDK_BUTTON_PRESS_MASK
                                                | GDK_BUTTON_RELEASE_MASK
                                                | GDK_POINTER_MOTION_MASK
                                                | GDK_POINTER_MOTION_HINT_MASK);
return(canvas);
}

/***************************/
/* default camera location */
/***************************/
void gl_camera_init(struct model_pak *model)
{
/*
VEC3SET(model->render.cam_x, 0.0, 0.0, -2.0*model->rmax);
VEC3SET(model->render.cam_f, 0.0, 0.0, 0.0);
VEC3SET(model->render.cam_o, 0.0, 0.0, 1.0);
*/
}

/****************************************/
/* setup the camera and projection mode */
/****************************************/
#define DEBUG_INIT_PROJ 0
void gl_init_projection(struct model_pak *data)
{
gdouble r, s, w, h, x, y;
gdouble pix2ang;
gdouble mat[9];
gdouble upv[3] = {0.0, 0.0, 1.0};
gdouble cam[3] = {0.0, 0.0, 0.0};

/* viewing */
glMatrixMode(GL_MODELVIEW);
glLoadIdentity();
/* get the model axes transformation matrix */
init_rotmat(mat);
/* transform camera position & up vector */
vecmat(mat, cam);
vecmat(mat, upv);

/* scaling affects placement to avoid near/far clipping */
r = RMAX_FUDGE * data->rmax;
/* yet another magic number (0.427) - works reasonably well */
pix2ang = 0.427 * (gdouble) sysenv.size / data->rmax;

/* if not periodic - may need some extra help fitting within the window */
if (!data->periodic && data->id != MORPH)
  {
/* add a bit extra due to the sizes of rendered spheres */
  switch (sysenv.render.type)
    {
    case BALL_STICK:
      r += 2.0 * sysenv.render.ball_rad;
      break;
    case CPK:
      r += 2.0 * sysenv.render.cpk_scale;
      break;
    }
  }
sysenv.rsize = r;

#if DEBUG_INIT_PROJ
printf("      size = %d\n", sysenv.size);
printf(" real size = %f\n", sysenv.rsize);
printf("      rmax = %f\n", data->rmax);
printf("     scale = %f\n", data->scale);
printf("    vpdist = %f\n", sysenv.render.vp_dist);
printf("   pix2ang = %f\n", pix2ang);
#endif

/* mouse translation simply shifts the camera */
cam[0] -= data->offset[0] / pix2ang;
cam[1] -= data->offset[1] / pix2ang;
/* move camera back to cope with scale magnification */
cam[2] = -(sysenv.render.vp_dist+data->scale)*r;

gluLookAt(cam[0],cam[1],cam[2],cam[0],cam[1],0.0,upv[0],upv[1],upv[2]);

/* projection */
glMatrixMode(GL_PROJECTION);
glLoadIdentity();

/* scaling affects projection to avoid near/far clipping */
/* NB: near/far clipping - dist from camera? */
/* if so, makes sense to have near clip plane = 0 */
/* NEW - allow rectangular canvas */
w = sysenv.width;
h = sysenv.height;
s = sysenv.size;
x = r * w/s;
y = r * h/s;

if (sysenv.render.perspective)
  {
  glFrustum(-x,x,-y,y,sysenv.render.vp_dist*r,
            (sysenv.render.vp_dist+2.0*data->scale)*r);
  }
else
  {
  glOrtho(-x,x,-y,y,0.0,(sysenv.render.vp_dist+2.0*data->scale)*r);
  }

/* store matrices for proj/unproj operations */
glGetIntegerv(GL_VIEWPORT, viewport);
glGetDoublev(GL_MODELVIEW_MATRIX, mvmatrix);
glGetDoublev(GL_PROJECTION_MATRIX, projmatrix);

/* NEW - opengl -> gdis coordinate conversion */
VEC3SET(&gl_acm[0], -1.0,  0.0,  0.0);
VEC3SET(&gl_acm[3],  0.0,  0.0,  1.0);
VEC3SET(&gl_acm[6],  0.0, -1.0,  0.0);
}

/***************************/
/* setup all light sources */
/***************************/
void gl_init_lights(struct model_pak *data)
{
gint i;
gfloat light[4];
gdouble x, tmp[4];
GSList *list;
struct light_pak *ldata;

/* go through all OpenGL lights (enable/disable as required) */
list = sysenv.render.light_list;
for (i=GL_LIGHT0 ; i<=GL_LIGHT7 ; i++)
  {
/* do we have an active light */
  if (list)
    {
    glEnable(i);
/* get light data */
    ldata = (struct light_pak *) list->data;
/* position/direction */
    ARR3SET(light, ldata->x);

ARR3SET(tmp, light);
vecmat(gl_acm, tmp);
ARR3SET(light, tmp);

    switch (ldata->type)
      {
      case DIRECTIONAL:
        light[3] = 0.0;
        break;
      case POSITIONAL:
      default:
        VEC3MUL(light, -1.0);
        light[3] = 1.0;
      }
    glLightfv(i, GL_POSITION, light);
/* light properties */
    ARR3SET(light, ldata->colour);
    VEC3MUL(light, ldata->ambient);
    glLightfv(i, GL_AMBIENT, light);

    ARR3SET(light, ldata->colour);
    VEC3MUL(light, ldata->diffuse);
    glLightfv(i, GL_DIFFUSE, light);

    ARR3SET(light, ldata->colour);
    VEC3MUL(light, ldata->specular);
    glLightfv(i, GL_SPECULAR, light);
/* next */
    list = g_slist_next(list);
    }
  else
    glDisable(i);
  }

/* halo diminishing function */
for (i=0 ; i<halo_segments ; i++)
  {
  x = (gdouble) i / (gdouble) halo_segments;
  x *= x;
  halo_fade[i] = exp(-5.0 * x);
  }
}

/*********************************************/
/* window to real space conversion primitive */
/*********************************************/
void gl_get_world_coords(gint x, gint y, gdouble *w)
{
gint ry;
GLdouble r[3];

ry = sysenv.height - y - 1;
/* place close to the near clipping plane (z=0.0) so object remains on top */
/* FIMXE - may get clipping problems for objects that aren't flat (eg axes) */
gluUnProject(x, ry, 0.1, mvmatrix, projmatrix, viewport, &r[0], &r[1], &r[2]);

ARR3SET(w, r);
}

/*********************************************/
/* real space to window conversion primitive */
/*********************************************/
void gl_get_window_coords(gdouble *w, gint *x)
{
GLdouble r[3];

gluProject(w[0], w[1], w[2], mvmatrix, projmatrix, viewport, &r[0], &r[1], &r[2]);

x[0] = r[0];
x[1] = sysenv.height - r[1] - 1;
}

/*******************************************/
/* set opengl RGB colour for gdis RGB data */
/*******************************************/
void set_gl_colour(gint *rgb)
{
gdouble col[3];

col[0] = (gdouble) *rgb;
col[1] = (gdouble) *(rgb+1);
col[2] = (gdouble) *(rgb+2);
VEC3MUL(col, 1.0/65535.0);
glColor4f(col[0], col[1], col[2], 1.0);
}

/*************************************************************/
/* adjust fg colour for visibility against current bg colour */
/*************************************************************/
void make_fg_visible(void)
{
gdouble fg[3], bg[3];

#define F_COLOUR_SCALE 65535.0
ARR3SET(fg, sysenv.render.fg_colour);
ARR3SET(bg, sysenv.render.bg_colour);
VEC3MUL(fg, F_COLOUR_SCALE);
VEC3MUL(bg, F_COLOUR_SCALE);
/* XOR to get a visible colour */
fg[0] = (gint) bg[0] ^ (gint) F_COLOUR_SCALE;
fg[1] = (gint) bg[1] ^ (gint) F_COLOUR_SCALE;
fg[2] = (gint) bg[2] ^ (gint) F_COLOUR_SCALE;
VEC3MUL(fg, 1.0/F_COLOUR_SCALE);
ARR3SET(sysenv.render.fg_colour, fg);

/* adjust label colour for visibility against the current background */
ARR3SET(fg, sysenv.render.label_colour);
VEC3MUL(fg, F_COLOUR_SCALE);
/* XOR to get a visible colour */
fg[0] = (gint) bg[0] ^ (gint) F_COLOUR_SCALE;
fg[1] = (gint) bg[1] ^ (gint) F_COLOUR_SCALE;
VEC3MUL(fg, 1.0/F_COLOUR_SCALE);
/* force to zero, so we get yellow (not white) for a black background */
fg[2] = 0.0;
ARR3SET(sysenv.render.label_colour, fg);

/* adjust title colour for visibility against the current background */
/*
ARR3SET(fg, sysenv.render.title_colour);
VEC3MUL(fg, F_COLOUR_SCALE);
*/
/* force to zero, so we get cyan (not white) for a black background */
fg[0] = 0.0;
/* XOR to get a visible colour */
fg[0] = (gint) bg[0] ^ (gint) F_COLOUR_SCALE;
fg[1] = (gint) bg[1] ^ (gint) F_COLOUR_SCALE;
fg[2] = (gint) bg[2] ^ (gint) F_COLOUR_SCALE;

/* faded dark blue */
fg[0] *= 0.3;
fg[1] *= 0.65;
fg[2] *= 0.85;

/* apricot */
/*
fg[0] *= 0.9;
fg[1] *= 0.7;
fg[2] *= 0.4;
*/

VEC3MUL(fg, 1.0/F_COLOUR_SCALE);
ARR3SET(sysenv.render.title_colour, fg);

/*
printf("fg: %lf %lf %lf\n",fg[0],fg[1],fg[2]);
*/
}

/*******************************/
/* return approx. string width */
/*******************************/
gint gl_text_width(gchar *str)
{
return(strlen(str) * gl_fontsize);
}

/******************************/
/* print at a window position */
/******************************/
void gl_print_win(gint x, gint y, gchar *str)
{
gdouble w[3];

/* the use of 3 coords allows us to put text above everything else */
gl_get_world_coords(x, y, w);
glRasterPos3f(w[0], w[1], w[2]); 

glListBase(font_offset);
glCallLists(strlen(str), GL_UNSIGNED_BYTE, str);
}

/*****************************/
/* print at a world position */
/*****************************/
void gl_print(gdouble x, gdouble y, gdouble z, gchar *str)
{
/* set the raster position & draw the text */
glRasterPos3f(x,y,z); 

glListBase(font_offset);
glCallLists(strlen(str), GL_UNSIGNED_BYTE, str);
}

/********************************/
/* vertex at 2D screen position */
/********************************/
void gl_vertex_win(gint x, gint y)
{
gdouble w[3];

gl_get_world_coords(x, y, w);
glVertex3dv(w);
}

/*********************************/
/* draw a box at screen position */
/*********************************/
void gl_draw_box(gint px1, gint py1, gint px2, gint py2)
{
glBegin(GL_LINE_LOOP);
gl_vertex_win(px1, py1);
gl_vertex_win(px1, py2);
gl_vertex_win(px2, py2);
gl_vertex_win(px2, py1);
glEnd();
}

/********************************/
/* OpenGL atom location routine */
/********************************/
#define DEBUG_GL_SEEK_BOND 0
GSList *gl_seek_bond(GtkWidget *w, gint x, gint y, struct model_pak *data)
{
gdouble r[3];
gdouble dx, dy, d2;
gdouble tol;
GSList *list, *match=NULL;
GdkGLContext *glcontext;
GdkGLDrawable *gldrawable;
struct bond_pak *bdata;
struct core_pak *core1, *core2;

/* is there an OpenGL canvas? */
glcontext = gtk_widget_get_gl_context(w);
gldrawable = gtk_widget_get_gl_drawable(w);
if (!gdk_gl_drawable_gl_begin(gldrawable, glcontext))
  return(NULL);

gl_get_world_coords(x, y, r);

/* this is probably close to as good as we can do, given */
/* all the different display modes (eg b&s, cpk, stick) */
tol = data->scale / data->rmax;
list = data->bonds;
while (list != NULL)
  {
  bdata = (struct bond_pak *) list->data; 
  core1 = bdata->atom1;
  core2 = bdata->atom2;

  dx = 0.5*data->scale*(core1->rx[0] + core2->rx[0]) - r[0];
  dy = 0.5*data->scale*(core1->rx[1] + core2->rx[1]) - r[1];

  d2 = dx*dx + dy*dy;
  if (d2 < tol)
    {
/* keep searching - return the best match */
    tol = d2;
    match = list;
    }
  list = g_slist_next(list);
  }

return(match);
}

/***************************************************/
/* compute atoms that lie within the selection box */
/***************************************************/
#define DEBUG_GL_SELECT_BOX 0
void gl_select_box(GtkWidget *w)
{
gint i, tmp, x[2];
gdouble wbox1[3], wbox2[3], r[3], piv[3];
GSList *list, *ilist=NULL;
GdkGLContext *glcontext;
GdkGLDrawable *gldrawable;
struct model_pak *data;
struct core_pak *core;
struct image_pak *image;

/* is there an OpenGL canvas? */
glcontext = gtk_widget_get_gl_context(w);
gldrawable = gtk_widget_get_gl_drawable(w);
if (!gdk_gl_drawable_gl_begin(gldrawable, glcontext))
  return;
gdk_gl_drawable_gl_end(gldrawable);

/* valid model */
data = sysenv.active_model;
if (!data)
  return;
if (data->graph_active)
  return;
if (data->picture_active)
  return;

/* ensure box limits have the correct order (ie low to high) */
for (i=0 ; i<2 ; i++)
  {
  if (data->select_box[i] > data->select_box[i+2])
    {
    tmp = data->select_box[i];
    data->select_box[i] = data->select_box[i+2];
    data->select_box[i+2] = tmp;
    }
  }

/* very small box is same as a single click */
if (abs(data->select_box[0] - data->select_box[2]) < 5)
  {
  if (abs(data->select_box[1] - data->select_box[3]) < 5)
    {
/* FIXME - try and cope with perspective projection */
    if (sysenv.render.perspective)
      {
      core = seek_coord2d(data->select_box[0], data->select_box[1], data);
      }
    else
      {
      pixel_coord_map(data->select_box[0], data->select_box[1], r, data);
      core = seek_coord3d(r, data);
      }

    if (core)
      {
      switch (sysenv.select_mode)
        {
        case ATOM_LABEL:
          select_all_labels(core, data);
          break;
        case CORE:
          select_add_core(core, data);
          break;
        case ELEM:
          select_all_elem(core->atom_code, data);
          break;
        case ELEM_MOL:
          select_all_elem_in_mol(core, data);
          break;
        case MOL:
          select_add_mol(core->mol, data);
          break;
        case FRAGMENT:
          select_add_fragment(core, data);
          break;
        case REGION:
          select_add_region(core, data);
          break;
        }
      }
    return;
    }
  }

if (sysenv.render.perspective)
  {
/* compare and select if... */
  for (list=data->cores ; list ; list=g_slist_next(list))
    {
    core = (struct core_pak *) list->data;
    if (core->status & DELETED)
      continue;

    ARR3SET(r, core->rx);
    VEC3MUL(r, data->scale);

    gl_get_window_coords(r, x);

/* within box */
    if (x[0] > data->select_box[0] && x[0] < data->select_box[2])
      {
      if (x[1] > data->select_box[1] && x[1] < data->select_box[3])
        {
        switch(sysenv.select_mode)
          {
          case CORE:
            select_add_core(core, data);
            break;
          case MOL:
            select_add_mol(core->mol, data);
            break;
          }
        }
      }
    }
  }
else
  {
/* get real coords for selection box */
  gl_get_world_coords(data->select_box[0], data->select_box[1], wbox1);
  gl_get_world_coords(data->select_box[2], data->select_box[3], wbox2);

/* account for model scaling */
  wbox1[0] /= data->scale;
  wbox1[1] /= data->scale;
  wbox2[0] /= data->scale;
  wbox2[1] /= data->scale;

/* NEW - cope with perioidic images */
  do
    {
    if (ilist)
      {
/* retrieve periodic image vector */
      image = (struct image_pak *) ilist->data;
      ARR3SET(piv, image->rx);
      ilist = g_slist_next(ilist);
      }
    else
      {
      VEC3SET(piv, 0.0, 0.0, 0.0);
      ilist = data->images;
      }

/* compare and select if... */
    for (list=data->cores ; list ; list=g_slist_next(list))
      {
      core = (struct core_pak *) list->data;
      if (core->status & DELETED)
        continue;

     r[0] = core->rx[0] + piv[0]; 
     r[1] = core->rx[1] + piv[1]; 

/* within box */
      if (r[0] > wbox1[0] && r[0] < wbox2[0])
        {
        if (r[1] > wbox1[1] && r[1] < wbox2[1])
          {
          switch(sysenv.select_mode)
            {
            case CORE:
              select_add_core(core, data);
              break;
            case MOL:
              select_add_mol(core->mol, data);
              break;
            }
          }
        }
      }
    }
  while (ilist);
  }

/* done */
redraw_canvas(SINGLE);
}

/************************************/
/* compute sphere radius for a core */
/************************************/
gdouble gl_get_radius(struct core_pak *core, struct model_pak *model)
{
gdouble radius;
struct elem_pak elem;

radius = model->scale;

switch (core->render_mode)
  {
  case CPK:
/* TODO - calling get_elem_data() all the time is inefficient */
    get_elem_data(core->atom_code, &elem, model);
    radius *= sysenv.render.cpk_scale * elem.vdw;
    break;

  case LIQUORICE:
/* only one bond - omit as it's a terminating atom */
/* FIXME - this will skip isolated atoms with one periodic bond */
    if (g_slist_length(core->bonds) == 1)
      radius *= -1.0;
/* more than one bond - put in a small sphere to smooth bond joints */
/* no bonds (isolated) - use normal ball radius */
    if (core->bonds)
      radius *= sysenv.render.stick_rad;
    else
      radius *= sysenv.render.ball_rad;
    break;

  case STICK:
    radius *= sysenv.render.stick_rad;
    if (core->bonds)
      radius *= -1.0;
    break;

  case BALL_STICK:
    radius *= sysenv.render.ball_rad;
    break;
  }
return(radius);
}

/*****************************/
/* vertex array atom drawing */
/*****************************/
void gl_draw_solid_atoms(struct model_pak *model)
{
gdouble radius, x[3], colour[4];
gpointer va;
GSList *list;
struct core_pak *core;

va = va_init();
va_make_sphere(va);

for (list=model->cores ; list ; list=g_slist_next(list))
  {
  core = (struct core_pak *) list->data;

  if (core->status & (HIDDEN | DELETED))
    continue;
  if (core->render_wire)
    continue;
  if (core->ghost)
    continue;

/* set colour */
  ARR3SET(colour, core->colour);
  VEC3MUL(colour, 1.0/65535.0);
  colour[3] = core->colour[3];
  glColor4dv(colour);

  radius = gl_get_radius(core, model);
  if (radius > 0.0)
    {
    ARR3SET(x, core->rx);
    VEC3MUL(x, model->scale);
    va_draw_sphere(va, x, radius);
    }
  }
va_free(va);
}

/**************************/
/* atom drawing primitive */
/**************************/
void gl_draw_atom_list(gint mode, GSList *list, struct model_pak *data)
{
gint omit_atom;
gdouble radius;
gdouble vec[3], colour[4];
GSList *item;
struct point_pak sphere;
struct core_pak *core;
struct elem_pak elem;

gl_init_sphere(&sphere);

for (item=list ; item ; item=g_slist_next(item))
  {
  core = (struct core_pak *) item->data;

/* set colour */
  ARR3SET(colour, core->colour);
  VEC3MUL(colour, 1.0/65535.0);
  colour[3] = core->colour[3];
  glColor4dv(colour);

/* set radius */
  omit_atom = FALSE;
  radius = data->scale;
  switch (mode)
    {
    case CPK:
/* FIXME - calling get_elem_data() all the time is inefficient */
      get_elem_data(core->atom_code, &elem, data);
      radius *= sysenv.render.cpk_scale * elem.vdw;
      break;

    case LIQUORICE:
/* only one bond - omit as it's a terminating atom */
/* FIXME - this will skip isolated atoms with one periodic bond */
      if (g_slist_length(core->bonds) == 1)
        omit_atom = TRUE;
/* more than one bond - put in a small sphere to smooth bond joints */
/* no bonds (isolated) - use normal ball radius */
      if (core->bonds)
        radius *= sysenv.render.stick_rad;
      else
        radius *= sysenv.render.ball_rad;
      break;

    case STICK:
      radius *= sysenv.render.stick_rad;
      if (core->bonds)
        omit_atom = TRUE;
      break;

    case BALL_STICK:
      radius *= sysenv.render.ball_rad;
      break;
    }

/* skip? */
   if (omit_atom)
     continue;

#define EXPERIMENTAL 1
/* TODO - implement fractional image_limits */
/* possibility is draw extra whole unit (ie plus bonds) then use clipping planes to trim */
#if EXPERIMENTAL
{
gint i, j, a, b, c, flag, limit[6], test[6];
gdouble t[4], frac[6], whole;
struct mol_pak *mol;

/* set up integer limits */
for (i=6 ; i-- ; )
  {
  frac[i] = modf(data->image_limit[i], &whole);
  limit[i] = (gint) whole;
/* if we have a fractional part - extend the periodic image boundary */
  if (frac[i] > FRACTION_TOLERANCE)
    {
/*
printf("i = %d, frac = %f\n", i, frac[i]);
*/
    test[i] = TRUE;
    limit[i]++;
    }
  else
    test[i] = FALSE;
  }

limit[0] *= -1;
limit[2] *= -1;
limit[4] *= -1;

/* setup for pic iteration */
a = limit[0];
b = limit[2];
c = limit[4];

for (;;)
  {
/* image increment */
  if (a == limit[1])
    {
    a = limit[0];
    b++;
    if (b == limit[3])
      {
      b = limit[2];
      c++;
      if (c == limit[5])
        break;
      }
    }

  VEC3SET(t, a, b, c);

/* NEW - include fractional cell images */
/* TODO - include the testing as part of image increment testing? (ie above) */
flag = TRUE;
for (i=0 ; i<data->periodic ; i++)
  {
/* +ve fractional extent test */
  if (test[2*i+1])
  if (t[i] == limit[2*i+1]-1) 
    {
    for (j=0 ; j<data->periodic ; j++)
      if (test[2*j+1])
        {
/* check molecule centroid, rather than atom coords */
        mol = core->mol;

        if (mol->centroid[j] > frac[2*j+1])
          flag = FALSE;
        }
    }

/* -ve fractional extent test */
  if (test[2*i])
  if (t[i] == limit[2*i]) 
    {
/* + ve fractional extent test */
    for (j=0 ; j<data->periodic ; j++)
      if (test[2*j])
        {
/* check molecule centroid, rather than atom coords */
        mol = core->mol;

/* TODO - pre-sub 1.0 from frac for this test */
        if (mol->centroid[j] < (1.0-frac[2*j]))
          flag = FALSE;
        }
    }

  }

if (flag)
  {

  t[3] = 0.0;
  vec4mat(data->display_lattice, t);

  ARR3SET(vec, core->rx);
  ARR3ADD(vec, t);
  VEC3MUL(vec, data->scale);

  gl_draw_sphere(&sphere, vec, radius);
  }

  a++;
  }

}
#else
/* original + image iteration */
  ilist = NULL;
  do
    {
    ARR3SET(vec, core->rx);
    if (ilist)
      {
/* image translation */
      image = (struct image_pak *) ilist->data;
      ARR3ADD(vec, image->rx);
      ilist = g_slist_next(ilist);
      }
    else
      ilist = data->images;

    VEC3MUL(vec, data->scale);

/* draw the atom */
    gl_draw_sphere(&sphere, vec, radius);
    }
  while (ilist);
#endif

  }

gl_free_points(&sphere);
}

/**********************************/
/* draw the selection halo/circle */
/**********************************/
void gl_draw_halo_list(GSList *list, struct model_pak *data)
{
gint h, omit_atom;
gdouble radius, dr;
gdouble vec[3], halo[4];
GSList *item, *ilist;
struct point_pak circle;
struct core_pak *core;
struct image_pak *image;

/* variable quality halo */
h = 2 + sysenv.render.sphere_quality;
h = h*h;
gl_init_circle(&circle, h);

/* halo colour */
VEC4SET(halo, 1.0, 0.95, 0.45, 1.0);
for (item=list ; item ; item=g_slist_next(item))
  {
  core = (struct core_pak *) item->data;

/* set colour */
/*
ARR3SET(halo, core->colour);
VEC3MUL(halo, 1.0/65535.0);
*/
  glColor4dv(halo);

/* set radius */
  omit_atom = FALSE;
  radius = data->scale;
  switch (core->render_mode)
    {
    case CPK:
      radius *= sysenv.render.cpk_scale * elements[core->atom_code].vdw;
      break;

    case LIQUORICE:
/* only one bond - omit as it's a terminating atom */
/* FIXME - this will skip isolated atoms with one periodic bond */
      if (g_slist_length(core->bonds) == 1)
        omit_atom = TRUE;
/* more than one bond - put in a small sphere to smooth bond joints */
/* no bonds (isolated) - use normal ball radius */
      if (core->bonds)
        radius *= sysenv.render.stick_rad;
      else
        radius *= sysenv.render.ball_rad;
      break;

    case STICK:
      radius *= sysenv.render.stick_rad;
      if (core->bonds)
        omit_atom = TRUE;
      break;

    case BALL_STICK:
      radius *= sysenv.render.ball_rad;
      break;
    }

/* halo ring size increment */
/* a fn of the radius? eg 1/2 */
  dr = 0.6*radius/halo_segments;

/* original + image iteration */
  ilist = NULL;
  do
    {
    ARR3SET(vec, core->rx);
    if (ilist)
      {
/* image translation */
      image = (struct image_pak *) ilist->data;
      ARR3ADD(vec, image->rx);
      ilist = g_slist_next(ilist);
      }
    else
      ilist = data->images;

    VEC3MUL(vec, data->scale);

    if (sysenv.render.halos)
      {
/* halo fade loop */
      for (h=0 ; h<halo_segments ; h++)
        {
        halo[3] = halo_fade[h];

        glColor4dv(halo);
        gl_draw_ring(&circle, vec, radius+h*dr-0.5*dr, radius+h*dr+0.5*dr);
        }
/* reset halo transparancy */
      halo[3] = 1.0;
      }
    else
      gl_draw_ring(&circle, vec, 1.2*radius, 1.4*radius);
    }
  while (ilist);
  }

gl_free_points(&circle);
}

/**************************/
/* draw the model's atoms */
/**************************/
void gl_draw_atoms(gint mode, gint wire, struct model_pak *data)
{
struct core_pak *core;
GSList *clist, *list;

/* checks */
g_assert(data != NULL);
if (!data->show_cores)
  return;

/*  construct list of atoms to be drawn & pass those */
list = NULL;
for (clist=data->cores ; clist ; clist=g_slist_next(clist))
  {
  core = (struct core_pak *) clist->data;

  if (core->status & (HIDDEN | DELETED))
    continue;
  if (core->ghost)
    continue;
  if (core->render_mode != mode)
    continue;
  if (core->render_wire != wire)
    continue;

  list = g_slist_prepend(list, core);
  }

gl_draw_atom_list(mode, list, data);

g_slist_free(list);
}

/*****************************************/
/* translucent atom depth buffer sorting */
/*****************************************/
gint gl_depth_sort(struct core_pak *c1, struct core_pak *c2)
{
if (c1->rx[2] > c2->rx[2])
  return(-1);
return(1);
}

/*******************************/
/* draw the model's ghost atom */
/*******************************/
void gl_draw_ghost_atoms(gint mode, struct model_pak *data)
{
struct core_pak *core;
GSList *clist, *list;

/*  construct list of atoms to be drawn & pass those */
list = NULL;
for (clist=data->cores ; clist ; clist=g_slist_next(clist))
  {
  core = (struct core_pak *) clist->data;

  if (core->status & (HIDDEN | DELETED))
    continue;
  if (core->render_mode != mode)
    continue;
  if (core->ghost)
    {
    core->colour[3] = sysenv.render.ghost_opacity;
    list = g_slist_prepend(list, core);
    }
  }

/* NEW - sort for back to front drawing */
list = g_slist_sort(list, (gpointer) gl_depth_sort);

/* perform the appropriate drawing */
gl_draw_atom_list(mode, list, data);

g_slist_free(list);
}

/***********************/
/* draw model's shells */
/***********************/
void gl_draw_shells(struct model_pak *data)
{
gint omit_atom;
gdouble radius;
gdouble vec[3], colour[4];
GSList *list, *ilist;
struct point_pak sphere;
struct shel_pak *shel;
struct image_pak *image;

/* set appropriate radius */
radius = data->scale;
switch (sysenv.render.type)
  {
  case BALL_STICK:
    radius *= sysenv.render.ball_rad;
    break;
  }

gl_init_sphere(&sphere);

for (list=data->shels ; list ; list=g_slist_next(list))
  {
  shel = (struct shel_pak *) list->data;
  if (shel->status & (DELETED | HIDDEN))
    continue;

/* shell colour */
  ARR3SET(colour, shel->colour);
  VEC3MUL(colour, 1.0/65535.0);
/* translucency */
  colour[3] = 0.5;

/* position */
  ilist=NULL;
  do
    {
    ARR3SET(vec, shel->rx);

    if (ilist)
      {
/* image */
      image = (struct image_pak *) ilist->data;

      ARR3ADD(vec, image->rx);

      ilist = g_slist_next(ilist);
      }
    else
      {
      ilist = data->images;
      }

  VEC3MUL(vec, data->scale);

/* set appropriate radius */
  omit_atom = FALSE;
  switch (sysenv.render.type)
    {
    case CPK:
      radius = data->scale * elements[shel->atom_code].vdw;
      break;

    case LIQUORICE:
/*
      if (core->bonds)
        omit_atom = TRUE;
      if (core->bonds)
        radius = data->scale * sysenv.render.stick_rad;
      else
        radius = data->scale * sysenv.render.ball_rad;
*/
      omit_atom = TRUE;

      break;

    case STICK:
      radius = data->scale * sysenv.render.stick_rad;
/*
      if (core->bonds)
        omit_atom = TRUE;
*/
      break;
    }

  if (!omit_atom)
    {
    glColor4dv(colour);
    gl_draw_sphere(&sphere, vec, radius);
    }

    }
  while (ilist);
  }

glEnable(GL_LIGHTING);

gl_free_points(&sphere);
}

/*******************************************/
/* draw model pipes (separate bond halves) */
/*******************************************/
/* stage (drawing stage) ie colour materials/lines/etc. */
/* TODO - only do minimum necessary for each stage */
void gl_draw_pipes(gint line, GSList *pipe_list, struct model_pak *data)
{
guint i;
gdouble v1[3], v2[3];
GSList *list, *ilist;
struct point_pak circle;
struct pipe_pak *pipe;
struct image_pak *image;

/* setup for bond drawing */
if (!line)
  {
  i = data->scale * sysenv.size / (10.0 * data->rmax);
  i++;
  if (i > sysenv.render.cylinder_quality)
    gl_init_circle(&circle, sysenv.render.cylinder_quality);
  else
    gl_init_circle(&circle, i);
  }

/* enumerate the supplied pipes (half bonds) */
for (list=pipe_list ; list ; list=g_slist_next(list))
  {
  pipe = (struct pipe_pak *) list->data;

/* original + image iteration */
  ilist = NULL;
  do
    {
/* original */
    ARR3SET(v1, pipe->v1);
    ARR3SET(v2, pipe->v2);
    if (ilist)
      {
      image = (struct image_pak *) ilist->data;
/* image */
      ARR3ADD(v1, image->rx);
      ARR3ADD(v2, image->rx);
      ilist = g_slist_next(ilist);
      }
    else
      ilist = data->images;

    VEC3MUL(v1, data->scale);
    VEC3MUL(v2, data->scale);

    glColor4dv(pipe->colour);

    if (line)
      {
      glBegin(GL_LINES);
      glVertex3dv(v1);
      glVertex3dv(v2);
      glEnd();
      }
    else
      gl_draw_cylinder(&circle, v1, v2, pipe->radius);

    }
  while (ilist);
  }

/* cleanup after bond drawing */
if (!line)
  gl_free_points(&circle);
}

/***********************/
/* draw all bond types */
/***********************/
void gl_draw_bonds(gint line, gint wire, gint ghost, struct model_pak *model)
{
GSList *list;

list = render_get_pipes(line, wire, ghost, model);
if (list)
  {
  gl_draw_pipes(line, list, model);
  free_slist(list);
  }
}

/***************************/
/* draw crystal morphology */
/***************************/
#define DEBUG_DRAW_MORPH 0
void gl_draw_morph(struct model_pak *data)
{
gdouble v1[3], n[3];
GSList *list1, *list2;
struct plane_pak *plane;
struct vertex_pak *v;

/* checks */
g_assert(data != NULL);

/* turn lighting off for wire frame drawing - looks esp ugly */
/* when hidden (stippled) lines and normal lines are overlayed */
if (0.5*sysenv.render.wire_surface)
  glDisable(GL_LIGHTING);

/* draw hidden lines first */
if (sysenv.render.wire_surface && sysenv.render.wire_show_hidden)
  {
  glEnable(GL_LINE_STIPPLE);
  glLineStipple(1, 0x0303);
  glLineWidth(sysenv.render.frame_thickness);

  for (list1=data->planes ; list1 ; list1=g_slist_next(list1))
    {
    plane = (struct plane_pak *) list1->data;
    if (!plane->present)
      continue;

/* start the face */
    glBegin(GL_POLYGON);
/* surface normal */
    ARR3SET(n, plane->norm);
    vecmat(data->rotmat, n);
    normalize(n, 3);
    glNormal3dv(n);
    for (list2=plane->vertices ; list2 ; list2=g_slist_next(list2))
      {
      v = (struct vertex_pak *) list2->data;

      ARR3SET(v1, v->rx);
      VEC3MUL(v1, data->scale);
      glVertex3dv(v1);
      }
    glEnd();
    }
  }

/* draw the visible facets */
glLineWidth(sysenv.render.frame_thickness);
glDisable(GL_LINE_STIPPLE);
for (list1=data->planes ; list1 ; list1=g_slist_next(list1))
  {
  plane = (struct plane_pak *) list1->data;
  if (!plane->present)
    continue;
  if (!plane->visible)
    continue;

#if DEBUG_DRAW_MORPH
printf("(%f %f %f) : %d vertices\n", plane->m[0], plane->m[1], plane->m[2], g_slist_length(plane->vertices));
#endif

/* start the face */
  glBegin(GL_POLYGON);
/* surface normal */
  ARR3SET(n, plane->norm);
  vecmat(data->rotmat, n);
  normalize(n, 3);
  glNormal3dv(n);
  for (list2=plane->vertices ; list2 ; list2=g_slist_next(list2))
    {
    v = (struct vertex_pak *) list2->data;

#if DEBUG_DRAW_MORPH
P3VEC("  : ", v->rx);
#endif

    ARR3SET(v1, v->rx);
    VEC3MUL(v1, data->scale);
    glVertex3dv(v1);
    }
  glEnd();
  }
/* if wire frame draw - turn lighting back on */
if (sysenv.render.wire_surface)
  glEnable(GL_LIGHTING);
}

/***********************************/
/* draw the cartesian/lattice axes */
/***********************************/
void gl_draw_axes(gint mode, struct canvas_pak *canvas, struct model_pak *data)
{
gint i;
gdouble x1[3], x2[3];
gchar label[3];

/* axes type setup */
if (data->axes_type == CARTESIAN)
  strcpy(label, " x");
else
  strcpy(label, " a");

/* position - upper left corner */
gl_get_world_coords(sysenv.x+40, sysenv.y+40, x1);

if (mode)
  {
/* set colour */
  glColor4f(sysenv.render.fg_colour[0], sysenv.render.fg_colour[1],
            sysenv.render.fg_colour[2], 1.0);
/* draw the axes */
  for (i=3 ; i-- ; )
    {
    ARR3SET(x2, x1);
    ARR3ADD(x2, data->axes[i].rx);

/* vector cylinder width = 1/200th the size of rmax */
    draw_vector(x1, x2, 0.005*data->rmax);
    }
  }
else
  {
/* draw the labels - offset by fontsize? */
  glColor4f(sysenv.render.title_colour[0], sysenv.render.title_colour[1],
            sysenv.render.title_colour[2], 1.0);
  for (i=0 ; i<3 ; i++)
    {
    ARR3SET(x2, x1);
    ARR3ADD(x2, data->axes[i].rx);
    gl_print(x2[0], x2[1], x2[2], label);
    label[1]++;
    }
  }
}

/*******************************************/
/* draw the cell frame for periodic models */
/*******************************************/
void gl_draw_cell(struct model_pak *data)
{
gint i, j;
gdouble v1[3], v2[3], v3[3], v4[3];

/* draw the opposite ends of the frame */
for (i=0 ; i<5 ; i+=4)
  {
  glBegin(GL_LINE_LOOP);
  ARR3SET(v1, data->cell[i+0].rx);
  ARR3SET(v2, data->cell[i+1].rx);
  ARR3SET(v3, data->cell[i+2].rx);
  ARR3SET(v4, data->cell[i+3].rx);
  VEC3MUL(v1, data->scale);
  VEC3MUL(v2, data->scale);
  VEC3MUL(v3, data->scale);
  VEC3MUL(v4, data->scale);
  glVertex3dv(v1);
  glVertex3dv(v2);
  glVertex3dv(v3);
  glVertex3dv(v4);
  glEnd();
  }
/* draw the sides of the frame */
glBegin(GL_LINES);
for (i=4 ; i-- ; )
  {
  j = i+4;
/* retrieve coordinates */
  ARR3SET(v1, data->cell[i].rx);
  ARR3SET(v2, data->cell[j].rx);
/* scale */
  VEC3MUL(v1, data->scale);
  VEC3MUL(v2, data->scale);
/* draw */
  glVertex3dv(v1);
  glVertex3dv(v2);
  }
glEnd();
}

/***************************************/
/* draw the cell frame periodic images */
/***************************************/
void gl_draw_cell_images(struct model_pak *model)
{
gint i, j;
gdouble v1[3], v2[3], v3[3], v4[3];
GSList *ilist;
struct image_pak *image;

/* image iteration (don't do original) */
ilist = model->images;
for (ilist=model->images ; ilist ; ilist=g_slist_next(ilist))
  {
/* image translation */
  image = (struct image_pak *) ilist->data;

/* draw the opposite ends of the frame */
  for (i=0 ; i<5 ; i+=4)
    {
    glBegin(GL_LINE_LOOP);
    ARR3SET(v1, model->cell[i+0].rx);
    ARR3SET(v2, model->cell[i+1].rx);
    ARR3SET(v3, model->cell[i+2].rx);
    ARR3SET(v4, model->cell[i+3].rx);
    ARR3ADD(v1, image->rx);
    ARR3ADD(v2, image->rx);
    ARR3ADD(v3, image->rx);
    ARR3ADD(v4, image->rx);
    VEC3MUL(v1, model->scale);
    VEC3MUL(v2, model->scale);
    VEC3MUL(v3, model->scale);
    VEC3MUL(v4, model->scale);
    glVertex3dv(v1);
    glVertex3dv(v2);
    glVertex3dv(v3);
    glVertex3dv(v4);
    glEnd();
    }
/* draw the sides of the frame */
  glBegin(GL_LINES);
  for (i=4 ; i-- ; )
    {
    j = i+4;
/* retrieve coordinates */
    ARR3SET(v1, model->cell[i].rx);
    ARR3SET(v2, model->cell[j].rx);
    ARR3ADD(v1, image->rx);
    ARR3ADD(v2, image->rx);
/* scale */
    VEC3MUL(v1, model->scale);
    VEC3MUL(v2, model->scale);
/* draw */
    glVertex3dv(v1);
    glVertex3dv(v2);
    }
  glEnd();
  }
}

/*********************/
/* draw measurements */
/*********************/
void gl_draw_measurements(struct model_pak *model)
{
gint type;
gdouble colour[3], a1[3], a2[3], a3[3], a4[3], v1[3], v2[3], v3[3], n[3];
GSList *list;

/* draw the lines */
for (list=model->measure_list ; list ; list=g_slist_next(list))
  {
  type = measure_type_get(list->data);

  measure_colour_get(colour, list->data);
  glColor4f(colour[0], colour[1], colour[2], 1.0);

  switch (type)
    {
    case MEASURE_BOND:
    case MEASURE_DISTANCE:
    case MEASURE_INTER:
    case MEASURE_INTRA:
      measure_coord_get(v1, 0, list->data);
      measure_coord_get(v2, 1, list->data);
      VEC3MUL(v1, model->scale);
      VEC3MUL(v2, model->scale);
      glBegin(GL_LINES);
      glVertex3dv(v1);
      glVertex3dv(v2);
      glEnd();
      break;

    case MEASURE_ANGLE:
      measure_coord_get(v1, 0, list->data);
      measure_coord_get(v2, 1, list->data);
      measure_coord_get(v3, 2, list->data);
      VEC3MUL(v1, model->scale);
      VEC3MUL(v2, model->scale);
      VEC3MUL(v3, model->scale);
/* NB: central atom should be first */
      draw_arc(v2, v1, v3);
      break;

    case MEASURE_TORSION:
/* get constituent core coordinates */
      measure_coord_get(a1, 0, list->data);
      measure_coord_get(a2, 1, list->data);
      measure_coord_get(a3, 2, list->data);
      measure_coord_get(a4, 3, list->data);
/* middle 2 cores define the axis */
      ARR3SET(n, a3);
      ARR3SUB(n, a2);
      normalize(n, 3);
/* arm 1 */
      ARR3SET(v3, a1);
      ARR3SUB(v3, a2);
      proj_vop(v1, v3, n);
      normalize(v1, 3);
/* arm 2 */
      ARR3SET(v3, a4);
      ARR3SUB(v3, a3);
      proj_vop(v2, v3, n);
      normalize(v2, 3);
/* axis centre */
      ARR3SET(v3, a2);
      ARR3ADD(v3, a3);
      VEC3MUL(v3, 0.5);
/* arm endpoints are relative to axis centre */
      ARR3ADD(v1, v3);
      ARR3ADD(v2, v3);
/* scale */
      VEC3MUL(v1, model->scale);
      VEC3MUL(v2, model->scale);
      VEC3MUL(v3, model->scale);
/* draw arc */
      draw_arc(v3, v1, v2);
/* draw lines */
      glBegin(GL_LINE_STRIP);
      glVertex3dv(v1);
      glVertex3dv(v3);
      glVertex3dv(v2);
      glEnd();
      break;

    }
  }
}

/***************************/
/* draw morphology indices */
/***************************/
void gl_draw_miller(struct model_pak *data)
{
gchar *label;
gdouble vec[3];
GSList *plist;
struct plane_pak *plane;

/* draw facet labels */
plist = data->planes;
while (plist != NULL)
  {
  plane = (struct plane_pak *) plist->data;
  if (plane->present && plane->visible)
    {
/* TODO - scale the font with data->scale? (vanishes if too small) */
/* print the hkl label */
    label = g_strdup_printf("%d%d%d", plane->index[0],
                                      plane->index[1],
                                      plane->index[2]);
    ARR3SET(vec, plane->rx);
    VEC3MUL(vec, data->scale);
    gl_print(vec[0], vec[1], vec[2], label);
    g_free(label);

/* TODO - vector font (display list) with number + overbar number */
/*
    glBegin(GL_LINES);
    if (plane->index[0] < 0)
      {
      glVertex3d(plane->rx, plane->ry-gl_fontsize, plane->rz);
      glVertex3d(plane->rx+gl_fontsize, plane->ry-gl_fontsize, plane->rz);
      }
    if (plane->index[1] < 0)
      {
      glVertex3d(plane->rx+1*gl_fontsize, plane->ry-gl_fontsize, plane->rz);
      glVertex3d(plane->rx+2*gl_fontsize, plane->ry-gl_fontsize, plane->rz);
      }
    if (plane->index[2] < 0)
      {
      glVertex3d(plane->rx+2*gl_fontsize, plane->ry-gl_fontsize, plane->rz);
      glVertex3d(plane->rx+3*gl_fontsize, plane->ry-gl_fontsize, plane->rz);
      }
    glEnd();
*/

    }
  plist = g_slist_next(plist);
  }
}

/************************************************/
/* draw the colour scale for molecular surfaces */
/************************************************/
void gl_draw_colour_scale(gint x, gint y, struct model_pak *data)
{
gint i, n;
gdouble z1, dz;
gdouble colour[3], w1[3], w2[3];
GString *text;

/* init */
n = data->gulp.epot_divisions;
z1 = data->gulp.epot_max;
dz = (data->gulp.epot_max - data->gulp.epot_min)/ (gdouble) (n-1);

/* colour boxes */
glPolygonMode(GL_FRONT, GL_FILL);
glBegin(GL_QUADS);
for (i=0 ; i<n ; i++)
  {
  switch (data->ms_colour_method)
    {
    case MS_HIRSHFELD:
      ms_hfs_colour(colour, z1, data);
      break;

    default:
      ms_epot_colour(colour, z1, data);
    }

  glColor3f(colour[0], colour[1], colour[2]);
  z1 -= dz;

  gl_get_world_coords(x, y+i*20, w1);
  gl_get_world_coords(x, y+19+i*20, w2);
  glVertex3dv(w1);
  glVertex3dv(w2);

  gl_get_world_coords(x+19, y+19+i*20, w1);
  gl_get_world_coords(x+19, y+i*20, w2);
  glVertex3dv(w1);
  glVertex3dv(w2);
  }
glEnd();

/* init */
text = g_string_new(NULL);
z1 = data->gulp.epot_max;
glColor3f(1.0, 1.0, 1.0);

/* box labels */
for (i=0 ; i<n ; i++)
  {
  g_string_sprintf(text, "%6.2f", z1);
  gl_print_win(x+30, y+i*20+18, text->str);
  z1 -= dz;
  }
g_string_free(text, TRUE);
}

/**************************************/
/* draw text we wish to be unobscured */
/**************************************/
void gl_draw_text(struct model_pak *data)
{
gint i, j, nv=0, type;
gchar *text;
gdouble q, v1[3], v2[3], v3[3];
GSList *list, *item;
GString *label;
struct vec_pak *p1, *p2;
struct spatial_pak *spatial;
struct core_pak *core[4];
struct shel_pak *shell;

/* print mode */
text = get_mode_label(data);
gl_print_win(sysenv.x+sysenv.width-gl_text_width(text), sysenv.height - 20, text);
g_free(text);

/* NEW _ replaced by the model properties pane */
/* print some useful info */
/*
if (data->show_title)
  {
  if (data->title)
    {
    text = g_strdup(data->title);
    gl_print_win(sysenv.x+sysenv.width-gl_text_width(text), sysenv.y+40, text);
    g_free(text);
    }
  }
*/

/* print current frame */
if (data->show_frame_number)
  {
  if (data->animation)
    {
    text = g_strdup_printf("Frame [%d:%d]", data->cur_frame, data->num_frames-1);
    gl_print_win(sysenv.x+sysenv.width-gl_text_width(text), sysenv.y + 80, text);
    g_free(text);
    }
  }

/* print energy */
if (data->show_energy)
  {
/* Eatt & Esurf are now the only energies printed here */
/* NB: the total energy should be put in the title string */
  if (data->periodic == 2)
    {
    if (data->gulp.eatt[1] == 0.0)
      text = g_strdup_printf("Ea = %f %s", data->gulp.eatt[0],
                                           data->gulp.eatt_units);
    else
      text = g_strdup_printf("Ea = %f %s", data->gulp.eatt[1],
                                           data->gulp.eatt_units);
/* print Eatt */
    gl_print_win(sysenv.x, sysenv.height - 40, text);
    g_free(text);

/* get Esurf */
    if (data->gulp.esurf[1] == 0.0)
      text = g_strdup_printf("Es = %f %s",data->gulp.esurf[0],
                                          data->gulp.esurf_units);
    else
      text = g_strdup_printf("Es = %f %s",data->gulp.esurf[1],
                                          data->gulp.esurf_units);

/* print Elatt/Esurf */
    gl_print_win(sysenv.x, sysenv.height - 20, text);
    g_free(text);
    }
  }

/* hkl labels */
if (data->num_vertices && data->morph_label)
  gl_draw_miller(data);

/* unit cell lengths */
if (data->show_cell_lengths)
  {
  j=2;
  for (i=0 ; i<data->periodic ; i++)
    {
    j += pow(-1, i) * (i+1);
    text = g_strdup_printf("%5.2f", data->pbc[i]);
    ARR3SET(v1, data->cell[0].rx);
    ARR3ADD(v1, data->cell[j].rx);
    VEC3MUL(v1, 0.5*data->scale);
    gl_print(v1[0], v1[1], v1[2], text);
    g_free(text);
    }
  }

/* epot scale */
if (data->ms_colour_scale)
  gl_draw_colour_scale(sysenv.x+1, 80, data);

/* the following text is likely to have partial */
/* overlapping, so XOR text fragments for clearer display */
glEnable(GL_COLOR_LOGIC_OP);
glLogicOp(GL_XOR);
glColor4f(1.0, 1.0, 1.0, 1.0);

/* TODO - all atom related printing -> construct a string */
label = g_string_new(NULL);
/* loop over atoms */
for (list=data->cores ; list ; list=g_slist_next(list))
  {
  core[0] = (struct core_pak *) list->data;
  if (core[0]->status & (DELETED | HIDDEN))
    continue;

  label = g_string_assign(label, "");

/* setup atom labels */
  if (data->show_atom_labels)
    {
    g_string_sprintfa(label, "(%s)", core[0]->label);
    }
  if (data->show_atom_types)
    {
    if (core[0]->atom_type)
      {
      g_string_sprintfa(label, "(%s)", core[0]->atom_type);
      }
    else
      {
      g_string_sprintfa(label, "(?)");
      }
    }

/* get atom charge, add shell charge (if any) to get net result */
  if (data->show_atom_charges)
    {
    q = core[0]->charge;
    if (core[0]->shell)
      {
      shell = (struct shel_pak *) core[0]->shell;
      q += shell->charge;
      }
    g_string_sprintfa(label, "{%6.3f}", q);
    }

/* print */
  if (label->str)
    {
    ARR3SET(v1, core[0]->rx);
    VEC3MUL(v1, data->scale);
    gl_print(v1[0], v1[1], v1[2], label->str);
    }
  }
g_string_free(label, TRUE);

/* print net charge */
if (data->show_charge)
  {
  text = g_strdup_printf("Q = %6.3f", data->gulp.qsum);
  gl_print_win(sysenv.x, sysenv.height - 60, text);
  g_free(text);
  }

/* geom measurement labels */
if (data->show_geom_labels)
for (list=data->measure_list ; list ; list=g_slist_next(list))
  {
  type = measure_type_get(list->data);
  switch(type)
    {
    case MEASURE_BOND:
    case MEASURE_DISTANCE:
    case MEASURE_INTER:
    case MEASURE_INTRA:
      measure_coord_get(v1, 0, list->data);
      measure_coord_get(v2, 1, list->data);
      ARR3ADD(v1, v2);
      VEC3MUL(v1, 0.5);
      VEC3MUL(v1, data->scale);
      gl_print(v1[0], v1[1], v1[2], measure_value_get(list->data));
      break;

    case MEASURE_ANGLE:
/* angle is i-j-k */
      measure_coord_get(v1, 0, list->data);
      measure_coord_get(v2, 1, list->data);
      measure_coord_get(v3, 2, list->data);
      VEC3MUL(v1, data->scale);
      VEC3MUL(v2, data->scale);
      VEC3MUL(v3, data->scale);
/* angle label */
/* FIXME - should use a similar process to the draw_arc code to */
/* determine which arm is shorter & use that to determine label position */
      ARR3ADD(v1, v2);
      ARR3ADD(v1, v3);
      VEC3MUL(v1, 0.3333);
      gl_print(v1[0], v1[1], v1[2], measure_value_get(list->data));
      break;
    }
  }

/* spatial object labels */
list = data->spatial;
while (list)
  {
  spatial = (struct spatial_pak *) list->data;
  switch(spatial->type)
    {
    case SPATIAL_VECTOR:
/* FIXME - give this it's own label on/off flag */
      if (!data->show_atom_labels)
        break;
      item = (GSList *) spatial->data;
      p1 = (struct vec_pak *) item->data;
      item = g_slist_next(item);
      p2 = (struct vec_pak *) item->data;
      ARR3SET(v1, p1->rx);
      ARR3SET(v2, p2->rx);
      VEC3MUL(v1, data->scale);
      VEC3MUL(v2, data->scale);
      VEC3MUL(v1, 0.5);
      ARR3ADD(v1, v2);
      VEC3MUL(v1, 0.67);
      text = g_strdup_printf("%d", nv);
      gl_print(v1[0], v1[1], v1[2], text);
      g_free(text);
      nv++;
      break;
    }
  list = g_slist_next(list);
  }

glDisable(GL_COLOR_LOGIC_OP);
}

/********************************/
/* draw a ribbon special object */
/********************************/
void gl_draw_ribbon(struct object_pak *odata, struct model_pak *data)
{
gint i;
gdouble len, vec1[3], vec2[3];
GSList *rlist;
struct ribbon_pak *ribbon;
GLfloat ctrl[8][3];

/* go through the ribbon segment list */
rlist = (GSList *) odata->data;
while (rlist != NULL)
  {
  ribbon = (struct ribbon_pak *) rlist->data;

  glColor4f(ribbon->colour[0], ribbon->colour[1], ribbon->colour[2],
                                            sysenv.render.transmit);

/* end points */
  ARR3SET(&ctrl[0][0], ribbon->r1);
  ARR3SET(&ctrl[3][0], ribbon->r2);

/* get distance between ribbon points */
  ARR3SET(vec1, ribbon->x1);
  ARR3SUB(vec1, ribbon->x2);
  len = VEC3MAG(vec1);

/* shape control points */
  ARR3SET(&ctrl[1][0], ribbon->r1);
  ARR3SET(&ctrl[2][0], ribbon->r2);

/* segment length based curvature - controls how flat it is at the cyclic group */
  ARR3SET(vec1, ribbon->o1);
  VEC3MUL(vec1, len*sysenv.render.ribbon_curvature);
  ARR3ADD(&ctrl[1][0], vec1);
  ARR3SET(vec2, ribbon->o2);
  VEC3MUL(vec2, len*sysenv.render.ribbon_curvature);
  ARR3ADD(&ctrl[2][0], vec2);

/* compute offsets for ribbon thickness */
  crossprod(vec1, ribbon->n1, ribbon->o1);
  crossprod(vec2, ribbon->n2, ribbon->o2);
  normalize(vec1, 3);
  normalize(vec2, 3);

/* thickness vectors for the two ribbon endpoints */
  VEC3MUL(vec1, 0.5*sysenv.render.ribbon_thickness);
  VEC3MUL(vec2, 0.5*sysenv.render.ribbon_thickness);

/* ensure these are pointing the same way */
  if (via(vec1, vec2, 3) > PI/2.0)
    {
    VEC3MUL(vec2, -1.0);
    }

/* init the bottom edge control points */
  ARR3SET(&ctrl[4][0], &ctrl[0][0]);
  ARR3SET(&ctrl[5][0], &ctrl[1][0]);
  ARR3SET(&ctrl[6][0], &ctrl[2][0]);
  ARR3SET(&ctrl[7][0], &ctrl[3][0]);
/* lift points to make the top edge */
  ARR3ADD(&ctrl[0][0], vec1);
  ARR3ADD(&ctrl[1][0], vec1);
  ARR3ADD(&ctrl[2][0], vec2);
  ARR3ADD(&ctrl[3][0], vec2);
/* lower points to make the bottom edge */
  ARR3SUB(&ctrl[4][0], vec1);
  ARR3SUB(&ctrl[5][0], vec1);
  ARR3SUB(&ctrl[6][0], vec2);
  ARR3SUB(&ctrl[7][0], vec2);

  for (i=0 ; i<8 ; i++)
    {
    VEC3MUL(&ctrl[i][0], data->scale);
    }
  
/* drawing */
  glMap2f(GL_MAP2_VERTEX_3, 
          0.0, 1.0, 3, 4,
          0.0, 1.0, 3*4, 2,
          &ctrl[0][0]);
  glEnable(GL_MAP2_VERTEX_3);
  glEnable(GL_AUTO_NORMAL);
  glMapGrid2f(sysenv.render.ribbon_quality, 0.0, 1.0, 3, 0.0, 1.0);
  glEvalMesh2(GL_FILL, 0, sysenv.render.ribbon_quality, 0, 3);

  rlist = g_slist_next(rlist);
  }
}

/**************************/
/* spatial object drawing */
/**************************/
/* NB: different setup for vectors/planes - draw in seperate iterations */
void gl_draw_spatial(gint type, struct model_pak *data)
{
gdouble vec1[3], vec2[3];
GSList *list, *list1, *list2, *ilist;
struct vec_pak *p1, *p2, *p3;
struct spatial_pak *spatial;
struct image_pak *image;

list = data->spatial;
while (list)
  {
  spatial = (struct spatial_pak *) list->data;
  if (spatial->type == type)
    {
    switch(type)
      {
/* TODO - this should replace SPATIAL_VECTOR */
      case SPATIAL_VECTORS:
        list1 = (GSList *) spatial->data;
        list2 = g_slist_next(list1);
        while (list1 && list2)
          {
          p1 = (struct vec_pak *) list1->data;
          p2 = (struct vec_pak *) list2->data;
          ARR3SET(vec1, p1->rx);
          ARR3SET(vec2, p2->rx);
          VEC3MUL(vec1, data->scale);
          VEC3MUL(vec2, data->scale);
          draw_vector(vec1, vec2, data->scale*0.04);
          list1 = g_slist_next(list2);
          list2 = g_slist_next(list1);
          }
        break;

      case SPATIAL_VECTOR:
        list2 = (GSList *) spatial->data;
        p1 = (struct vec_pak *) list2->data;
        list2 = g_slist_next(list2);
        p2 = (struct vec_pak *) list2->data;
        ARR3SET(vec1, p1->rx);
        ARR3SET(vec2, p2->rx);
        VEC3MUL(vec1, data->scale);
        VEC3MUL(vec2, data->scale);
        draw_vector(vec1, vec2, data->scale*0.04);
        break;

      case SPATIAL_PLANE:
        list2 = (GSList *) spatial->data;
        glBegin(GL_POLYGON);
        while (list2)
          {
          p1 = (struct vec_pak *) list2->data;
          ARR3SET(vec1, p1->rx);
          VEC3MUL(vec1, data->scale);
          glVertex3dv(vec1);
          list2 = g_slist_next(list2);
          }
        glEnd();
        break;

      case SPATIAL_MOLSURF:
      case SPATIAL_TRIANGLE_STRIP:
      case SPATIAL_POLYGON:
      case SPATIAL_POLYHEDRA:
/* compute normal, for proper lighting */
        list2 = (GSList *) spatial->data;
        p1 = (struct vec_pak *) list2->data;
        list2 = g_slist_next(list2);
        p2 = (struct vec_pak *) list2->data;
        list2 = g_slist_next(list2);
        p3 = (struct vec_pak *) list2->data;

/* enumerate periodic images */
        ilist=NULL;
        do
          {
          if (ilist)
            {
/* image */
            image = (struct image_pak *) ilist->data;
            ARR3SET(vec2, image->rx);
            ilist = g_slist_next(ilist);
            }
          else
            {
/* original */
            VEC3SET(vec2, 0.0, 0.0, 0.0);
            ilist = data->images;
            }

/* enumerate vertices */
          glBegin(spatial->method);
          list2 = (GSList *) spatial->data;
          while (list2)
            {
            p1 = (struct vec_pak *) list2->data;

            ARR3SET(vec1, vec2);
            ARR3ADD(vec1, p1->rx);
            VEC3MUL(vec1, data->scale);
            glColor4f(p1->colour[0], p1->colour[1], p1->colour[2],
                                          sysenv.render.transmit);
            glNormal3dv(p1->rn);
            glVertex3dv(vec1);
            list2 = g_slist_next(list2);
            }
          glEnd();
          }
        while (ilist);
        break;
      }
    }
  list = g_slist_next(list);
  }
}

/***************************************/
/* draw a picture in the OpenGL window */
/***************************************/
#if DRAW_PICTURE
void gl_picture_draw(struct canvas_pak *canvas, struct model_pak *model)
{
GdkPixbuf *raw_pixbuf, *scaled_pixbuf;
GError *error;

g_assert(model->picture_active != NULL);

/* read in the picture */
error = NULL;
raw_pixbuf = gdk_pixbuf_new_from_file(model->picture_active, &error);

/* error checking */
if (!raw_pixbuf)
  {
  if (error)
    {
    printf("%s\n", error->message);
    g_error_free(error);
    }
  else
    printf("Failed to load: %s\n", (gchar *) model->picture_active);
  return;
  }

/* scale and draw the picture */
scaled_pixbuf = gdk_pixbuf_scale_simple(raw_pixbuf,
                  canvas->width, canvas->height, GDK_INTERP_TILES);

gdk_draw_pixbuf((canvas->glarea)->window, NULL, scaled_pixbuf,
                0, 0, 0, 0, canvas->width, canvas->height,
                GDK_RGB_DITHER_NONE, 0, 0);
}
#endif

/************************/
/* main drawing routine */
/************************/
void draw_objs(struct canvas_pak *canvas, struct model_pak *data)
{
gint i;
gulong time;
gdouble r;
gfloat specular[4], fog[4];
gdouble fog_mark;
struct object_pak *obj_data=NULL;
GSList *list, *pipes[4];

time = mytimer();

/* NEW - transformation recording */
if (data->mode == RECORD && sysenv.moving)
  {
  struct transform_pak *transform;

/* TODO - transformation type extension eg operate on a single atom??? */
  transform = g_malloc(sizeof(struct transform_pak));
  data->transform_list = g_slist_append(data->transform_list, transform);

/* init transformation */
  transform->id = ROTATION;
  memcpy(transform->matrix, data->rotmat, 9*sizeof(gdouble));
  VEC3SET(transform->vector, data->offset[0], data->offset[1], 0.0);
  transform->scalar = data->scale;

  data->num_frames++;
  }

/* setup the lighting */
gl_init_lights(data);

/* scaling affects placement to avoid near/far clipping */
r = RMAX_FUDGE * data->rmax;

/* main drawing setup */
glFrontFace(GL_CCW);
glEnable(GL_LIGHTING);
glDepthMask(GL_TRUE);
glEnable(GL_DEPTH_TEST);
glDepthFunc(GL_LESS);
glEnable(GL_CULL_FACE);
glShadeModel(GL_SMOOTH);
glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
glLineWidth(sysenv.render.line_thickness);



/* NEW - vertex arrarys */
glEnableClientState(GL_VERTEX_ARRAY);
glEnableClientState(GL_NORMAL_ARRAY);



/* turn off antialiasing, in case it was previously on */
glDisable(GL_LINE_SMOOTH);
glDisable(GL_POINT_SMOOTH);
glDisable(GL_BLEND);

render_make_pipes(pipes, data);

/* depth queuing via fog */
/* FIXME: the stupid translate command for isolated atoms/spheres */
/* is what screws up this fog stuff  - molecules drawn in stick */
/* or liquorice/cylinder mode are perfectly fine, but spheres */
/* and crosses are not */
/* basically, spheres will all have the fog color */
/* at the origin superimposed on them */
if (sysenv.render.fog)
  {
/*
printf("  fog start = %f\n", sysenv.render.fog_start);
printf("fog density = %f\n", sysenv.render.fog_density);
*/

  glEnable(GL_FOG);
  ARR3SET(fog, sysenv.render.bg_colour);
  glFogf(GL_FOG_MODE, GL_LINEAR);
  glFogfv(GL_FOG_COLOR, fog);
  glHint(GL_FOG_HINT, GL_DONT_CARE);
/* FIXME - why does changing this do nothing??? */
  glFogf(GL_FOG_DENSITY, 0.35);
/* convert (intuitive) dialog value to required value */

/* FIXME - this is getting close, but still could do with some tweaking  */
  fog_mark = (sysenv.render.vp_dist+data->scale-1.0+2.0*sysenv.render.fog_start)*r;
  glFogf(GL_FOG_START, fog_mark);
/* NB: fog marks should not be equal as this turns depth queueing off */
  fog_mark += 0.1 + 2.0*(1.0 - sysenv.render.fog_density)*r;
  glFogf(GL_FOG_END, fog_mark);
  }
else
  glDisable(GL_FOG);

/* solid drawing */
glPolygonMode(GL_FRONT, GL_FILL);
glLightModeli(GL_LIGHT_MODEL_TWO_SIDE, GL_FALSE);

/* material shininess control (atoms) */
VEC3SET(specular, sysenv.render.ahl_strength 
                , sysenv.render.ahl_strength 
                , sysenv.render.ahl_strength);
glMaterialf(GL_FRONT, GL_SHININESS, sysenv.render.ahl_size);
glMaterialfv(GL_FRONT, GL_SPECULAR, specular);

/* colour-only change for spheres */
glEnable(GL_COLOR_MATERIAL);
glColorMaterial(GL_FRONT, GL_AMBIENT_AND_DIFFUSE);


/* TODO - speedup - if atoms are v small - disable lighting */
/* CURRENT - replace with vertex arrays */
gl_draw_atoms(CPK, FALSE, data);
gl_draw_atoms(BALL_STICK, FALSE, data);
gl_draw_atoms(LIQUORICE, FALSE, data);
gl_draw_atoms(STICK, FALSE, data);

/*
gl_draw_solid_atoms(data);
*/


/* material shininess control (surfaces) */
VEC3SET(specular, sysenv.render.shl_strength 
                , sysenv.render.shl_strength 
                , sysenv.render.shl_strength);
glMaterialf(GL_FRONT, GL_SHININESS, sysenv.render.shl_size);
glMaterialfv(GL_FRONT, GL_SPECULAR, specular);

/* draw vector objects */
glColorMaterial(GL_FRONT, GL_AMBIENT_AND_DIFFUSE);
glColor4f(sysenv.render.fg_colour[0], 
          sysenv.render.fg_colour[1],
          sysenv.render.fg_colour[2], 1.0);
gl_draw_spatial(SPATIAL_VECTOR, data);
gl_draw_spatial(SPATIAL_VECTORS, data);

if (data->show_axes)
  gl_draw_axes(TRUE, canvas, data);

/* normal bonds */
gl_draw_pipes(FALSE, pipes[0], data);

/* wire frame atom drawing */
glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);

if (sysenv.render.antialias)
  {
  glEnable(GL_LINE_SMOOTH);
  glEnable(GL_POINT_SMOOTH);
  glEnable(GL_BLEND);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  }


/* CURRENT - replace with vertex arrays */
gl_draw_atoms(CPK, TRUE, data);
gl_draw_atoms(BALL_STICK, TRUE, data);
gl_draw_atoms(LIQUORICE, TRUE, data);
gl_draw_atoms(STICK, TRUE, data);


/* double sided drawing for all spatial objects */
glLightModeli(GL_LIGHT_MODEL_TWO_SIDE, GL_TRUE);
glColorMaterial(GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE);
glDisable(GL_CULL_FACE);

/* draw spatial objects in wire/translucent mode */
if (!sysenv.render.wire_surface)
  {
  glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);

/* START - translucent drawing */
/* NB: make depth buffer read only - for translucent objects */
/* see the red book p229 */
  glDepthMask(GL_FALSE);
/* alpha blending for translucent surfaces */
  glEnable(GL_BLEND);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  }

/* morphology drawing */
if (data->num_vertices)
  {
  glLineWidth(sysenv.render.frame_thickness);
  glColor4f((float) sysenv.render.morph_colour[0], 
            (float) sysenv.render.morph_colour[1],
            (float) sysenv.render.morph_colour[2], sysenv.render.transmit);
  gl_draw_morph(data);
  glLineWidth(sysenv.render.line_thickness);
  }

/* FIXME - disable translucency until the front to back issue is sorted */
glDisable(GL_BLEND);
glDepthMask(GL_TRUE);

/* draw the ribbons */
list = data->ribbons;
while (list)
  {
  obj_data = (struct object_pak *) list->data;
  switch(obj_data->type)
    {
    case RIBBON:
      gl_draw_ribbon(obj_data, data);
      break;
    }
  list = g_slist_next(list);
  }

/* FIXME - translucent spatials are not drawn back to front & can look funny */
gl_draw_spatial(SPATIAL_PLANE, data);
gl_draw_spatial(SPATIAL_POLYGON, data);
gl_draw_spatial(SPATIAL_POLYHEDRA, data);
gl_draw_spatial(SPATIAL_TRIANGLE, data);
gl_draw_spatial(SPATIAL_TRIANGLE_STRIP, data);
gl_draw_spatial(SPATIAL_MOLSURF, data);

/* FIXME - re-enable translucency */
glEnable(GL_BLEND);
glDepthMask(GL_FALSE);

/* translucent ghost atoms */
glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
if (data->show_cores)
  {
  gl_draw_ghost_atoms(CPK, data);
  gl_draw_ghost_atoms(BALL_STICK, data);
  gl_draw_ghost_atoms(LIQUORICE, data);
  gl_draw_ghost_atoms(STICK, data);
  }

/* ghost bonds */
/* NB: back to front ordering */
pipes[1] = render_sort_pipes(pipes[1]);
gl_draw_pipes(FALSE, pipes[1], data);

/* translucent shells */
if (data->show_shells)
  gl_draw_shells(data);

/* selection outlines */
glDisable(GL_LIGHTING);
gl_draw_halo_list(data->selection, data);

/* END - translucent drawing */
glDepthMask(GL_TRUE);


/* at this point, should have completed all colour_material routines */
/* ie only simple line/text drawing stuff after this point */
glEnable(GL_CULL_FACE);
glDisable(GL_COLOR_MATERIAL);
glDisable(GL_LIGHTING);
glShadeModel(GL_FLAT);
if (sysenv.render.antialias)
  {
  glEnable(GL_LINE_SMOOTH);
  glEnable(GL_POINT_SMOOTH);
  glEnable(GL_BLEND);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  }

/* always visible foreground colour */
glColor4f(sysenv.render.fg_colour[0], 
          sysenv.render.fg_colour[1],
          sysenv.render.fg_colour[2], 1.0);

/* unit cell drawing */
if (data->show_cell && data->periodic)
  {
  glLineWidth(sysenv.render.frame_thickness);
  gl_draw_cell(data);
  }

/* draw wire/line bond types */
glLineWidth(sysenv.render.stick_thickness);
glColor4f(1.0, 0.85, 0.5, 1.0);
glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);

/* wire frame cylinders bonds */
glDisable(GL_CULL_FACE);
gl_draw_pipes(FALSE, pipes[2], data);
glEnable(GL_CULL_FACE);

/* stick (line) bonds */
gl_draw_pipes(TRUE, pipes[3], data);

/* axes labels */
glLineWidth(1.0);
if (data->show_axes)
  gl_draw_axes(FALSE, canvas, data);

/* set up stippling */
glEnable(GL_LINE_STIPPLE);
glLineStipple(1, 0x0F0F);

/* periodic cell images */
glColor3f(0.8, 0.7, 0.6);
if (data->show_cell_images)
  gl_draw_cell_images(data);

/* active/measurements colour (yellow) */
glColor4f(sysenv.render.label_colour[0],
          sysenv.render.label_colour[1],
          sysenv.render.label_colour[2], 1.0);

/* selection box */
glLineWidth(1.5);
if (data->box_on)
  gl_draw_box(data->select_box[0], data->select_box[1],
              data->select_box[2], data->select_box[3]);

/* measurements */
glLineWidth(sysenv.render.geom_line_width);
gl_draw_measurements(data);
glDisable(GL_LINE_STIPPLE);

/* text drawing - accept all fragments */
glDepthFunc(GL_ALWAYS);
glColor4f(sysenv.render.fg_colour[0], 
          sysenv.render.fg_colour[1],
          sysenv.render.fg_colour[2], 1.0);
gl_draw_text(data);

/* free all pipes */
for (i=4 ; i-- ; )
  free_slist(pipes[i]);

/* save timing info */
data->redraw_current = mytimer() - time;
data->redraw_cumulative += data->redraw_current;
data->redraw_count++;
}

/************************/
/* total canvas refresh */
/************************/
#define DEBUG_OPENGL_DRAW 0
void gl_draw(struct canvas_pak *canvas, struct model_pak *data)
{
gint flag;
gdouble sq, cq;
GdkGLContext *glcontext;
GdkGLDrawable *gldrawable;
PangoFontDescription *pfd;

#if DEBUG_OPENGL_DRAW
printf("gl_draw()\n");
#endif

/* CURRENT - stereo piracy */
if (sysenv.stereo && !sysenv.stereo_fullscreen)
  {
/* NEW - configure windowed stereo (ie setup viewport) */
  stereo_init_window(canvas);
/* draw in the stereo window */
  stereo_expose_event(canvas->glarea, NULL);
  return;
  }

/* is there anything to draw on? */
glcontext = gtk_widget_get_gl_context(canvas->glarea);
gldrawable = gtk_widget_get_gl_drawable(canvas->glarea);
if (!gdk_gl_drawable_gl_begin(gldrawable, glcontext))
  return;

/* non stereo double-buffered mode */
/* CURRENT - I think this helps when switching to/from stereo */
glDrawBuffer(GL_BACK);

/* NB: need to have this here (not in configure event) */
/* in order for the render dialog to change it */
glClearColor(sysenv.render.bg_colour[0], sysenv.render.bg_colour[1],
             sysenv.render.bg_colour[2], 0.0);

/* auto foreground colour */
make_fg_visible();

/* construct all half bond (ie pipe) lists */
/* clear the canvas */
glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT | GL_STENCIL_BUFFER_BIT);

if (!data)
  {
/* exit if no data */
/*
  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT | GL_STENCIL_BUFFER_BIT);
*/
  gdk_gl_drawable_swap_buffers(gldrawable);
  gdk_gl_drawable_gl_end(gldrawable);
  atom_properties_update(NULL, NULL);
  return;
  }
else
  {
/* clear the redraw flag */
  data->redraw = FALSE;
/* clear the atom info box  */
  if (g_slist_length(data->selection) != 1)
    atom_properties_update(NULL, NULL);
  }

/* setup transformation matrices */
gl_init_projection(data);

/* low quality render when scaling/rotating */
sq = sysenv.render.sphere_quality;
cq = sysenv.render.cylinder_quality;
if (sysenv.moving && sysenv.render.fast_rotation)
  {
  sysenv.render.sphere_quality = 1;
  sysenv.render.cylinder_quality = 5;
  }

/*
glClearStencil(0x4);
*/
glClearStencil(0x0);

/* pango fonts for OpenGL */
if (font_offset < 0)
  {
  font_offset = glGenLists(128);
  if (font_offset)
    { 
    pfd = pango_font_description_from_string(sysenv.gl_fontname);
    if (!gdk_gl_font_use_pango_font(pfd, 0, 128, font_offset))
      show_text(ERROR, "Failed to set up Pango font for OpenGL.\n");
    gl_fontsize = pango_font_description_get_size(pfd) / PANGO_SCALE;
    pango_font_description_free(pfd); 
    }
  else
    show_text(ERROR, "Failed to allocate display lists for OpenGL fonts.\n");
  }

/* draw special display types */
flag = TRUE;
if (data->graph_active)
  {
  graph_draw(canvas, data);
  flag = FALSE;
  }

#if DRAW_PICTURE
if (data->picture_active)
  {
  sysenv.render.sphere_quality = sq;
  sysenv.render.cylinder_quality = cq;

/* NEW - not sure why, but the drawable_end call must come after the */
/* picture_draw - also, swap_buffers must not be called at all */

  gl_picture_draw(canvas, data);
  gdk_gl_drawable_gl_end(gldrawable);

  return;
  }
#endif

/* draw model */
if (flag)
  draw_objs(canvas, data);

/* all done */
sysenv.render.sphere_quality = sq;
sysenv.render.cylinder_quality = cq;

gdk_gl_drawable_swap_buffers(gldrawable);
gdk_gl_drawable_gl_end(gldrawable);
}

/**************************/
/* handle redraw requests */ 
/**************************/
#define DEBUG_REDRAW_HANDLER 0
gint redraw_handler(gpointer *dummy)
{
gint m;
static gint n=1;
gdouble time;
GSList *list;
struct model_pak *model;
struct canvas_pak *canvas;

/* do we have to do anything? */
if (sysenv.redraw)
  {
/* NEW - fullscreen stereo piracy */
  if (sysenv.stereo && sysenv.stereo_fullscreen)
    {
    stereo_draw();
    return(TRUE);
    }

/* TODO - the redraw_canvas() call will create a list of */
/* glarea's that need to be updated by this routine */
/* ie instead of updating them all as is done currently */
/* loop and draw if we have a pending model and a canvas */

/* NB: the active canvas should be redrawn last - since any */
/* subsequent OpenGL calls (eg get_world_coords()) will refer to this canvas */
/* ... OR does the GL context begin/end stuff fix this issue??? */
  for (list=sysenv.canvas_list ; list ; list=g_slist_next(list))
    {
    canvas = (struct canvas_pak *) list->data;

/*
printf("canvas: %p\nactive: %d\nmodel: %p\nglarea: %p\n",
        canvas, canvas->active, canvas->model, canvas->glarea);
*/

    model = canvas->model;
    if (model)
      {
      if (!model->redraw)
        return(TRUE);

      gl_draw(canvas, model);

      model->redraw = FALSE;

/* timing analysis */
     if (model->redraw_count >= 10)
       {
       time = model->redraw_cumulative;
       time /= model->redraw_count;
       time /= 1000.0;

#if DEBUG_REDRAW_HANDLER
printf("[redraw] cumulative = %d us : average = %.1f ms : freq = %d x 25ms\n",
        model->redraw_cumulative, time, n);
#endif

       model->redraw_count = 0;
       model->redraw_cumulative = 0;

/* adjust redraw frequency? */
        m = 1 + time/25;
        if (m > n)
          {
          n = m;
#if DEBUG_REDRAW_HANDLER
printf("increasing delay between redraw: %d\n", n);
#endif
          g_timeout_add(n*25, (GSourceFunc) &redraw_handler, NULL);
          return(FALSE);
          }
        }
      else
        {
/* redraw frequency test */
        if (n > 1 && model->redraw_count)
          {
          time = model->redraw_cumulative;
          time /= model->redraw_count;
          time /= 1000.0;
          m = 1 + time/25;
/* adjust redraw frequency? */
          if (m < n)
            {
            n = m;
#if DEBUG_REDRAW_HANDLER
printf("decreasing delay between redraw: %d\n", n);
#endif
            g_timeout_add(n*25, (GSourceFunc) &redraw_handler, NULL);
            return(FALSE);
            }
          }
        }
      }
    else
      {
/* clear the canvas */
      gl_draw(canvas, NULL);
      }
    }
/* clear the redraw flags */
  sysenv.redraw = FALSE;
  }

return(TRUE);
}

/****************************/
/* schedule redraw requests */ 
/****************************/
void redraw_canvas(gint action)
{
GSList *list;
struct model_pak *data;

switch (action)
  {
  case SINGLE:
    data = sysenv.active_model;
    if (data)
      data->redraw = TRUE;
    break;
  case ALL:
    for (list=sysenv.mal ; list ; list=g_slist_next(list))
      {
      data = (struct model_pak *) list->data;
      data->redraw = TRUE;
      }
    break;
  }
sysenv.redraw = TRUE;
}

