/*
Copyright (C) 2003 by Sean David Fleming

sean@power.curtin.edu.au

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

#include <math.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>
#include <unistd.h>
#include <sys/stat.h>

#include "gdis.h"
#include "coords.h"
#include "matrix.h"
#include "edit.h"
#include "file.h"
#include "graph.h"
#include "morph.h"
#include "model.h"
#include "measure.h"
#include "project.h"
#include "analysis.h"
#include "render.h"
#include "select.h"
#include "space.h"
#include "surface.h"
#include "opengl.h"
#include "interface.h"
#include "dialog.h"
#include "zone.h"

/* the main pak structures */
extern struct sysenv_pak sysenv;
extern struct elem_pak elements[];
/* main model database */
struct model_pak *models[MAX_MODELS];

/*****************************/
/* default model orientation */
/*****************************/
void init_rotmat(gdouble *mat)
{
VEC3SET(&mat[0], 1.0, 0.0, 0.0);
VEC3SET(&mat[3], 0.0, 0.0,-1.0);
VEC3SET(&mat[6], 0.0, 1.0, 0.0);
/*
VEC3SET(&mat[0], 1.0, 0.0, 0.0);
VEC3SET(&mat[3], 0.0, 1.0, 0.0);
VEC3SET(&mat[6], 0.0, 0.0, 1.0);
*/
}

/***********************************/
/* unified initialization/cleaning */
/***********************************/
void template_model(struct model_pak *data)
{
static gint n=0;

/* NB: all the important defines should go here */
/* ie those values that might cause a core dump due to */
/* code that requires they at least have a value assigned  */
data->id = -1;
data->locked = FALSE;
data->protein = FALSE;
data->num_atoms = 0;
data->num_asym = 0;
data->num_shells = 0;
data->num_geom = 0;
data->num_images = 0;
data->num_frames = 1;
data->cur_frame = 0;
data->expected_cores = 0;
data->expected_shells = 0;
data->header_size = 0;
data->frame_size = 0;
data->file_size = 0;
data->trj_swap = FALSE;
data->selection = NULL;
data->atom_info = -1;
data->shell_info = -1;
data->has_sof = FALSE;
data->sof_colourize = FALSE;
data->construct_pbc = FALSE;
data->redraw = FALSE;          /* wait for more setup before drawing */
data->redraw_count = 0;
data->redraw_cumulative = 0;
data->redraw_time = 0;
data->colour_scheme = ELEM;
/* linked list init */
data->cbu = TRUE;
data->cores = NULL;
data->shels = NULL;
data->bonds = NULL;
data->ubonds = NULL;
data->moles = NULL;
data->planes = NULL;
data->ghosts = NULL;
data->images = NULL;
data->layer_list = NULL;
data->measure_list = NULL;
data->graph_list = NULL;
data->picture_list = NULL;
data->transform_list = NULL;
data->frame_data_list = NULL;
data->graph_active = NULL;
data->picture_active = NULL;
data->project = NULL;
data->zones = NULL;
data->num_zones = 0;
VEC3SET(data->zone_min, 0.0, 0.0, 0.0);
VEC3SET(data->zone_max, 1.0, 1.0, 1.0);
VEC3SET(data->zone_div, 1, 1, 1);
VEC3SET(data->zone_dx, 0.0, 0.0, 0.0);

/* display flags for the drawing area */
data->show_names = FALSE;
data->show_title = TRUE;
data->show_frame_number = TRUE;
data->show_hbonds = FALSE;
data->show_energy = FALSE;
data->show_charge = FALSE;
data->show_atom_charges = FALSE;
data->show_atom_labels = FALSE;
data->show_atom_types = FALSE;
data->show_geom_labels = TRUE;
data->show_cores = TRUE;
data->show_shells = FALSE;
data->show_axes = TRUE;
data->show_cell = TRUE;
data->show_cell_images = TRUE;
data->show_cell_lengths = FALSE;

/* NEW */
data->property_table = g_hash_table_new_full(g_str_hash, g_str_equal, g_free, property_free);
data->property_list = NULL;

data->ribbons = NULL;
data->spatial = NULL;
data->phonons = NULL;
data->ir_list = NULL;
data->raman_list = NULL;
data->phonon_slider = NULL;
data->current_phonon = -1;
data->pulse_count = 0;
data->pulse_direction = 1;
data->num_phonons = 0;
data->show_eigenvectors = FALSE;
data->gulp.phonon = FALSE;
data->gulp.num_kpoints = 0;
VEC3SET(data->gulp.kpoints, 0.0, 0.0, 0.0);

/* VdW surface calc */
data->ms_colour_scale = FALSE;

/* init flags */
data->periodic = 0;
data->grafted = FALSE;
data->fractional = FALSE;
data->done_pbonds = FALSE;
data->axes_type = CARTESIAN;
data->box_on = FALSE;
data->asym_on = FALSE;
/* bonding modes */
data->build_molecules = TRUE;
data->build_hydrogen = FALSE;
data->build_polyhedra = FALSE;
data->build_zeolite = FALSE;

/* can we eliminate da? I don't like it. */
data->da = 0.0;
data->scale = 1.0;
data->rmax = RMAX_FUDGE;
/* init the model orientation (change to unit_mat() or something sensible) */
init_rotmat(data->rotmat);
init_rotmat(data->irotmat);
/* init translation & rotation */
VEC3SET(data->offset, 0.0, 0.0, 0.0);
VEC3SET(data->angle, 0.0, 0.0, 0.0);
/* this is now necessary due to the new latvec code */
/* ie standard cartesian axes */
data->pbc[0] = 1.0;
data->pbc[1] = 1.0;
data->pbc[2] = 1.0;
data->pbc[3] = 0.5*PI;
data->pbc[4] = 0.5*PI;
data->pbc[5] = 0.5*PI;
data->volume = 0.0;
data->area = 0.0;
/* init lattice matrix as this (for safety) */
make_latmat(data);

/* space group info */
data->sginfo.lookup = TRUE;
data->sginfo.spacenum = 0;
data->sginfo.lattice = 0;
data->sginfo.pointgroup = 0;
data->sginfo.cellchoice = 0;
data->sginfo.order = 0;
data->sginfo.centric = 0;
/*
data->sginfo.spacename = g_strdup("P 1");
data->sginfo.latticename = g_strdup("None");
*/
data->sginfo.spacename = NULL;
data->sginfo.latticename = NULL;
data->sginfo.matrix = NULL;
data->sginfo.offset = NULL;

/* symmetry info */
data->symmetry.num_symops = 0;
data->symmetry.symops = g_malloc(sizeof(struct symop_pak));
data->symmetry.num_items = 1;
data->symmetry.items = g_malloc(2*sizeof(gchar *));
*(data->symmetry.items) = g_strdup("none");
*((data->symmetry.items)+1) = NULL;
data->symmetry.pg_entry = NULL;
data->symmetry.summary = NULL;
data->symmetry.pg_name = g_strdup("unknown");
/* periodic images */
data->image_limit[0] = 0.0;
data->image_limit[1] = 1.0;
data->image_limit[2] = 0.0;
data->image_limit[3] = 1.0;
data->image_limit[4] = 0.0;
data->image_limit[5] = 1.0;

data->region_max = 0;
/* existence */
data->region_empty[REGION1A] = FALSE;
data->region_empty[REGION2A] = TRUE;
data->region_empty[REGION1B] = TRUE;
data->region_empty[REGION2B] = TRUE;
/* lighting */
data->region[REGION1A] = TRUE;
data->region[REGION2A] = FALSE;
data->region[REGION1B] = FALSE;
data->region[REGION2B] = FALSE;
/* morphology */
data->morph_type = DHKL;
data->num_vertices = 0;
data->num_planes = 0;
data->vertices = NULL;

data->basename = g_strdup_printf("model_%d", g_slist_index(sysenv.mal, data));
strcpy(data->filename, data->basename);
n++;

/* default potential library */
strcpy(data->gulplib, LIBRARY);

/* gulp template init */
data->gulp.maxcyc=0;
data->gulp.energy = 0.0 ;
data->gulp.no_esurf=FALSE;
data->gulp.esurf[0]=0.0;
data->gulp.esurf[1]=0.0;
data->gulp.esurf_units=g_strdup(" ");
data->gulp.no_eatt=FALSE;
data->gulp.eatt[0]=0.0;
data->gulp.eatt[1]=0.0;
data->gulp.eatt_units=g_strdup(" ");
data->gulp.sbulkenergy=0.0;
data->gulp.sdipole=999999999.0;
data->gulp.sdipole_tolerance=0.005;
data->gulp.gnorm = -1.0;
data->gulp.qsum = 0.0;
data->gulp.no_exec = FALSE;
data->gulp.run = E_SINGLE;
data->gulp.dock = FALSE;
data->gulp.qeq = FALSE;
data->gulp.free = FALSE;
data->gulp.zsisa = FALSE;
data->gulp.compare = FALSE;
data->gulp.nosym = FALSE;
data->gulp.fix = FALSE;
data->gulp.method = CONP;
/* dynamics info */
data->gulp.frame_time = 0.0;
data->gulp.frame_ke = 0.0;
data->gulp.frame_pe = 0.0;
data->gulp.frame_temp = 0.0;
/* NEW - original frame data (stored since reading in trajectory overwrites) */
data->gulp.orig_fractional = FALSE;
data->gulp.orig_construct_pbc = FALSE;
data->gulp.orig_cores = NULL;
data->gulp.orig_shells = NULL;
data->gulp.ff_list = NULL;
data->gulp.species_list = NULL;
data->gulp.element_list = NULL;
/* potgrid replacement */
data->gulp.epot_autoscale = TRUE;
data->gulp.epot_vecs = NULL;
data->gulp.epot_vals = NULL;
data->gulp.epot_divisions = 0;
data->gulp.epot_min = 999999999.9;
data->gulp.epot_max = -999999999.9;

data->gulp.optimiser = -1;
data->gulp.optimiser2 = SWITCH_OFF;
data->gulp.switch_type = -1;
data->gulp.switch_value = 0.1;
data->gulp.unit_hessian = FALSE;
data->gulp.ensemble = -1;
data->gulp.temp = 0.0;
VEC3SET(data->gulp.super, 1, 1, 1);
data->gulp.coulomb = NOBUILD;
data->gulp.libfile = g_strdup(LIBRARY);
data->gulp.srcfile = NULL;
data->gulp.extra = NULL;
data->gulp.temp_file = g_strdup("none");
data->gulp.dump_file = g_strdup("none");
data->gulp.out_file = g_strdup("none");
data->gulp.trj_file = NULL;
data->gulp.mov_file = NULL;

/* GAMESS defaults */
data->gamess.units = GMS_ANGS;
data->gamess.exe_type = GMS_RUN;
data->gamess.run_type = GMS_ENERGY;
data->gamess.scf_type = GMS_RHF;
data->gamess.basis = GMS_MNDO;
data->gamess.ngauss = 0;
data->gamess.opt_type = GMS_QA;
data->gamess.total_charge = 0;
data->gamess.multiplicity = 1;
data->gamess.time_limit = 600;
data->gamess.mwords = 1;
data->gamess.num_p = 0;
data->gamess.num_d = 0;
data->gamess.num_f = 0;
data->gamess.have_heavy_diffuse = FALSE;
data->gamess.have_hydrogen_diffuse = FALSE;
data->gamess.converged = FALSE;
data->gamess.wide_output = FALSE;
data->gamess.nstep = 20;
data->gamess.maxit = 30;
data->gamess.srcfile = NULL;
data->gamess.title = g_strdup("none");
data->gamess.temp_file = g_strdup("none");
data->gamess.out_file = g_strdup("none");
data->gamess.energy = data->gamess.max_grad = data->gamess.rms_grad = 0.0;
data->gamess.have_energy = data->gamess.have_max_grad = data->gamess.have_rms_grad = FALSE;

data->abinit.energy = data->abinit.max_grad = data->abinit.rms_grad = 0.0;

data->nwchem.energy = data->nwchem.max_grad = data->nwchem.rms_grad = 0.0;
data->nwchem.have_energy = data->nwchem.have_max_grad = data->nwchem.have_rms_grad = data->nwchem.min_ok = FALSE;

data->castep.energy = data->castep.max_grad = data->castep.rms_grad = 0.0;
data->castep.have_energy = data->castep.have_max_grad = data->castep.have_rms_grad = data->castep.min_ok = FALSE;

/* SIESTA defaults */
data->siesta.energy = data->siesta.max_grad = 0.0;
data->siesta.have_energy = data->siesta.have_max_grad = 0.0;

/* diffraction defaults */
VEC3SET(data->diffract.theta, 0.0, 90.0, 0.1);
data->diffract.wavelength = 1.54180;
data->diffract.asym = 0.18;
data->diffract.u = 0.0;
data->diffract.v = 0.0;
data->diffract.w = 0.0;

/* surface creation init */
data->surface.optimise = FALSE;
data->surface.converge_eatt = FALSE;
data->surface.converge_r1 = TRUE;
data->surface.converge_r2 = TRUE;
data->surface.include_polar = FALSE;
data->surface.ignore_bonding = FALSE;
data->surface.create_surface = FALSE;
data->surface.true_cell = FALSE;
data->surface.keep_atom_order = FALSE;
data->surface.miller[0] = 1;
data->surface.miller[1] = 0;
data->surface.miller[2] = 0;
data->surface.shift = 0.0;
data->surface.dspacing = 0.0;
data->surface.region[0] = 1;
data->surface.region[1] = 1;
VEC3SET(data->surface.depth_vec, 0.0, 0.0, 0.0);

get_rot_matrix(data->latmat, IDENTITY, NULL, 3);
get_rot_matrix(data->ilatmat, IDENTITY, NULL, 3);
get_rot_matrix(data->rlatmat, IDENTITY, NULL, 3);
get_rot_matrix(data->rotmat, IDENTITY, NULL, 3);
get_rot_matrix(data->irotmat, IDENTITY, NULL, 3);

/* no non-default element data for the model (yet) */
data->num_elem = 0;
data->elements = NULL;
data->unique_atom_list = NULL;
/* file pointer */
data->animation = FALSE;
data->animating = FALSE;
data->anim_fix = FALSE;
data->anim_loop = FALSE;
data->afp = NULL;
data->anim_confine = PBC_CONFINE_ATOMS;
data->anim_speed = 5.0;
data->frame_list = NULL;
data->title = NULL;

/* NEW */
data->analysis = analysis_new();
}

/************************************/
/* standard preparation for display */
/************************************/
#define DEBUG_PREP_MODEL 0
gint prep_model(struct model_pak *data)
{
g_return_val_if_fail(data != NULL, 1);

#if DEBUG_PREP_MODEL
printf("start prep: %d\n", data->number);
#endif

/* init the original values */
data->num_asym = g_slist_length(data->cores);

/* create the all important lattice matrix before anything else */
make_latmat(data);

/* convert input cartesian coords to fractional */
if (!data->fractional)
  latmat_fudge(data);

/* initialize the spatial partitioning */
zone_init(data);

/* get initial linkages */
/* NB: full calc is done after remove_duplicates, but this is still needed */
/* so that a full cell of shells can be built from the asymmetric unit */
shell_make_links(data);

/* mark the largest region label */
data->region_max = region_max(data);

if (data->periodic == 3)
  {
  if (!data->sginfo.spacenum)
    data->sginfo.spacenum=1;
  if (space_lookup(data))
    show_text(ERROR, "Error in Space Group lookup.\n");
  }
else
  remove_duplicates(data);


#if OLD_CRAP
/* periodic setup */
switch(data->periodic)
  {
  case 3:
/*
  case 2:
*/
    if (!data->sginfo.spacenum)
      data->sginfo.spacenum=1;
    if (space_lookup(data))
      show_text(ERROR, "Error in Space Group lookup.\n");
    break;

  default:
/* NEW - remove duplicate cores which can cause connectivity  */
/* wierdness and stop the ribbon code (for eg) from working */
/* also removes symmetry generated equivalents */
/* NB: fn_fill_cell() in space_lookup() also calls this */
    remove_duplicates(data);
  }
#endif

/* multi-frame file is always an animation */
if (data->num_frames > 1)
  data->animation = TRUE;

/* calculate how many frames an associated gulp trg file has */
if (data->animation)
  {
  guint n, header_size, frame_size;
  gdouble num_frames;
  gchar *filename;
  struct stat buff;
  FILE *fp;

/* calculate frame size for num_frame calculation */
  if (data->id == GULP && !data->file_size)
    {
    header_size = 6*sizeof(int) + sizeof(double);
/* sizeof data */
    data->expected_cores = g_slist_length(data->cores);
    data->expected_shells = g_slist_length(data->shels);
    n = data->expected_cores + data->expected_shells;
/* cope with the supercell cell multiplication */
    n *= data->gulp.super[0] * data->gulp.super[1] * data->gulp.super[2];

/* NB: a bit messy, but the supercell isn't created until later */
/* (see file_load() - after the entire file has been processed) */

/* sizeof RECORD */
    frame_size = 14 * sizeof(int);
    frame_size += (4 + 6*n) * sizeof(double);

if (data->gulp.ensemble == NPT)
  {
  frame_size += 4 * sizeof(int);

/* nstrains (cell velocities) */
  switch (data->periodic)
    {
    case 3:
      n=6;
      break;
    case 2:
      n=3;
      break;
    case 1:
      n=1;
      break;
    default:
      n=0;
    }

/* cell vectors */
  n += 9;

  frame_size += n * sizeof(double);
  }

/* stat the file (MUST have the full path) */
    filename = g_strdup_printf("%s/%s", sysenv.cwd, data->gulp.trj_file);
/* get num frames (as a float - for debugging) */
    if (stat(filename, &buff))
      {
      printf("Warning: bad or missing GULP trajectory file.\n");
      data->animation = FALSE;
      }
    else
      {
/* attempt to read the header */
fp = fopen(filename, "r");
if (fp)
  read_trj_header(fp, data);
else
  printf("Failed to open %s.\n", filename);

      num_frames = buff.st_size - header_size;
      num_frames /= (gdouble) frame_size;
      data->num_frames = (guint) num_frames;
      data->header_size = header_size;
      data->frame_size = frame_size;
      data->file_size = (guint) buff.st_size;

#if DEBUG_PREP_MODEL
printf("  num cores: %d\n", data->expected_cores);
printf(" num shells: %d\n", data->expected_shells);
printf("header size: %d\n", data->header_size);
printf(" frame size: %d\n", data->frame_size);
printf("  file size: %d\n", data->file_size);
printf(" num frames: %f\n", num_frames);
#endif

if ((num_frames - data->num_frames) > FRACTION_TOLERANCE)
  {
  printf("Bad GULP trajectory file.\n");
  }
fclose(fp);

      }
    g_free(filename);
    }
  }

/* init */
data->num_atoms = g_slist_length(data->cores);
data->num_shells = g_slist_length(data->shels);
data->mode = FREE;

#if DEBUG_PREP_MODEL
printf(" num atoms: %d\n", data->num_atoms);
printf("num shells: %d\n", data->num_shells);
#endif

/* display coords & connectivity */
init_objs(INIT_COORDS, data);
calc_bonds(data);
calc_mols(data);

/* auto unfrag? */
if (sysenv.unfragment)
  unfragment(data);

/* init the colour scheme */
model_colour_scheme(data->colour_scheme, data);

/* final surface setup */
if (data->periodic == 2)
  {
/* order by z, unless input core order should be kept */
  if (!data->surface.keep_atom_order)
    {
    sort_coords(data);
    init_objs(CENT_COORDS, data);
    }
  }

#if DEBUG_PREP_MODEL
printf("end prep: %d\n", data->number);
#endif

return(0);
}

/*****************************************/
/* connectivity update for current model */
/*****************************************/
void refresh_connectivity(struct model_pak *data)
{
g_return_if_fail(data != NULL);

init_objs(REFRESH_BONDS, data);
calc_bonds(data);
calc_mols(data);
}

/*********************************************************/
/* convenience routine for duplicating a list of strings */
/*********************************************************/
GSList *slist_gchar_dup(GSList *src)
{
GSList *list, *dest=NULL;

for (list=src ; list ; list=g_slist_next(list))
  dest = g_slist_prepend(dest, g_strdup(list->data));

if (dest)
  dest = g_slist_reverse(dest);

return(dest);
}

/*************************************************************/
/* generic call for an slist containing one freeable pointer */
/*************************************************************/
void free_slist(GSList *list)
{
GSList *item;

for (item=list ; item ; item=g_slist_next(item))
  g_free(item->data);

g_slist_free(list);
list=NULL;
}

/***********************************************************/
/* generic call for a list containing one freeable pointer */
/***********************************************************/
void free_list(GList *list)
{
GList *item;

for (item=list ; item ; item=g_list_next(item))
  g_free(item->data);

g_list_free(list);
list=NULL;
}

/*************************************************/
/* TODO - put stuff like this in a gamess.c file */
/*************************************************/
void gamess_data_free(struct model_pak *model)
{
g_free(model->gamess.title);
g_free(model->gamess.temp_file);
g_free(model->gamess.out_file);
}

/***********************************/
/* free the entire model structure */
/***********************************/
#define DEBUG_FREE_MODEL 0
void free_model(struct model_pak *data)
{
/* check */
g_return_if_fail(data != NULL);

#if DEBUG_FREE_MODEL
printf("freeing string data...\n");
#endif

g_free(data->basename);
g_free(data->symmetry.pg_name);
g_free(data->title);

space_free(&data->sginfo);

#if DEBUG_FREE_MODEL
printf("freeing zone data...\n");
#endif

zone_free(data);

#if DEBUG_FREE_MODEL
printf("freeing coord data...\n");
#endif

/* free lists with associated data */
free_core_list(data);
free_slist(data->images);
free_list(data->frame_list);
free_slist(data->layer_list);
free_slist(data->picture_list);
graph_free_list(data);
free_slist(data->phonons);
free_slist(data->ir_list);
free_slist(data->raman_list);

/* destroy the property table and list */
g_hash_table_destroy(data->property_table);
g_slist_free(data->property_list);

gamess_data_free(data);

/* NEW - GULP files with associated trajectory only */
if (data->gulp.orig_cores)
  free_slist(data->gulp.orig_cores);
if (data->gulp.orig_shells)
  free_slist(data->gulp.orig_shells);

free_gulp_data(data);

#if DEBUG_FREE_MODEL
printf("freeing symmetry data...\n");
#endif
g_free(data->symmetry.symops);
g_strfreev(data->symmetry.items);

#if DEBUG_FREE_MODEL
printf("freeing measurements...\n");
#endif
meas_prune_model(data);
measure_free_all(data);

#if DEBUG_FREE_MODEL
printf("freeing vertices... (%d)\n", data->num_vertices);
#endif

plane_data_free(data->planes);
g_slist_free(data->planes);
data->planes = NULL;
data->num_planes = 0;

/* TODO - free adj list (not data) for each vertices */
free_slist(data->vertices);

#if DEBUG_FREE_MODEL
printf("freeing element data...\n");
#endif
if (data->num_elem)
  g_free(data->elements);

/*
#if DEBUG_FREE_MODEL
printf("closing animation stream...\n");
#endif
if (data->afp)
  fclose(data->afp);
*/

analysis_free(data);
}

/************************/
/* delete a given model */
/************************/
#define DEBUG_DELETE_MODEL 0
void model_delete(struct model_pak *model)
{
GSList *list;
struct model_pak *m;

/* checks */
if (!model)
  return;
if (model->locked)
  {
  show_text(ERROR, "Model is locked.\n");
  return;
  }

if (!g_slist_find(sysenv.mal, model))
  {
  printf("WARNING: unregistered model.\n");
  }
else
  {
#if DEBUG_DELETE_MODEL
printf("shuffling model numbers...\n");
#endif
/* shuffle the data->number to account */
/* for the deletion of model */
  for (list=sysenv.mal ; list ; list=g_slist_next(list))
    {
    m = list->data;
    if (m == model)
      break;
    m->number--; 
    }
  sysenv.mal = g_slist_remove(sysenv.mal, model);
  }

#if DEBUG_DELETE_MODEL
printf("freeing assoc. dialogs...\n");
#endif

dialog_destroy_model(model);

/* free the model's pointers */
free_model(model);
g_free(model);

/* update */
canvas_shuffle();
redraw_canvas(ALL);

#if DEBUG_DELETE_MODEL
printf("delete_model() completed.\n");
#endif
}

/**************************/
/* convenient signal hook */
/**************************/
void delete_active()
{
model_delete(sysenv.active_model);
}

/*******************************/
/* replacement for ASSIGN call */
/*******************************/
struct model_pak *model_new(void)
{
struct model_pak *model;

model = g_malloc(sizeof(struct model_pak));
sysenv.mal = g_slist_append(sysenv.mal, model);

template_model(model);
return(model);
}

/*******************************************/
/* get pointer to a requested model's data */
/*******************************************/
#define DEBUG_PTR 0
struct model_pak *model_ptr(gint model, gint mode)
{
struct model_pak *ptr=NULL;

/* how were we called? */
switch (mode)
  {
  case RECALL:
/* FIXME - there are an awful lot of calls to this; check for routines */
/* that pass the model number, when they could pass the model ptr instead */
#if DEBUG_PTR
printf("Request for model %d [%d,%d], ptr %p\n", model, 0, sysenv.num_models, ptr);
#endif
    ptr = (struct model_pak *) g_slist_nth_data(sysenv.mal, model);
    break;
  }
return(ptr);
}

/***********************************************/
/* model property list modification primitives */
/***********************************************/

struct property_pak
{
guint rank;
gchar *label;
gchar *value;
};

/*************************/
/* destruction primitive */
/*************************/
void property_free(gpointer ptr_property)
{
struct property_pak *p = ptr_property;

/* NB: the label/key is free'd elsewhere */
g_free(p->value);
g_free(p);
}

/*********************/
/* ranking primitive */
/*********************/
gint property_ranking(gpointer ptr_p1, gpointer ptr_p2)
{
struct property_pak *p1 = ptr_p1, *p2 = ptr_p2;

if (p1->rank < p2->rank)
  return(-1);
if (p1->rank > p2->rank)
  return(1);
return(0);
}

/***************************************/
/* add a pre-ranked new model property */
/***************************************/
/* NB: property is not displayed if rank = 0 */
void property_add_ranked(guint rank,
                         const gchar *key,
                         const gchar *value,
                         struct model_pak *model)
{
struct property_pak *p;

g_assert(model != NULL);

/* check if label already exists */
p = g_hash_table_lookup(model->property_table, key);
if (p)
  {
/* update existing property */
  g_free(p->value); 
  p->value = g_strdup(value);
  p->rank = rank;
  model->property_list = g_slist_remove(model->property_list, p);
  }
else
  {
/* create new property */
  p = g_malloc(sizeof(struct property_pak));

  p->value = g_strdup(value);
  p->label = g_strdup(key);
  p->rank = rank;
  g_hash_table_replace(model->property_table, p->label, p);
  }

/* update sorted property list */
model->property_list = g_slist_insert_sorted(model->property_list, p, (gpointer) property_ranking);
}

/****************************************/
/* property value extraction primitives */
/****************************************/
gchar *property_lookup(gchar *key, struct model_pak *model)
{
struct property_pak *p;

g_assert(model != NULL);

p = g_hash_table_lookup(model->property_table, key);
if (p)
  return(p->value);
return(NULL);
}

guint property_rank(gpointer ptr_property)
{
struct property_pak *p = ptr_property;

return(p->rank);
}

gchar *property_label(gpointer ptr_property)
{
struct property_pak *p = ptr_property;

return(p->label);
}

gchar *property_value(gpointer ptr_property)
{
struct property_pak *p = ptr_property;

return(p->value);
}

