/*
 * GNoise
 *
 * Copyright (C) 1999-2001 Dwight Engen
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: dlgprefs.c,v 1.5 2001/08/31 02:04:22 dengen Exp $
 *
 */

#include <errno.h>
#include <fcntl.h>
#include <pthread.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <gtk/gtk.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "gnoise.h"


struct _prefs_sym
{
    char	name[40];
    size_t	offset;
    int		datasz;
};


static void prefs_apply(GtkButton *button, gpointer user_data);

static GtkWidget *win_prefs = NULL;
static const prefs_sym_t prefs_symtable[] =
{
    {"DCACHE_SAVE_FILES",  offsetof(prefs_t, dcache_save),        sizeof(gboolean)},
    {"SAMPLE_DISPLAY_FMT", offsetof(prefs_t, display_fmt),        sizeof(smpl_fmt_enum_t)},
    {"PLAYBACK_TRACKED",   offsetof(prefs_t, playback_tracked),   sizeof(gboolean)},
    {"PLAYBACK_SCROLLED",  offsetof(prefs_t, playback_scrolled),  sizeof(gboolean)},
    {"SHOW_TIME",          offsetof(prefs_t, show_time),          sizeof(gboolean)},
    {"SELECTION_BOND",     offsetof(prefs_t, selection_bond),     sizeof(gboolean)},
    {"DISABLE_UNDO",       offsetof(prefs_t, disable_undo),       sizeof(gboolean)},
    {"REMEMBER_OPEN_DIR",  offsetof(prefs_t, remember_open_dir),  sizeof(gboolean)},
    {"RECORD_CHANNELS",    offsetof(prefs_t, record.channels),    sizeof(chnl_indx)},
    {"RECORD_SAMPLE_RATE", offsetof(prefs_t, record.sample_rate), sizeof(ulong)},
    {"RECORD_SAMPLE_BITS", offsetof(prefs_t, record.sample_bits), sizeof(ulong)},
    {"TB_TRANSPORT",       offsetof(prefs_t, tb_transport),       sizeof(gboolean)},
    {"TB_MARKERS",         offsetof(prefs_t, tb_markers),         sizeof(gboolean)},
};
#define PREFS_COUNT (sizeof(prefs_symtable)/sizeof(prefs_symtable[0]))


void
on_dlgprefs(GtkMenuItem *menuitem, gpointer user_data)
{
    gchar			widget_name[80];
    GtkToggleButton		*tb;
    GtkNotebook			*nb;

    win_prefs = create_PreferencesDialog();

    /* general prefs */
    if (prefs.dcache_save)
    {
	tb = GTK_TOGGLE_BUTTON(lookup_widget(win_prefs, "SaveDCache"));
	gtk_toggle_button_set_active(tb, TRUE);
    }

    if (prefs.playback_tracked)
    {
	tb = GTK_TOGGLE_BUTTON(lookup_widget(win_prefs, "TrackPlayback"));
	gtk_toggle_button_set_active(tb, TRUE);
    }

    if (prefs.playback_scrolled)
    {
	tb = GTK_TOGGLE_BUTTON(lookup_widget(win_prefs, "PlaybackScrolled"));
	gtk_toggle_button_set_active(tb, TRUE);
    }

    if (prefs.selection_bond)
    {
	tb = GTK_TOGGLE_BUTTON(lookup_widget(win_prefs, "BondChannelSelection"));
	gtk_toggle_button_set_active(tb, TRUE);
    }

    if (prefs.show_time)
    {
	tb = GTK_TOGGLE_BUTTON(lookup_widget(win_prefs, "ShowTime"));
	gtk_toggle_button_set_active(tb, TRUE);
    }

    if (prefs.disable_undo)
    {
	tb = GTK_TOGGLE_BUTTON(lookup_widget(win_prefs, "DisableUndo"));
	gtk_toggle_button_set_active(tb, TRUE);
    }

    if (prefs.remember_open_dir)
    {
	tb = GTK_TOGGLE_BUTTON(lookup_widget(win_prefs, "RememberOpenDir"));
	gtk_toggle_button_set_active(tb, TRUE);
    }

    /* recording defaults */
    sprintf(widget_name, "_%dRButton", (int)prefs.record.sample_rate);
    tb = GTK_TOGGLE_BUTTON(lookup_widget(win_prefs, widget_name));
    gtk_toggle_button_set_active(tb, TRUE);

    sprintf(widget_name, "_%dBitRButton", (int)prefs.record.sample_bits);
    tb = GTK_TOGGLE_BUTTON(lookup_widget(win_prefs, widget_name));
    gtk_toggle_button_set_active(tb, TRUE);

    if (prefs.record.channels == 1)
	tb = GTK_TOGGLE_BUTTON(lookup_widget(win_prefs, "MonoRButton"));
    else
	tb = GTK_TOGGLE_BUTTON(lookup_widget(win_prefs, "StereoRButton"));
    gtk_toggle_button_set_active(tb, TRUE);


    /* display fmt defaults */
    sprintf(widget_name, "DisplayFmt%d", (int)prefs.display_fmt);
    tb = GTK_TOGGLE_BUTTON(lookup_widget(win_prefs, widget_name));
    gtk_toggle_button_set_active(tb, TRUE);


    /* toolbars */
    if (prefs.tb_transport)
    {
	tb = GTK_TOGGLE_BUTTON(lookup_widget(win_prefs, "TransportTB"));
	gtk_toggle_button_set_active(tb, TRUE);
    }

    if (prefs.tb_markers)
    {
	tb = GTK_TOGGLE_BUTTON(lookup_widget(win_prefs, "MarkersTB"));
	gtk_toggle_button_set_active(tb, TRUE);
    }

    nb = GTK_NOTEBOOK(lookup_widget(win_prefs, "PrefsNotebook"));
    gtk_notebook_set_page(nb, (gint)user_data);
    gtk_widget_show(win_prefs);
}

void
on_dlgprefs_close(GtkButton *button, gpointer user_data)
{
    if (win_prefs)
	gtk_widget_destroy(win_prefs);
    win_prefs = NULL;
}


void
on_dlgprefs_apply(GtkButton *button, gpointer user_data)
{
    prefs_apply(button, user_data);
    on_dlgprefs_close(button, user_data);
}


void
on_dlgprefs_save(GtkButton *button, gpointer user_data)
{
    prefs_apply(button, user_data);
    prefs_save();
    on_dlgprefs_close(button, user_data);
}

gboolean
on_dlgprefs_destroy(GtkWidget *widget, GdkEvent *event, gpointer user_data)
{
    win_prefs = NULL;
    return FALSE;
}



static void
prefs_apply(GtkButton *button, gpointer user_data)
{
    gint			i;
    gint			bit[] = {8, 16};
    gint			rate[] = {8000, 11025, 22050, 44100};
    gchar			widget_name[80];
    GtkToggleButton		*tb;

    memset(&prefs, 0, sizeof(prefs));

    /* general prefs */
    tb = GTK_TOGGLE_BUTTON(lookup_widget(win_prefs, "SaveDCache"));
    if (tb->active)
	prefs.dcache_save = TRUE;
    tb = GTK_TOGGLE_BUTTON(lookup_widget(win_prefs, "TrackPlayback"));
    if (tb->active)
	prefs.playback_tracked = TRUE;
    tb = GTK_TOGGLE_BUTTON(lookup_widget(win_prefs, "PlaybackScrolled"));
    if (tb->active)
	prefs.playback_scrolled = TRUE;
    tb = GTK_TOGGLE_BUTTON(lookup_widget(win_prefs, "BondChannelSelection"));
    if (tb->active)
	prefs.selection_bond = TRUE;
    tb = GTK_TOGGLE_BUTTON(lookup_widget(win_prefs, "ShowTime"));
    if (tb->active)
    {
	prefs.show_time = TRUE;
	queue_cmd(CMD_WIDGET_SHOW, "Timebar");
    } else {
	queue_cmd(CMD_WIDGET_HIDE, "Timebar");
    }
    tb = GTK_TOGGLE_BUTTON(lookup_widget(win_prefs, "DisableUndo"));
    if (tb->active)
	prefs.disable_undo = TRUE;
    tb = GTK_TOGGLE_BUTTON(lookup_widget(win_prefs, "RememberOpenDir"));
    if (tb->active)
	prefs.remember_open_dir = TRUE;

    /* recording defaults */
    /* determine sample rate */
    for(i=0; i < sizeof(rate)/sizeof(rate[0]); i++)
    {
	sprintf(widget_name, "_%dRButton", rate[i]);
	tb = GTK_TOGGLE_BUTTON(lookup_widget(win_prefs, widget_name));
	if (tb && tb->active)
	    break;
    }
    prefs.record.sample_rate = rate[i];

    /* determine bits */
    for(i=0; i < sizeof(bit)/sizeof(bit[0]); i++)
    {
	sprintf(widget_name, "_%dBitRButton", bit[i]);
	tb = GTK_TOGGLE_BUTTON(lookup_widget(win_prefs, widget_name));
	if (tb && tb->active)
	    break;
    }
    prefs.record.sample_bits = bit[i];

    /* determine channels */
    tb = GTK_TOGGLE_BUTTON(lookup_widget(win_prefs, "MonoRButton"));
    if (tb && tb->active)
	prefs.record.channels = 1;
    else
	prefs.record.channels = 2;


    /* display fmt */
    for(i=1; i < SMPL_FMT_LAST; i++)
    {
	sprintf(widget_name, "DisplayFmt%d", i);
	tb = GTK_TOGGLE_BUTTON(lookup_widget(win_prefs, widget_name));
	if (tb && tb->active)
	    break;
    }
    prefs.display_fmt = i;


    /* toolbars */
    tb = GTK_TOGGLE_BUTTON(lookup_widget(win_prefs, "TransportTB"));
    if (tb->active)
    {
	prefs.tb_transport = TRUE;
	queue_cmd(CMD_WIDGET_SHOW, "TransportTB");
    } else {
	queue_cmd(CMD_WIDGET_HIDE, "TransportTB");
    }

    tb = GTK_TOGGLE_BUTTON(lookup_widget(win_prefs, "MarkersTB"));
    if (tb->active)
    {
	prefs.tb_markers = TRUE;
	queue_cmd(CMD_WIDGET_SHOW, "MarkersTB");
    } else {
	queue_cmd(CMD_WIDGET_HIDE, "MarkersTB");
    }
}



gboolean
prefs_save(void)
{
    gchar		*filename;
    gchar		buf[80];
    int			fd;
    int			i;
    const prefs_sym_t	*sym;
    gboolean		rc = FALSE;

    filename = g_strconcat(g_get_home_dir(), "/.gnoise/gnoiserc", NULL);
    fd = open(filename, O_WRONLY | O_CREAT | O_TRUNC, 0644);
    if (fd < 0)
    {
	log("PREFS", "Unable to open %s %s\n", filename, strerror(errno));
	goto out1;
    }

    for(sym = prefs_symtable, i=0; i < PREFS_COUNT; sym++, i++)
    {
	switch(sym->datasz)
	{
	    case 1:
		snprintf(buf, sizeof(buf), "%s=%d\n", sym->name,
			*(gint8 *)((char *)&prefs+sym->offset));
		break;
	    case 2:
		snprintf(buf, sizeof(buf), "%s=%d\n", sym->name,
			*(gint16 *)((char *)&prefs+sym->offset));
		break;
	    case 4:
		snprintf(buf, sizeof(buf), "%s=%d\n", sym->name,
			*(gint32 *)((char *)&prefs+sym->offset));
		break;
	}

	if (write(fd, buf, strlen(buf)) != strlen(buf))
	{
	    log("PREFS", "Unable to write %s %s\n", filename, strerror(errno));
	    goto out2;
	}
    }

    rc = TRUE;

out2:
    close(fd);
out1:
    g_free(filename);
    return rc;
}



gboolean
prefs_load(void)
{
    int			fd;
    int			size;
    int			i;
    int			offset;
    gchar		buf[80];
    gchar		*filename;
    gchar		*ch;
    gboolean		rc = FALSE;
    const prefs_sym_t	*sym;

    /* initialize to defaults in case loading fails */
    prefs.record.channels = 2;
    prefs.record.sample_rate = 44100;
    prefs.record.sample_bits = 16;
    prefs.dcache_save = 1;
    prefs.playback_tracked = 1;
    prefs.playback_scrolled = 1;
    prefs.selection_bond = 1;
    prefs.show_time = 1;
    prefs.disable_undo = 0;
    prefs.remember_open_dir = 1;
    prefs.display_fmt = SMPL_FMT_HHMMSS_SSS;
    prefs.tb_transport = 1;
    prefs.tb_markers = 0;

    filename = g_strconcat(g_get_home_dir(), "/.gnoise/gnoiserc", NULL);
    fd = open(filename, O_RDONLY | O_NONBLOCK);
    if (fd < 0)
    {
	log("PREFS", "Unable to open %s %s, using defaults\n", filename, strerror(errno));
	goto out1;
    }

    for(offset = 0;;)
    {
	size = read(fd, buf, sizeof(buf)-1);
	if (size < 0)
	{
	    log("PREFS", "Unable to read %s %s\n", filename, strerror(errno));
	    goto out2;
	}
	buf[size] = 0;

	if (!(ch = strchr(buf, '\n')) || size == 0)
	    break;
	*ch = 0;
	offset += ch-buf+1;
	lseek(fd, offset, SEEK_SET);

	if (!(ch = strchr(buf, '=')))
	    break;
	*ch++ = 0;

	/* now buf has name, ch points to value */

	for(sym = prefs_symtable, i=0; i < PREFS_COUNT; sym++, i++)
	{
	    if (strcmp(sym->name, buf))
		continue;

	    switch(sym->datasz)
	    {
		case 1:
		    *(gint8 *)((char *)&prefs+sym->offset) = strtol(ch, NULL, 0);
		    break;
		case 2:
		    *(gint16 *)((char *)&prefs+sym->offset) = strtol(ch, NULL, 0);
		    break;
		case 4:
		    *(gint32 *)((char *)&prefs+sym->offset) = strtol(ch, NULL, 0);
		    break;
	    }
	}
    }

    rc = TRUE;

out2:
    close(fd);
out1:
    g_free(filename);

    return rc;
}
