/*
**  BounceWindowController.m
**
**  Copyright (c) 2001, 2002, 2003
**
**  Author: Ludovic Marcotte <ludovic@Sophos.ca>
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or
**  (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "BounceWindowController.h"

#ifndef MACOSX
#include "BounceWindow.h"
#endif

#include "GNUMail+TaskManager.h"
#include "Constants.h"
#include "Task.h"
#include "Utilities.h"

#include <Pantomime/Constants.h>
#include <Pantomime/InternetAddress.h>
#include <Pantomime/Message.h>
#include <Pantomime/MimeUtility.h>
#include <Pantomime/NSData+Extensions.h>
#include <Pantomime/NSString+Extensions.h>
#include <Pantomime/Parser.h>

#include <time.h>


//
//
//
@implementation BounceWindowController

- (id) initWithWindowNibName: (NSString *) windowNibName
{
  NSDictionary *allAccounts;
#ifndef MACOSX
  BounceWindow *theBounceWindow;
#endif
  
  allAccounts = [Utilities allEnabledAccounts];

  // We first verify if we have at least one transport agent defined
  if ( !allAccounts || [allAccounts count] == 0 )
    {
      NSRunAlertPanel(_(@"Error!"),
		      _(@"You must have at least one transport agent defined and enabled.\nSee Preferences -> Account."),
		      _(@"OK"), // default
		      NULL,     // alternate
		      NULL);
      
      AUTORELEASE(self);
      return nil;
    }

#ifdef MACOSX
  
  self = [super initWithWindowNibName: windowNibName];
  
#else

  theBounceWindow = [[BounceWindow alloc] initWithContentRect: NSMakeRect(100,100,514,165)
				       styleMask: NSTitledWindowMask|NSResizableWindowMask|NSClosableWindowMask
				       backing: NSBackingStoreBuffered
				       defer: NO];
  
  self = [super initWithWindow: theBounceWindow];
  
  [theBounceWindow layoutWindow];
  [theBounceWindow setDelegate: self];
  
  // We link our outlets
  sendButton = theBounceWindow->sendButton;
  addressesButton = theBounceWindow->addressesButton;
  toField = theBounceWindow->toField;
  ccField = theBounceWindow->ccField;
  bccField = theBounceWindow->bccField;
  accountPopUpButton = theBounceWindow->accountPopUpButton;
  transportMethodPopUpButton = theBounceWindow->transportMethodPopUpButton;

  RELEASE(theBounceWindow);
#endif

  [[self window] setTitle: _(@"Bounce a message")];
  
  // We load our personal profiles and our transport methods
  [Utilities loadAccountsInPopUpButton: accountPopUpButton  select: nil];
  [Utilities loadTransportMethodsInPopUpButton: transportMethodPopUpButton];
 
  [self accountSelectionHasChanged: nil];

  // We finally set our autosave window frame name and restore the one from the user's defaults.
  [[self window] setFrameAutosaveName: @"BounceWindow"];
  [[self window] setFrameUsingName: @"BounceWindow"];
 
  return self;
}


//
//
//
#ifdef MACOSX
- (void) awakeFromNib
{
  NSToolbar *aToolbar;
  
  aToolbar = [[NSToolbar alloc] initWithIdentifier: @"BounceWindowToolbar"];
  [aToolbar setDelegate: self];
  [aToolbar setAllowsUserCustomization: YES];
  [aToolbar setAutosavesConfiguration: YES];
  [[self window] setToolbar: aToolbar];
  RELEASE(aToolbar);
}
#endif


//
//
//
- (void) dealloc
{
  NSDebugLog(@"BounceWindowController: -dealloc");

  // We remove our observer for our two notifications
  [[NSNotificationCenter defaultCenter]
    removeObserver: self
    name: AccountsHaveChanged
    object: nil];

#ifdef MACOSX
  RELEASE(sendButton);
  RELEASE(addressesButton);
#endif

  RELEASE(message);

  [super dealloc];
}


//
// action methods
//
- (IBAction) send : (id) sender
{
  NSMutableData *aMessageAsData;
  NSData *rawSource, *aData;
  NSAutoreleasePool *pool;
  NSDictionary *allValues;
  NSRange aRange;

  Task *aTask;
  int op;

  // We create our local pool
  pool = [[NSAutoreleasePool alloc] init];
  
  // We create our mutable string
  aMessageAsData = [[NSMutableData alloc] init];

  // We get the raw source of the message
  rawSource = [[self message] rawSource];
  
  // We get our headers delimiter
  aRange = [rawSource rangeOfCString: "\n\n"];

  if (aRange.length == 0)
    {
      NSRunAlertPanel(_(@"Error!"),
  		      _(@"Unable to create a valid bounced message.\nPlease, report this as a bug."),
  		      _(@"OK"),
  		      NULL,
  		      NULL);
      return;
    }
  
  // We append the content of our headers
  aData = [rawSource subdataToIndex: aRange.location + 1];

  // If we have our "From " separator, we remove it since we don't want to send
  // this by SMTP.
  if ( [aData hasCPrefix: "From "] )
    {
      NSRange r;
      
      r = [aData rangeOfCString: "\n"];

      if (r.length > 0)
	{
	  aData = [aData subdataWithRange: NSMakeRange(r.location + 1, [aData length] - r.location - 1)];
	}
    }

  // We append all our headers
  [aMessageAsData appendData: aData];

  // We append our set of new headers
  if ( ![self _appendAdditionalHeadersToData: aMessageAsData] )
    {
      NSRunAlertPanel(_(@"Error!"),
  		      _(@"Please, verify that the To/Cc/Bcc fields are correct. Also, you must have at\nleast one valid E-Mail address in the To field."),
  		      _(@"OK"),
  		      NULL,
  		      NULL);
      return;
    }
  
  // We sync our popup
  [transportMethodPopUpButton synchronizeTitleAndSelectedItem];

  // We get our transport method type
  allValues = [[[[NSUserDefaults standardUserDefaults] objectForKey: @"ACCOUNTS"]
		 objectForKey: [Utilities accountNameForTransportMethodItemTitle: [transportMethodPopUpButton titleOfSelectedItem]]]
		objectForKey: @"SEND"];
  
  if ( [[allValues objectForKey: @"TRANSPORT_METHOD"] intValue] == TRANSPORT_SMTP )
    {
      op = SEND_SMTP;
    }
  else
    {
      op = SEND_SENDMAIL;
    }

  // We append our header delimiter
  [aMessageAsData appendCString: "\n"];

  // We finally append the content of our message
  [aMessageAsData appendData: [rawSource subdataFromIndex: aRange.location + 2]];
  
  aTask = [[Task alloc] init];
  [aTask setOp: op];
  [aTask setMessage: aMessageAsData];
  [aTask setKey:  [Utilities accountNameForItemTitle: [accountPopUpButton titleOfSelectedItem]]];
  [aTask setSendingKey: [Utilities accountNameForTransportMethodItemTitle: [transportMethodPopUpButton titleOfSelectedItem]]];
  [[NSApp delegate] addTask: aTask];
  
  RELEASE(aTask);
  
  RELEASE(aMessageAsData);
  RELEASE(pool);

  // Since we're done, we close our window.
  [self close];
}


//
//
//
- (IBAction) accountSelectionHasChanged: (id) sender
{
  NSDictionary *theAccount;
  NSString *aString;
  NSRange aRange;
  int i;

  // We synchronize our selection from the popup
  [accountPopUpButton synchronizeTitleAndSelectedItem];

  theAccount = [[[NSUserDefaults standardUserDefaults] objectForKey: @"ACCOUNTS"]
		 objectForKey: [Utilities accountNameForItemTitle: [accountPopUpButton titleOfSelectedItem]]];
  
  for (i = 0; i < [transportMethodPopUpButton numberOfItems]; i++)
    {
      if ( [[[theAccount objectForKey: @"SEND"] objectForKey: @"TRANSPORT_METHOD"] intValue] == TRANSPORT_MAILER )
	{
	  aString = [[theAccount objectForKey: @"SEND"] objectForKey: @"MAILER_PATH"];
	}
      else
	{
	  aString = [[theAccount objectForKey: @"SEND"] objectForKey: @"SMTP_HOST"];
	}

      aRange = [[transportMethodPopUpButton itemTitleAtIndex: i] rangeOfString: aString];

      if ( aRange.length )
	{
	  [transportMethodPopUpButton selectItemAtIndex: i];
	  [transportMethodPopUpButton synchronizeTitleAndSelectedItem];
	  return;
	}
    }
  
  [transportMethodPopUpButton selectItemAtIndex: 0];
  [transportMethodPopUpButton synchronizeTitleAndSelectedItem];
}


//
// delegate methods
//
- (void) windowWillClose: (NSNotification *) theNotification
{
  if ( [GNUMail lastAddressTakerWindowOnTop] == self )
    {
      [GNUMail setLastAddressTakerWindowOnTop: nil];
    }

  AUTORELEASE(self);
}


//
//
//
- (void) windowDidBecomeMain: (NSNotification *) theNotification;
{
  [GNUMail setLastAddressTakerWindowOnTop: self];
  
  // Make the "Deliver" menu item send the mail
  [[[NSApp delegate] deliverMenuItem] setTarget: self];
  [[[NSApp delegate] deliverMenuItem] setAction: @selector(send:)];
}


//
//
//
- (void) windowDidResignMain: (NSNotification *) theNotification
{
  // We disable our Deliver menu item.
  [[[NSApp delegate] deliverMenuItem] setTarget: nil];
  [[[NSApp delegate] deliverMenuItem] setAction: NULL];
}


//
//
//
- (void) windowDidLoad
{
#ifdef MACOSX
  [addressesButton setTarget: [NSApp delegate]];
  [addressesButton setAction: @selector(showAddressBook:)];
#endif

  // We add our observer for our notification
  [[NSNotificationCenter defaultCenter]
    addObserver: self
    selector: @selector(_loadAccounts)
    name: AccountsHaveChanged
    object: nil];
}


//
// access/mutation methods
//
- (Message *) message
{
  return message;
}


//
//
//
- (void) setMessage: (Message *) theMessage
{
  RETAIN(theMessage);
  RELEASE(message);
  message = theMessage;
}


//
//
//
- (void) takeToAddress: (NSArray *) theAddress
{
  [Utilities appendAddress: theAddress
	     toTextField: toField];
}


//
//
//
- (void) takeCcAddress: (NSArray *) theAddress
{
  [Utilities appendAddress: theAddress
	     toTextField: ccField];
}


//
//
//
- (void) takeBccAddress: (NSArray *) theAddress
{
  [Utilities appendAddress: theAddress
	     toTextField: bccField];
}

@end


//
// private methods
//
@implementation BounceWindowController (Private)

- (BOOL) _appendAdditionalHeadersToData: (NSMutableData *) theData
{
  InternetAddress *anInternetAddress;
  NSCalendarDate *aCalendarDate;
  NSDictionary *allValues;

  NSDictionary *aLocale;

  // We first verify if at least a recipient To has been define
  if ( ![[[toField stringValue] stringByTrimmingWhiteSpaces] length] )
    {
      return NO;
    }
  
  // We get our locale in English
#ifndef MACOSX
  aLocale = [NSDictionary dictionaryWithContentsOfFile: [[NSBundle bundleForLibrary: @"gnustep-base"]
							  pathForResource: @"English"
							  ofType: nil
							  inDirectory: @"Languages"]];
#else
  aLocale = [NSDictionary dictionaryWithContentsOfFile: [[NSBundle bundleForClass:[NSObject class]]
							  pathForResource: @"English"
							  ofType: nil
							  inDirectory: @"Languages"] ];
#endif

  // We set the Resent-Date
#ifndef MACOSX
  tzset();
  
  aCalendarDate = [[[NSDate alloc] init] dateWithCalendarFormat:@"%a, %d %b %Y %H:%M:%S %z"
					 timeZone: [NSTimeZone timeZoneWithAbbreviation: 
								 [NSString stringWithCString: tzname[1]]] ];
#else
  aCalendarDate = [[[NSDate alloc] init] dateWithCalendarFormat:@"%a, %d %b %Y %H:%M:%S %z"
					 timeZone: [NSTimeZone systemTimeZone] ];
#endif
  
  [theData appendCFormat: @"Resent-Date: %@\n", [aCalendarDate descriptionWithLocale: aLocale]];
  
  // We get our account from our user defaults
  allValues = [[[[NSUserDefaults standardUserDefaults] objectForKey: @"ACCOUNTS"]
		 objectForKey: [Utilities accountNameForItemTitle: [accountPopUpButton titleOfSelectedItem]]]
		objectForKey: @"PERSONAL"];
  
  // We set the Resent-From
  anInternetAddress = [[InternetAddress alloc] initWithPersonal: [allValues objectForKey: @"NAME"]
					       andAddress: [allValues objectForKey: @"EMAILADDR"]];
  
  [theData appendCString: "Resent-From: "];
  [theData appendData: [anInternetAddress dataValue]];
  [theData appendCString: "\n"];
  RELEASE(anInternetAddress);
  
  // We set the Resent-To
  [theData appendCString: "Resent-To: "];
  [theData appendData: [[toField stringValue] dataUsingEncoding: NSASCIIStringEncoding]];
  [theData appendCString: "\n"];
    
  // We set the ReSent-Cc, if we need to.
  if ( [[[ccField stringValue] stringByTrimmingWhiteSpaces] length] )
    {
      [theData appendCString: "Resent-Cc: "];
      [theData appendData: [[ccField stringValue] dataUsingEncoding: NSASCIIStringEncoding]];
      [theData appendCString: "\n"];
    }

  // We set the ReSent-Bcc, if we need to.
  if ( [[[bccField stringValue] stringByTrimmingWhiteSpaces] length] )
    {
      [theData appendCString: "Resent-Bcc: "];
      [theData appendData: [[bccField stringValue] dataUsingEncoding: NSASCIIStringEncoding]];
      [theData appendCString: "\n"];
    }

  // We set the ReSent-Message-ID
  [theData appendCString: "Resent-Message-ID: <"];
  [theData appendData: [MimeUtility generateOSID]];
  [theData appendCString: ">\n"];
  
  return YES;
}


//
//
//
- (void) _loadAccounts
{
  [Utilities loadAccountsInPopUpButton: accountPopUpButton 
	     select: nil];
  
  [Utilities loadTransportMethodsInPopUpButton: transportMethodPopUpButton];
}

@end
