/*
 * Copyright (C) 2002,2003 Pascal Haakmat.
 */

/* still using GtkCList: */
#undef GTK_DISABLE_DEPRECATED

#include <math.h>
#include <errno.h>
#include <dlfcn.h>
#include <dirent.h>
#include <ctype.h>
#include <wordexp.h>
#include <gtk/gtk.h>
#include <glade/glade.h>
#include "../pref.h"
#include "../modutils.h"
#include "ladspa.h"

#define MAX_AUDIO_INPUTS 10
#define MAX_AUDIO_OUTPUTS 10
#define MAX_CONTROLS 60
#define MAX_PRESETS 20
#define MAX_PLUGINS 200

#ifdef HAVE_GNOME2
#define LADSPA_GLADE_FILE  "ladspa-2.glade"
#else
#define LADSPA_GLADE_FILE  "ladspa.glade"
#endif

module modinfo;

struct ladspa_module {
    char *filename;
    void *handle;
    LADSPA_Descriptor *desc;
};

struct ladspa_module ladspa_modules[MAX_PLUGINS];
int ladspa_count = 0;
const char *ladspa_path;

typedef struct {
    module_id id;
    shell *shl;
    GtkWidget *dialog;
    GtkCList *clist;
    GtkTable *controls_table;
    GtkCombo *presets;
    GtkViewport *viewport;
    GtkLabel *author;
    GtkLabel *copyright;
    GtkLabel *audio_io_ports;
    GtkTooltips *tips;
    LADSPA_Descriptor *currently_selected;
    LADSPA_Handle *handle;
    int controls_count;
    GtkWidget *controls[MAX_CONTROLS];
    GtkWidget *envelope1[MAX_CONTROLS];
    GtkWidget *envelope2[MAX_CONTROLS];
    LADSPA_Data control_values[MAX_CONTROLS];
    LADSPA_Data control_min[MAX_CONTROLS];
    LADSPA_Data control_max[MAX_CONTROLS];
    int envelopes_enabled[MAX_CONTROLS];
    int envelope1_connect_count;
    LADSPA_Data *envelope1_connections[MAX_CONTROLS];
    LADSPA_Data envelope1_defaults[MAX_CONTROLS];
    LADSPA_Data envelope1_min[MAX_CONTROLS];
    LADSPA_Data envelope1_max[MAX_CONTROLS];
    int envelope2_connect_count;
    LADSPA_Data *envelope2_connections[MAX_CONTROLS];
    LADSPA_Data envelope2_defaults[MAX_CONTROLS];
    LADSPA_Data envelope2_min[MAX_CONTROLS];
    LADSPA_Data envelope2_max[MAX_CONTROLS];
    int audio_input_count;
    int audio_input_map[MAX_AUDIO_INPUTS];
    int audio_output_count;
    int audio_output_map[MAX_AUDIO_OUTPUTS];
    LADSPA_Data *in_buf[MAX_AUDIO_INPUTS];
    LADSPA_Data *out_buf[MAX_AUDIO_OUTPUTS];
} ladspa_params;

void
ladspa_process_core(ladspa_params *p,
                    AFframecount frame_offset,
                    AFframecount frame_count,
                    struct marker_list *ml) {
    AFframecount seg_count, seg_remain = frame_count, seg_offset = 0;
    LADSPA_Descriptor *desc = p->currently_selected;
    double adjust = 0, delta;
    int i, seg_size = (int)(pref_get_as_float("ladspa_segment_time") *
        p->shl->sr->rate);
    
    for(seg_count = MIN(seg_remain, seg_size); 
        seg_remain; 
        seg_count = MIN(seg_remain, seg_size)) {
        
        /* Point audio ports at correct positions. */
        
        for(i = 0; i < p->audio_input_count; i++)
            desc->connect_port(p->handle, p->audio_input_map[i],
                               &p->in_buf[i][seg_offset]);
        
        for(i = 0; i < p->audio_output_count; i++)
            desc->connect_port(p->handle, p->audio_output_map[i], 
                               &p->out_buf[i][seg_offset]);
        
        /* Twiddle control ports via envelopes. */
        
        if(p->envelope1_connect_count)
            adjust = marker_list_slope_value(ml, frame_offset + seg_offset,
                                             MARKER_SLOPE);
        for(i = 0; i < p->envelope1_connect_count; i++) {
            if(adjust > 0) 
                delta = p->envelope1_max[i] - p->envelope1_defaults[i];
            else 
                delta = p->envelope1_defaults[i] - p->envelope1_min[i];
            *(p->envelope1_connections[i]) =
                p->envelope1_defaults[i] + (adjust * delta);
            //            DEBUG("min: %f, max: %f, default: %f, value: %f\n", 
            //                  p->envelope1_min[i], p->envelope1_max[i], p->envelope1_defaults[i], *(p->envelope1_connections[i]));
            
        }

        if(p->envelope2_connect_count)
            adjust = marker_list_slope_value(ml, frame_offset + seg_offset,
                                             MARKER_SLOPE_AUX);
        
        for(i = 0; i < p->envelope2_connect_count; i++) {
            if(adjust > 0) 
                delta = p->envelope2_max[i] - p->envelope2_defaults[i];
            else 
                delta = p->envelope2_defaults[i] - p->envelope2_min[i];
            *(p->envelope2_connections[i]) =
                p->envelope2_defaults[i] + (adjust * delta);
            //            DEBUG("min: %f, max: %f, default: %f, value: %f\n", 
            //                  p->envelope2_min[i], p->envelope2_max[i], p->envelope2_defaults[i], *(p->envelope2_connections[i]));
            
        }


        for(i = 0; i < p->envelope2_connect_count; i++) 
            *(p->envelope2_connections[i]) =
                p->envelope2_defaults[i] + (adjust * p->envelope2_defaults[i]);
        
        desc->run(p->handle, seg_count);
        
        seg_offset += seg_count;
        seg_remain -= seg_count;
    }
}

/*
 * Process a multichannel run, that is apply the plugin to the 
 * selected range on all selected channels simultaneously.
 */
void 
ladspa_process_many(shell *shl,
                    ladspa_params *p,
                    track_map_t map,                    
                    AFframecount start, 
                    AFframecount end) {
    AFframecount frame_offset = start, frame_count = end - start, 
        frame_count_old = frame_count, frames_processed = 0, rmax, 
        r[MAX_AUDIO_INPUTS], count;
    int t, i, err, envelope_track = 0;

    for(t = 0; t < shl->sr->channels; t++) {
        if((1 << t) & map) {
            envelope_track = t;
            break;
        }
    }
    
    do {
        count = MIN(frame_count, EFFECT_BUF_SIZE);
        rmax = 0;
        
        for(t = 0, i = 0; (i < p->audio_input_count && 
                           t < shl->sr->channels); t++) {
            if((1 << t) & map) {                
                r[i] = track_float_frames_get(shl->sr->tracks[t],
                                              (float *)(p->in_buf[i]),
                                              frame_offset,
                                              count);
              
                if(r[i] != count) 
                    /* Got fewer frames than expected so clear buffer
                       until end. */
                    memset(&p->in_buf[i][r[i]], 
                           (count - r[i]) * sizeof(LADSPA_Data), '\0');
                if(r[i] > rmax)
                    rmax = r[i];
                i++;
            }
        }
        
        if(!rmax)
            break;

        ladspa_process_core(p, frame_offset, rmax, 
                            shl->sr->markers[envelope_track]);
        gui_yield(); 
        
        for(t = 0, i = 0; (i < p->audio_output_count &&
                           t < shl->sr->channels); t++) {
            if((1 << t) & map) {
                err = track_float_frames_replace(shl->sr->tracks[t],
                                                 (float *)(p->out_buf[i]),
                                                 frame_offset,
                                                 rmax);
                //                DEBUG("replacing %ld frames at %ld on track %d from buf %d, err: %d\n",
                //                      rmax, frame_offset, t, i, err);
                i++;
            }
        }
        
        frame_offset += rmax;
        frame_count -= rmax;
        frames_processed += rmax;
        gtk_progress_bar_set_fraction((shl)->progress,  
                                      CLAMP((float)frames_processed / 
                                            (float)(frame_count_old), 0, 1));
    } while(!shl->module_cancel_requested && rmax > 0 && frame_count > 0);
    
}

/*
 * Process a mono run, that is apply the plugin to the selected range
 * on every selected track in succession.
 */
void
ladspa_process_1on1(shell *shl,
                    ladspa_params *p,
                    track_map_t map,                    
                    AFframecount start, 
                    AFframecount end) {
    AFframecount frame_offset = start, frame_count = end - start, 
        frame_count_old = frame_count, frames_processed = 0, r, count;
    int t;

    for(t = 0; t < shl->sr->channels; t++) {
        if((1 << t) & map) {
            
            frame_offset = start;
            frame_count = end - start;
            frame_count_old = frame_count;
            frames_processed = 0;
            
            do {
                count = frame_count > EFFECT_BUF_SIZE ? 
                    EFFECT_BUF_SIZE : frame_count;
                
                if(p->in_buf[0])
                    r = track_float_frames_get(shl->sr->tracks[t],
                                               (float *)(p->in_buf[0]),
                                               frame_offset,
                                               count);
                else
                    r = count;
                
                ladspa_process_core(p, frame_offset, r,
                                    shl->sr->markers[t]);                
                track_float_frames_replace(shl->sr->tracks[t],
                                           (float *)(p->out_buf[0]),
                                           frame_offset,
                                           r);
                frame_offset += r;
                frame_count -= r;
                frames_processed += r;
                gtk_progress_bar_set_fraction((shl)->progress,  
                                              CLAMP((float)frames_processed / 
                                                    (float)(frame_count_old), 0, 1));
                gui_yield(); 
            } while(!shl->module_cancel_requested &&
                    r > 0 && frame_count > 0);
        }
    }

}



int
ladspa_get_port_count_by_type(const LADSPA_Descriptor *psDescriptor,
                              const LADSPA_PortDescriptor iType) {
    unsigned long lCount;
    unsigned long lIndex;
  
    lCount = 0;
    for (lIndex = 0; lIndex < psDescriptor->PortCount; lIndex++)
        if ((psDescriptor->PortDescriptors[lIndex] & iType) == iType)
            lCount++;
  
    return lCount;
}

void
ladspa_load(const char *path) {
    void *handle;
    char *error;
    int i, audio_output_count, audio_input_count;
    LADSPA_Descriptor *desc;
    LADSPA_Descriptor *(*ladspa_desc)(unsigned long i);

    handle = dlopen(path, RTLD_NOW);
    if (!handle) {
        //        FAIL("could not open %s: %s\n", path, dlerror());
        return;
    }

    ladspa_desc = dlsym(handle, "ladspa_descriptor");
    if((error = dlerror()) != NULL) {
        //        FAIL("could not open %s: %s\n", path, error);
        dlclose(handle);
        return;
    }

    for(i = 0; (desc = ladspa_desc(i)); i++) {
        audio_output_count = 
            ladspa_get_port_count_by_type(desc, 
                                          LADSPA_PORT_AUDIO | 
                                          LADSPA_PORT_OUTPUT);
        audio_input_count = 
            ladspa_get_port_count_by_type(desc, 
                                          LADSPA_PORT_AUDIO | 
                                          LADSPA_PORT_INPUT);

        /* If plugin is unsupported, skip it. */

        if(audio_output_count > MAX_AUDIO_OUTPUTS ||audio_input_count > MAX_AUDIO_INPUTS) {
            FAIL("LADSPA plugin %s (%s/%ld) not supported: too many inputs or outputs (%d inputs, %d outputs)\n", desc->Name, desc->Label, desc->UniqueID, audio_input_count, audio_output_count);
            continue;
        }

        /* Don't load too many plugins. */

        if(ladspa_count >= MAX_PLUGINS) {
            FAIL("LADSPA plugin %s (%s/%ld) not loaded because maximum amount of LADSPA plugins was reached, increase MAX_PLUGINS (loaded: %d, max: %d).\n", desc->Name, desc->Label, desc->UniqueID, ladspa_count, MAX_PLUGINS);
            continue;
        }

        ladspa_modules[ladspa_count].desc = desc;
        ladspa_modules[ladspa_count].handle = handle;
        ladspa_count++;
        /*
          DEBUG("\t%s (%lu/%s) %ld\n", 
          desc->Name,
          desc->UniqueID,
          desc->Label,
          desc->PortCount);
        */
    }
}

int
ladspa_compare(const void *a, 
               const void *b) {
    return strcmp(((struct ladspa_module *)a)->desc->Name, 
                  ((struct ladspa_module *)b)->desc->Name);
}

void
ladspa_collect(const char *path) {
    char tmp[4096];
    struct dirent **namelist;
    int i, n;

    n = scandir(path, &namelist, 0, alphasort);
    if (n < 0) {
        FAIL("unable to load modules in %s: %s\n", path, strerror(errno));
        return;
    }

    for(i = 0; i < n; i++) {
        snprintf(tmp, 4096, "%s/%s", path, namelist[i]->d_name);
        free(namelist[i]);
        //	DEBUG("tmp: %s\n", tmp);
        ladspa_load(tmp);
    }

    free(namelist);

    qsort(ladspa_modules, ladspa_count, sizeof(struct ladspa_module), ladspa_compare);

    return;

}


void 
ladspa_init() {
    char *path, *p;
    wordexp_t exp;
    int r, i;
    ladspa_path = getenv("LADSPA_PATH");
    if(!ladspa_path) {
        ladspa_path = pref_get_as_string("ladspa_path");
        FAIL("Warning: You do not have a LADSPA_PATH "
             "environment variable set. Trying default %s.\n", ladspa_path);
    }
    path = strdup(ladspa_path);

    while((p = strsep(&path, ":"))) {
        r = wordexp(p, &exp, 0);
        if(r) {
            FAIL("unable to initalize LADSPA plugins in path %s, "
                 "invalid path?\n",
                 p);
            continue;
        }
        DEBUG("collecting LADSPA plugins in path %s\n", p);
        for(i = 0; i < exp.we_wordc; i++) 
            ladspa_collect(exp.we_wordv[i]);
        wordfree(&exp);
    }
    free(path);
}

module *
module_new() {

    ladspa_init();
    MODULE_INIT(&modinfo, 
		"LADSPA Plugins",
		"Pascal Haakmat",
		"Copyright (C) 2002,2003");
    return &modinfo;
}

void
module_destroy(module *mod) {
    int i;
    for(i = 0; i < ladspa_count; i++) {
        DEBUG("releasing %s\n",
              ladspa_modules[i].desc->Name);
        dlclose(ladspa_modules[i].handle);
    }
}

void
on_apply_clicked(GtkWidget *w,
                 gpointer user_data) {
    int i;
    struct _module_state *module_state = 
        g_object_get_data(G_OBJECT(gtk_widget_get_toplevel(w)),
                          "user_data");
    ladspa_params *p = (ladspa_params *)module_state->data;
    GtkAdjustment *adj;

    for(i = 0; i < p->controls_count; i++) {
        p->envelopes_enabled[i] = 0;
        if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(p->envelope1[i])))
            p->envelopes_enabled[i] |= MARKER_SLOPE;
        if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(p->envelope2[i])))
            p->envelopes_enabled[i] |= MARKER_SLOPE_AUX;
        if(GTK_IS_TOGGLE_BUTTON(p->controls[i]))
            p->control_values[i] = 
                gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(p->controls[i])) ? 0 : 1;
        else {
            adj = gtk_range_get_adjustment(GTK_RANGE(p->controls[i]));
            p->control_values[i] = adj->value;
        }
    }

    action_do(ACTION_MODULE_EXECUTE_NEW(WITH_UNDO, 
                                        p->shl,
                                        p->id));
}

void
on_ok_clicked(GtkWidget *w,
              gpointer user_data) {
    struct _module_state *module_state = 
        g_object_get_data(G_OBJECT(gtk_widget_get_toplevel(w)),
                          "user_data");
    ladspa_params *p = (ladspa_params *)module_state->data;
    on_apply_clicked(w, NULL);
    gtk_object_destroy(GTK_OBJECT(p->dialog));
}

void
on_close_clicked(GtkWidget *w,
                 gpointer user_data) {
    struct _module_state *module_state = 
        g_object_get_data(G_OBJECT(gtk_widget_get_toplevel(w)),
                          "user_data");
    ladspa_params *p = (ladspa_params *)module_state->data;
    gtk_object_destroy(GTK_OBJECT(p->dialog));
}

void
on_dialog_destroy(GtkWidget *w,
                  gpointer user_data) {
    struct _module_state *module_state = 
        g_object_get_data(G_OBJECT(gtk_widget_get_toplevel(w)),
                          "user_data");
    module_state->is_open = 0;
    free(module_state->data);
}

void
on_delete_preset_clicked(GtkWidget *w,
                         gpointer user_data) {
    /* FIXME: this function shares a lot of code with
       on_presets_entry_activate() below. */
    
    struct _module_state *module_state = 
        g_object_get_data(G_OBJECT(gtk_widget_get_toplevel(w)),
                          "user_data");
    ladspa_params *p = (ladspa_params *)module_state->data;
    LADSPA_Descriptor *desc = p->currently_selected;
    GList *children;
    GtkLabel *label;
    int i;
    int num_presets = 0;
    const char *s, *selected_preset = gtk_entry_get_text(GTK_ENTRY(p->presets->entry));
    char *name[MAX_PRESETS];
    char path[256];

    if(strcmp("", selected_preset) == 0)
        return;

    children = gtk_container_children(GTK_CONTAINER(p->presets->list));
    for(i = 0; i < MAX_PRESETS && children; children = children->next) {
        label = GTK_LABEL(GTK_BIN(children->data)->child);
#ifdef HAVE_GNOME2
        s = gtk_label_get_text(label);
#else
        gtk_label_get(label, (char **)&s);
#endif
        if(strcmp(s, "") == 0 || strcmp(s, selected_preset) == 0) 
            continue;
        name[i++] = strdup(s);
        num_presets = i;
    }

    children = NULL;
    for(i = 0; i < num_presets; i++) 
        children = g_list_append(children, name[i]);
    
    snprintf(path, 256, "/gnusound/%s/Presets", desc->Name);
    gnome_config_set_vector(path, num_presets, (const char * const *) name);
    gnome_config_sync();

    children = g_list_insert(children, "", 0);
    gtk_combo_set_popdown_strings(p->presets, children);
    
    for(i = 0; i < num_presets; i++)
        free(name[i]);
}

void
on_presets_list_select_child(GtkList *l,
                             GtkWidget *child,
                             gpointer user_data) {
    struct _module_state *module_state = 
        g_object_get_data(G_OBJECT(l),
                          "user_data");
    ladspa_params *p = (ladspa_params *)module_state->data;
    const char *s;
    int num_values, i;
    char **preset_values;
    char path[256];
    GList *items;
    GtkAdjustment *adjust;

#ifdef HAVE_GNOME2
    s = gtk_label_get_text(GTK_LABEL(GTK_BIN(child)->child));
#else
    gtk_label_get(GTK_LABEL(GTK_BIN(child)->child), (char **)&s);
#endif

    if(!p->currently_selected || strcmp("", s) == 0)
        return;

    /* Recall values for preset. */

    snprintf(path, 256, "/gnusound/%s/%s", p->currently_selected->Name, s);
    gnome_config_get_vector(path, &num_values, &preset_values);

    items = NULL;
    for(i = 0; i < p->controls_count && i < num_values; i++) {
        if(GTK_IS_TOGGLE_BUTTON(p->controls[i])) 
            gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(p->controls[i]), 
                                         atof(preset_values[i]) > 0);
        else {
            adjust = gtk_range_get_adjustment(GTK_RANGE(p->controls[i]));
            gtk_adjustment_set_value(adjust, atof(preset_values[i]));
        }
    }

    if(preset_values) {
        for(i = 0; i < num_values; i++)
            g_free(preset_values[i]);
        g_free(preset_values);
    }
}

void
on_presets_entry_activate(GtkEditable *w,
                          gpointer user_data) {
    struct _module_state *module_state = 
        g_object_get_data(G_OBJECT(gtk_widget_get_toplevel(GTK_WIDGET(w))),
                          "user_data");
    ladspa_params *p = (ladspa_params *)module_state->data;
    LADSPA_Descriptor *desc = p->currently_selected;
    GList *children;
    GtkLabel *label;
    GtkAdjustment *adjust;
    char path[256];
    const char *s;
    char *new_preset_name;
    char *name[MAX_PRESETS];
    char *values[MAX_CONTROLS];
    int i;
    int num_controls = ladspa_get_port_count_by_type(desc, 
                                                     LADSPA_PORT_CONTROL | 
                                                     LADSPA_PORT_INPUT);
    int num_presets = 1;

    s = gtk_entry_get_text(GTK_ENTRY(w));
    if(strcmp(s, "") == 0)
        return;

    /* Construct array of preset names. */

    new_preset_name = strdup(s);

    name[0] = new_preset_name;
    children = gtk_container_children(GTK_CONTAINER(p->presets->list));
    for(i = 1; i < MAX_PRESETS && children; children = children->next) {
        label = GTK_LABEL(GTK_BIN(children->data)->child);
#ifdef HAVE_GNOME2
        s = gtk_label_get_text(label);
#else
        gtk_label_get(label, (char **)&s);
#endif
        if(strcmp(s, "") == 0 || strcmp(s, new_preset_name) == 0) 
            continue;
        name[i++] = strdup(s);
        num_presets = i;
    }

    /* Make linked list from array. */

    for(i = 0; i < num_presets; i++) 
        children = g_list_append(children, name[i]);
    
    /* Save the array of preset names. */

    snprintf(path, 256, "/gnusound/%s/Presets", desc->Name);
    gnome_config_set_vector(path, num_presets, (const char * const *) name);

    /* Construct array of preset values. */

    for(i = 0; i < p->controls_count && i < num_controls; i++) {
        values[i] = mem_alloc(20);
        if(GTK_IS_TOGGLE_BUTTON(p->controls[i])) {
            snprintf(values[i], 20, "%d", 
                     gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(p->controls[i])) ? 1 : 0); 
        } else {
            adjust = gtk_range_get_adjustment(GTK_RANGE(p->controls[i]));
            snprintf(values[i], 20, "%f", adjust->value); 
        }
    }

    /* Save the array of preset values. */

    snprintf(path, 256, "/gnusound/%s/%s", desc->Name, new_preset_name);
    gnome_config_set_vector(path, num_controls, (const char * const *) values);
    gnome_config_sync();
    for(i = 0; i < num_controls; i++)
        mem_free(values[i]);

    /* Set the new array of preset names. */

    children = g_list_insert(children, "", 0);
    gtk_combo_set_popdown_strings(p->presets, children);

    for(i = 0; i < num_presets; i++) 
        mem_free(name[i]);    
}

void
make_widget_from_port(ladspa_params *p,
                      int port,
                      GtkWidget **v,
                      LADSPA_Data *defalt,
                      LADSPA_Data *min,
                      LADSPA_Data *max) {
    GtkAdjustment *adjust;
    LADSPA_Descriptor *desc = p->currently_selected;
    LADSPA_PortRangeHintDescriptor hint = desc->PortRangeHints[port].HintDescriptor;
    LADSPA_Data upper, lower, value = 0;
    double step_increment, page_increment, page_size;

    *defalt = 0;
    *min = -10000;
    *max = 10000;

    if(LADSPA_IS_HINT_TOGGLED(hint)) {
        *v = gtk_check_button_new_with_label(desc->PortNames[port]);
        if(LADSPA_IS_HINT_DEFAULT_0(hint)) {
            value = 0;
            gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(*v),
                                         FALSE);
        }
        if(LADSPA_IS_HINT_DEFAULT_1(hint)) {
            value = 1;
            gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(*v),
                                         TRUE);
        }
        *min = 0;
        *max = 1;
    } else {
        lower = 0;
        upper = 100000;
        value = 100;
        if(LADSPA_IS_HINT_BOUNDED_BELOW(hint)) 
            lower = desc->PortRangeHints[port].LowerBound;
        if(LADSPA_IS_HINT_BOUNDED_ABOVE(hint)) 
            upper = desc->PortRangeHints[port].UpperBound;
        
        if(LADSPA_IS_HINT_SAMPLE_RATE(hint)) {
            lower = lower * p->shl->sr->rate;
            upper = upper * p->shl->sr->rate;
        }
        
        step_increment = (upper - lower) / 1000;
        page_increment = (upper - lower) / 100;
        page_size = (upper - lower) / 100;
        
        if(LADSPA_IS_HINT_DEFAULT_MINIMUM(hint)) 
            value = lower;
        if(LADSPA_IS_HINT_DEFAULT_MAXIMUM(hint)) 
            value = upper;
        
        if(LADSPA_IS_HINT_DEFAULT_LOW(hint)) {
            if(LADSPA_IS_HINT_LOGARITHMIC(hint))
                value = lower * .75 + upper * .25;
            else 
                value = exp(log(lower) * .75 + log(upper) * .25);
        }
        if(LADSPA_IS_HINT_DEFAULT_MIDDLE(hint)) {
            if(LADSPA_IS_HINT_LOGARITHMIC(hint))
                value = lower * .5 + upper * .5;
            else
                value = exp(log(lower) * .5 + log(upper) * .5);
        }
        if(LADSPA_IS_HINT_DEFAULT_HIGH(hint)) {
            if(LADSPA_IS_HINT_LOGARITHMIC(hint))
                value = lower * .25 + upper * .75;
            else
                value = exp(log(lower) * .25 + log(upper) * .75);
        }
        if(LADSPA_IS_HINT_DEFAULT_0(hint)) 
            value = 0;
        if(LADSPA_IS_HINT_DEFAULT_1(hint)) 
            value = 1;
        if(LADSPA_IS_HINT_DEFAULT_100(hint)) 
            value = 100;
        if(LADSPA_IS_HINT_DEFAULT_440(hint)) 
            value = 440;
        if(LADSPA_IS_HINT_INTEGER(hint)) {
            step_increment = page_increment = page_size = 1;
        }
        upper += page_increment;
        adjust = GTK_ADJUSTMENT(gtk_adjustment_new(value, 
                                                   lower,
                                                   upper, 
                                                   step_increment,
                                                   page_increment, 
                                                   page_size));
        //                DEBUG("lower: %f, upper: %f, value: %f, step_increment: %f, page_increment: %f, page_size: %f\n", lower, upper, value, step_increment, page_increment, page_size);
        //                v = gtk_spin_button_new(adjust, 1, decimals);
        *v = gtk_hscale_new(adjust);
        gtk_scale_set_digits(GTK_SCALE(*v), 2);
        if(LADSPA_IS_HINT_INTEGER(hint)) {
            //            adjust->upper += .9;
            gtk_scale_set_digits(GTK_SCALE(*v), 0);
        }
        *min = lower;
        *max = upper;
    }
    *defalt = value;
}

void
on_clist_key_press_event(GtkCList *clist,
                         GdkEventKey *event,
                         gpointer user_data) {
    int i, key = event->keyval;
    for(i = 0; i < ladspa_count; i++) {
        if(tolower(ladspa_modules[i].desc->Name[0]) == 
           tolower(key)) {
            gtk_clist_moveto(clist, i, 0, 0, 0); 
            gtk_clist_select_row(clist, i, 0);
            break;
        }
    }

    /* Flicker a bit to show that we received the command. */

    gtk_widget_queue_draw(GTK_WIDGET(clist));
}

void    
on_clist_select_row(GtkCList *clist,
                    gint row,
                    gint column,
                    GdkEventButton *event,
                    gpointer user_data) {
    int i, num_presets;
    char **preset_names, path[256];
    LADSPA_Descriptor *desc = ladspa_modules[row].desc;
    struct _module_state *module_state = 
        g_object_get_data(G_OBJECT(gtk_widget_get_toplevel(GTK_WIDGET(clist))),
                          "user_data");
    ladspa_params *p = (ladspa_params *)module_state->data;
    GtkLabel *l;
    GtkWidget *hbox;
    GList *items = NULL;

    printf("selected: %s, ports: %ld\n",
           desc->Name,
           desc->PortCount);

    /* Recall presets. */

    snprintf(path, 256, "/gnusound/%s/Presets", desc->Name);
    DEBUG("path: %s\n", path);
    gnome_config_get_vector(path, &num_presets, &preset_names);

    items = g_list_append(items, "");
    for(i = 0; i < num_presets; i++) 
        items = g_list_append(items, preset_names[i]);
    
    gtk_combo_set_popdown_strings(p->presets, items);

    if(preset_names) {
        for(i = 0; i < num_presets; i++)
            g_free(preset_names[i]);
        g_free(preset_names);
    }

    /* Destroy previous interface. */

    if(p->controls_table) {
        gtk_object_unref(GTK_OBJECT(p->controls_table));
        gtk_container_remove(GTK_CONTAINER(p->viewport), 
                             GTK_WIDGET(p->controls_table));
    }

    /* Setup new interface. */

    snprintf(path, 256, "%d/%d", 
             ladspa_get_port_count_by_type(desc, 
                                           LADSPA_PORT_AUDIO | 
                                           LADSPA_PORT_INPUT),
             ladspa_get_port_count_by_type(desc, 
                                           LADSPA_PORT_AUDIO | 
                                           LADSPA_PORT_OUTPUT));
    p->currently_selected = desc;
    p->controls_count = 0;
    gtk_label_set_text(p->author, desc->Maker);
    gtk_label_set_text(p->copyright, desc->Copyright);
    gtk_label_set_text(p->audio_io_ports, path);
    p->controls_table = GTK_TABLE(gtk_table_new(0, 2, FALSE));
    gtk_object_ref(GTK_OBJECT(p->controls_table));
    gtk_widget_show(GTK_WIDGET(p->controls_table));
    gtk_container_add(GTK_CONTAINER(p->viewport), 
                      GTK_WIDGET(p->controls_table));

    for(i = 0; i < desc->PortCount; i++) {
        //        DEBUG("portname: %s\n", desc->PortNames[i]);
        if(LADSPA_IS_PORT_CONTROL(desc->PortDescriptors[i]) &&
           LADSPA_IS_PORT_INPUT(desc->PortDescriptors[i])) {
            if(p->controls_count >= MAX_CONTROLS) {
                FAIL("no room for any more controls, skipping control %s\n",
                     desc->PortNames[i]);
                continue;
            }
            gtk_table_resize(p->controls_table, p->controls_count+1, 2);
            l = GTK_LABEL(gtk_label_new(desc->PortNames[i]));
            gtk_label_set_justify(l, GTK_JUSTIFY_RIGHT);
            gtk_misc_set_alignment(GTK_MISC(l), 1, 1);
            gtk_widget_show(GTK_WIDGET(l));
            gtk_table_attach(p->controls_table,
                             GTK_WIDGET(l),
                             0, 1, p->controls_count, p->controls_count+1,
                             GTK_FILL, GTK_FILL,
                             6, 0);
            DEBUG("port: %s, hints: 0x%x\n",
                  desc->PortNames[i],
                  desc->PortRangeHints[i].HintDescriptor);
            
            make_widget_from_port(p, i,
                                  &p->controls[p->controls_count], 
                                  &p->control_values[p->controls_count], 
                                  &p->control_min[p->controls_count], 
                                  &p->control_max[p->controls_count]);
            p->envelope1[p->controls_count] = gtk_toggle_button_new_with_label("Env1");
            p->envelope2[p->controls_count] = gtk_toggle_button_new_with_label("Env2");
            gtk_tooltips_set_tip(p->tips, p->envelope1[p->controls_count],
                                 "Assigns this parameter to the primary envelope.", ""); 
            gtk_tooltips_set_tip(p->tips, p->envelope2[p->controls_count],
                                 "Assigns this parameter to the auxiliary envelope.", "");
            hbox = gtk_hbox_new(FALSE, 0);
            gtk_box_pack_start(GTK_BOX(hbox), p->controls[p->controls_count], TRUE, TRUE, 0);
            gtk_box_pack_start(GTK_BOX(hbox), p->envelope1[p->controls_count], FALSE, FALSE, 0);
            gtk_box_pack_start(GTK_BOX(hbox), p->envelope2[p->controls_count], FALSE, FALSE, 0);
            gtk_widget_show(p->controls[p->controls_count]);
            gtk_widget_show(p->envelope1[p->controls_count]);
            gtk_widget_show(p->envelope2[p->controls_count]);
            gtk_widget_show(hbox);
            
            gtk_table_attach(p->controls_table,
                             hbox,
                             1, 2, p->controls_count, p->controls_count+1,
                             GTK_EXPAND | GTK_FILL, GTK_FILL,
                             0, 0);
            p->controls_count++;
        }
    }
}

void
module_open(module_id id,
            shell *shl, 
            int undo) {
    GladeXML *xml;
    GtkWidget *w;
    char path[4096];
    int i;
    ladspa_params *p = mem_calloc(1, sizeof(ladspa_params));
    
    if(!p) {
        gui_alert("Not enough memory.");
        return;
    }

    if(ladspa_count == 0) {
        gui_alert("Did not find any LADSPA plugins.\n"
                  "Check your LADSPA_PATH environment variable.\n"
                  "Download LADSPA plugins at http://ladspa.org.");
        return;
    }

    snprintf(path, 4096, "%s/%s", dirname(modules[id].fname), LADSPA_GLADE_FILE);
    DEBUG("loading interface %s\n", path);
    xml = glade_xml_new(path, "dialog", NULL);
    
    if(!xml) {
        gui_alert("LADSPA plugins: could not load interface %s.", path);
        free(p);
        return;
    }

    shl->module_state[id].is_open = 1;
    shl->module_state[id].data = p;
    p->id = id;
    p->shl = shl;
    p->dialog = GTK_WIDGET(glade_xml_get_widget(xml, "dialog"));
    p->clist = GTK_CLIST(glade_xml_get_widget(xml, "clist1"));
    p->viewport = GTK_VIEWPORT(glade_xml_get_widget(xml, "viewport1"));
    p->presets = GTK_COMBO(glade_xml_get_widget(xml, "presets"));
    p->author = GTK_LABEL(glade_xml_get_widget(xml, "author"));
    p->copyright = GTK_LABEL(glade_xml_get_widget(xml, "copyright"));
    p->audio_io_ports = GTK_LABEL(glade_xml_get_widget(xml, "audio_io_ports"));
    p->tips = gtk_tooltips_new();
    g_signal_connect(GTK_OBJECT(p->dialog), "destroy", 
                     G_CALLBACK(on_dialog_destroy), shl);
    g_signal_connect(GTK_OBJECT(p->clist), "select_row",
                     G_CALLBACK(on_clist_select_row), shl);
    g_signal_connect(GTK_OBJECT(p->clist), "key_press_event",
                     G_CALLBACK(on_clist_key_press_event), shl);
    g_signal_connect(GTK_OBJECT(p->presets->entry), "activate",
                     G_CALLBACK(on_presets_entry_activate), shl);
    g_signal_connect(GTK_OBJECT(p->presets->list), "select_child",
                     G_CALLBACK(on_presets_list_select_child), shl);
    w = GTK_WIDGET(glade_xml_get_widget(xml, "ok"));
    g_signal_connect(GTK_OBJECT(w), "clicked", 
                     G_CALLBACK(on_ok_clicked), shl);
    w = GTK_WIDGET(glade_xml_get_widget(xml, "close"));
    g_signal_connect(GTK_OBJECT(w), "clicked", 
                     G_CALLBACK(on_close_clicked), shl);
    w = GTK_WIDGET(glade_xml_get_widget(xml, "apply"));
    g_signal_connect(GTK_OBJECT(w), "clicked", 
                     G_CALLBACK(on_apply_clicked), shl);
    w = GTK_WIDGET(glade_xml_get_widget(xml, "delete_preset"));
    g_signal_connect(GTK_OBJECT(w), "clicked", 
                     G_CALLBACK(on_delete_preset_clicked), shl);
    g_object_set_data(G_OBJECT(p->dialog), "user_data",
                      GINT_TO_POINTER(&shl->module_state[id]));
    g_object_set_data(G_OBJECT(p->presets->list), "user_data",
                      GINT_TO_POINTER(&shl->module_state[id]));
    g_object_unref(G_OBJECT(xml));

    for(i = 0; i < ladspa_count; i++) 
        gtk_clist_append(p->clist, (gchar **)&ladspa_modules[i].desc->Name);
    gtk_combo_disable_activate(p->presets);
    gtk_clist_select_row(p->clist, 0, 0);

}

void
module_close(mod_state *module_state) {
    gtk_object_destroy(GTK_OBJECT(((ladspa_params *)module_state->data)->dialog));
}

action_group *
module_execute(shell *shl, 
               int undo) {
    AFframecount start = shl->select_start,
        end = shl->select_end;
    track_map_t map = shl->select_channel_map;
    int i, j, t, id = shl->active_module_id;
    ladspa_params *p;
    action_group *undo_ag = NULL;
    LADSPA_Descriptor *desc;
    LADSPA_Data dummy;
    int selected_tracks = 0;

    if(!shl->module_state[id].is_open) { 
        gui_alert("LADSPA does not have defaults.");
        return NULL;
    }

    p = mem_calloc(1, sizeof(ladspa_params));

    if(!p) {
        gui_alert("Not enough memory to copy data structures.");
        return NULL;
    }

    memcpy(p, shl->module_state[id].data, sizeof(ladspa_params));

    if(!p->currently_selected) {
        free(p);
        gui_alert("No plugin selected.");
        return NULL;
    }

    desc = p->currently_selected;
    
    p->audio_output_count = 
        ladspa_get_port_count_by_type(desc, 
                                      LADSPA_PORT_AUDIO | LADSPA_PORT_OUTPUT);
    p->audio_input_count = 
        ladspa_get_port_count_by_type(desc, 
                                      LADSPA_PORT_AUDIO | LADSPA_PORT_INPUT);

    for(t = 0; t < shl->sr->channels; t++) 
        if((1 << t) & map) 
            selected_tracks++;

    if((p->audio_input_count > 1 || p->audio_output_count > 1) &&
       selected_tracks != MAX(p->audio_input_count, p->audio_output_count)) {
        gui_alert("This plugin needs %d inputs and %d outputs. \n"
                  "You need to select %d tracks.",
                  p->audio_input_count, p->audio_output_count,
                  MAX(p->audio_input_count, p->audio_output_count));
        free(p);
        return NULL;
    }

    p->handle = desc->instantiate(desc,
                                  shl->sr->rate);
    
    if(!p->handle) {
        gui_alert("Cannot instantiate plugin \n%s\n",
                  desc->Name);
        free(p);
        return NULL;
    }

    /* Construct envelope -> control port arrays. */

    p->envelope1_connect_count = p->envelope2_connect_count = 0;
    for(i = 0; i < p->controls_count; i++) {
        if(p->envelopes_enabled[i] & MARKER_SLOPE) {
            p->envelope1_connections[p->envelope1_connect_count] = 
                &p->control_values[i];
            p->envelope1_defaults[p->envelope1_connect_count] =
                p->control_values[i];
            p->envelope1_min[p->envelope1_connect_count] =
                p->control_min[i];
            p->envelope1_max[p->envelope1_connect_count] =
                p->control_max[i];
            p->envelope1_connect_count++;
        }
        if(p->envelopes_enabled[i] & MARKER_SLOPE_AUX) {
            p->envelope2_connections[p->envelope2_connect_count] = 
                &p->control_values[i];
            p->envelope2_defaults[p->envelope2_connect_count] =
                p->control_values[i];
            p->envelope2_min[p->envelope2_connect_count] =
                p->control_min[i];
            p->envelope2_max[p->envelope2_connect_count] =
                p->control_max[i];
            p->envelope2_connect_count++;
        }
    }

    /* Map port numbers for audio inputs and outputs. */

    p->audio_input_count = p->audio_output_count = 0;
    for(i = 0, j = 0; i < desc->PortCount; i++) {
        if(LADSPA_IS_PORT_AUDIO(desc->PortDescriptors[i])) {
            if(LADSPA_IS_PORT_INPUT(desc->PortDescriptors[i]))
                p->audio_input_map[p->audio_input_count++] = i;
            else
                p->audio_output_map[p->audio_output_count++] = i;
        }
    }

    /* Allocate I/O buffers. */

    for(i = 0; i < 10; i++) {
        p->in_buf[i] = NULL;
        p->out_buf[i] = NULL;
    }
    for(i = 0; i < p->audio_input_count; i++) {
        p->in_buf[i] = mem_calloc(sizeof(float), EFFECT_BUF_SIZE);
        if(!p->in_buf[i]) {
            for(i = i-1; i; i--)
                free(p->in_buf[i]);
            FAIL("Not enough memory to allocate input buffers!");
            free(p);
            gui_alert("Could not allocate input buffers!");
            return NULL;
        }
    }
    for(i = 0; i < p->audio_output_count; i++) {
        p->out_buf[i] = mem_calloc(sizeof(float), EFFECT_BUF_SIZE);
        if(!p->out_buf[i]) {
            for(i = i-1; i; i--) 
                free(p->out_buf[i]);
            for(i = 0; i < p->audio_input_count; i++) 
                free(p->in_buf[i]);
            FAIL("Not enough memory to allocate output buffer!");
            free(p);
            gui_alert("Could not allocate output buffer!");
            return NULL;
        }
    }

    /* Connect control ports. */

    for(i = 0, j = 0; i < desc->PortCount; i++) {
        if(LADSPA_IS_PORT_CONTROL(desc->PortDescriptors[i])) {
            printf("connecting control port: %s\n", desc->PortNames[i]);
            if(LADSPA_IS_PORT_INPUT(desc->PortDescriptors[i])) {
                desc->connect_port(p->handle, i, &p->control_values[j++]);
            } else {
                desc->connect_port(p->handle, i, &dummy);
            }
        }
    }

    if(desc->activate)
        desc->activate(p->handle);

    /* Do it. */

    rwlock_rlock(&shl->sr->rwl);

    if(undo) 
        undo_ag = action_group_undo_create(shl,
                                           map, 
                                           start,
                                           end - start,
                                           start,
                                           end - start);



    if(p->audio_input_count < 2 && p->audio_output_count < 2)
        ladspa_process_1on1(shl,
                            p,
                            map, start, end);
    else
        ladspa_process_many(shl,
                            p,
                            map, start, end);
        
    rwlock_runlock(&shl->sr->rwl);

    gtk_progress_bar_set_fraction(shl->progress, 0);
    
    for(i = 0; i < p->audio_input_count; i++)
        if(p->in_buf[i])
            mem_free(p->in_buf[i]);

    for(i = 0; i < p->audio_output_count; i++)
        if(p->out_buf[i])
            mem_free(p->out_buf[i]);

    if(desc->deactivate)
        desc->deactivate(p->handle);

    if(desc->cleanup)
        desc->cleanup(p->handle);

    free(p);
    return undo_ag;

}
