/*
 * GPG Keys 
 * cdlgserversearch.cpp
 * (c) 2001 Peter Mathiasson <peter@mathiasson.nu>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2, as
 * published by the Free Software Foundation
 */

#include <qcombobox.h>
#include <qpushbutton.h>
#include <qmessagebox.h>
#include <qlineedit.h>
#include <qstatusbar.h>
#include <qlabel.h>
#include <qlistview.h>
#include <qlayout.h>
#include <qpopupmenu.h>
#include <qstringlist.h>
#include "cdlgserversearch.h"
#include "clistviewitem.h"
#include "ckeyserver.h"
#include "cgpgproc.h"
#include "functions.h"
#include "config.h"

/* --------------------------------------------------------------------------------- */

CDlgServerSearch::CDlgServerSearch( QWidget *parent, const QString searchString ) :
CDlgServerSearchUI( parent, 0, TRUE )
{

    /* Create Keyserver Object */
    keyServer = new CKeyServer( this );
    connect( keyServer, SIGNAL(statusChange(int)), this, SLOT(statusChange(int)) );    
    connect( keyServer, SIGNAL(itemFound(QString,QString,QString,QString)),
             this, SLOT(itemFound(QString,QString,QString,QString)) );
    connect( keyServer, SIGNAL(searchDone()), this, SLOT(searchDone()) );
    connect( keyServer, SIGNAL(getDone(QString)), this, SLOT(doImport(QString)) );

    /* Search Button */
    connect( btnSearch, SIGNAL(clicked()), this, SLOT(startSearch()) );

    /* Search ListView */
    connect( lvSearch, SIGNAL(contextMenuRequested(QListViewItem*,const QPoint&,int)),
             this, SLOT(contextMenuRequested(QListViewItem*,const QPoint&,int)) );
    
    /* Create Popup Menu */
    popupMenu = new QPopupMenu( this );
    popupMenu->insertItem( tr( "&Import Key" ), this, SLOT(importKey()), CTRL+Key_I );

    /* Create Statusbar */
    statusBar = new QStatusBar(this);
    CDlgServerSearchUILayout->addWidget( statusBar );

    /* Set Fonts */
    setFont( readConfigFont( "stdfont", STD_FONT ) );
    lvSearch->setFont( readConfigFont( "fixedfont", FIX_FONT ) );

    /* Auto Search */
    if ( searchString != QString::null ) {
        editSearch->setText( searchString );
        startSearch();
    }

}

/* --------------------------------------------------------------------------------- */

void CDlgServerSearch::startSearch()
{
    /* Check input box */
    if ( !editSearch->text().length() ) {
        QMessageBox::critical( this, tr( "Error" ), tr( "No input string" ),
                               QMessageBox::Ok | QMessageBox::Default, QMessageBox::NoButton );
        return;
    }

    lvSearch->clear();
    keyServer->search( editSearch->text() );
}

/* --------------------------------------------------------------------------------- */

void CDlgServerSearch::statusChange( int newStatus )
{
    QString newText;
    switch( newStatus ) {
    
    case CKeyServer::Idle:
        statusBar->message( tr( "Idle" ), 2000 );
        return;
    case CKeyServer::Connecting:
        newText = tr( "Connecting..." );
        break;
    case CKeyServer::Sending:
        newText = tr( "Sending Request..." );
        break;
    case CKeyServer::Waiting:
        newText = tr( "Waiting for reply..." );
        break;
    case CKeyServer::Recieving:
        newText = tr( "Recieving data..." );
        break;
    default:
        break;
    
    }
    statusBar->message( newText );
}

/* --------------------------------------------------------------------------------- */

void CDlgServerSearch::itemFound( QString type, QString name, QString date, QString keyID )
{

    /* Public Key */
    if ( type == "pub" ) {

        /* Create ListView Item */
        itemPub = new CListViewItem( lvSearch, CListViewItem::PUBLICKEY );
        itemPub->setText( 0, fromUtf8( name ) );        
        itemPub->setText( 1, keyID );
        itemPub->setText( 2, date );

    }

    /* Signature */
    else if ( type == "sig" ) {

        /* Create ListView Item */
        QListViewItem *itemTemp = new CListViewItem( itemPub, CListViewItem::SIGNATURE );
        itemTemp->setText( 0, fromUtf8( name ) );
        itemTemp->setText( 1, keyID );
        
    }

}

/* --------------------------------------------------------------------------------- */

void CDlgServerSearch::searchDone()
{
    QString tempString;
    tempString.sprintf( tr( "Done: %d item(s) found" ), keyServer->entriesFound );
    statusBar->message( tempString );
}

/* --------------------------------------------------------------------------------- */

void CDlgServerSearch::contextMenuRequested( QListViewItem *tmpItem, const QPoint &pos, int )
{
    /* Abort if no item selected */
    if ( !tmpItem )
        return;

    /* Display Menu */
    CListViewItem *item = (CListViewItem*)tmpItem;
    if ( item->type == CListViewItem::PUBLICKEY )
        popupMenu->exec( pos );
}

/* --------------------------------------------------------------------------------- */

void CDlgServerSearch::importKey()
{
    /* Get Import Item */
    importItem = (CListViewItem*)lvSearch->currentItem();

    /* Check Item */
    if ( !importItem )
        return;
    if ( importItem->type != CListViewItem::PUBLICKEY )
        return;

    /* Display Message Box*/
    int retval = QMessageBox::information( this, "GPG Keys", tr( "Proceed with import" ),
                                           QMessageBox::Yes | QMessageBox::Default,
                                           QMessageBox::No | QMessageBox::Escape );
    
    if ( retval == QMessageBox::Yes )
        keyServer->getKey( importItem->text( 1 ) );    
}

/* --------------------------------------------------------------------------------- */

void CDlgServerSearch::doImport( QString key )
{
    /* Create Argument List */
    QStringList args;
    args << "--import";

    /* Set Status Bar Message */
    statusBar->message( tr( "Importing Key..." ) );

    /* Run and Check Return Code */
    switch ( CGPGProc::run( this, args, key ) ) {
    
    case CGPGProc::RUNFAILED:
        runFailed( statusBar );
        break;
    case CGPGProc::OK:
        statusBar->message( tr( "Key Imported!" ), 3000 );
        emit newPublic();
        break;
    case CGPGProc::ERROR:
        statusBar->message( tr( "Import failed!" ), 3000 );
        break;

    }
}

/* --------------------------------------------------------------------------------- */
