/*
    GQ -- a GTK-based LDAP client
    Copyright (C) 1998-2003 Bert Vermeulen
    Copyright (C) 2002-2003 Peter Stamfest

    This program is released under the Gnu General Public License with
    the additional exemption that compiling, linking, and/or using
    OpenSSL is allowed.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/* $Id: browse.c,v 1.95 2003/11/03 21:14:41 stamfest Exp $ */

#include <glib.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include <errno.h>
#include <string.h>

#include <config.h>

#include "common.h"
#include "configfile.h"
#include "browse.h"

#include "dn-browse.h"
#include "server-browse.h"
#include "ref-browse.h"

#include "mainwin.h"
#include "template.h"
#include "util.h"
#include "tinput.h"
#include "errorchain.h"
#include "ldif.h"
#include "formfill.h"
#include "encode.h"
#include "input.h"
#include "debug.h"
#include "ldapops.h"
#include "search.h"
#include "i18n.h"
#include "state.h"
#include "utf8-compat.h"
#include "progress.h"
#include "prefs.h"

#ifdef BROWSER_DND
#include "browse-dnd.h"
#endif

/*  #include "../icons/warning.xpm" */

static void destroy_browse_mode(GtkWidget *w, struct tab *tab);

static gboolean button_press_on_tree_item(GtkWidget *tree,
					  GdkEventButton *event,
					  struct tab *tab);


static void tree_row_refresh(GtkMenuItem *menuitem, struct tab *tab);

void record_path(struct tab *tab, browse_entry *entry, 
		 GtkCTree *ctreeroot, GtkCTreeNode *node)
{
     GtkCTreeRow *row;
     browse_entry *e;
     int type = -1;

     if (BROWSETAB(tab)->cur_path) {
	  g_list_foreach(BROWSETAB(tab)->cur_path, (GFunc) g_free, NULL);
	  g_list_free(BROWSETAB(tab)->cur_path);
     }

     BROWSETAB(tab)->cur_path = NULL;

     for ( ; node ; node = row->parent ) {
	  row = GTK_CTREE_ROW(node);
	  e = (browse_entry *) gtk_ctree_node_get_row_data(ctreeroot, node);

	  /* currently it is sufficient to keep changes in entry types only */
	  if (e && e->type != type) {
	       GString *str = g_string_sized_new(128);
	       g_string_sprintf(str, "%d:%s", 
				e->type, e->base_methods->get_name(e, TRUE));

	       BROWSETAB(tab)->cur_path =
		    g_list_insert(BROWSETAB(tab)->cur_path, str->str, 0);

	       g_string_free(str, FALSE);

	       type = e->type;
	  }
     }
}

/* A GtkDestroyNotify callback to be used as a destroy function for
   browse_entry objects attached to gtk objects */
void destroy_browse_entry(browse_entry *entry) 
{
     if (!entry) return;
     assert(entry->base_methods);
     if (entry->base_methods->destroy) {
	  entry->base_methods->destroy(entry);
     } else {
	  free(entry);
     }
}


GtkCTreeNode *dn_browse_single_add(const char *dn,
				   GtkCTree *ctree,
				   GtkCTreeNode *node)
{
     char **exploded_dn = NULL;
#if GTK_MAJOR >= 2
     const char *labels[] = { NULL, NULL };
#else
     char *labels[] = { NULL, NULL };
#endif
     char *dummy[] = { "dummy", NULL };
     dn_browse_entry *new_entry;
     GtkCTreeNode *new_item, *added = NULL;
     int ctx;

     ctx = error_new_context(_("Exploding DN"), GTK_WIDGET(ctree));

     if (config->show_rdn_only) {
	  /* explode DN */
	  exploded_dn = gq_ldap_explode_dn(dn, FALSE);
	  
	  if (exploded_dn == NULL) {
	       /* problem with DN */
	       /*  	  printf("problem dn: %s\n", dn); */
	       error_push(ctx, _("Cannot explode DN '%s'. Maybe problems with quoting or special characters. See RFC 2253 for details of DN syntax."), dn);
	       
	       goto fail;
	  }
#if GTK_MAJOR >= 2
	  labels[0] = exploded_dn[0];
#else
	  labels[0] = decoded_string(exploded_dn[0]);
#endif
     } else {
#if GTK_MAJOR >= 2
	  labels[0] = dn;
#else
	  labels[0] = decoded_string(dn);
#endif
     }

     new_entry = (dn_browse_entry *) new_dn_browse_entry(dn);
     
     added = gtk_ctree_insert_node(ctree,
				   node, NULL,
				   (char**) labels,  /* bug in the GTK2 API: should be const */
				   0,
				   NULL, NULL, NULL, NULL,
				   FALSE, FALSE);
     
     gtk_ctree_node_set_row_data_full(ctree,
				      added,
				      new_entry,
				      (GtkDestroyNotify) destroy_browse_entry);
     
     /* add dummy node */
     new_item = gtk_ctree_insert_node(ctree,
				      added, NULL,
				      dummy, 
				      0,
				      NULL, NULL, NULL, NULL,
				      TRUE, FALSE);

 fail:
#if GTK_MAJOR < 2
     if (labels[0]) free(labels[0]);
#endif
     if (exploded_dn) gq_exploded_free(exploded_dn);

     error_flush(ctx);

     return added;
}








/*
 * adds a single server to the root node
 */
static void add_single_server_internal(GtkCTree *ctree,
				       struct ldapserver *server)
{
     browse_entry *entry;
     GtkCTreeNode *new_item;
     char *labels[2] = { NULL, NULL };

     if (ctree && server) {
	  labels[0] = server->name;
	  entry = new_server_browse_entry(server);

	  new_item = gtk_ctree_insert_node(ctree,
					   NULL, NULL,
					   labels, 
					   0,
					   NULL, NULL, NULL, NULL,
					   FALSE, FALSE);

	  gtk_ctree_node_set_row_data_full(ctree,
					   new_item,
					   entry,
					   (GtkDestroyNotify) destroy_browse_entry);

	  /* add dummy nodes to get expansion capability  */

	  labels[0] = "DUMMY";

	  new_item = gtk_ctree_insert_node(ctree,
					   new_item, NULL,
					   labels, 
					   0,
					   NULL, NULL, NULL, NULL,
					   TRUE, FALSE);

     }
}

#if DEAD_CODE
void add_single_server(struct tab *tab, struct ldapserver *server)
{
     GtkCTree *ctree;

     ctree = BROWSETAB(tab)->ctreeroot;
     add_single_server_internal(ctree, server);

}
#endif

/*
 * add all configured servers to the root node
 */
static void add_all_servers(GtkCTree *ctree)
{
     struct ldapserver *server;
     int server_cnt = 0;
     GList *I;
     
     for( I = config->servers ; I ; I = g_list_next(I)) {
	  server = (struct ldapserver *) I->data;
	  add_single_server_internal(ctree, server);
	  server_cnt++;
     }

     statusbar_msg(ngettext("One server found",
			    "%d servers found", server_cnt),
		   server_cnt);
}


/* GTK callbacks translating to the correct entry object method calls */
static void tree_row_expanded(GtkCTree *ctree,
			      GtkCTreeNode *ctree_node,
			      struct tab *tab)
{
     browse_entry *entry;

     entry = (browse_entry *) gtk_ctree_node_get_row_data(ctree, ctree_node);

     if (!entry) return;
     assert(entry->base_methods);
     if (entry->base_methods->expand) {
	  int ctx = error_new_context(_("Expanding subtree"),
				      GTK_WIDGET(ctree));

	  entry->base_methods->expand(entry, ctx, ctree, ctree_node, tab);
	  record_path(tab, entry, ctree, ctree_node);

	  error_flush(ctx);
     }
}


static void tree_row_selected(GtkCTree *ctree, GtkCTreeNode *node,
			      int column, struct tab *tab)
{
     browse_entry *entry;
     struct inputform *iform;

     /* avoid recursive calls to this handler - this causes crashes!!! */
     /* This is a legitimate use of
	gtk_object_set_data/gtk_object_get_data */

     if (gtk_object_get_data(GTK_OBJECT(ctree), "in-tree_row_selected")) {
#if GTK_MAJOR >= 2
	  g_signal_stop_emission_by_name(GTK_OBJECT(ctree),
					 "tree-select-row");
#else
	  gtk_signal_emit_stop_by_name(GTK_OBJECT(ctree),
				       "tree-select-row");
#endif
	  return;
     }
     
     gtk_object_set_data(GTK_OBJECT(ctree), "in-tree_row_selected", (gpointer) 1);
     
/*      if (gtk_signal_n_emissions_by_name(GTK_OBJECT(ctree),  */
/* 					"tree-select-row") > 1) { */
/* 	  g_signal_stop_emission_by_name(GTK_OBJECT(ctree), */
/* 					 "tree-select-row"); */
/* 	  return; */
/*      } */
     
     entry = (browse_entry *) gtk_ctree_node_get_row_data(ctree, node);
     
     /* delete old struct inputform (if any) */
     iform = BROWSETAB(tab)->inputform;
     if(iform) {
	  /* but first get current hide status */
	  /* HACK: store hide status it in the browse-tab data-structure */
	  BROWSETAB(tab)->hidden = iform->hide_status;

	  inputform_free(iform);
	  BROWSETAB(tab)->inputform = NULL;
     }

     if (entry) {
	  assert(entry->base_methods);
	  if (entry->base_methods->select) {

	       /* do not update right-hand pane if update-lock is set */
	       if (! BROWSETAB(tab)->update_lock) {
		    int ctx = error_new_context(_("Selecting entry"),
						GTK_WIDGET(ctree));
		    entry->base_methods->select(entry, ctx, ctree, node, tab);
		    error_flush(ctx);
	       }
	  }
	  BROWSETAB(tab)->tree_row_selected = node;
     }
     
     gtk_object_remove_data(GTK_OBJECT(ctree), "in-tree_row_selected");

}


static void tree_row_refresh(GtkMenuItem *menuitem, struct tab *tab)
{
     GtkCTree *ctree;
     GtkCTreeNode *node;
     browse_entry *entry;

     ctree = BROWSETAB(tab)->ctreeroot;
     node = BROWSETAB(tab)->tree_row_popped_up;
     entry = (browse_entry *) gtk_ctree_node_get_row_data(ctree, node);

     if (!entry) return;
     assert(entry->base_methods);
     if (entry->base_methods->refresh) {
	  int ctx = error_new_context(_("Refreshing entry"),
				      GTK_WIDGET(ctree));
	  
	  /* delete old struct inputform (if any) */
	  struct inputform *iform = BROWSETAB(tab)->inputform;
	  if(iform) {
	       /* but first get current hide status */
	       /* HACK: store hide status it in the browse-tab
		  data-structure */
	       BROWSETAB(tab)->hidden = iform->hide_status;
	       
	       inputform_free(iform);
	       BROWSETAB(tab)->inputform = NULL;
	  }

	  entry->base_methods->refresh(entry, ctx, ctree, node, tab);
	  error_flush(ctx);
     }
}

/* server & ref */
void tree_row_close_connection(GtkMenuItem *menuitem, struct tab *tab)
{
     GtkCTree *ctree;
     GtkCTreeNode *node;
     struct ldapserver *server;

     ctree = BROWSETAB(tab)->ctreeroot;
     node = BROWSETAB(tab)->tree_row_popped_up;
     server = server_from_node(ctree, node);
     close_connection(server, TRUE);

     statusbar_msg(_("Closed connection to server %s"), server->name);
}



#if DEAD_CODE
static void tree_row_expand_all(GtkMenuItem *menuitem, struct tab *tab)
{
     GtkCTree *ctree;
     GtkCTreeNode *node;

     ctree = BROWSETAB(tab)->ctreeroot;
     node = BROWSETAB(tab)->tree_row_popped_up;
     if(!ctree || !node)
	  return;

     /* XXX er wait, does this turn it on? */
//     g_hash_table_insert(hash, "expand-all", "");

     gtk_ctree_expand(ctree, node);
/*       gtk_ctree_pre_recursive(ctree, node, gtk_ctree_expand, NULL); */

     /* XXX and off again? */
//     g_hash_table_remove(hash, "expand-all");

}
#endif

static void browse_save_snapshot(int error_context, 
				 char *state_name, struct tab *tab)
{
     char *tmp;
     state_value_set_list(state_name, "open-path", BROWSETAB(tab)->cur_path);

#if GTK_MAJOR >= 2
     if (BROWSETAB(tab)->mainpane)
	  state_value_set_int(state_name, "gutter-pos", 
			      gtk_paned_get_position(GTK_PANED(BROWSETAB(tab)->mainpane)));
#endif
     /* the state of the show empty attributes toggle button */


     tmp = g_malloc(strlen(state_name) + 10);
     strcpy(tmp, state_name);
     strcat(tmp, ".input");
     
     if (BROWSETAB(tab)->inputform) {
	  save_input_snapshot(error_context, BROWSETAB(tab)->inputform, tmp);
     } else {
	  rm_value(tmp);
     }

     g_free(tmp);
}

static int cmp_name(browse_entry *entry, const char *name)
{
     gchar *c;
     int rc; 

     if (entry == NULL) return -1;
     c = entry->base_methods->get_name(entry, TRUE);
     rc = strcasecmp(name, c);
     g_free(c);
     return rc;
}

static void browse_restore_snapshot(int context, 
				    char *state_name, struct tab *tab,
				    struct pbar_win *progress)
{
     GtkCTree *ctree = BROWSETAB(tab)->ctreeroot;
     int gutter = state_value_get_int(state_name, "gutter-pos", -1);
     const GList *path = state_value_get_list(state_name, "open-path");
     char *tmp;

     struct ldapserver *server = NULL;

     GtkCTreeNode *node = gtk_ctree_node_nth(GTK_CTREE(ctree), 0);

     if (path) {
	  const GList *I;
	  for (I = path ; I ; I = g_list_next(I)) {
	       const char *s = I->data;
	       const char *c = g_utf8_strchr(s, -1, ':');
	       char *ep;
	       long type = strtol(s, &ep, 10);
 
	       if (progress->cancelled) break;

	       if (progress) {
		    update_progress(progress, _("Opening %s"), c + 1);
	       }

	       if (c == ep) {
		    switch(type) {
		    case DN_BROWSE_ENTRY_ID:
			 node = show_dn(context, ctree, node, c + 1, FALSE);
			 break;
		    case REF_BROWSE_ENTRY_ID:
			 gtk_ctree_expand(ctree, node);
			 node = 
			      gtk_ctree_find_by_row_data_custom(ctree,
								node,
								(gpointer) (c + 1), 
								(GCompareFunc) cmp_name);
			 break;
		    case SERVER_BROWSE_ENTRY_ID:
			 server = server_by_name(c + 1);
			 if (server != NULL) {
			      node = tree_node_from_server_dn(ctree, server,
							      "");
			 } else {
			      /* FIXME - popup error? */
			      node = NULL;
			 }
			 break;
		    default:
			 node = NULL;
			 break;
		    }
	       }
	       if (node == NULL) break;
	  }
	  if (node) gtk_ctree_select(ctree, node);
     }

     if (!progress->cancelled) {
	  if (gutter > 0) {
	       gtk_paned_set_position(GTK_PANED(BROWSETAB(tab)->mainpane),
				      gutter);
	  }
	  
	  tmp = g_malloc(strlen(state_name) + 10);
	  strcpy(tmp, state_name);
	  strcat(tmp, ".input");
	  
	  if (BROWSETAB(tab)->inputform) {
	       restore_input_snapshot(context, BROWSETAB(tab)->inputform, tmp);
	  }
	  
	  g_free(tmp);
     }
}

void set_update_lock(struct tab *tab)
{
     assert(tab);
     BROWSETAB(tab)->update_lock++;
}

void release_update_lock(struct tab *tab)
{
     assert(tab);
     BROWSETAB(tab)->update_lock--;
}

static struct tab_vtab browse_vtab = { browse_save_snapshot, 
				       browse_restore_snapshot };

struct tab *new_browsemode()
{
     GtkWidget *ctreeroot, *browsemode_vbox, *spacer;
     GtkWidget *mainpane, *pane2_vbox, *pane1_scrwin, *pane2_scrwin;
     struct tab_browse *modeinfo;
     struct tab *tab = g_malloc0(sizeof(struct tab));
     tab->type = BROWSE_MODE;

     modeinfo = calloc(sizeof(struct tab_browse), 1);
     tab->modeinfo = modeinfo;
     tab->vtab = &browse_vtab;

     browsemode_vbox = gtk_vbox_new(FALSE, 0);

     spacer = gtk_hbox_new(FALSE, 0);
     gtk_widget_show(spacer);
     gtk_box_pack_start(GTK_BOX(browsemode_vbox), spacer, FALSE, FALSE, 3);

     mainpane = gtk_hpaned_new();
     gtk_container_border_width(GTK_CONTAINER(mainpane), 2);
     gtk_widget_show(mainpane);
     gtk_box_pack_start(GTK_BOX(browsemode_vbox), mainpane, TRUE, TRUE, 0);
     BROWSETAB(tab)->mainpane = mainpane;

     ctreeroot = gtk_ctree_new(1, 0);
     modeinfo->ctreeroot = (GtkCTree *) ctreeroot;

     gtk_clist_set_selection_mode(GTK_CLIST(ctreeroot), GTK_SELECTION_BROWSE);
     gtk_clist_set_column_auto_resize(GTK_CLIST(ctreeroot), 0, TRUE);
     if (config->sort_browse) {
	  gtk_clist_set_auto_sort(GTK_CLIST(ctreeroot), TRUE);
     }

     gtk_signal_connect(GTK_OBJECT(ctreeroot), "tree-select-row",
      			GTK_SIGNAL_FUNC(tree_row_selected), tab);
     gtk_signal_connect(GTK_OBJECT(ctreeroot), "tree-expand",
			GTK_SIGNAL_FUNC(tree_row_expanded), tab);
     gtk_signal_connect(GTK_OBJECT(ctreeroot), "button_press_event",
			GTK_SIGNAL_FUNC(button_press_on_tree_item), tab);

#ifdef BROWSER_DND
     browse_dnd_setup(ctreeroot, tab);
#endif /* BROWSER_DND */

     add_all_servers(GTK_CTREE(ctreeroot));
     gtk_widget_show(ctreeroot);

     pane1_scrwin = gtk_scrolled_window_new(NULL, NULL);
     gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(pane1_scrwin),
				    GTK_POLICY_AUTOMATIC,
				    GTK_POLICY_AUTOMATIC);
     gtk_widget_show(pane1_scrwin);
#if GTK_MAJOR < 2
     gtk_widget_set_usize(pane1_scrwin, 300, -1); /* FIXME: needed in gtk 1.2? */
#endif
     gtk_paned_set_position(GTK_PANED(mainpane), 300);
     gtk_paned_add1(GTK_PANED(mainpane), pane1_scrwin);
     gtk_container_add(GTK_CONTAINER(pane1_scrwin), ctreeroot);

     pane2_scrwin = gtk_scrolled_window_new(NULL, NULL);
     gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(pane2_scrwin),
				    GTK_POLICY_AUTOMATIC,
				    GTK_POLICY_AUTOMATIC);
     gtk_widget_show(pane2_scrwin);
     modeinfo->pane2_scrwin = pane2_scrwin;

     pane2_vbox = gtk_vbox_new(FALSE, 5);
     gtk_widget_show(pane2_vbox);
     modeinfo->pane2_vbox = pane2_vbox;

     gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(pane2_scrwin), pane2_vbox);

     gtk_paned_add2(GTK_PANED(mainpane), pane2_scrwin);

     gtk_widget_show(browsemode_vbox);

     /* prepare for proper cleanup */
     gtk_signal_connect(GTK_OBJECT(browsemode_vbox), "destroy",
			GTK_SIGNAL_FUNC(destroy_browse_mode),
			(gpointer) tab);

     tab->content = browsemode_vbox;
     gtk_object_set_data(GTK_OBJECT(tab->content), "tab", tab);
     return tab;
}

struct ldapserver *server_from_node(GtkCTree *ctreeroot, 
				    GtkCTreeNode *node)
{
     GtkCTreeRow *row = NULL;
     browse_entry *entry;

     for ( ; node ; node = row->parent ) {
	  row = GTK_CTREE_ROW(node);
	  entry = (browse_entry *) gtk_ctree_node_get_row_data(ctreeroot,
							       node);
	  if (IS_SERVER_ENTRY(entry)) {
	       return ((server_browse_entry*) entry)->server;
	  }
	  if (IS_REF_ENTRY(entry)) {
	       return ((ref_browse_entry*) entry)->server;
	  }
     }
     return NULL;
}

static int dn_row_compare_func(dn_browse_entry *row_data, char *dn)
{
     if (row_data == NULL || !IS_DN_ENTRY(row_data)) return 1;

     return strcasecmp(dn, row_data->dn);
}

/*
 * returns this DN's GtkCTreeNode
 */

GtkCTreeNode *node_from_dn(GtkCTree *ctreeroot, 
			   GtkCTreeNode *top,
			   char *dn)
{
     return gtk_ctree_find_by_row_data_custom(GTK_CTREE(ctreeroot),
					      top, dn, 
					      (GCompareFunc) dn_row_compare_func);
}

#if DEAD_CODE
static GtkCTreeNode *node_from_entry(GtkCTree *ctreeroot, 
				     GtkCTreeNode *top,
				     browse_entry *e)
{
     return gtk_ctree_find_by_row_data(GTK_CTREE(ctreeroot),
				       top, e);
}
#endif



struct server_dn {
     const struct ldapserver *server;
     const char *dn;
     struct ldapserver *currserver;
     GtkCTreeNode *found;
};

static void tree_node_from_server_dn_check_func(GtkCTree *ctree,
						GtkCTreeNode *node,
						struct server_dn *sd)
{
     browse_entry *e;

     e = (browse_entry *) gtk_ctree_node_get_row_data(ctree, node);
     
     if (e == NULL) return;
     if (IS_SERVER_ENTRY(e)) {
	  sd->currserver = ((server_browse_entry *)e)->server;
	  if (strlen(sd->dn) == 0 && sd->currserver == sd->server) {
	       sd->found = node;
	  }
	  return;
     }
     if (IS_DN_ENTRY(e)) {
	  if (strcasecmp(sd->dn, ((dn_browse_entry *)e)->dn) == 0 &&
	      sd->currserver && sd->currserver == sd->server) {
	       sd->found = node;
	  }
	  return;
     }
}

/* NOTE: used by server_node_from_server() as well */

GtkCTreeNode *tree_node_from_server_dn(GtkCTree *ctree, 
				       struct ldapserver *server,
				       const char *dn)
{
     GtkCTreeNode *thenode;

     struct server_dn *sd = g_malloc(sizeof(struct server_dn));
     sd->server = server;
     sd->dn = dn;
     sd->currserver = NULL;
     sd->found = NULL;

     ldapserver_ref(server);
     gtk_ctree_pre_recursive(ctree,
			     NULL, /* root */
			     (GtkCTreeFunc) tree_node_from_server_dn_check_func,
			     sd);

     thenode = sd->found;
     ldapserver_unref(server);

     g_free(sd);
     return thenode;
}


static GtkCTreeNode *server_node_from_server(GtkCTree *ctree,
					     struct ldapserver *server)
{

     return tree_node_from_server_dn(ctree, server, "");
}

#if DEAD_CODE
static void remove_single_server_internal(GtkCTree *ctree, struct ldapserver *server)
{
     GtkCTreeNode *node;

     node = server_node_from_server(ctree, server);
     if(node)
	  gtk_ctree_remove_node(ctree, node);

}


static void remove_single_server(struct tab *tab, struct ldapserver *server)
{
     remove_single_server_internal(BROWSETAB(tab)->ctreeroot, server);
}
#endif

static void destroy_browse_mode(GtkWidget *w, struct tab *tab)
{
     assert(tab);

     if(BROWSETAB(tab)->cur_path) {
	  g_list_foreach(BROWSETAB(tab)->cur_path, (GFunc) g_free, NULL);
	  g_list_free(BROWSETAB(tab)->cur_path);
     }

     if(BROWSETAB(tab)->inputform) {
	  inputform_free(BROWSETAB(tab)->inputform);
	  BROWSETAB(tab)->inputform = NULL;
     }

     g_free(tab->modeinfo);
     tab->modeinfo = NULL;

     g_free(tab);
}


void refresh_subtree_new_dn(int error_context, 
			    GtkCTree *ctree, 
			    GtkCTreeNode *node,
			    const char *newdn, 
			    int options)
{
     GtkCTreeNode *new_node;
     dn_browse_entry *e = 
	  (dn_browse_entry *) gtk_ctree_node_get_row_data(ctree, node);

     if (IS_DN_ENTRY(e)) {
	  GtkCTreeNode *parent, *sibling;
	  char **exploded_dn;
	  char *labels[2] = { NULL, NULL };
	  gboolean is_leaf, is_expanded;
	  int n_expand = 2;

	  /* mark the entry to be uncached before we check for it again */
	  e->uncache = TRUE;
	  gtk_clist_freeze(GTK_CLIST(ctree));

	  gtk_ctree_get_node_info(ctree,
				  node,
				  NULL, /* text */
				  NULL, /* spacing */
				  NULL, /* pixmap_closed */
				  NULL, /* mask_closed */
				  NULL, /* pixmap_opened */
				  NULL, /* mask_opened */
				  &is_leaf, /* is_leaf */
				  &is_expanded);

	  if (newdn) {
	       parent = GTK_CTREE_ROW(node)->parent;
	       sibling = GTK_CTREE_ROW(node)->sibling;

	       /* disconnecting entry from row doesn't work without calling
		  the destroy notify function - thus copy the entry */
	       e = (dn_browse_entry *) new_dn_browse_entry(newdn ? newdn : e->dn);
	       
	       gtk_ctree_unselect(ctree, node);
	       
	       exploded_dn = gq_ldap_explode_dn(e->dn, FALSE);
	       if (exploded_dn == NULL) {
		    /* parsing problem */
	       }

#if GTK_MAJOR >= 2
	       labels[0] = exploded_dn[0];
#else
	       labels[0] = decoded_string(exploded_dn[0]);
#endif       

	       /* add a new entry - alternatively we could just set
                  the new info (new rdn) for the existing node, but we
                  would nevertheless have to remove child nodes and
                  mark the node as unvisited. Therefore we just start
                  with a completely fresh tree */
	       new_node = gtk_ctree_insert_node(ctree,
						parent, sibling,
						labels, 
						0,
						NULL, NULL, NULL, NULL,
						FALSE, FALSE);
	       
	       gtk_ctree_node_set_row_data_full(ctree,
						new_node,
						e,
						(GtkDestroyNotify) destroy_browse_entry);
#if GTK_MAJOR < 2
	       free(labels[0]);
#endif
	       /* add dummy node to have something to expand */
	       labels[0] = "DUMMY";
	       
	       gtk_ctree_insert_node(ctree,
				     new_node, NULL,
				     labels, 0,
				     NULL, NULL, NULL, NULL,
				     TRUE, FALSE);
	       
	       /* select the newly added node */
	       gtk_ctree_select(ctree, new_node);
	       
	       /* delete old node */
	       gtk_ctree_remove_node(ctree, node);
	       node = new_node;

	       gq_exploded_free(exploded_dn);
	  } else {
	       /* ! newdn */
	       e->seen = FALSE;

	       /* make the node unexpanded */
	       if (is_expanded) 
		    gtk_ctree_toggle_expansion(ctree, node);
	  }
	  /* toggle expansion at least twice to fire the expand
	     callback (NOTE: do we need this anymore?) */
	  
	  if (options & REFRESH_FORCE_EXPAND) {
	       n_expand = 1;
	  } else if (options & REFRESH_FORCE_UNEXPAND) {
	       n_expand = 0;
	  } else if (is_expanded) {
	       n_expand = 1;
	  } else {
	       n_expand = 0;
	  }
	  
	  while (n_expand-- > 0) {
	       gtk_ctree_toggle_expansion(ctree, node);
	  }

	  show_server_dn(error_context, 
			 ctree, server_from_node(ctree, node), e->dn, TRUE);


	  gtk_clist_thaw(GTK_CLIST(ctree));
     }
}

void refresh_subtree(int error_context, 
		     GtkCTree *ctree, 
		     GtkCTreeNode *node)
{
     refresh_subtree_new_dn(error_context, ctree, node, NULL, 0);
}

void refresh_subtree_with_options(int error_context, 
				  GtkCTree *ctree,
				  GtkCTreeNode *node,
				  int options)
{
     refresh_subtree_new_dn(error_context, ctree, node, NULL, options);
}

GtkCTreeNode *show_server_dn(int context, 
			     GtkCTree *tree, 
			     struct ldapserver *server, const char *dn,
			     gboolean select_node)
{
     GtkCTreeNode *node = tree_node_from_server_dn(tree, server, "");
     if (node) {
	  gtk_ctree_expand(tree, node);
	  return show_dn(context, tree, node, dn, select_node);
     }
     return NULL;
}

GtkCTreeNode *show_dn(int error_context,
		      GtkCTree *tree, GtkCTreeNode *node, const char *dn, 
		      gboolean select_node)
{
     char **dnparts;
     int i;
     GString *s;
     GtkCTreeNode *found = NULL;
     char *attrs[] = { LDAP_NO_ATTRS, NULL };

     if (!dn) return NULL;
     if (!tree) return NULL;
     
     s = g_string_new("");
     dnparts = gq_ldap_explode_dn(dn, 0);
     
     for(i = 0 ; dnparts[i] ; i++) {
     }

     for(i-- ; i >= 0 ; i--) {
	  if (*dnparts[i] == 0) continue;  /* skip empty DN elements (ie always the last one???) - ah, forget to think about it. */
	  g_string_insert(s, 0, dnparts[i]);

/*  	  printf("try %s at %08lx\n", s->str, node); */
	  if (node) gtk_ctree_expand(tree, node);

	  found = node_from_dn(tree, node, s->str);

	  if (found) {
	       node = found;
	  } else if (node) {
	       /* check if the object with this dn actually exists. If
                  it does, we add it to the tree by hand, as we
                  probably cannot see it due to a size limit */

	       struct ldapserver *server = server_from_node(tree, node);
	       LDAP *ld;

	       if ((ld = open_connection(error_context, server)) != NULL) {
		    LDAPMessage *res, *e;
		    LDAPControl c;
		    LDAPControl *ctrls[2] = { NULL, NULL } ;
		    int rc;

		    c.ldctl_oid			= LDAP_CONTROL_MANAGEDSAIT;
		    c.ldctl_value.bv_val	= NULL;
		    c.ldctl_value.bv_len	= 0;
		    c.ldctl_iscritical		= 1;
		    
		    ctrls[0] = &c;

		    rc = ldap_search_ext_s(ld, s->str,
					   LDAP_SCOPE_BASE, 
					   "(objectClass=*)",
					   attrs,
					   0, 
					   ctrls,	/* serverctrls */
					   NULL,	/* clientctrls */
					   NULL,	/* timeout */
					   LDAP_NO_LIMIT,	/* sizelimit */
					   &res);

		    if (rc == LDAP_SUCCESS) {
			 e = ldap_first_entry(ld, res);
			 if (e) {
			      /* have it!! */
			      char *dn2 = ldap_get_dn(ld, e);
			      found = dn_browse_single_add(dn2, tree, node);
			      if (dn2) ldap_memfree(dn2);

			      node = found;
			 }
		    } else {
			 /* FIXME report error */
		    }

		    if (res) ldap_msgfree(res);
		    res = NULL;

		    close_connection(server, FALSE);
	       } else {
		    /* ERROR - cannot open connection to server,
		       either it was just restarted or the server is
		       down. In order to not prolong the time for
		       deeply nested DNs to finally fail just break
		       out of this DN parts loop.
		    */
		    break;
	       }
	  }

/*  	  else break; */

	  g_string_insert(s, 0, ",");
     }

     gq_exploded_free(dnparts);
     g_string_free(s, TRUE);

     if (found && select_node) {
	  gtk_ctree_select(tree, found);
	  gtk_ctree_node_moveto(tree, node,
				0,
				0.5, 0);
     }

     if (found) return found;
     return NULL;
}

#if DEAD_CODE
static void _gtk_ctree_toggle_is_leaf(GtkCTree * ctree, GtkCTreeNode * node)
{
     gchar *text, *temp;
     guint8 spacing;
     GdkPixmap *pixmap_closed, *pixmap_opened;
     GdkBitmap *mask_closed, *mask_opened;
     gboolean is_leaf, expanded;

     g_return_if_fail(ctree != NULL);
     g_return_if_fail(GTK_IS_CTREE(ctree));

     gtk_ctree_get_node_info(ctree, node,
			     &text, &spacing,
			     &pixmap_closed, &mask_closed,
			     &pixmap_opened, &mask_opened,
			     &is_leaf, &expanded);
     temp = g_strdup(text);
     gtk_ctree_set_node_info(ctree, node,
			     temp, spacing,
			     pixmap_closed, mask_closed,
			     pixmap_opened, mask_opened,
			     is_leaf, expanded);
     g_free(temp);
}
#endif


/*
 * Button pressed on a tree item. Button 3 gets intercepted and puts up
 * a popup menu, all other buttons get passed along to the default handler
 *
 * the return values here are related to handling of button events only.
 */
static gboolean button_press_on_tree_item(GtkWidget *tree,
					  GdkEventButton *event,
					  struct tab *tab)
{
     GtkWidget *ctreeroot;
     GtkWidget *root_menu, *menu, *menu_item, *label;
     GtkCTreeNode *ctree_node;
     dn_browse_entry *entry;

     if (event->type == GDK_BUTTON_PRESS && event->button == 3
	 && event->window == GTK_CLIST(tree)->clist_window) {
	  int row, column, rc;
	  char *name;
     
	  rc = gtk_clist_get_selection_info(GTK_CLIST(tree), event->x, event->y, &row, &column);
	  if (rc == 0)
	       return TRUE;
    
	  ctree_node = gtk_ctree_node_nth(GTK_CTREE(tree), row);
	  if (ctree_node == 0)
	       return TRUE;

	  ctreeroot = GTK_WIDGET(BROWSETAB(tab)->ctreeroot);

	  entry = (dn_browse_entry *) gtk_ctree_node_get_row_data(GTK_CTREE(ctreeroot), GTK_CTREE_NODE(ctree_node));

  	  if (entry == NULL) return TRUE;
	  if (entry->base_methods->popup) {
	       /* The get_name method already does UTF-8 decoding */
	       name = entry->base_methods->get_name((browse_entry *) entry, FALSE);
	       
	       BROWSETAB(tab)->tree_row_popped_up = ctree_node;
	       
	       /* hack, hack. Need to pass both the ctree_node and tab... */
	       BROWSETAB(tab)->selected_ctree_node = ctree_node;
	       
	       root_menu = gtk_menu_item_new_with_label("Root");
	       gtk_widget_show(root_menu);
	       menu = gtk_menu_new();

	       gtk_menu_item_set_submenu(GTK_MENU_ITEM(root_menu), menu);
	       
	       label = gtk_menu_item_new_with_label(name);
	       gtk_widget_set_sensitive(label, FALSE);
	       gtk_widget_show(label);
	       
	       gtk_menu_append(GTK_MENU(menu), label);
	       gtk_menu_set_title(GTK_MENU(menu), name);
	       
#if GTK_MAJOR >= 2
	       menu_item = gtk_separator_menu_item_new(); 
#else
	       menu_item = gtk_tearoff_menu_item_new();
#endif
	       gtk_menu_append(GTK_MENU(menu), menu_item);
	       gtk_widget_set_sensitive(menu_item, FALSE);
	       gtk_widget_show(menu_item);
	       
	       if (name) g_free(name);

	       /* The common Refresh item */
	       menu_item = gtk_menu_item_new_with_label(_("Refresh"));
	       gtk_menu_append(GTK_MENU(menu), menu_item);
	       gtk_widget_show(menu_item);
	       
	       gtk_signal_connect(GTK_OBJECT(menu_item), "activate",
				  GTK_SIGNAL_FUNC(tree_row_refresh),
				  (gpointer) tab);

	       entry->base_methods->popup((browse_entry*) entry, menu, 
					  ctreeroot, ctree_node, tab);

	       gtk_menu_popup(GTK_MENU(menu), NULL, NULL, NULL, NULL,
			      event->button, event->time);
	       
	       gtk_signal_emit_stop_by_name(GTK_OBJECT(ctreeroot), 
					    "button_press_event");
	       return(TRUE);
	  }
     }
#if GTK_MAJOR >= 2
     return(FALSE);
#else
     return(TRUE);  /* may be a bug to return TRUE unconditionally in gtk 1.2 */
#endif
}


typedef struct {
     GList *list;
} cb_server_list;

static void get_current_servers_list_check_func(GtkCTree *ctree,
						GtkCTreeNode *node,
						cb_server_list *csl)
{
     browse_entry *e;
     e = (browse_entry *) gtk_ctree_node_get_row_data(ctree, node);
     
     if (e == NULL) return;
     if (IS_SERVER_ENTRY(e)) {
	  struct ldapserver *thisserver = ((server_browse_entry *)e)->server;

	  csl->list = g_list_append(csl->list, thisserver);
	  ldapserver_ref(thisserver);
     }
}


void update_browse_serverlist(struct tab *tab)
{
     GtkCTree *ctree;
     struct ldapserver *server;
     cb_server_list csl;
     GList *l, *I;

     ctree = BROWSETAB(tab)->ctreeroot;
     /* walk the list of ldapservers, add any not yet in the show list */

     for (I = config->servers ; I ; I = g_list_next(I)) {
	  server = (struct ldapserver *) I->data;
	  if (server_node_from_server(ctree, server) == NULL)
	       add_single_server_internal(ctree, server);
     }

     csl.list = NULL;
     gtk_ctree_pre_recursive_to_depth(ctree,
				      NULL, /* root node */
				      1,    /* max depth */
				      (GtkCTreeFunc) get_current_servers_list_check_func,
				      &csl);

     /* walk list of shown servers, remove any no longer in config */
     for (l = csl.list ; l ; l=l->next) {
	  int found = 0;
	  struct ldapserver *thisserver = l->data;
	  GtkCTreeNode *node = server_node_from_server(ctree, thisserver);

	  /* is this server still in the list of configured servers? */
	  if (g_list_find(config->servers, thisserver)) {
	       found = 1;
	  }

	  /* deleted servers */
	  if (!found) {
	       if (node) gtk_ctree_remove_node(ctree, node);
	  } else {
	       /* renamed servers ? */
	       char *currtext = NULL;
	       gtk_ctree_get_node_info(ctree,
				       node,
				       &currtext, /* gchar **text */
				       NULL, /* spacing */
				       NULL, /* GdkPixmap **pixmap_closed */
				       NULL, /* GdkBitmap **mask_closed */
				       NULL, /* GdkPixmap **pixmap_opened */
				       NULL, /* GdkBitmap **mask_opened */
				       NULL, /* gboolean *is_leaf */
				       NULL  /* gboolean *expanded */
				       );

	       if (currtext && strcmp(currtext, thisserver->name) != 0) {
		    gtk_ctree_node_set_text(ctree, node, 0, thisserver->name);
	       }
	  }
     }

     if (csl.list) {
	  g_list_foreach(csl.list, (GFunc) ldapserver_unref, NULL);
	  g_list_free(csl.list);
     }
}

/* 
   Local Variables:
   c-basic-offset: 5
   End:
*/
