/* gradio-main.c
 * Contains functions to build and display the GRadio interface,
 * and for reading/writing the .gradiorc file and initializing the program
 */

/* (c) 1998-2000 by Keith Wesolowski (wesolows@foobazco.org) 

 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version. 

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details. 

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 */

#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include "gradio.h"

radio_status gr_stat;

int gradio_init (int argc, char **argv)
{
    FILE *rcfile;
    char rcfilename[256];
    char *homedir = getenv ("HOME");
    int i, ignored;
    
    /* For clo checking */
    int opt;
    int cli_only = FALSE;
    int volume = -1, mute = -1, freq = -1;

    /* Initially, use the compiled-in device. */
    strncpy (gr_stat.dev_name, DEV_NAME, 128);

    if (argc == 1)
	cli_only = FALSE;
    else
	while ((opt = getopt (argc, argv, "chVv:f:m:d:")) != EOF)
	    switch (opt) {
	    case 'c':
		cli_only = TRUE;
		break;
	    case 'v':
		volume = atoi (optarg);
		break;
	    case 'f':
		freq = (int)(atof (optarg) * 1000. + 0.5);
		break;
	    case 'm':
		mute = (optarg[0] != '-');
		break;
	    case 'd':
		strncpy (gr_stat.dev_name, optarg, sizeof (gr_stat.dev_name));
		break;
	    case 'V':
		printf ("GRadio version %s\n", VERSION);
		exit (0);
	    case 'h':
	    default:
		gradio_usage (argv[0]);
		break;
	    }

    /* We can't actually do anything until after we've had a chance
     * to set the device name.
     */
    
    /* Let there br POWER */
    gr_stat.power = TRUE;
    gr_stat.memset = -1;

    /* Read in (or create) the rcfile */
    /* The rcfile format is as follows:
     * {ioport}<newline> --- Now always 0 as of 0.9.10
     * {current frequency}<newline>
     * {volume level}<tab>{muted}<newline>
     * {memory freq 1}<newline>
     * .
     * .
     * .
     * {memory freq 8}<newline>
     */

    /* Thanks to v4l-only, no long pause for volume init is needed any more! */
    gr_stat.muted = FALSE;
    snprintf (rcfilename, 256, "%s/.gradiorc", (homedir != NULL) 
	      ? homedir : ".");
    if ((rcfile = fopen (rcfilename, "r")) == NULL) {
	/* We can't read an rcfile, so make one up */
	gradio_get_interface ();
	if ((rcfile = fopen (rcfilename, "w")) == NULL) {
	    /* Can't make up an rcfile. Process what we can and then stop */
	    if (cli_only) {
		if (mute >= 0) gradio_mute (mute);
		if (volume >= 0) gradio_setvol (volume);
		if (freq >= 0) gradio_setfreq (freq);
	    }
	    return cli_only;
	}
	fprintf (rcfile, "0\n"); /* Compatibility with 0.9.<10 */
	fprintf (rcfile, "%d\n", GR_FREQ_DEFAULT);
	fprintf (rcfile, "0 0\n");
	for (i=0; i<8; i++) {
	    fprintf (rcfile, "%d\n", GR_FREQ_DEFAULT);
	    gr_stat.mem[i] = GR_FREQ_DEFAULT;
	}
	fclose (rcfile);
	if (cli_only) {
	    if (mute >= 0) gradio_mute (mute);
	    if (volume >= 0) gradio_setvol (volume);
	    if (freq >= 0) gradio_setfreq (freq);
	} else {
	    gradio_mute (FALSE); 
	    gradio_setvol (GR_VOL_DEFAULT);
	    gradio_setfreq (GR_FREQ_DEFAULT);
	}
	return cli_only;
    }
    /* OK, there's an rcfile. Read and process it. */
    fscanf (rcfile, "%x", &ignored); /* Compatibility with 0.9.<10 */
    gradio_get_interface ();
    fscanf (rcfile, "%d", &(gr_stat.freq));
    fscanf (rcfile, "%d", &(gr_stat.vol));
    fscanf (rcfile, "%d", &(gr_stat.muted));
    for (i=0; i<8; i++)
	fscanf (rcfile, "%d", &(gr_stat.mem[i]));
    if (cli_only) {
	if (mute >= 0) gr_stat.muted = mute;
	if (volume >= 0) gr_stat.vol = volume;
	if (freq >= 0) gr_stat.freq = freq;
    }
    gradio_mute (gr_stat.muted);
    gradio_setvol (gr_stat.vol);
    gradio_setfreq (gr_stat.freq);

    return cli_only;
}

void gradio_save_rcfile (void)
{
    FILE *rcfile;
    char rcfilename[256];
    char *homedir = getenv ("HOME");
    int i;

    snprintf (rcfilename, 256, "%s/.gradiorc", (homedir != NULL) 
	      ? homedir : ".");
    if ((rcfile = fopen (rcfilename, "w")) == NULL)
	return;
    fprintf (rcfile, "0\n"); /* Compat (see above) */
    fprintf (rcfile, "%d\n", gr_stat.freq);
    fprintf (rcfile, "%d %d\n", gr_stat.vol, gr_stat.muted);
    for (i=0; i<8; i++)
	fprintf (rcfile, "%d\n", gr_stat.mem[i]);
    fclose (rcfile);
}

void gradio_usage (char *name)
{
    fprintf (stderr,
	     "Usage: %s [-c [-v level] [-f freq] [-m +|-]] [-d filename] [-h] [-V]\n"
	     "Options:\n\n"
	     "-c\t\tUse command line interface only\n"
	     "-v\t\tSet volume to level. Valid arguments are integers 0-11\n"
	     "-f\t\tSet frequency to freq MHz\n"
	     "-m\t\tMute (+) or unmute (-) the radio\n"
	     "-d\t\tUse filename as the radio device\n\n"
	     "-h\t\tShow usage information\n"
	     "-V\t\tShow version information and exit\n", name);
    
    exit (1);
}

int main (int argc, char *argv[])
{
    GtkWidget *main_window;
    GtkWidget *main_box;
    GtkWidget *main_box_left;
    GtkWidget *main_box_right;
    GtkWidget *main_box_bottom;
    GtkWidget *volume_box;
    GtkWidget *tuner_box;
    GtkWidget *memory_box;
    GtkWidget *control_box;
    GtkWidget *display;

    /* This returns TRUE if the gtk interface is not being used,
     * and if that is the case, it does whatever is needed, so we can quit
     * here.
     */
    if (gradio_init (argc, argv)) {
	gradio_save_rcfile ();
	return 0;
    }

    gtk_init (&argc, &argv);

    /* main_window inits */
    main_window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title (GTK_WINDOW (main_window), "GRadio");
    gtk_window_position (GTK_WINDOW (main_window), GTK_WIN_POS_NONE);
    gtk_window_set_policy (GTK_WINDOW (main_window), FALSE, FALSE, TRUE);
    gtk_signal_connect (GTK_OBJECT (main_window), "delete_event",
			GTK_SIGNAL_FUNC (gradio_delete_event_cb), NULL);
    gtk_signal_connect (GTK_OBJECT (main_window), "destroy",
			GTK_SIGNAL_FUNC (gradio_destroy_cb), NULL);
    /* ends (main_window inits) */

    main_box = gtk_hbox_new (FALSE, 2);
    main_box_left = gtk_vbox_new (FALSE, 2);
    main_box_right = gtk_vbox_new (FALSE, 2);
    main_box_bottom = gtk_hbox_new (FALSE, 2);

    gtk_widget_realize (main_window);

    /* let other functions build the rest */
    display = gradio_get_display (main_window); /* Must be first for inits */
    volume_box = gradio_get_volume_box (main_window);
    tuner_box = gradio_get_tuner_box (main_window);
    memory_box = gradio_get_memory_box (main_window);
    control_box = gradio_get_control_box (main_window);
    
    gtk_widget_show (display);
    gtk_widget_show (tuner_box); 
    gtk_widget_show (control_box);
    gtk_widget_show (memory_box);
    gtk_widget_show (volume_box); 
    
    gtk_container_border_width (GTK_CONTAINER (main_window), 6);

    gtk_box_pack_start (GTK_BOX (main_box_bottom), control_box, TRUE, 
			FALSE, 0);
    gtk_box_pack_start (GTK_BOX (main_box_bottom), memory_box, TRUE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (main_box_left), display, TRUE, FALSE, 5);
    gtk_box_pack_start (GTK_BOX (main_box_left), main_box_bottom, TRUE, 
			FALSE, 5);
    gtk_box_pack_start (GTK_BOX (main_box_right), tuner_box, FALSE, FALSE, 14);
    gtk_box_pack_end (GTK_BOX (main_box_right), volume_box, FALSE, FALSE, 5);

    gtk_widget_show (main_box_bottom);
    gtk_widget_show (main_box_left);
    gtk_widget_show (main_box_right);

    gtk_box_pack_start (GTK_BOX (main_box), main_box_left, TRUE, FALSE, 4);
    gtk_box_pack_start (GTK_BOX (main_box), main_box_right, TRUE, FALSE, 4);
    gtk_widget_show (main_box);

    gtk_container_add (GTK_CONTAINER (main_window), main_box);

    gtk_widget_show (main_window);

    gtk_main();

    return 0;
}
