# -*- coding: utf-8 -*-
#
# Gramps - a GTK+/GNOME based genealogy program
#
# Copyright (C) 2003-2004  Donald N. Allingham
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#

# Written by Julio Sanchez <julio.sanchez@gmail.com>

# $Id: rel_es.py,v 1.1.2.4 2004/10/21 17:39:40 rshura Exp $

#-------------------------------------------------------------------------
#
# GRAMPS modules
#
#-------------------------------------------------------------------------

import RelLib
from Relationship import apply_filter as getallancestors
from gettext import gettext as _

#-------------------------------------------------------------------------
#
#
#
#-------------------------------------------------------------------------

_level_name_male = [ "", "primero", "segundo", "tercero", "cuarto", "quinto",
                "sexto", "séptimo", "octavo", "noveno", "décimo", "undécimo",
                "duodécimo", "decimotercero", "decimocuarto", "decimoquinto",
                "decimosexto", "decimoséptimo", "decimoctavo", "decimonono",
                "vigésimo", ]

# Short forms (in apocope) used before names
_level_name_male_a = [ "", "primer", "segundo", "tercer", "cuarto", "quinto",
                "sexto", "séptimo", "octavo", "noveno", "décimo", "undécimo",
                "duodécimo", "decimotercer", "decimocuarto", "decimoquinto",
                "decimosexto", "decimoséptimo", "decimoctavo", "decimonono",
                "vigésimo"]

_level_name_female = [ "", "primera", "segunda", "tercera", "cuarta", "quinta",
                "sexta", "séptima", "octava", "novena", "décima", "undécima",
                "duodécima", "decimotercera", "decimocuarta", "decimoquinta",
                "decimosexta", "decimoséptima", "decimoctava", "decimonona",
                "vigésima"]

_level_name_plural = [ "", "primeros", "segundos", "terceros", "cuartos",
                "quintos", "sextos", "séptimos", "octavos", "novenos",
                "décimos", "undécimos", "duodécimos", "decimoterceros",
                "decimocuartos", "decimoquintos", "decimosextos",
                "decimoséptimos", "decimoctavos", "decimononos",
                "vigésimos", ]

# Higher-order terms (after trastatarabuelo) on this list are not standard,
# but then there is no standard naming scheme at all for this in Spanish.
# Check http://www.genealogia-es.com/guia3.html that echoes a proposed
# scheme that has got some reception in the Spanish-language genealogy
# community.  This plugin implements a scheme based on this list.  When
# this list is not enough, ordinals are used based on the same idea, i.e.
# bisabuelo is 'segundo abuelo' and so on.  When we run out of ordinals
# we resort to N-ésimo notation, that is sort of understandable if in
# context.
_parents_level = [ "", "padres", "abuelos", "bisabuelos", "tatarabuelos",
                   "trastatarabuelos", "pentabuelos", "hexabuelos",
                   "heptabuelos", "octabuelos", "eneabuelos", "decabuelos"]

_father_level = [ "", "padre", "abuelo", "bisabuelo", "tatarabuelo",
                  "trastatarabuelo", "pentabuelo", "hexabuelo",
                  "heptabuelo", "octabuelo", "eneabuelo", "decabuelo"]

_mother_level = [ "", "madre", "abuela", "bisabuela", "tatarabuela",
                  "trastatarabuela", "pentabuela", "hexabuela",
                  "heptabuela", "octabuela", "eneabuela", "decabuela"]

_son_level = [ "", "hijo", "nieto", "bisnieto",
               "tataranieto", "trastataranieto", ]

_daughter_level = [ "", "hija", "nieta", "bisnieta",
                    "tataranieta", "trastataranieta", ]

_sister_level = [ "", "hermana", "tía", "tía abuela",
                  "tía bisabuela", ]

_brother_level = [ "", "hermano", "tío", "tío abuelo",
                   "tío bisabuelo", ]

_nephew_level = [ "", "sobrino", "sobrino nieto", "sobrino bisnieto", ]

_niece_level = [ "", "sobrina", "sobrina nieta", "sobrina bisnieta", ]

#-------------------------------------------------------------------------
#
#
#
#-------------------------------------------------------------------------
def apply_filter(person,index,plist,pmap):
    if person == None:
        return
    plist.append(person)
    pmap[person.getId()] = index
    
    family = person.getMainParents()
    if family != None:
        apply_filter(family.getFather(),index+1,plist,pmap)
        apply_filter(family.getMother(),index+1,plist,pmap)

def get_male_cousin(level):
    if level<len(_level_name_male):
        return "primo %s" % (_level_name_male[level]) 
    else:
        return "primo %d-ésimo" % level 

def get_female_cousin(level):
    if level<len(_level_name_female):
        return "prima %s" % (_level_name_female[level]) 
    else:
        return "prima %d-ésima" % level 

def get_distant_uncle(level):
    if level<len(_level_name_male):
        return "tío %s" % (_level_name_male[level]) 
    else:
        return "tío %d-ésimo" % level 

def get_distant_aunt(level):
    if level<len(_level_name_female):
        return "tía %s" % (_level_name_female[level]) 
    else:
        return "tía %d-ésima" % level 

def get_distant_nephew(level):
    if level<len(_level_name_male):
        return "sobrino %s" % (_level_name_male[level]) 
    else:
        return "sobrino %d-ésimo" % level 

def get_distant_niece(level):
    if level<len(_level_name_female):
        return "sobrina %s" % (_level_name_female[level]) 
    else:
        return "sobrina %d-ésima" % level 

def get_male_relative(level):
    if level<len(_level_name_male_a):
        return "pariente en %s grado" % (_level_name_male_a[level])
    else:
        return "pariente en %d-ésimo grado" % level

def get_female_relative(level):
    if level<len(_level_name_male_a):
        return "pariente en %s grado" % (_level_name_male_a[level])
    else:
        return "pariente en %d-ésimo grado" % level

def get_parents(level):
    if level<len(_parents_level):
        return _parents_level[level]
    elif (level-1)<len(_level_name_plural):
        return "%s abuelos" % (_level_name_plural[level-1])
    else:
        return "%d-ésimos abuelos" % (level-1)

def get_father(level):
    if level<len(_father_level):
        return _father_level[level]
    elif (level-1)<len(_level_name_male_a):
        return "%s abuelo" % (_level_name_male_a[level-1])
    else:
        return "%d-ésimo abuelo" % (level-1)

def get_son(level):
    if level<len(_son_level):
        return _son_level[level]
    elif (level-1)<len(_level_name_male_a):
        return "%s nieto" % (_level_name_male_a[level-1])
    else:
        return "%d-ésimo nieto" % (level-1)

def get_mother(level):
    if level<len(_mother_level):
        return _mother_level[level]
    elif (level-1)<len(_level_name_female):
        return "%s abuela" % (_level_name_female[level-1])
    else:
        return "%d-ésima abuela" % (level-1)

def get_daughter(level):
    if level<len(_daughter_level):
        return _daughter_level[level]
    elif (level-1)<len(_level_name_female):
        return "%s nieta" % (_level_name_female[level-1])
    else:
        return "%d-ésima nieta" % (level-1)

def get_aunt(level):
    if level<len(_sister_level):
        return _sister_level[level]
    elif (level-2)<len(_level_name_female):
        return "%s tía abuela" % (_level_name_female[level-2])
    else:
        return "%d-ésima tía abuela" % (level-2)

def get_uncle(level):
    if level<len(_brother_level):
        return _brother_level[level]
    elif (level-2)<len(_level_name_male_a):
        return "%s tío abuelo" % (_level_name_male_a[level-2])
    else:
        return "%d-ésimo tío abuelo" % (level-2)

def get_nephew(level):
    if level<len(_nephew_level):
        return _nephew_level[level]
    elif (level-1)<len(_level_name_male_a):
        return "%s sobrino nieto" % (_level_name_male_a[level-1])
    else:
        return "%d-ésimo sobrino nieto" % (level-1)

def get_niece(level):
    if level<len(_niece_level):
        return _niece_level[level]
    elif (level-1)<len(_level_name_female):
        return "%s sobrina nieta" % (_level_name_female[level-1])
    else:
        return "%d-ésima sobrina nieta" % (level-1)

def is_spouse(orig,other):
    for f in orig.getFamilyList():
        if other == f.getFather() or other == f.getMother():
            return 1
    return 0

def get_relationship(orig_person,other_person):
    """
    returns a string representing the relationshp between the two people,
    along with a list of common ancestors (typically father,mother) 
    """
    firstMap = {}
    firstList = []
    secondMap = {}
    secondList = []
    common = []
    rank = 9999999

    if orig_person == None:
        return ("undefined",[])

    if orig_person == other_person:
        return ('', [])
    if is_spouse(orig_person,other_person):
        if other_person.getGender() == RelLib.Person.male:
            return ("marido",[])
        else:
            return ("mujer",[])

    try:
        apply_filter(orig_person,0,firstList,firstMap)
        apply_filter(other_person,0,secondList,secondMap)
    except RuntimeError,msg:
        return (_("Relationship loop detected"),None)
    
    for person in firstList:
        if person in secondList:
            new_rank = firstMap[person.getId()]
            if new_rank < rank:
                rank = new_rank
                common = [ person ]
            elif new_rank == rank:
                common.append(person)

    firstRel = -1
    secondRel = -1

    length = len(common)
    
    if length == 1:
        person = common[0]
        secondRel = firstMap[person.getId()]
        firstRel = secondMap[person.getId()]
    elif length == 2:
        p1 = common[0]
        secondRel = firstMap[p1.getId()]
        firstRel = secondMap[p1.getId()]
    elif length > 2:
        person = common[0]
        secondRel = firstMap[person.getId()]
        firstRel = secondMap[person.getId()]

    if firstRel == -1:
        return ("",[])
    elif firstRel == 0:
        if secondRel == 0:
            return ('',common)
        elif other_person.getGender() == RelLib.Person.male:
            return (get_father(secondRel),common)
        else:
            return (get_mother(secondRel),common)
    elif secondRel == 0:
        if other_person.getGender() == RelLib.Person.male:
            return (get_son(firstRel),common)
        else:
            return (get_daughter(firstRel),common)
    elif firstRel == 1:
        if other_person.getGender() == RelLib.Person.male:
            return (get_uncle(secondRel),common)
        else:
            return (get_aunt(secondRel),common)
    elif secondRel == 1:
        if other_person.getGender() == RelLib.Person.male:
            return (get_nephew(firstRel-1),common)
        else:
            return (get_niece(firstRel-1),common)
    elif firstRel == 2 and secondRel == 2:
        if other_person.getGender() == RelLib.Person.male:
            return ('primo hermano',common)
        else:
            return ('prima hermana',common)
    elif firstRel == secondRel:
        if other_person.getGender() == RelLib.Person.male:
            return (get_male_cousin(firstRel-1),common)
        else:
            return (get_female_cousin(firstRel-1),common)
    elif firstRel == secondRel+1:
        if other_person.getGender() == RelLib.Person.male:
            return (get_distant_nephew(secondRel),common)
        else:
            return (get_distant_niece(secondRel),common)
    elif firstRel+1 == secondRel:
        if other_person.getGender() == RelLib.Person.male:
            return (get_distant_uncle(firstRel),common)
        else:
            return (get_distant_aunt(firstRel),common)
    else:
        if other_person.getGender() == RelLib.Person.male:
            return (get_male_relative(firstRel+secondRel),common)
        else:
            return (get_female_relative(firstRel+secondRel),common)

def get_grandparents_string(orig_person,other_person):
    """
    returns a string representing the relationshp between the two people,
    along with a list of common ancestors (typically father,mother) 
    """
    firstMap = {}
    firstList = []
    secondMap = {}
    secondList = []
    common = []
    rank = 9999999

    if orig_person == None:
        return ("undefined",[])

    if orig_person == other_person:
        return ('', [])

    apply_filter(orig_person,0,firstList,firstMap)
    apply_filter(other_person,0,secondList,secondMap)
    
    for person in firstList:
        if person in secondList:
            new_rank = firstMap[person.getId()]
            if new_rank < rank:
                rank = new_rank
                common = [ person ]
            elif new_rank == rank:
                common.append(person)

    firstRel = -1
    secondRel = -1

    length = len(common)
    
    if length == 1:
        person = common[0]
        secondRel = firstMap[person.getId()]
        firstRel = secondMap[person.getId()]
    elif length == 2:
        p1 = common[0]
        secondRel = firstMap[p1.getId()]
        firstRel = secondMap[p1.getId()]
    elif length > 2:
        person = common[0]
        secondRel = firstMap[person.getId()]
        firstRel = secondMap[person.getId()]

    if firstRel == 0:
        if secondRel == 0:
            return ('',common)
        else:
            return (get_parents(secondRel),common)
    else:
        return None

#-------------------------------------------------------------------------
#
# Register this function with the Plugins system 
#
#-------------------------------------------------------------------------
from Plugins import register_relcalc, register_parents

register_relcalc(get_relationship,
    ["es","ES","es_ES","espanol","Espanol","es_ES.UTF8","es_ES@euro","es_ES.UTF8@euro",
            "spanish","Spanish", "es_ES.UTF-8", "es_ES.utf-8", "es_ES.utf8"])

register_parents(get_grandparents_string,
    ["es","ES","es_ES","espanol","Espanol","es_ES.UTF8","es_ES@euro","es_ES.UTF8@euro",
            "spanish","Spanish", "es_ES.UTF-8", "es_ES.utf-8", "es_ES.utf8"])
