/* form editor. This presents the user with a modal window
 * containing various input forms representing the fields of a struct */

#include "config.h"

#ifdef HAVE_GNOME
#include <gnome.h>
#else
#include <gtk/gtk.h>
#endif

#include <stdlib.h>
#include <string.h>
#include <stdarg.h>

#include "int.h"
#include "updatehandlers.h"

/* uncomment for debugging */
/* #define DEBUG */

/* starts a new line. */
#define FORMS_NEWLINE 0
/* defines a text entry. Label text and a pointer to a string follows */
#define FORMS_ENTRY 1
/* defines a checkbox. Label text and a pointer to an int follows */
#define FORMS_CHECKBOX 2

typedef struct
{
   GtkWidget *dialog;
   GList **updatehandlers;
} forms_internaldata;

typedef struct
{
   /* the edit widget */
   GtkWidget *form;
   /* referencing the destination.
    * Destination will be written when ok signal is being received.
    * Until then, the value stored in destination can be used as undo reference */
   gpointer data;
} forms_fielddata;

/* destroy window on delete event */
gint forms_deleteevent(GtkWidget *w,
		       GdkEvent *e,
		       gpointer data)
{
   return FALSE;
};

/* Generic cleanup callback. Called for both the main forms dialog
 * and it's entry fields to free private data structures */
void forms_destroy(GtkWidget *w,
		   gpointer data)
{
   free(data);
};

void forms_cancel(GtkWidget *w,
		  gpointer data)
{
   gtk_widget_destroy(((forms_internaldata*)data)->dialog);
};
   
void forms_ok(GtkWidget *w,
	      gpointer data)
{
   forms_internaldata *i=(forms_internaldata*)data;

   if (i->updatehandlers!=NULL)
     updatehandlers_call(*i->updatehandlers);   
   gtk_widget_destroy(i->dialog);
};

void forms_entry_ok(GtkWidget *w,
		    gpointer data)
{
   forms_fielddata *i=(forms_fielddata*)data;

   const char *entrycontent =gtk_entry_get_text(GTK_ENTRY(i->form));
   strcpy((char*)i->data,entrycontent);
};

void forms_entry_undo(GtkWidget *w,
		      gpointer data)
{
   forms_fielddata *i=(forms_fielddata*)data;

   gtk_entry_set_text(GTK_ENTRY(i->form),(char*)i->data);
};

void forms_checkbox_ok(GtkWidget *w,
		       gpointer data)
{
   forms_fielddata *i=(forms_fielddata*)data;

   *((int*)i->data)=GTK_TOGGLE_BUTTON(i->form)->active;
};

void forms_checkbox_undo(GtkWidget *w,
			 gpointer data)
{
   forms_fielddata *i=(forms_fielddata*)data;

   gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(i->form),
				*((int*)i->data));
};

void forms_create(char *title,
		  int fields,
		  GList **updatehandlers,
		  ...)
{
   GtkWidget *w=NULL;
   GtkWidget *cl=NULL;
   
   GtkWidget *okbutton=NULL;
   GtkWidget *cancelbutton=NULL;
   GtkWidget *undobutton=NULL;
   
   va_list param_pt;
   forms_internaldata *info=NULL;
   forms_fielddata *fd=NULL;
   int x;
   int etype=FORMS_NEWLINE;
   
   info=(forms_internaldata*)malloc(sizeof(forms_internaldata));
   info->updatehandlers=updatehandlers;
   
   w=gtk_dialog_new();
   gtk_signal_connect(GTK_OBJECT(w),"delete_event",
		      GTK_SIGNAL_FUNC(forms_deleteevent),NULL);
   gtk_signal_connect(GTK_OBJECT(w),"destroy",
		      GTK_SIGNAL_FUNC(forms_destroy),info);
   gtk_widget_show(w);
   gtk_window_set_position(GTK_WINDOW(w),GTK_WIN_POS_CENTER);
   info->dialog=w;
   gtk_window_set_title(GTK_WINDOW(w),
			title);
   gtk_window_set_modal(GTK_WINDOW(w),
			1);

#ifdef HAVE_GNOME   
   okbutton=gnome_stock_button(GNOME_STOCK_BUTTON_OK);
#else   
   okbutton=gtk_button_new_with_label(_("OK"));
#endif   
   gtk_box_pack_start(GTK_BOX(GTK_DIALOG(w)->action_area),
		      okbutton,0,0,0);
   gtk_widget_show(okbutton);
   
#ifdef HAVE_GNOME
   cancelbutton=gnome_stock_button(GNOME_STOCK_BUTTON_CANCEL);
#else
   cancelbutton=gtk_button_new_with_label(_("Cancel"));
#endif   
   gtk_box_pack_start(GTK_BOX(GTK_DIALOG(w)->action_area),
		      cancelbutton,0,0,0);
   gtk_widget_show(cancelbutton);
   
#ifdef HAVE_GNOME
   undobutton=gnome_stock_button(GNOME_STOCK_BUTTON_PREV);
#else   
   undobutton=gtk_button_new_with_label(_("Undo"));
#endif   
   gtk_box_pack_start(GTK_BOX(GTK_DIALOG(w)->action_area),
		      undobutton,0,0,0);
   gtk_widget_show(undobutton);

   va_start(param_pt,updatehandlers);
#ifdef DEBUG
   printf ("forms_create: reading %i fields...\n",
	   fields);
#endif   
   for (x=0;x<fields;x++)
     {
	if (etype==FORMS_NEWLINE)
	  {
	     cl=gtk_hbox_new(0,0);
	     gtk_box_pack_start(GTK_BOX(GTK_DIALOG(w)->vbox),
				cl,
				0,0,5);
	     gtk_widget_show(cl);
	     /* fd is used to determine wether to connect the
	      * destroy event of the main dialog window 
	      * to a handler freeing the memory used by fd */
	     fd=NULL;
	  };
	/* etype is preinitialized with FORMS_NEWLINE to make sure
	 * we got a valid vbox to write into */
	etype=va_arg(param_pt,int);
	switch (etype)
	  {	     
	   case FORMS_ENTRY:
	     fd=(forms_fielddata*)malloc(sizeof(forms_fielddata));
	     
	     fd->form=gtk_label_new(va_arg(param_pt,char *));
	     gtk_box_pack_start(GTK_BOX(cl),fd->form,0,0,5);
	     gtk_widget_show(fd->form);
	     
	     fd->data=(gpointer)va_arg(param_pt,char *);
	     fd->form=gtk_entry_new();
	     gtk_entry_set_text(GTK_ENTRY(fd->form),(char*)fd->data);
	     gtk_box_pack_start(GTK_BOX(cl),fd->form,0,0,0);
	     gtk_widget_show(fd->form);
#ifdef DEBUG
	     printf ("forms_create: created text entry '%s'\n",
		     (char*)fd->data);
#endif
	     
	     gtk_signal_connect(GTK_OBJECT(okbutton),"clicked",
				forms_entry_ok,(gpointer)fd);
	     gtk_signal_connect(GTK_OBJECT(undobutton),"clicked",
				forms_entry_undo,(gpointer)fd);
	     break;
	   case FORMS_CHECKBOX:
	     fd=(forms_fielddata*)malloc(sizeof(forms_fielddata));

	     fd->form=gtk_check_button_new_with_label(va_arg(param_pt,char *));
	     gtk_box_pack_start(GTK_BOX(cl),fd->form,0,0,0);
	     gtk_widget_show(fd->form);
	     
	     fd->data=(gpointer)va_arg(param_pt,int *);
	     gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(fd->form),
					  *((int*)fd->data));
	     
#ifdef DEBUG
	     printf ("forms_create: created checkbox\n");
#endif
	     gtk_signal_connect(GTK_OBJECT(okbutton),"clicked",
				forms_checkbox_ok,(gpointer)fd);
	     gtk_signal_connect(GTK_OBJECT(undobutton),"clicked",
				forms_checkbox_undo,(gpointer)fd);
	     break;
	   default:
	     /* do this to prevent a memory hole in case of a wrong
	      * forms parameter list */
	     fd=NULL;
	  };
	if (fd!=NULL)
	  /* destroy field data when main dialog gets destroyed 
	   * if fd is valid for this element */
	  gtk_signal_connect(GTK_OBJECT(w),"destroy",
			     GTK_SIGNAL_FUNC(forms_destroy),(gpointer)fd);
     };
   va_end(param_pt);      
   gtk_signal_connect(GTK_OBJECT(okbutton),"clicked",
		      GTK_SIGNAL_FUNC(forms_ok),(gpointer)info);
   gtk_signal_connect(GTK_OBJECT(cancelbutton),"clicked",
		      GTK_SIGNAL_FUNC(forms_cancel),(gpointer)info);
};
  
