/* piping functions for data filters and cd writing */

#ifndef PIPING_H
#define PIPING_H

#include <gtk/gtk.h>
#include <time.h>    /* FD_ZERO / FD_SET */

/* we might need quite a lot of those for the new cdrecord
 * call scheme where all tracks have to be listed at the commandline */
#define PIPING_MAX_ARGS 1024
#define PIPING_PIDWATCH_TIMEOUT 500

typedef int(*piping_finishcb)(int status,gpointer data);

/* that kind of function is called when forking to a background function using
 * piping_create_function() */
typedef void(*piping_pipefunc)(int c_stdin,int c_stdout,int c_stderr,gpointer data);

typedef struct 
{
	int  pid;
	int  status;  // status as given back by waitpid
	int  handle;  // handle for timer
	piping_finishcb finishcb;
	gpointer data;
} piping_watch;

#define PIPING_WATCHALL 0xff
#define PIPING_WATCHSTDOUT 0x01
#define PIPING_WATCHSTDERR 0x02

/* This function will parse command calls like "cat test.txt|sort",
 * that is multiple commands separated by the piping symbol.
 * Will return 0 on success */
int piping_createcommandchain(char *commandchain,int *in,int *out);

/* initialize in,out,err with -1 respectively to let piping_create
 * assign filedescriptor to them, pass NULL to ignore a specific
 * pipe completely. Returns PID of child process */
int piping_create(char *command,int *in,int *out,int *err);
/* run a function in background and connect to the main process using pipes
 * the same way as in piping_create */
int piping_create_wait(char *command,int *in,int *out,int *err);
/* same as above,just waiting for input before running client */
int piping_create_function(piping_pipefunc,gpointer,int *in,int *out,int *err);
/* call a client and store the output specified by mask to the buffer 
 * until maxoutput count is reached */
void piping_create_getoutput(char *command,
			     char *buf,
			     int maxoutput,
			     char watchmask);
/* scan the whole output of a command, applying function to each file.
 * returns the first function result != 0 */
int piping_create_scantextoutput(char *command,
				 char *function,
				 char watchmask);
piping_watch *piping_pidwatch_add(int pid,
				  piping_finishcb,
				  gpointer data);
void piping_pidwatch_remove(piping_watch*);


/* check whether executable is available and can actually be executed.
 * To determine the truth of this claim, we check the current exec path
 * if there's no '/' in the path 
 * 
 * This function returns TRUE if char *exec is referencing an executable file */
int piping_isvalid_exec(char *exec);

/* same as above, only it accepts a whole command incl. commandline parameters.
 * Parameters are ignored and cut off, though.
 * This is just cor convenience */
int piping_isvalid_command(char *command);

/* returns true if in a given chain e.g. "shorten -x|sox xxx" all
 * files exist. If message != NULL, 
 * dependency informations will pop up for every command in the chain
 * that couldn't get found */
int piping_isvalid_commandchain(char *chain,const char *message);

#endif PIPING_H
