/*
 * Guifications - The end all, be all, toaster popup plugin
 * Copyright (C) 2003-2004 Gary Kramlich
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include <gtk/gtk.h>

#include "gf_action.h"
#include "gf_display.h"
#include "gf_internal.h"
#include "gf_item.h"
#include "gf_item_icon.h"
#include "gf_item_text.h"
#include "gf_menu.h"
#include "gf_stock.h"

static GtkWidget *
gf_menu_make_item(GtkWidget *image, const gchar *text) {
	GtkWidget *menu_item, *hbox, *label;

	menu_item = gtk_menu_item_new();

	hbox = gtk_hbox_new(FALSE, 4);
	gtk_container_add(GTK_CONTAINER(menu_item), hbox);

	if(image)
		gtk_box_pack_start(GTK_BOX(hbox), image, FALSE, FALSE, 0);

	label = gtk_label_new(_(text));
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);

	gtk_widget_show_all(menu_item);

	return menu_item;
}

GtkWidget *
gf_menu_build(GfMenuItemBuilder builder) {
	GtkWidget *menu;
	gint i, max;

	if(builder == gf_menu_position)
		max = 4;
	else if(builder == gf_menu_zoom)
		max = 8;
	else if(builder == gf_menu_mouse)
		max = gf_actions_count();
	else if(builder == gf_menu_event)
		max = gf_events_count();
	else if(builder == gf_menu_item_position)
		max = 9;
	else if(builder == gf_menu_item_type)
		max = 3;
	else if(builder == gf_menu_item_icon_type)
		max = 3;
	else if(builder == gf_menu_item_icon_size)
		max = 7;
	else if(builder == gf_menu_item_text_clipping)
		max = 4;
	else
		return NULL;

	menu = gtk_menu_new();

	for(i = 0; i < max; i++)
		builder(menu, i);

	gtk_widget_show_all(menu);

	return menu;
}

GtkWidget *
gf_menu_position(GtkWidget *menu, gint item) {
	GtkWidget *menu_item = NULL, *image;

	g_return_val_if_fail(menu, NULL);

	switch(item) {
		case GF_DISPLAY_POSITION_NW:
			image = gtk_image_new_from_stock(GF_STOCK_WINDOW_POSITION_NORTH_WEST,
											 GTK_ICON_SIZE_MENU);
			menu_item = gf_menu_make_item(image, _("Top Left"));
			break;
		case GF_DISPLAY_POSITION_NE:
			image = gtk_image_new_from_stock(GF_STOCK_WINDOW_POSITION_NORTH_EAST,
											 GTK_ICON_SIZE_MENU);
			menu_item = gf_menu_make_item(image, _("Top Right"));
			break;
		case GF_DISPLAY_POSITION_SW:
			image = gtk_image_new_from_stock(GF_STOCK_WINDOW_POSITION_SOUTH_WEST,
											 GTK_ICON_SIZE_MENU);
			menu_item = gf_menu_make_item(image, _("Bottom Left"));
			break;
		case GF_DISPLAY_POSITION_SE:
			image = gtk_image_new_from_stock(GF_STOCK_WINDOW_POSITION_SOUTH_EAST,
											 GTK_ICON_SIZE_MENU);
			menu_item = gf_menu_make_item(image, _("Bottom Right"));
			break;
		case GF_DISPLAY_POSITION_UNKNOWN:
		default:
			break;
	}

	if(menu_item)
		gtk_menu_shell_append(GTK_MENU_SHELL(menu), menu_item);

	return menu_item;
}

GtkWidget *
gf_menu_zoom(GtkWidget *menu, gint item) {
	GtkWidget *menu_item = NULL, *image;

	g_return_val_if_fail(menu, NULL);

	switch(item) {
		case GF_DISPLAY_ZOOM_200:
			image = gtk_image_new_from_stock(GF_STOCK_WINDOW_ZOOM_IN,
											 GTK_ICON_SIZE_MENU);
			menu_item = gf_menu_make_item(image, _("200%"));
			break;
		case GF_DISPLAY_ZOOM_175:
			image = gtk_image_new_from_stock(GF_STOCK_WINDOW_ZOOM_IN,
											 GTK_ICON_SIZE_MENU);
			menu_item = gf_menu_make_item(image, _("175%"));
			break;
		case GF_DISPLAY_ZOOM_150:
			image = gtk_image_new_from_stock(GF_STOCK_WINDOW_ZOOM_IN,
											 GTK_ICON_SIZE_MENU);
			menu_item = gf_menu_make_item(image, _("150%"));
			break;
		case GF_DISPLAY_ZOOM_125:
			image = gtk_image_new_from_stock(GF_STOCK_WINDOW_ZOOM_IN,
											 GTK_ICON_SIZE_MENU);
			menu_item = gf_menu_make_item(image, _("125%"));
			break;
		case GF_DISPLAY_ZOOM_100:
			image = gtk_image_new_from_stock(GF_STOCK_WINDOW_ZOOM_NORMAL,
											 GTK_ICON_SIZE_MENU);
			menu_item = gf_menu_make_item(image, _("100%"));
			break;
		case GF_DISPLAY_ZOOM_75:
			image = gtk_image_new_from_stock(GF_STOCK_WINDOW_ZOOM_OUT,
											 GTK_ICON_SIZE_MENU);
			menu_item = gf_menu_make_item(image, _("75%"));
			break;
		case GF_DISPLAY_ZOOM_50:
			image = gtk_image_new_from_stock(GF_STOCK_WINDOW_ZOOM_OUT,
											 GTK_ICON_SIZE_MENU);
			menu_item = gf_menu_make_item(image, _("50%"));
			break;
		case GF_DISPLAY_ZOOM_25:
			image = gtk_image_new_from_stock(GF_STOCK_WINDOW_ZOOM_OUT,
											 GTK_ICON_SIZE_MENU);
			menu_item = gf_menu_make_item(image, _("25%"));
			break;
		case GF_DISPLAY_ZOOM_UNKNOWN:
		default:
			break;
	}

	if(menu_item)
		gtk_menu_shell_append(GTK_MENU_SHELL(menu), menu_item);

	return menu_item;
}

GtkWidget *
gf_menu_mouse(GtkWidget *menu, gint item) {
	GtkWidget *menu_item = NULL;

	g_return_val_if_fail(menu, NULL);

	menu_item = gf_menu_make_item(NULL, gf_actions_get_nth_i18n(item));

	if(menu_item)
		gtk_menu_shell_append(GTK_MENU_SHELL(menu), menu_item);

	return menu_item;
}

GtkWidget *
gf_menu_event(GtkWidget *menu, gint item) {
	GtkWidget *menu_item = NULL;

	g_return_val_if_fail(menu, NULL);

	menu_item = gf_menu_make_item(NULL, gf_events_get_nth_name(item));

	if(menu_item)
		gtk_menu_shell_append(GTK_MENU_SHELL(menu), menu_item);

	return menu_item;
}

GtkWidget *
gf_menu_item_position(GtkWidget *menu, gint item) {
	GtkWidget *menu_item = NULL, *image = NULL;

	g_return_val_if_fail(menu, NULL);

	switch(item) {
		case GF_ITEM_POSITION_NW:
			image = gtk_image_new_from_stock(GF_STOCK_ITEM_POSITION_NORTH_WEST,
											 GTK_ICON_SIZE_MENU);
			break;
		case GF_ITEM_POSITION_N:
			image = gtk_image_new_from_stock(GF_STOCK_ITEM_POSITION_NORTH,
											 GTK_ICON_SIZE_MENU);
			break;
		case GF_ITEM_POSITION_NE:
			image = gtk_image_new_from_stock(GF_STOCK_ITEM_POSITION_NORTH_EAST,
											 GTK_ICON_SIZE_MENU);
			break;
		case GF_ITEM_POSITION_W:
			image = gtk_image_new_from_stock(GF_STOCK_ITEM_POSITION_WEST,
											 GTK_ICON_SIZE_MENU);
			break;
		case GF_ITEM_POSITION_C:
			image = gtk_image_new_from_stock(GF_STOCK_ITEM_POSITION_CENTER,
											 GTK_ICON_SIZE_MENU);
			break;
		case GF_ITEM_POSITION_E:
			image = gtk_image_new_from_stock(GF_STOCK_ITEM_POSITION_EAST,
											 GTK_ICON_SIZE_MENU);
			break;
		case GF_ITEM_POSITION_SW:
			image = gtk_image_new_from_stock(GF_STOCK_ITEM_POSITION_SOUTH_WEST,
											 GTK_ICON_SIZE_MENU);
			break;
		case GF_ITEM_POSITION_S:
			image = gtk_image_new_from_stock(GF_STOCK_ITEM_POSITION_SOUTH,
											 GTK_ICON_SIZE_MENU);
			break;
		case GF_ITEM_POSITION_SE:
			image = gtk_image_new_from_stock(GF_STOCK_ITEM_POSITION_SOUTH_EAST,
											 GTK_ICON_SIZE_MENU);
			break;
		case GF_ITEM_POSITION_UNKNOWN:
		default:
			break;

	}

	menu_item = gf_menu_make_item(image,
								  gf_item_position_to_string(item, TRUE));

	if(menu_item)
		gtk_menu_shell_append(GTK_MENU_SHELL(menu), menu_item);

	return menu_item;
}

GtkWidget *
gf_menu_item_type(GtkWidget *menu, gint item) {
	GtkWidget *menu_item = NULL;

	switch(item) {
		case GF_ITEM_TYPE_ICON:
			menu_item = gf_menu_make_item(NULL,
										  gf_item_type_to_string(item, TRUE));
			break;
		case GF_ITEM_TYPE_IMAGE:
			menu_item = gf_menu_make_item(NULL,
										  gf_item_type_to_string(item, TRUE));
			break;
		case GF_ITEM_TYPE_TEXT:
			menu_item = gf_menu_make_item(NULL,
										  gf_item_type_to_string(item, TRUE));
			break;
		case GF_ITEM_TYPE_UNKNOWN:
		default:
			break;
	}

	if(menu_item)
		gtk_menu_shell_append(GTK_MENU_SHELL(menu), menu_item);

	return menu_item;
}

GtkWidget *
gf_menu_item_icon_type(GtkWidget *menu, gint item) {
	GtkWidget *menu_item = NULL;

	g_return_val_if_fail(menu, NULL);

	switch(item) {
		case GF_ITEM_ICON_TYPE_BUDDY:
			menu_item = gf_menu_make_item(NULL, _("Buddy"));
			break;
		case GF_ITEM_ICON_TYPE_PROTOCOL:
			menu_item = gf_menu_make_item(NULL, _("Protocol"));
			break;
		case GF_ITEM_ICON_TYPE_STATUS:
			menu_item = gf_menu_make_item(NULL, _("Status"));
			break;
		case GF_ITEM_ICON_TYPE_UNKNOWN:
		default:
			break;
	}

	if(menu_item)
		gtk_menu_shell_append(GTK_MENU_SHELL(menu), menu_item);

	return menu_item;
}

GtkWidget *
gf_menu_item_icon_size(GtkWidget *menu, gint item) {
	GtkWidget *menu_item = NULL, *image;

	g_return_val_if_fail(menu, NULL);

	switch(item) {
		case GF_ITEM_ICON_SIZE_HUGE:
			image = gtk_image_new_from_stock(GF_STOCK_ITEM_ICON_SIZE_HUGE,
											 GTK_ICON_SIZE_MENU);
			menu_item = gf_menu_make_item(image, _("Huge (144x144)"));
			break;
		case GF_ITEM_ICON_SIZE_LARGE:
			image = gtk_image_new_from_stock(GF_STOCK_ITEM_ICON_SIZE_LARGE,
											 GTK_ICON_SIZE_MENU);
			menu_item = gf_menu_make_item(image, _("Large (96x96)"));
			break;
		case GF_ITEM_ICON_SIZE_BIG:
			image = gtk_image_new_from_stock(GF_STOCK_ITEM_ICON_SIZE_BIG,
											 GTK_ICON_SIZE_MENU);
			menu_item = gf_menu_make_item(image, _("Big (64x64)"));
			break;
		case GF_ITEM_ICON_SIZE_NORMAL:
			image = gtk_image_new_from_stock(GF_STOCK_ITEM_ICON_SIZE_NORMAL,
											 GTK_ICON_SIZE_MENU);
			menu_item = gf_menu_make_item(image, _("Normal (48x48)"));
			break;
		case GF_ITEM_ICON_SIZE_LITTLE:
			image = gtk_image_new_from_stock(GF_STOCK_ITEM_ICON_SIZE_LITTLE,
											 GTK_ICON_SIZE_MENU);
			menu_item = gf_menu_make_item(image, _("Little (32x32)"));
			break;
		case GF_ITEM_ICON_SIZE_SMALL:
			image = gtk_image_new_from_stock(GF_STOCK_ITEM_ICON_SIZE_SMALL,
											 GTK_ICON_SIZE_MENU);
			menu_item = gf_menu_make_item(image, _("Small (24x24)"));
			break;
		case GF_ITEM_ICON_SIZE_TINY:
			image = gtk_image_new_from_stock(GF_STOCK_ITEM_ICON_SIZE_TINY,
											 GTK_ICON_SIZE_MENU);
			menu_item = gf_menu_make_item(image, _("Tiny (16x16)"));
			break;
		case GF_ITEM_ICON_SIZE_UNKNOWN:
		default:
			break;
	}

	if(menu_item)
		gtk_menu_shell_append(GTK_MENU_SHELL(menu), menu_item);

	return menu_item;
}

GtkWidget *
gf_menu_item_text_clipping(GtkWidget *menu, gint item) {
	GtkWidget *menu_item = NULL, *image;

	g_return_val_if_fail(menu, NULL);

	switch(item) {
		case GF_ITEM_TEXT_CLIPPING_TRUNCATE:
			image = gtk_image_new_from_stock(GF_STOCK_ITEM_TEXT_CLIPPING_TRUNCATE,
											 GTK_ICON_SIZE_MENU);
			menu_item = gf_menu_make_item(image, _("Truncate"));
			break;
		case GF_ITEM_TEXT_CLIPPING_ELLIPSIS_END:
			image = gtk_image_new_from_stock(GF_STOCK_ITEM_TEXT_CLIPPING_ELLIPSIS_END,
											 GTK_ICON_SIZE_MENU);
			menu_item = gf_menu_make_item(image, _("Ellipsis at the end"));
			break;
		case GF_ITEM_TEXT_CLIPPING_ELLIPSIS_MIDDLE:
			image = gtk_image_new_from_stock(GF_STOCK_ITEM_TEXT_CLIPPING_ELLIPSIS_MIDDLE,
											 GTK_ICON_SIZE_MENU);
			menu_item = gf_menu_make_item(image, _("Ellipsis in the middle"));
			break;
		case GF_ITEM_TEXT_CLIPPING_ELLIPSIS_START:
			image = gtk_image_new_from_stock(GF_STOCK_ITEM_TEXT_CLIPPING_ELLIPSIS_START,
											 GTK_ICON_SIZE_MENU);
			menu_item = gf_menu_make_item(image, _("Ellipsis at the beginning"));
			break;
		case GF_ITEM_TEXT_CLIPPING_UNKNOWN:
		default:
			break;
	}

	if(menu_item)
		gtk_menu_shell_append(GTK_MENU_SHELL(menu), menu_item);

	return menu_item;
}
