/****************************************************************************
 *                           DFIOFilter.cc
 *
 * Author: Matthew Ballance
 * Desc:   Base class for the DFIO Filter class. Each DFIO Filter will
 *         provide an implementation of this class
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 ****************************************************************************/
#include "DFIOFilter.h"
#include "CmdSwitcher.h"
#include "WidgetMgrTypes.h"
#include "WidgetManager.h"
#include <ctype.h>

/************************************************************
 * DFIOFilter_InstCmd()
 ************************************************************/
static Int32 DFIOFilter_InstCmd(
        ClientData        clientData,
        Tcl_Interp       *interp,
        int               argc,
        Char            **argv)
{
    DFIOFilter    *filter = (DFIOFilter *)clientData;
    return filter->InstCmd(argc-1, &argv[1]);
}

/************************************************************
 * DFIOFilter()
 ************************************************************/
DFIOFilter::DFIOFilter(
        Tcl_Interp  *interp,
        const char  *inst_name,
        const char  *name,
        const char  *desc,
        const char  *suffixes) : 
    instName(inst_name), filterName(name), filterDesc(desc),
    filterCapability(FiltCapabilty_Import|FiltCapabilty_Export)
{
    Tcl_Obj          *tmp;
    const char       *dptr, *eptr;
    Uint32      idx = 0;

    this->interp = interp;
    Tcl_CreateCommand(interp, instName.value(), 
            (Tcl_CmdProc *)DFIOFilter_InstCmd, this, 0);

    dptr = suffixes;
    filtSuffixes = Tcl_NewListObj(0, 0);

    /**** Process suffixes string... ****/
    while ((dptr = strchr(dptr, '.'))) {

        eptr = dptr;
        /**** Scan ahead until we hit whitespace... ****/
        while (*eptr && !isspace((int)*eptr)) {
            eptr++;
        }

        /**** Okay, now should have a segment... ****/
        if ((eptr-dptr) > 0) {
            tmp = Tcl_NewStringObj(dptr, eptr-dptr);
            Tcl_ListObjAppendElement(interp, filtSuffixes, tmp);
        }

        dptr = eptr;
    }
}

typedef enum {
    DFC_Import,
    DFC_Export,
    DFC_Id,
    DFC_Suffixes,
    DFC_NumCmds
} DFCCmds;

static CmdSwitchStruct    dfio_filter_cmds[] = {
    { "import",            DFC_Import          },
    { "export",            DFC_Export          },
    { "id",                DFC_Id              },
    { "suffixes",          DFC_Suffixes        },
    { "",                  0                   }
};

/************************************************************
 * InstCmd()
 ************************************************************/
Int32 DFIOFilter::InstCmd(Uint32 argc, Char **argv)
{
    Int32        cmd, ret;
    DFIO        *dfio;
    FileAccObj   fobj("matt", FileAccObj::OpenMode_Read);

    if (argc < 1) {
        Tcl_AppendResult(interp, "too few args", 0);
        return TCL_ERROR;
    }

    cmd = CmdSwitch(dfio_filter_cmds, argv[0]);

    switch (cmd) {
        case DFC_Import:
            if (argc < 3) {
                Tcl_AppendResult(interp, "too few args: import $dfio $file", 0);
                return TCL_ERROR;
            }

            dfio = (DFIO *)WidgetMgr_GetObjHandle(interp,
                    WIDGET_TYPE_DFIO, argv[1]);

            if (!dfio) {
                Tcl_AppendResult(interp, "no DFIO named ", argv[1], 0);
                return TCL_ERROR;
            }

            fobj(argv[2], FileAccObj::OpenMode_Read);
            if (!fobj.okay) {
                Tcl_AppendResult(interp, "cannot open ", argv[2], 0);
                return TCL_ERROR;
            }

            ret = Import(&dfio, fobj);
            fobj.close();

            if (ret < 0) {
                return TCL_ERROR;
            }
            break;

       /**** export $dfio $file ****/
        case DFC_Export:
            if (argc < 3) {
                Tcl_AppendResult(interp, "too few args: export $dfio $file", 0);
                return TCL_ERROR;
            }

            dfio = (DFIO *)WidgetMgr_GetObjHandle(interp,
                    WIDGET_TYPE_DFIO, argv[1]);
            
            if (!dfio) {
                Tcl_AppendResult(interp, "no DFIO named ", argv[1], 0);
                return TCL_ERROR;
            }

            fobj(argv[2], FileAccObj::OpenMode_Write);

            if (!fobj.okay) {
                Tcl_AppendResult(interp, "cannot open ", argv[2], 0);
                return TCL_ERROR;
            }

            ret = Export(&dfio, fobj);
            fobj.close();

            if (ret < 0) {
                Tcl_AppendResult(interp, "export failed", 0);
                return TCL_ERROR;
            }

            break;

        case DFC_Id:
            Tcl_SetObjResult(interp, Tcl_NewStringObj(filterName.value(),
                        filterName.length()));
            break;

        case DFC_Suffixes:
            Tcl_SetObjResult(interp, Tcl_DuplicateObj(filtSuffixes));
            break;

        default:
            Tcl_AppendResult(interp, "unknown filter sub-cmd ", argv[0], 0);
            break;
    }

    return TCL_OK;
}


