// =============================================================================
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviScriptObjectController"

#include "kvi_debug.h"
#include "kvi_error.h"
#include "kvi_script_button.h"
#include "kvi_script_checkbox.h"
#include "kvi_script_combobox.h"
#include "kvi_script_dns.h"
#include "kvi_script_file.h"
#include "kvi_script_label.h"
#include "kvi_script_layout.h"
#include "kvi_script_lineedit.h"
#include "kvi_script_objectclassdefinition.h"
#include "kvi_script_objectcontroller.h"
#include "kvi_script_object.h"
#include "kvi_script_popupmenu.h"
#include "kvi_script_qtwrapper.h"
#include "kvi_script_socket.h"
#include "kvi_script_spinbox.h"
#include "kvi_script_splitter.h"
#include "kvi_script_tabwidget.h"
#include "kvi_script_textedit.h"
#include "kvi_script_timer.h"
#include "kvi_script_toolbar.h"
#include "kvi_script_toolbutton.h"
#include "kvi_script_treeview.h"
#include "kvi_script_treeview_item.h"
#include "kvi_script_widget.h"
#include "kvi_userparser.h"

// Global class definitions
KviScriptObjectClassDefinition      *g_pClassDefinitions  = 0;
// Global list of object controllers
QPtrList<KviScriptObjectController> *g_pObjectControllers = 0;

/**
 * This will overflow at 2^32 (4.294.967.296) and when it happens,
 * the lowest numbered objects will SURELY be dead.
 * Otherwise all the objects will occupy several terabytes of memory.
 */
static unsigned int g_uObjectUniqueId = 1;

/**
 *  Script object controller
 *  Unique for each KviFrame object
 */
KviScriptObjectController::KviScriptObjectController(KviUserParser *pParser)
{
	m_pUserParser     = pParser;
	m_pTopLevelObject = 0;

	// First one creates the list
	if( !g_pObjectControllers ) {
		// Global init
		g_pObjectControllers = new QPtrList<KviScriptObjectController>;
		g_pObjectControllers->setAutoDelete(false);
		allocateBuiltInClassDefinitions();
	}

	// Register ourself
	g_pObjectControllers->append(this);

	// And create the toplevel object
	createTopLevelObject();
}

KviScriptObjectController::~KviScriptObjectController()
{
	// Remove the children one by one
	if( m_pTopLevelObject ) {
		delete m_pTopLevelObject;
		m_pTopLevelObject = 0;
	}

	// Last one removes the list
	g_pObjectControllers->removeRef(this);
	if( g_pObjectControllers ) {
		if( g_pObjectControllers->count() == 0 ) {
			// The last one out closes the door
			delete g_pObjectControllers;
			g_pObjectControllers = 0;
			delete g_pClassDefinitions;
			g_pClassDefinitions = 0;
		}
	}
}

KviFrame *KviScriptObjectController::mainFrame()
{
	return m_pUserParser->m_pFrm;
}

void KviScriptObjectController::createTopLevelObject()
{
	m_pTopLevelObject = new KviScriptObject(this, 0, "frame", g_pClassDefinitions);
}

void KviScriptObjectController::dumpClasses(KviWindow *wnd)
{
	g_pClassDefinitions->dump(wnd);
}

void KviScriptObjectController::dumpObjects(KviWindow *wnd)
{
	m_pTopLevelObject->dump(wnd);
}

void KviScriptObjectController::allocateBuiltInClassDefinitions()
{
	// Object
	g_pClassDefinitions = new KviScriptObjectClassDefinition("object", 0, true);
	KviScriptObject::initializeClassDefinition(g_pClassDefinitions);
	// Classes that derive just from object
	void *misc_classes[] =
	{
		// Name,                  init function
		(void *) "dns",           (void *) KviScriptDns::initializeClassDefinition,
		(void *) "file",          (void *) KviScriptFile::initializeClassDefinition,
		(void *) "layout",        (void *) KviScriptLayout::initializeClassDefinition,
		(void *) "socket",        (void *) KviScriptSocket::initializeClassDefinition,
		(void *) "timer",         (void *) KviScriptTimer::initializeClassDefinition,
		(void *) "treeviewitem",  (void *) KviScriptTreeViewItem::initializeClassDefinition
	};
	// Classes that derive from widget
	void *widget_classes[] =
	{
		// Name,                  init function
		(void *) "button",        (void *) KviScriptButton::initializeClassDefinition,
		(void *) "checkbox",      (void *) KviScriptCheckBox::initializeClassDefinition,
		(void *) "combobox",      (void *) KviScriptComboBox::initializeClassDefinition,
		(void *) "label",         (void *) KviScriptLabel::initializeClassDefinition,
		(void *) "lineedit",      (void *) KviScriptLineEdit::initializeClassDefinition,
		(void *) "popupmenu",     (void *) KviScriptPopupMenu::initializeClassDefinition,
		(void *) "qtwrapper",     (void *) KviScriptQtWrapper::initializeClassDefinition,
		(void *) "spinbox",       (void *) KviScriptSpinBox::initializeClassDefinition,
		(void *) "splitter",      (void *) KviScriptSplitter::initializeClassDefinition,
		(void *) "tabwidget",     (void *) KviScriptTabWidget::initializeClassDefinition,
		(void *) "textedit",      (void *) KviScriptTextEdit::initializeClassDefinition,
		(void *) "toolbar",       (void *) KviScriptToolBar::initializeClassDefinition,
		(void *) "toolbutton",    (void *) KviScriptToolButton::initializeClassDefinition,
		(void *) "treeview",      (void *) KviScriptTreeView::initializeClassDefinition
	};

	KviScriptObjectClassDefinition *d1 = 0;
	void (*pFunc) (KviScriptObjectClassDefinition *) = 0;
	for( int i = 0; i < (int) (sizeof(misc_classes) / sizeof(misc_classes[0])); i++ ) {
		d1 = new KviScriptObjectClassDefinition((char *) misc_classes[i], g_pClassDefinitions, true);
		g_pClassDefinitions->addChildClassDefinition(d1);
		pFunc = (void(*) (KviScriptObjectClassDefinition *)) misc_classes[++i];
		pFunc(d1);
	}

	// Widget
	d1 = new KviScriptObjectClassDefinition("widget", g_pClassDefinitions, true);
	KviScriptWidget::initializeClassDefinition(d1);
	g_pClassDefinitions->addChildClassDefinition(d1);

	// OK, the widget is ready, now we can add classes that derive from it
	KviScriptObjectClassDefinition *d2 = 0;
	for( int i = 0; i < (int) (sizeof(widget_classes) / sizeof(widget_classes[0])); i++ ) {
		d2 = new KviScriptObjectClassDefinition((char *) widget_classes[i], d1, true);
		d1->addChildClassDefinition(d2);
		pFunc = (void(*) (KviScriptObjectClassDefinition *)) widget_classes[++i];
		pFunc(d2);
	}
}

void KviScriptObjectController::globalReset()
{
	for( KviScriptObjectController *c = g_pObjectControllers->first(); c; c = g_pObjectControllers->next() ) {
		delete c->m_pTopLevelObject;
		c->m_pTopLevelObject = 0;
	}

	__range_valid(g_pClassDefinitions);
	if( g_pClassDefinitions ) {
		delete g_pClassDefinitions;
		g_pClassDefinitions = 0;
		allocateBuiltInClassDefinitions();
	}

	for(KviScriptObjectController *c = g_pObjectControllers->first(); c; c = g_pObjectControllers->next() ) {
		c->createTopLevelObject();
	}
}

bool KviScriptObjectController::killClass(const char *szClassName)
{
	KviScriptObjectClassDefinition *d = lookupClassDefinition(szClassName);
	if( !d ) return false;

	killClass(d);
	return true;
}

void KviScriptObjectController::killClass(KviScriptObjectClassDefinition *d)
{
	// First of all, delete the derived classes
	if( d->childClasses() ) {
		QPtrList<KviScriptObjectClassDefinition> l(*(d->childClasses()));
		for( KviScriptObjectClassDefinition *chdef = l.first(); chdef; chdef = l.next() ) {
			killClass(chdef);
		}
	}

	// Now remove all the objects with THIS class
	for( KviScriptObjectController *c = g_pObjectControllers->first(); c; c = g_pObjectControllers->next() ) {
		c->killAllLocalObjectsWithClass(d->getClass());
		if( !(c->m_pTopLevelObject) ) {
			// We were deleting the "object" class (so the toplevel object too!)
			__range_valid(d == g_pClassDefinitions); // Object class destroyed!
			c->createTopLevelObject();
		}
	}

	// Finally delete the class definition, if it is not built-in
	if( !(d->isBuiltin()) ) {
		// Has at least one parent class!
		d->inheritedClass()->childClasses()->removeRef(d); // Oh! A "deleted" path! :)
	}
}

void KviScriptObjectController::killAllLocalObjectsWithClass(const char *szClassName)
{
	KviScriptObject *o;
	while( (o = findLocalObjectByClass(szClassName)) )
		delete o;
}

KviScriptObject *KviScriptObjectController::topLevelObject()
{
	return m_pTopLevelObject;
}

/**
 * Application-wide unique ID
 */
void KviScriptObjectController::getUniqueId(KviStr &idBuf)
{
	idBuf.setNum(g_uObjectUniqueId);
	idBuf.prepend("@");
	g_uObjectUniqueId++;
}

KviScriptObjectClassDefinition *KviScriptObjectController::lookupClassDefinition(const char *szClass)
{
	if( !g_pClassDefinitions ) {
		debug("No existing class definitions");
		return 0;
	}
	if( kvi_strEqualCI(g_pClassDefinitions->getClass(), szClass) )
		return g_pClassDefinitions;
	return g_pClassDefinitions->lookupChildClassDefinition(szClass);
}

KviScriptObject *KviScriptObjectController::allocateObject(
	const char *szClass, const char *szName, KviScriptObject *par, QPtrList<KviStr> *params, bool *pbDeletedParams)
{
	// Built-in classes
	__range_valid(par);

	KviScriptObject *o = 0;
	KviScriptObjectClassDefinition *d = lookupClassDefinition(szClass);
	if( d ) {
		KviScriptObjectClassDefinition *builtin = d;
		while( !builtin->isBuiltin() ) {
			builtin = builtin->inheritedClass();
			if( !builtin )
				debug("Oops... something went wrong with class definitions?");
		}
		if( kvi_strEqualCI(builtin->getClass(), "object") ) {
			o = new KviScriptObject(this, par, szName, d);
		} else if( kvi_strEqualCI(builtin->getClass(), "socket") ) {
			o = new KviScriptSocket(this, par, szName, d);
		} else if( kvi_strEqualCI(builtin->getClass(), "timer") ) {
			o = new KviScriptTimer(this, par, szName, d);
		} else if( kvi_strEqualCI(builtin->getClass(), "dns") ) {
			o = new KviScriptDns(this, par, szName, d);
		} else if( kvi_strEqualCI(builtin->getClass(), "widget") ) {
			o = new KviScriptWidget(this, par, szName, d);
			if( !(((KviScriptWidget *) o)->init(0)) ) {
				// Failed to create it
				delete o;
				return 0;
			}
		} else if( kvi_strEqualCI(builtin->getClass(), "file") ) {
			o = new KviScriptFile(this, par, szName, d);
			if( !(((KviScriptFile *) o)->init(params)) ) {
				// Failed to create it
				delete o;
				return 0;
			}
		} else if( kvi_strEqualCI(builtin->getClass(), "button") ) {
			o = new KviScriptButton(this, par, szName, d);
			if( !(((KviScriptButton *) o)->init(params)) ) {
				// Failed to create it
				delete o;
				return 0;
			}
		} else if( kvi_strEqualCI(builtin->getClass(), "splitter") ) {
			o = new KviScriptSplitter(this, par, szName, d);
			if( !(((KviScriptSplitter *) o)->init(params)) ) {
				// Failed to create it
				delete o;
				return 0;
			}
		} else if( kvi_strEqualCI(builtin->getClass(), "popupmenu") ) {
			o = new KviScriptPopupMenu(this, par, szName, d);
			if( !(((KviScriptPopupMenu *) o)->init(params)) ) {
				// Failed to create it
				delete o;
				return 0;
			}
		} else if( kvi_strEqualCI(builtin->getClass(), "toolbutton") ) {
			o = new KviScriptToolButton(this, par, szName, d);
			if( !(((KviScriptToolButton *) o)->init(params)) ) {
				// Failed to create it
				delete o;
				return 0;
			}
		} else if( kvi_strEqualCI(builtin->getClass(), "label") ) {
			o = new KviScriptLabel(this, par, szName, d);
			if( !(((KviScriptLabel *) o)->init(params)) ) {
				// Failed to create it
				delete o;
				return 0;
			}
		} else if( kvi_strEqualCI(builtin->getClass(), "layout") ) {
			o = new KviScriptLayout(this, par, szName, d);
			if( !(((KviScriptLayout *) o)->init(0)) ) {
				// Failed to create it
				delete o;
				return 0;
			}
		} else if( kvi_strEqualCI(builtin->getClass(), "lineedit") ) {
			o = new KviScriptLineEdit(this, par, szName, d);
			if( !(((KviScriptLineEdit *) o)->init(params)) ) {
				// Failed to create it
				delete o;
				return 0;
			}
		} else if( kvi_strEqualCI(builtin->getClass(), "textedit") ) {
			o = new KviScriptTextEdit(this, par, szName, d);
			if( !(((KviScriptTextEdit *) o)->init(params)) ) {
				// Failed to create it
				delete o;
				return 0;
			}
		} else if( kvi_strEqualCI(builtin->getClass(), "toolbar") ) {
			o = new KviScriptToolBar(this, par, szName, d);
			if( !(((KviScriptToolBar *) o)->init(params)) ) {
				// Failed to create it
				delete o;
				return 0;
			}
		} else if( kvi_strEqualCI(builtin->getClass(), "qtwrapper") ) {
			o = new KviScriptQtWrapper(this, par, szName, d);
			if( !(((KviScriptQtWrapper *) o)->init(params)) ) {
				// Failed to create it
				delete o;
				return 0;
			}
		} else if( kvi_strEqualCI(builtin->getClass(), "checkbox") ) {
			o = new KviScriptCheckBox(this, par, szName, d);
			if( !(((KviScriptCheckBox *) o)->init(params)) ) {
				// Failed to create it
				delete o;
				return 0;
			}
		} else if( kvi_strEqualCI(builtin->getClass(), "spinbox") ) {
			o = new KviScriptSpinBox(this, par, szName, d);
			if( !(((KviScriptSpinBox *) o)->init(params)) ) {
				// Failed to create it
				delete o;
				return 0;
			}
		} else if( kvi_strEqualCI(builtin->getClass(), "combobox") ) {
			o = new KviScriptComboBox(this, par, szName, d);
			if( !(((KviScriptComboBox *) o)->init(params)) ) {
				// Failed to create it
				delete o;
				return 0;
			}
		} else if( kvi_strEqualCI(builtin->getClass(), "tabwidget") ) {
			o = new KviScriptTabWidget(this, par, szName, d);
			if( !(((KviScriptTabWidget *) o)->init(params)) ) {
				// Failed to create it
				delete o;
				return 0;
			}
		} else if( kvi_strEqualCI(builtin->getClass(), "treeviewitem") )
		{
			o = new KviScriptTreeViewItem(this, par, szName, d);
			if( !(((KviScriptTreeViewItem *) o)->init(params)) ) {
				delete o;
				return 0;
			}
		} else if( kvi_strEqualCI(builtin->getClass(), "treeview") ) {
			o = new KviScriptTreeView(this, par, szName, d);
			if( !(((KviScriptTreeView *) o)->init(params)) ) {
				delete o;
				return 0;
			}
		}
		// Need to insert the default event handlers from all the inherited classes
		KviScriptObjectClassDefinition *def = d;
		while( def ) {
			for( KviScriptEventStruct *s = def->eventList()->first(); s; s = def->eventList()->next() ) {
				// Insert unless we have already got it from a class lower in inheritance tree
				if( !(o->hasEventHandler(s->szName.ptr())) )
					o->setEventHandler(s->szName.ptr(), s->szBuffer.ptr());
			}
			def = def->inheritedClass();
		}
	} else return 0;

	if( o ) {
		KviStr buffer;
		*pbDeletedParams = true; // The params will be deleted by callFunction!

		if( o->callFunction("constructor", params, buffer) != KVI_ERROR_Success ) {
			debug("Cannot execute object constructor: implementation failure");
			delete o;
			return 0;
		}

		if( kvi_strEqualCI(buffer.ptr(), "0") ) {
			debug("Object constructor failed: return value 0");
			delete o;
			return 0;
		}
	}
	return o;
}

KviScriptObject *KviScriptObjectController::findLocalObjectByClass(const char *szClass)
{
	if( kvi_strEqualCI(m_pTopLevelObject->getClass(), szClass) )
		return m_pTopLevelObject;
	return m_pTopLevelObject->findObjectByClass(szClass);
}

KviScriptObject *KviScriptObjectController::findObjectById(const char *szId)
{
	KviScriptObject *o;
	for( KviScriptObjectController *c = g_pObjectControllers->first(); c; c = g_pObjectControllers->next() ) {
		if( kvi_strEqualCI(c->m_pTopLevelObject->id(), szId) )
			return c->m_pTopLevelObject;
		o = c->m_pTopLevelObject->findObjectById(szId);
		if( o ) return o;
	}
	return 0;
}
