/* $Id: display.c,v 1.13 2000/01/24 00:37:41 fraserm Exp $
   $Log: display.c,v $
   Revision 1.13  2000/01/24 00:37:41  fraserm
   added 'e' chunk

   Revision 1.12  1999/12/22 01:53:57  fraserm
   fixed long descriptions

   Revision 1.11  1999/12/22 01:36:12  fraserm
   fixed rare core dump when stat() failed in newfnode()

   Revision 1.10  1999/12/21 02:42:12  fraserm
   changes as per 0.3 in CHANGELOG

   Revision 1.9  1999/12/14 02:07:01  fraserm
   fixed long date by removing spaces from display format

   Revision 1.8  1999/07/01 01:40:33  fraserm
   *** empty log message ***

   Revision 1.7  1999/06/15 23:45:56  fraserm
   fixed the COMMA_JUST bug (removed initial comma)

   Revision 1.6  1999/05/30 01:07:59  fraserm
   final fix to ensure that when filename is last, it isn't padded with spaces

   Revision 1.5  1999/05/26 23:51:04  fraserm
   added custom command stuff

   Revision 1.4  1999/05/16 19:47:33  fraserm
   changed n N and L to n +n N and +N

   Revision 1.3  1999/04/26 01:15:12  fraserm
   added d and D chunks

   Revision 1.2  1999/04/24 17:22:29  fraserm
   added raw timestamp

   Revision 1.1  1999/04/20 23:42:49  fraserm
   Initial revision

*/

/* display functions */

#include <stdio.h>
#include <sys/stat.h>
#include <sys/sysmacros.h>
#include <unistd.h>
#include <sys/types.h>
#include <errno.h>
#include <time.h>
#include <dirent.h>
#include <limits.h> /* needed for Solaris */
#include "limo.h"

extern int numcustcmds; /* number of custom commands defined */
extern int euid, egid;

/* display a time as an age, return the chunk width */
int disp_age(time_t t, int lng, int options, time_t ctime)
{
  long di, d; /* differences */
  char pm; /* plus or minus */
  int doprint = !(options & NOPRINT);

  if (doprint) {
    di = t - ctime;
    if (di < 0) {
      pm = ' ';
      d = -di;
    }
    else {
      pm = '+';
      d = di;
    }
  }
  if (lng == TIME_SHORT) {
    if (doprint) {
      if (d < 60*60) { /* < 60 minutes */
	printf("%c%02ldm%02ld", pm, d / 60, d % 60);
      }
      else if (d < 24*60*60) { /* < 24 hours */
	d = (d + 30) / 60; /* round to nearest minute */
	printf("%c%02ldh%02ld", pm, d / 60, d % 60);
      }
      else if (d < 100*24*60*60) { /* < 100 days */
	d = (d + 30*60) / (60*60); /* round to nearest hour */
	printf("%c%02ldd%02ld", pm, d / 24, d % 24);
      }
      else {
	printf("%c%4ldd", pm, (d + 12*60*60) / (24*60*60));
      }
    }
    return 7;
  }
  else {
    time_t hours, mins, secs;

    if (doprint) {
      secs = d % 60; d = d / 60;
      mins = d % 60; d = d / 60;
      hours = d % 24; d = d / 24;
      /* now d is the number of days */
      printf("%c%3ldd-%02ld:%02ld:%02ld", pm, d, hours, mins, secs);
    }
    return 14;
  }
}

/* display a time, return the chunk width */
int disp_time(time_t t, int lng, int options, time_t now)
{
  struct tm *tb; /* the broken-out time */
  time_t d; /* difference */
  char buf[32]; /* for strftime(), should be long enough */
  int doprint = !(options & NOPRINT);

  /* first, break out the time fields */
  tb = localtime(&t);
  if (lng == TIME_SHORT) {
    if (doprint) {
      if (tb == NULL) { /* manual doesn't mention this possibility, but lets
			   check anyway */
	printf("%-5s", "error");
      }
      else {
	if (t > now) { /* time is in the future */
	  strftime(buf, 32, "!%Y", tb);
	  printf("%5s", buf);
	}
	else {
	  d = now - t; /* difference */
	  if (d < 24*60*60) { /* < 24 hours */
	    strftime(buf, 32, "%H:%M", tb);
	  }
	  else if (d < 365*24*60*60) { /* < 1 year */
	    strftime(buf, 32, "%d%b", tb);
	  }
	  else if (d < 10*365*24*60*60) { /* < 50 years */
	    strftime(buf, 32, "%b%y", tb);
	  }
	  else { /* >= 100 years - yeah, right, will never reach this until
		    the clock is 64-bit */
	    strftime(buf, 32, "%Y", tb);
	  }
	  printf("%5s", buf);
	}
      }
    }
    return 5;
  }
  else {
    if (doprint) {
      if (tb == NULL) { /* manual doesn't mention this possibility, but lets
			   check anyway */
	printf("%-24s", "error");
      }
      else {
	strftime(buf, 32, "%a-%d-%b-%Y-%H:%M:%S", tb);
	printf("%-24s", buf);
      }
    }
    return 24;
  }
}

/* display an individual file
   returns the standard width of all output chunks */

int display_file(int argc, char *argv[], struct fnode *f, struct totals *tot,
		 char *format,
		 struct stat *cfile, int options, char *wd, int wdlen,
		 int namelen, int linklen, int *custlens, time_t now)
{
  char *i, *j;
  static char fnbuf[PATH_MAX + 1];
  static char lnbuf[PATH_MAX + 1];
  mode_t tm;
  off_t tb;
  struct passwd *pwd;
  struct group *grp;
  int totwid = 0;
  int doprint = !(options & NOPRINT); /* don't print for first one; it just
					 gets the length */
  int nameprintlen; /* what length to use to print the name */
  int relative = 0; /* whether times are shown as age or date */
  int rawtime = 0; /* whether long times are shown as raw */
  int showlink = 0; /* whether to show a link target */
  int lastchunk = 0; /* set when this is the last chunk */
  int custnum; /* which custom command this is */
  char r_eff, w_eff, x_eff; /* effective read, write and execute perms */
  char tempc, *temps; /* temporary characters and strings */

  /* first, compute totals if required */
  if (options & SHOW_TOTALS
      && !(options & NOPRINT)) {
    ++(tot->count);
    tot->bytes += f->s->st_size;
    tot->blocks += f->s->st_blocks;
  }
  /* print each chunk in turn */
  for (i = format; *i != '\0'; ++i) {
    /* if (*(i+1) == '\0') {
       lastchunk = 1;
       }
       if (totwid != 0 && *i != RELATIVE && *i != RAWTIME && *i != LINK) {
       if (doprint) fputs(" ", stdout);
       ++totwid;
       }
    */
    if (*i == OPT_PREFIX) { /* check for a format specifier */
      ++i; /* assume that this still points to a character, rather than end of
	      the string; it's checked in main() or set this way in main(),
	      don't want to waste time checking it again here */
      switch(*i) {
      case NAME:
      case NAMEID:
	if (*i == NAME) {
	  /* yes I know, these need tidying, I'll do a function or
	     macro later */
	  if (options & FULL_PATHNAMES && *(f->name) != '/') {
	    sprintf(fnbuf, "%s/%s", wd, f->name);
	    nameprintlen = wdlen + 1 + namelen;
	  }
	  else {
	    sprintf(fnbuf, "%s", f->name);
	    nameprintlen = namelen;
	  }
	}
	else {
	  if (f->s == NULL) { /* an error may have caused this */
	    tempc = '!';
	  }
	  else {
	    tm = f->s->st_mode;
	    tempc = (S_ISLNK(tm) ? '@' :
		     (S_ISDIR(tm) ? '/' :
		      (S_ISFIFO(tm) ? '|' :
		       (S_ISCHR(tm) ? '&' :
			(S_ISBLK(tm) ? '#' :
			 (S_ISSOCK(tm) ? ':' : ' '))))));
	  }
	  if (options & FULL_PATHNAMES && *(f->name) != '/') {
	    sprintf(fnbuf, "%s/%s%c", wd, f->name, tempc);
	    nameprintlen = wdlen + 1 + namelen + 1;
	  }
	  else {
	    sprintf(fnbuf, "%s%c", f->name, tempc);
	    nameprintlen = namelen + 1;
	  }
	}
	totwid += nameprintlen;
	if (showlink) {
	  nameprintlen += 4 + linklen;
	  totwid += 4 + linklen;
	  if (S_ISLNK(f->s->st_mode)) {
	    sprintf(lnbuf, " -> %s", f->linktarget);
	    strcat(fnbuf, lnbuf);
	  }
	}
	if (options & LAST_IN_ROW
	    && *(i+1) == '\0') { /* last file on the row, last chunk */
	  nameprintlen = 1;
	}
	if (options & QUOTE_UNPRINTABLE) {
	  for (j = fnbuf; *j != '\0'; ++j) {
	    if (*j < 32 || *j == 127) {
	      *j = '?';
	    }
	  }
	}
	if (doprint) printf("%-*s", nameprintlen, fnbuf);
	break;
      case '0': /* custom fields */
      case '1':
      case '2':
      case '3':
      case '4':
      case '5':
      case '6':
      case '7':
      case '8':
      case '9':
	custnum = *i - '0';
	if (custnum >= numcustcmds) {
	  if (doprint) printf("%c?", *i);
	  totwid += 2;
	}
	else {
	  if (doprint) printf("%*s", custlens[custnum], f->custvals[custnum]);
	  totwid += custlens[custnum];
	}
	break;
      case DESCSHORT:
	if (doprint) {
	  if (f->s == NULL) {
	    temps = "Err!";
	  }
	  else {
	    tm = f->s->st_mode;
	    temps = (S_ISLNK(tm) ? "Syml" :
		     (S_ISREG(tm) ? "File" :
		      (S_ISDIR(tm) ? "Dir" :
		       (S_ISCHR(tm) ? "ChSp" :
			(S_ISBLK(tm) ? "BlSp" :
			 (S_ISFIFO(tm) ? "FIFO" :
			  (S_ISSOCK(tm) ? "Sock" :
			   ("????"))))))));
	  }
	  printf("%-4s", temps);
	}
	totwid += 4;
	break;
      case DESCLONG:
	if (f->s == NULL) {
	  temps = "Error; type unknown";
	}
	else {
	  tm = f->s->st_mode;
	  temps = (S_ISLNK(tm) ? "Symbolic link" :
		   (S_ISREG(tm) ? "File" :
		    (S_ISDIR(tm) ? "Directory" :
		     (S_ISCHR(tm) ? "Character Special" :
		      (S_ISBLK(tm) ? "Block Special" :
		       (S_ISFIFO(tm) ? "FIFO" :
			(S_ISSOCK(tm) ? "Socket" :
			 ("????"))))))));
	}
	if (doprint) printf("%-17s", temps);
	totwid += 17;
	break;
      case SIZEBLOCKS:
	if (doprint) printf("%7ld", f->s->st_blocks);
	totwid += 7;
	break;
      case BYTESSHORT:
	if (doprint) {
	  tb = f->s->st_size;
	  if (tb <= 999) {
	    printf("%3ldb", tb);
	  }
	  else if (tb <= 9.9*1024) {
	    printf("%3.1fK", (float) (tb+512/10) / 1024);
	  }
	  else if (tb <= 999*1024) {
	    printf("%3ldK", (tb+512-1) / 1024);
	  }
	  else if (tb <= 9.9*1024*1024) {
	  printf("%3.1fM", (float) (tb+512*1024/10) / (1024*1024));
	  }
	  else if (tb <= 999*1024*1024) {
	    printf("%3ldM", (tb+512*1024) / (1024 * 1024));
	  }
	  else if (tb <= 9.9*1024*1024*1024) {
	    printf("%3.1fG", (float) (tb+512*1024*1024/10) / (1024*1024*1024));
	  }
	  else {
	    printf("%3ldG", (tb+512*1024*1024) / (1024 * 1024 * 1024));
	  }
	}
	totwid += 3;
	break;
      case BYTESLONG:
	if (doprint) {
	  if (S_ISCHR(f->s->st_mode) || S_ISBLK(f->s->st_mode)) {
	    printf("   %3d,%3d", major(f->s->st_rdev), minor(f->s->st_rdev));
	  }
	  else {
	    printf("%10ld", f->s->st_size);
	  }
	}
	totwid += 10;
	break;
      case MODESHORT:
	if (doprint) printf("%06o", f->s->st_mode);
	totwid += 6;
	break;
      case MODELONG:
	if (doprint) {
	  if (f->s == NULL) {
	    printf("??????????");
	  }
	  else {
	    tm = f->s->st_mode;
	    printf("%c%c%c%c%c%c%c%c%c%c",
		   S_ISDIR(tm) ? 'd' :
		   (S_ISCHR(tm) ? 'c' :
		    (S_ISBLK(tm) ? 'b' :
		     (S_ISLNK(tm) ? 'l' : '-'))),
		   (tm & S_IRUSR) ? 'r' : '-',
		   (tm & S_IWUSR) ? 'w' : '-',
		   (tm & S_IXUSR) ? ((tm & S_ISUID) ? 's' : 'x') : ((tm & S_ISUID) ? 'S' : '-'),
		   (tm & S_IRGRP) ? 'r' : '-',
		   (tm & S_IWGRP) ? 'w' : '-',
		   (tm & S_IXGRP) ? ((tm & S_ISGID) ? 's' : 'x') : ((tm & S_ISGID) ? 'S' : '-'),
		   (tm & S_IROTH) ? 'r' : '-',
		   (tm & S_IWOTH) ? 'w' : '-',
		   (tm & S_IXOTH) ? 'x' : '-');
	  }
	}
	totwid += 10;
	break;
      case EFFECTIVE_SHORT:
	if (doprint) {
	  if (f->s == NULL) {
	    printf("???");
	  }
	  else {
	    tm = f->s->st_mode;
	    if (euid == f->s->st_uid) { /* owned by user */
	      r_eff = (S_IRUSR & tm) ? 'r' : '-';
	      w_eff = (S_IWUSR & tm) ? 'w' : '-';
	      x_eff = (S_IXUSR & tm) ? 'x' : '-';
	    }
	    else if (egid == f->s->st_gid) { /* owned by group */
	      r_eff = (S_IRGRP & tm) ? 'r' : '-';
	      w_eff = (S_IWGRP & tm) ? 'w' : '-';
	      x_eff = (S_IXGRP & tm) ? 'x' : '-';
	    }
	    else { /* owned by someone else entirely */
	      r_eff = (S_IROTH & tm) ? 'r' : '-';
	      w_eff = (S_IWOTH & tm) ? 'w' : '-';
	      x_eff = (S_IXOTH & tm) ? 'x' : '-';
	    }
	    printf("%c%c%c", r_eff, w_eff, x_eff);
	  }
	}
	totwid += 3;
	break;
      case OWNERNUMBER:
	if (doprint) printf("%5d", f->s->st_uid);
	totwid += 5;
	break;
      case OWNERNAME:
	if (doprint) printf("%-8s", f->username);
	totwid += 8;
	break;
      case GROUPNUMBER:
	if (doprint) printf("%5d", f->s->st_gid);
	totwid += 5;
	break;
      case GROUPNAME:
	if (doprint) printf("%-8s", f->groupname);
	totwid += 8;
	break;
      case INODE:
	if (doprint) printf("%8ld", f->s->st_ino);
	totwid += 8;
      break;
      case LINKS:
	if (doprint) printf("%4d", f->s->st_nlink);
	totwid += 4;
	break;
      case ACCESSSHORT:
      case ACCESSLONG:
	if (rawtime) {
	  if (relative) {
	    if (doprint) printf("%10d",
				f->s->st_atime - ((options & USE_COMPARE_FILE)
						  ? cfile->st_atime
						  : now));
	  relative = 0;
	  }
	  else {
	    if (doprint) printf("%10d", f->s->st_atime);
	  }
	  totwid += 10;
	  rawtime = 0;
	}
	else {
	  if (relative) {
	    totwid += disp_age(f->s->st_atime,
			       (*i == ACCESSSHORT)?TIME_SHORT:TIME_LONG,
			       options,
			       (options & USE_COMPARE_FILE)
			       ? cfile->st_atime
			       : now);
	    relative = 0;
	  }
	  else {
	    totwid += disp_time(f->s->st_atime,
				(*i == ACCESSSHORT)?TIME_SHORT:TIME_LONG,
				options, now);
	  }
	}
	break;
      case MODIFYSHORT:
      case MODIFYLONG:
	if (rawtime) {
	  if (relative) {
	    if (doprint) printf("%10d",
				f->s->st_mtime - ((options & USE_COMPARE_FILE)
						  ? cfile->st_mtime
						  : now));
	    relative = 0;
	  }
	  else {
	    if (doprint) printf("%10d", f->s->st_mtime);
	  }
	  totwid += 10;
	  rawtime = 0;
	}
	else {
	  if (relative) {
	    totwid += disp_age(f->s->st_mtime,
			       (*i == MODIFYSHORT)?TIME_SHORT:TIME_LONG,
			       options,
			       (options & USE_COMPARE_FILE)
			       ? cfile->st_mtime
			       : now);
	    relative = 0;
	  }
	  else {
	    totwid += disp_time(f->s->st_mtime,
				(*i == MODIFYSHORT)?TIME_SHORT:TIME_LONG,
				options, now);
	  }
	}
	break;
      case CHANGESHORT:
      case CHANGELONG:
	if (rawtime) {
	  if (relative) {
	    if (doprint) printf("%10d",
				f->s->st_ctime - ((options & USE_COMPARE_FILE)
						  ? cfile->st_ctime
						  : now));
	    relative = 0;
	  }
	  else {
	    if (doprint) printf("%10d", f->s->st_ctime);
	  }
	  totwid += 10;
	  rawtime = 0;
	}
	else {
	  if (relative) {
	    totwid += disp_age(f->s->st_ctime,
			       (*i == CHANGESHORT)?TIME_SHORT:TIME_LONG,
			       options,
			       (options & USE_COMPARE_FILE)
			       ? cfile->st_ctime
			       : now);
	    relative = 0;
	  }
	  else {
	    totwid += disp_time(f->s->st_ctime,
				(*i == CHANGESHORT)?TIME_SHORT:TIME_LONG,
				options, now);
	  }
	}
	break;
      case RELATIVE:
	relative = 1; /* show next time as an age */
	break;
      case RAWTIME:
	rawtime = 1; /* show the next long time in the raw */
	break;
      case LINK:
	showlink = 1; /* show the link target */
	break;
      default:
	if (doprint) printf("%c?", *i);
	totwid += 2;
	break;
      }
      }
    else { /* not an option specifier, just a plain jane character */
      if (doprint)
	putchar(*i);
      ++totwid;
    }
  }
  return totwid;
}

/* display a list of files */

void display_file_list(int argc, char *argv[], struct flist *list,
		       int style, int width, char *format, struct stat *cfile,
		       int options, time_t now)
{
  int i, j, fwidth, cols, rows, curcol;
  struct totals ltotal;
  char wdbuf[PATH_MAX]; /* buffer for the current working directory */
  int wdlen;

  if (options & SHOW_TOTALS) {
    memset(&ltotal, 0, sizeof(ltotal));
  }
  if (options & FULL_PATHNAMES) {
    if (getcwd(wdbuf, PATH_MAX) == NULL) {
      fprintf(stderr, "%s: warning: unable to get current directory: %s\n",
	      argv[0], strerror(errno));
      *wdbuf = '\0';
    }
    wdlen = strlen(wdbuf);
  }
  /* if the list is empty, do nothing */
  if (list->size == 0) {
    return;
  }
  /* sort the list */
  flist_sort(argc, argv, list);
  /* find the standard output width by printing the first element */
  fwidth = display_file(argc, argv, list->fa[0], &ltotal, format, cfile,
			options | NOPRINT, wdbuf, wdlen,
			(style == COMMA_JUST) ? 1 : list->maxnamelen,
			list->maxlinklen, list->maxcustlens, now);
  if (width < 0) { /* this means that an explicit number of columns
		      was specified */
    cols = -width;
  }
  else {
    cols = (width - 1) / (fwidth + 1); /* calculate number of columns across
					  the screen */
    if (cols == 0) ++cols;
  }
  rows = list->size / cols; /* don't need this for COMMA_JUST, but who
			       cares? */
  if (list->size % cols != 0) ++rows; /* add one if not an even number of
					 files */
  if (style == COMMA_JUST) {
    for (i = 0; i < list->size; ++i) {
      if (i > 0) {
	fputs(",", stdout);
      }
      (void) display_file(argc, argv, list->fa[i], &ltotal, format, cfile,
			  options, wdbuf, wdlen, 1, 1, list->maxcustlens, now);
    }
    fputs("\n", stdout);
  }
  else if (style == COLUMN_DOWN) {
    for (i = 0; i < rows; ++i) {
      for (j = i, curcol = 1; j < list->size; j += rows, ++curcol) {
	if (j != i) {
	  printf(" ");
	}
	(void) display_file(argc, argv, list->fa[j], &ltotal, format, cfile,
			    options | (curcol == cols ? LAST_IN_ROW : 0),
			    wdbuf, wdlen,
			    list->maxnamelen, list->maxlinklen,
			    list->maxcustlens, now);
      }
      printf("\n");
    }
  }
  else if (style == COLUMN_ACROSS) {
    for (i = 0; i < list->size; i += cols) {
      for (j = i, curcol = 1; j < i+cols && j < list->size; ++j, ++curcol) {
	if (j != i) {
	  printf(" ");
	}
	(void) display_file(argc, argv, list->fa[j], &ltotal, format, cfile,
			    options | (curcol == cols ? LAST_IN_ROW : 0),
			    wdbuf, wdlen,
			    list->maxnamelen, list->maxlinklen,
			    list->maxcustlens, now);
      }
      printf("\n");
    }
  }
  else {
    fprintf(stderr, "%s: unknown print style '%c'\n", argv[0], style);
    exit(ERROR_PROC);
  }
  if (options & SHOW_TOTALS) {
    printf("%lu files, %lu blocks, %lu bytes\n",
	   ltotal.count, ltotal.blocks, ltotal.bytes);
  }
}

/* display a directory; fetch all the files from it, then add them to a list
   and call display_file_list() on it */

void display_directory(int argc, char *argv[], struct fnode *dn,
		       int style, int width, char *format, struct stat *cfile,
		       int options, time_t now)
{
  DIR *dp;
  struct flist l, dl;
  struct fnode *tmp;
  struct dirent *ent;
  char wdbuf[PATH_MAX];
  long i;

  if ((dp = opendir(dn->name)) == NULL) {
    fprintf(stderr, "%s: %s: %s\n", argv[0], dn->name, strerror(errno));
    return;
  }
  /* get the current directory */
  if (getcwd(wdbuf, (size_t)PATH_MAX) == NULL) {
    fprintf(stderr, "%s: can't get current directory (%s)\n",
	    argv[0], strerror(errno));
    return;
  }
  /* change to the directory in question, so all the stat()s work */
  if (chdir(dn->name) != 0) {
    fprintf(stderr, "%s: %s: %s\n", argv[0], dn->name, strerror(errno));
    return;
  }
  flist_clear(argc, argv, &l);
  if (options & RECURSE) {
    flist_clear(argc, argv, &dl);
  }
  do {
    errno = 0;
    if ((ent = readdir(dp)) == NULL) { /* either end of directory or error */
      if (errno != 0) { /* error, then */
	fprintf(stderr, "%s: readdir on %s: %s\n",
		argv[0], dn->name, strerror(errno));
	break;
      }
      else { /* not error, EOF */
	break;
      }
    }
    else { /* got valid entry, process it */
      if (!(options & SHOW_ALL)
	  && (strcmp(ent->d_name, ".") == 0
	      || strcmp(ent->d_name, "..") == 0)) {
	continue;
      }
      if (!(options & (SHOW_ALL|SHOW_ALL_LIMITED)) && *(ent->d_name) == '.') {
	continue;
      }
      if ((tmp = newfnode(argc, argv, ent->d_name, options)) == NULL) {
	(void)closedir(dp); /* again, no point in checking since we are going
			       to exit anyway */
	exit(ERROR_PROC);
      }
      /* right, got an entry, stat()ed it and everything, append it to the
	 list */
      if (flist_append(argc, argv, &l, tmp) == 0) {
	(void)closedir(dp);
	exit(ERROR_PROC);
      }
      if (options & RECURSE && S_ISDIR(tmp->s->st_mode)
	  && strcmp(".", ent->d_name) != 0
	  && strcmp("..", ent->d_name) != 0
	  && (!(options & SAME_FILESYSTEM)
	      || tmp->s->st_dev == dn->s->st_dev)) {
	/* append to the directory list only if it's a directory, and we care
	   about recursing, and it's not . or .., and it's in the same
	   filesystem if we care about that */
	if (flist_append(argc, argv, &dl, tmp) == 0) {
	  (void)closedir(dp);
	  exit(ERROR_PROC);
	}
      }
    }
  } while (1);
  if (closedir(dp) != 0) {
    fprintf(stderr, "%s: closedir of %s: %s\n",
	    argv[0], dn->name, strerror(errno));
  }
  /* okay, got the list, display it */
  if (options & SHOW_DIR_HEAD) { /* print the directory header */
    printf("\n%s%s%s (%ld):\n", (*(dn->name) == '/') ? "" : wdbuf,
	   (*(dn->name) == '/') ? "" : "/", dn->name, l.size);
  }
  display_file_list(argc, argv, &l, style, width, format, cfile, options,
		    now);
  if (options & RECURSE) {
    for (i = 0; i < dl.size; ++i) {
      display_directory(argc, argv, dl.fa[i], style, width, format, cfile,
			options, now);
    }
  }
  flist_free(&l); /* free the memory in the list */
  /* change back to the original directory */
  if (chdir(wdbuf) != 0) {
    fprintf(stderr, "%s: couldn't chdir back to %s: %s\n", argv[0], wdbuf, strerror(errno));
    exit(ERROR_PROC);
  }
}
