/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *  Lock Keys (formerly Led) Applet was writen by Jrgen Scheibengruber <mfcn@gmx.de>
 */

#include "config.h"

#include <X11/XKBlib.h>
#include <gdk/gdkx.h>
#include <gnome.h>
#include <libgnome/libgnome.h>
#include <panel-applet.h>
#include <panel-applet-gconf.h>
#include <string.h>
#include "../pixmaps/numlock_on.xpm"
#include "../pixmaps/capslock_on.xpm"
#include "../pixmaps/scrolllock_on.xpm"
#include "../pixmaps/numlock_off.xpm"
#include "../pixmaps/capslock_off.xpm"
#include "../pixmaps/scrolllock_off.xpm"
#include "../pixmaps/lock-keys-applet.xpm"

#define NUMLOCK_ON_ICON (const char**)numlock_on_xpm
#define CAPSLOCK_ON_ICON (const char**)capslock_on_xpm
#define SCROLLLOCK_ON_ICON (const char**)scrolllock_on_xpm
#define NUMLOCK_OFF_ICON (const char**)numlock_off_xpm
#define CAPSLOCK_OFF_ICON (const char**)capslock_off_xpm
#define SCROLLLOCK_OFF_ICON (const char**)scrolllock_off_xpm
#define APPLET_ICON (const char**)lock_keys_applet_xpm

enum
{
	CAPSLOCK  = 0,
	NUMLOCK,
	SCROLLLOCK
} Leds;

typedef struct
{
	PanelApplet *applet;
	GtkWidget *vbox, *hbox;
	GtkWidget *num_pix, *scroll_pix, *caps_pix;
	GtkWidget *show_cb[3];
	Display *rootwin;
	int xkbev, xkberr;
	
	gboolean on[3];
	gboolean show[3];
	
	GtkDialog *about, *settings;
	
	GtkTooltips *tooltips;
} LedApplet;

static const char 
led_applet_menu_xml [] =
	"<popup name=\"button3\">\n"
	"   <menuitem name=\"Properties Item\" verb=\"LedAppletProperties\" label=\"%s\"\n"
	"             pixtype=\"stock\" pixname=\"gtk-properties\"/>\n"
	"   <menuitem name=\"Help Item\" verb=\"LedAppletHelp\" label=\"%s\"\n"
	"             pixtype=\"stock\" pixname=\"gtk-help\"/>\n"
	"   <menuitem name=\"About Item\" verb=\"LedAppletAbout\" label=\"%s\"\n"
	"             pixtype=\"stock\" pixname=\"gnome-stock-about\"/>\n"
	"</popup>\n";

/* Reorder the icons in the applet, according to the orient and the size 
 * of the panel
 * at the moment this seems ok...
 */
static void
applet_reorder_icons(int size, int orient, LedApplet *applet)
{
	GtkBox *box;
	int count = 0;
	
	g_assert(applet);

	/* how many buttons do we have?
	 * if TRUE == 1 then we could do this easier,
	 * but who knows...
	 */
	count += applet->show[CAPSLOCK] ? 1 : 0;
	count += applet->show[NUMLOCK] ? 1 : 0;
	count += applet->show[SCROLLLOCK] ? 1 : 0;
	
	g_assert(count > 0);
	
	/* Extract the pixmaps out of the boxes, in case they 
	 * have been put there before...
	 * they have to be ref'ed before doing so
	 */
	if (applet->num_pix->parent)
	{
		gtk_widget_ref(applet->num_pix);
		gtk_container_remove(GTK_CONTAINER(applet->num_pix->parent), applet->num_pix);
	}
	if (applet->caps_pix->parent)
	{
		gtk_widget_ref(applet->caps_pix);
		gtk_container_remove(GTK_CONTAINER(applet->caps_pix->parent), applet->caps_pix);
	}
	if (applet->scroll_pix->parent)
	{
		gtk_widget_ref(applet->scroll_pix);
		gtk_container_remove(GTK_CONTAINER(applet->scroll_pix->parent), applet->scroll_pix);
	}
	
	/* Do we put the pixmaps into the vbox or the hbox?
	 * this depends on size of the panel and how many pixmaps are shown
	 */
	if ((orient == PANEL_APPLET_ORIENT_UP) || (orient == PANEL_APPLET_ORIENT_DOWN)) 
		box = GTK_BOX(size < 48 ? applet->hbox : applet->vbox);
	else
		box = GTK_BOX(size < 48 ? applet->vbox : applet->hbox);
	
	if ((count < 3) || (size < 48))
	{
		if (applet->show[CAPSLOCK])
			gtk_box_pack_start(box, applet->caps_pix, TRUE, TRUE, 0);
		if (applet->show[NUMLOCK])
			gtk_box_pack_start(box, applet->num_pix, TRUE, TRUE, 0);
		if (applet->show[SCROLLLOCK])
			gtk_box_pack_start(box, applet->scroll_pix, TRUE, TRUE, 0);
	}
	else
	{
		gtk_box_pack_start(GTK_BOX(applet->vbox), applet->caps_pix, TRUE, TRUE, 0);
		gtk_box_pack_start(GTK_BOX(applet->hbox), applet->num_pix, TRUE, TRUE, 0);
		gtk_box_pack_start(GTK_BOX(applet->hbox), applet->scroll_pix, TRUE, TRUE, 0);
	}
	
	gtk_widget_show_all(GTK_WIDGET(applet->applet));
}

/* panel has changed its size 
 */
static void
applet_change_size(PanelApplet *applet_widget, int size, LedApplet *applet)
{
	applet_reorder_icons(size, panel_applet_get_orient(applet_widget), applet);
}

/* panel has changed its orientation
 */
static void
applet_change_orient(PanelApplet *applet_widget, int orient, LedApplet *applet)
{
	applet_reorder_icons(panel_applet_get_size(applet_widget), orient, applet);
}

/* Set the icons according to the status of the keyboard leds
 */
static void
change_icons(LedApplet *applet)
{
	GdkPixbuf *num_pixbuf, *caps_pixbuf, *scroll_pixbuf;
	
	if (applet->on[NUMLOCK])
		num_pixbuf = gdk_pixbuf_new_from_xpm_data(NUMLOCK_ON_ICON);
	else
		num_pixbuf = gdk_pixbuf_new_from_xpm_data(NUMLOCK_OFF_ICON);
	
	if (applet->on[CAPSLOCK])
		caps_pixbuf = gdk_pixbuf_new_from_xpm_data(CAPSLOCK_ON_ICON);
	else
		caps_pixbuf = gdk_pixbuf_new_from_xpm_data(CAPSLOCK_OFF_ICON);
	
	if (applet->on[SCROLLLOCK])
		scroll_pixbuf = gdk_pixbuf_new_from_xpm_data(SCROLLLOCK_ON_ICON);
	else
		scroll_pixbuf = gdk_pixbuf_new_from_xpm_data(SCROLLLOCK_OFF_ICON);
		
	gtk_image_set_from_pixbuf(GTK_IMAGE(applet->num_pix), num_pixbuf);
	gtk_image_set_from_pixbuf(GTK_IMAGE(applet->caps_pix), caps_pixbuf);
	gtk_image_set_from_pixbuf(GTK_IMAGE(applet->scroll_pix), scroll_pixbuf);

	gdk_pixbuf_unref(num_pixbuf);
	gdk_pixbuf_unref(caps_pixbuf);
	gdk_pixbuf_unref(scroll_pixbuf);
}	

/* Free the memory of the applet struct
 * Save the current state of the applet
 */
static void
applet_destroy(PanelApplet *applet_widget, LedApplet *applet)
{
	g_assert(applet);
	g_free(applet);
	return;
}

/* Just a boring about box
 */
static void
about_cb(BonoboUIComponent *uic, gpointer data, const gchar *verbname)
{
	GdkPixbuf *icon;
	const char *authors[] = {"Jörgen Scheibengruber <mfcn@gmx.de>", NULL};
	LedApplet *applet = (LedApplet*)data;
	
	/* Feel free to put your names here translators :-) */
	char *translators = _("TRANSLATORS");

	if (applet->about)
	{
		gtk_window_present(GTK_WINDOW(applet->about));
		return;
	}
	icon = gdk_pixbuf_new_from_xpm_data(APPLET_ICON);
	
	applet->about = GTK_DIALOG(gnome_about_new (_("Keyboard Lock Keys"), VERSION, "Copyright 2002 Jörgen Scheibengruber",
				_("An applet that shows the state of your Capslock-, Numlock-, and Scroll-lock keys"),
				(const char **) authors, NULL, strcmp("TRANSLATORS", translators) ? translators : NULL, 
				icon));
 
	gdk_pixbuf_unref(icon);
	g_object_add_weak_pointer(G_OBJECT(applet->about), (gpointer*)&applet->about);
	gtk_widget_show(GTK_WIDGET(applet->about));
}

/* Opens gnome help application
 */
static void
help_cb (BonoboUIComponent *uic, gpointer data, const gchar *verbname)
{
	GError *error = NULL;

	gnome_help_display(PACKAGE, NULL, &error);
	if (error)
	{
		GtkWidget *dialog;
		dialog = gtk_message_dialog_new(NULL, 
				GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
				GTK_MESSAGE_ERROR, GTK_BUTTONS_OK,
				error->message);
		gtk_dialog_run (GTK_DIALOG (dialog));
		gtk_widget_destroy (dialog);
		g_error_free (error);
		error = NULL;
	}
}

/* Called when one of the applet->show_cb checkbutton is toggled...
 */
static void
show_cb_change_cb(GtkToggleButton *togglebutton, LedApplet *applet)
{
	gboolean active;
	GtkWidget *others[2];
	
	active = gtk_toggle_button_get_active(togglebutton);

	if (togglebutton == GTK_TOGGLE_BUTTON(applet->show_cb[CAPSLOCK]))
	{
		others[0] = applet->show_cb[NUMLOCK];
		others[1] = applet->show_cb[SCROLLLOCK];
		applet->show[CAPSLOCK] = active;
	}	
	else if (togglebutton == GTK_TOGGLE_BUTTON(applet->show_cb[NUMLOCK]))
	{
		others[0] = applet->show_cb[CAPSLOCK];
		others[1] = applet->show_cb[SCROLLLOCK];
		applet->show[NUMLOCK] = active;
	}	
	else if (togglebutton == GTK_TOGGLE_BUTTON(applet->show_cb[SCROLLLOCK]))
	{
		others[0] = applet->show_cb[CAPSLOCK];
		others[1] = applet->show_cb[NUMLOCK];
		applet->show[SCROLLLOCK] = active;
	}
	else
		g_assert_not_reached();

	/* Shows hides the icons according to the new settings */
	applet_reorder_icons(panel_applet_get_size(applet->applet), 
		panel_applet_get_orient(applet->applet), applet);
	
	if (active)
	{
		gtk_widget_set_sensitive(others[0], TRUE);
		gtk_widget_set_sensitive(others[1], TRUE);
		return;
	}
	
	if (!active && !gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(others[0])))
		gtk_widget_set_sensitive(others[1], FALSE);
	
	if (!active && !gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(others[1])))
		gtk_widget_set_sensitive(others[0], FALSE);
}

static void
settings_cb(BonoboUIComponent *uic, gpointer data, const gchar *verbname)
{
	LedApplet *applet = (LedApplet*)data;
	GtkWidget *hbox, *vbox1, *vbox2, *header_lbl, *dummy_lbl;
	int i, sum = 0, lo = 0, answer;
	char* header_str;
	
	g_assert(applet);
	
	if (applet->settings)
	{
		gtk_window_present(GTK_WINDOW(applet->settings));
		return;
	}
	applet->settings = 
		GTK_DIALOG(gtk_dialog_new_with_buttons(_("Lock Keys Preferences"), 
		NULL, GTK_DIALOG_DESTROY_WITH_PARENT | GTK_DIALOG_NO_SEPARATOR,
		GTK_STOCK_CLOSE, GTK_RESPONSE_ACCEPT, 
		GTK_STOCK_HELP, GTK_RESPONSE_HELP,
		NULL));

	/*gtk_container_set_border_width(GTK_CONTAINER(applet->settings), 12);
	gtk_box_set_spacing(GTK_BOX(applet->settings->vbox), 18);
	gtk_window_set_default_size(GTK_WINDOW(applet->settings), 240, -1);*/
	gtk_dialog_set_default_response(GTK_DIALOG(applet->settings), GTK_RESPONSE_ACCEPT);
	gtk_window_set_resizable(GTK_WINDOW(applet->settings), FALSE);
	
	vbox1 = gtk_vbox_new(FALSE, 6);
	gtk_container_set_border_width(GTK_CONTAINER(vbox1), 12);
	
	header_str = g_strconcat("<span weight=\"bold\">", _("Settings"), "</span>", NULL);
	header_lbl = gtk_label_new(header_str);
	gtk_label_set_use_markup(GTK_LABEL(header_lbl), TRUE);
	//gtk_label_set_justify(GTK_LABEL(header_lbl), GTK_JUSTIFY_LEFT);
	gtk_misc_set_alignment(GTK_MISC(header_lbl), 0, 0.5);
	g_free(header_str);
	gtk_box_pack_start(GTK_BOX(vbox1), header_lbl, TRUE, TRUE, 0);
	
	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox1), hbox, TRUE, TRUE, 0);
	
	dummy_lbl = gtk_label_new("    ");
	vbox2 = gtk_vbox_new(FALSE, 6);
	gtk_box_pack_start(GTK_BOX(hbox), dummy_lbl, TRUE, TRUE, 0);
	gtk_box_pack_start(GTK_BOX(hbox), vbox2, TRUE, TRUE, 0);
	
	applet->show_cb[CAPSLOCK] = gtk_check_button_new_with_mnemonic(_("Show c_aps-lock"));
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(applet->show_cb[CAPSLOCK]), applet->show[CAPSLOCK]);
	applet->show_cb[NUMLOCK] = gtk_check_button_new_with_mnemonic(_("Show _num-lock"));
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(applet->show_cb[NUMLOCK]), applet->show[NUMLOCK]);
	applet->show_cb[SCROLLLOCK] = gtk_check_button_new_with_mnemonic(_("Show _scroll-lock"));
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(applet->show_cb[SCROLLLOCK]), applet->show[SCROLLLOCK]);
	
	for (i=0;i<3;i++)
		if (applet->show[i])
		{
			sum++;
			lo = i;
		}
	
	if (sum < 2)
		gtk_widget_set_sensitive(applet->show_cb[lo], FALSE);
		
	gtk_box_pack_start(GTK_BOX(vbox2), applet->show_cb[CAPSLOCK], TRUE, TRUE, 0);
	gtk_box_pack_start(GTK_BOX(vbox2), applet->show_cb[NUMLOCK], TRUE, TRUE, 0);
	gtk_box_pack_start(GTK_BOX(vbox2), applet->show_cb[SCROLLLOCK], TRUE, TRUE, 0);
	
	gtk_widget_show_all(vbox1);
	gtk_container_add(GTK_CONTAINER(applet->settings->vbox), vbox1); 
	
	g_signal_connect(G_OBJECT(applet->show_cb[CAPSLOCK]), "toggled",
					GTK_SIGNAL_FUNC(show_cb_change_cb),
                    (gpointer)applet);

	g_signal_connect(G_OBJECT(applet->show_cb[NUMLOCK]), "toggled",
					GTK_SIGNAL_FUNC(show_cb_change_cb),
                    (gpointer)applet);

	g_signal_connect(G_OBJECT(applet->show_cb[SCROLLLOCK]), "toggled",
					GTK_SIGNAL_FUNC(show_cb_change_cb),
                    (gpointer)applet);

	answer = GTK_RESPONSE_HELP;
	while (answer == GTK_RESPONSE_HELP)
	{
		answer = gtk_dialog_run(applet->settings);
		
		if (answer == GTK_RESPONSE_ACCEPT)
		{
			panel_applet_gconf_set_bool(PANEL_APPLET(applet->applet), "capslock_show", applet->show[CAPSLOCK], NULL);
			panel_applet_gconf_set_bool(PANEL_APPLET(applet->applet), "numlock_show", applet->show[NUMLOCK], NULL);
			panel_applet_gconf_set_bool(PANEL_APPLET(applet->applet), "scrolllock_show", applet->show[SCROLLLOCK], NULL);

			gtk_widget_destroy(GTK_WIDGET(applet->settings));
			applet->settings = NULL;
			break;
		}
		help_cb(NULL, NULL, NULL);
	}
}

/* Get the mask used to set the keyboard leds
 * I "stole" this code from kkeyled
 * the thing with the mask of the CAPS Lock key is 
 * really ugly, but it seems to work
 */
static unsigned int 
get_lockmask(LedApplet *applet)
{
	int i;
	unsigned int mask = 0;
	XkbDescPtr xkb;

	g_assert(applet);
	
	if (!(xkb = XkbGetKeyboard(applet->rootwin, XkbAllComponentsMask, XkbUseCoreKbd)))
		return 0;
	
	if (!xkb->names)
		return 0;
    
	for(i = 0;i <= XkbNumVirtualMods;i++)
	{                                         
		unsigned int tmp = 0;
		char* name = NULL;
		if (!xkb->names->vmods[i])
			continue;

		name = XGetAtomName(xkb->dpy, xkb->names->vmods[i]);
		if (name)
		{
			if (!strcmp(name, "Caps Lock") && applet->on[CAPSLOCK])
			{
				XkbVirtualModsToReal(xkb, 1 << i, &tmp);
				if (!tmp)
					tmp = 2; /* Hack to make it work. Maybe a bug in xkb? */
			}
			if (!strcmp(name, "NumLock") && applet->on[NUMLOCK])
				XkbVirtualModsToReal(xkb, 1 << i, &tmp);
			if (!strcmp(name, "ScrollLock") && applet->on[SCROLLLOCK])
				XkbVirtualModsToReal(xkb, 1 << i, &tmp);
			mask += tmp;
		}
	}
	XkbFreeKeyboard(xkb, 0, True);
	return mask;
}


/* Retrieve the modifier lockmask and then set the status of the leds
 * according to applet->on[]
 */
void
set_ledstates(LedApplet *applet)
{
	unsigned int mask = 0;
	
	g_assert(applet);
	
	mask = get_lockmask(applet);
	
	XkbLockModifiers(applet->rootwin, XkbUseCoreKbd, mask, mask);
}

/* If the state of the leds has changed,
 * then the icons are set to the new state,
 * the gconf settings are updated and so is
 * the tooltip.
 */
void 
ledstates_changed(LedApplet *applet, unsigned int state)
{
	int i;
	char *buf, *on, *off;
	
	for (i=0;i<3;i++)
	{
		if (state & (1 << i))
			applet->on[i] = 1;
		else
			applet->on[i] = 0;
	}

	change_icons(applet);

	panel_applet_gconf_set_bool(PANEL_APPLET(applet->applet), "capslock_state", applet->on[CAPSLOCK], NULL);
	panel_applet_gconf_set_bool(PANEL_APPLET(applet->applet), "numlock_state", applet->on[NUMLOCK], NULL);
	panel_applet_gconf_set_bool(PANEL_APPLET(applet->applet), "scrolllock_state", applet->on[SCROLLLOCK], NULL);

	on = _("On");
	off = _("Off");
	buf = g_strdup_printf(_("Caps: %s Num: %s Scroll: %s"), applet->on[CAPSLOCK] ? on : off, 
		applet->on[NUMLOCK] ? on : off,applet->on[SCROLLLOCK] ? on : off);
	gtk_tooltips_set_tip(applet->tooltips, GTK_WIDGET(applet->applet), buf, "");
	g_free(buf);
}


/* Get the current state of the leds
 * call ledstates_changed();
 */
void
get_ledstates(LedApplet *applet)
{
	unsigned int state = 0;

	XkbGetIndicatorState(applet->rootwin, XkbUseCoreKbd, &state);

	ledstates_changed(applet, state);
}	

GdkFilterReturn
event_filter(GdkXEvent *gdkxevent, GdkEvent *event, LedApplet* applet)
{
	XkbEvent ev;
	memcpy(&ev.core, gdkxevent, sizeof(ev.core));
	
	if (ev.core.type == applet->xkbev + XkbEventCode)
	{
		if (ev.any.xkb_type == XkbIndicatorStateNotify)
			ledstates_changed(applet, ev.indicators.state);
	}
	
	return GDK_FILTER_CONTINUE;
}

/* Just check if it's possible to use the xkb extension
 */
static gboolean
init_xkb_extension(LedApplet *applet)
{
    int code;
    int maj = XkbMajorVersion;
    int min = XkbMinorVersion;
    
	if (!XkbLibraryVersion(&maj, &min))
		return FALSE;
	if (!XkbQueryExtension(applet->rootwin, &code, &applet->xkbev, &applet->xkberr, &maj, &min))
		return FALSE;
	return TRUE;
}
	
static const BonoboUIVerb
led_applet_menu_verbs [] = 
{
		BONOBO_UI_VERB("LedAppletProperties", settings_cb),
		BONOBO_UI_VERB("LedAppletHelp", help_cb),
		BONOBO_UI_VERB("LedAppletAbout", about_cb),
	
		BONOBO_UI_VERB_END
};

/* The "main" function
 */
gboolean
led_applet_factory(PanelApplet *applet_widget, const gchar *iid, gpointer data)
{
	GdkDrawable *drawable;
	char *buf;
	LedApplet *applet;
	GdkPixbuf *icon;
	GList *iconlist = NULL;
	
	if (strcmp (iid, "OAFIID:GNOME_LockKeysApplet"))
		return FALSE;

/* Set an icon for all windows
 */
	icon = gdk_pixbuf_new_from_xpm_data(APPLET_ICON);
	iconlist = g_list_append(NULL, (gpointer)icon);
	gtk_window_set_default_icon_list(iconlist);
	gdk_pixbuf_unref(icon);
	
	applet = g_malloc0(sizeof(LedApplet));
	applet->applet = applet_widget;
	
/* I use the root window as display... this is probably
 * not necessary, but it does not seem to cause problems
 * either 
 */
	drawable = gdk_get_default_root_window();
	g_assert(drawable);
	applet->rootwin = gdk_x11_drawable_get_xdisplay(drawable);
	
	applet->tooltips = gtk_tooltips_new();
	
	applet->vbox = gtk_vbox_new(FALSE, 0);
	applet->hbox = gtk_hbox_new(FALSE, 0);
	
	applet->num_pix = gtk_image_new();
	applet->caps_pix = gtk_image_new();
	applet->scroll_pix = gtk_image_new();

	if (!init_xkb_extension(applet))
	{
		GtkWidget *dialog;
		dialog = gtk_message_dialog_new(NULL, 
				GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT, 
				GTK_MESSAGE_ERROR, GTK_BUTTONS_OK,
				_("Could not initialize X Keyboard Extension."));
		gtk_dialog_run (GTK_DIALOG (dialog));
		gtk_widget_destroy (dialog);
		return FALSE;
	}
		
/* Only set the state of the leds if we already have entries in gconf
 * otherwise the applet would always turn of all leds when executed
 * manually
 */
	if (panel_applet_gconf_get_bool(PANEL_APPLET(applet->applet), "have_settings", NULL))
	{	
		applet->on[CAPSLOCK] = panel_applet_gconf_get_bool(PANEL_APPLET(applet->applet), "capslock_state", NULL);
		applet->on[NUMLOCK] = panel_applet_gconf_get_bool(PANEL_APPLET(applet->applet), "numlock_state", NULL);
		applet->on[SCROLLLOCK] = panel_applet_gconf_get_bool(PANEL_APPLET(applet->applet), "scrolllock_state", NULL);
		applet->show[CAPSLOCK] = panel_applet_gconf_get_bool(PANEL_APPLET(applet->applet), "capslock_show", NULL);
		applet->show[NUMLOCK] = panel_applet_gconf_get_bool(PANEL_APPLET(applet->applet), "numlock_show", NULL);
		applet->show[SCROLLLOCK] = panel_applet_gconf_get_bool(PANEL_APPLET(applet->applet), "scrolllock_show", NULL);
		if (!applet->show[CAPSLOCK] && !applet->show[NUMLOCK] && !applet->show[SCROLLLOCK])
			applet->show[CAPSLOCK] = TRUE;
		set_ledstates(applet);
	}
	else
	{
		applet->show[CAPSLOCK] = TRUE;
		applet->show[NUMLOCK] = TRUE;
		applet->show[SCROLLLOCK] = TRUE;
	}
	panel_applet_gconf_set_bool(PANEL_APPLET(applet->applet), "have_settings", TRUE, NULL);

	change_icons(applet);
	get_ledstates(applet);

	gtk_box_pack_end(GTK_BOX(applet->vbox), applet->hbox, TRUE, TRUE, 0);
	gtk_box_pack_start(GTK_BOX(applet->hbox), applet->caps_pix, TRUE, TRUE, 0);
	gtk_box_pack_start(GTK_BOX(applet->hbox), applet->num_pix, TRUE, TRUE, 0);
	gtk_box_pack_start(GTK_BOX(applet->hbox), applet->scroll_pix, TRUE, TRUE, 0);
	
/* the applet doesnt get a change_size/change_orient signal on startup
 * so it has to figure these two out manually
 */
	applet_reorder_icons(panel_applet_get_size(applet_widget), 
		panel_applet_get_orient(applet_widget), applet);
	
	gtk_container_add(GTK_CONTAINER(applet_widget), applet->vbox);
	gtk_widget_show_all(GTK_WIDGET(applet_widget));

	g_signal_connect(GTK_OBJECT(applet_widget), "change_size",
                           GTK_SIGNAL_FUNC(applet_change_size),
                           (gpointer)applet);
	g_signal_connect(GTK_OBJECT(applet_widget), "change_orient",
                           GTK_SIGNAL_FUNC(applet_change_orient),
                           (gpointer)applet);
	g_signal_connect(GTK_OBJECT(applet_widget), "destroy",
                           GTK_SIGNAL_FUNC(applet_destroy),
                           (gpointer)applet);

/* Change in 0.3: Don't use the timeout anymore, but instead use
 * gdk_window_add_filter()
 */
	if (!XkbSelectEvents(applet->rootwin, XkbUseCoreKbd, XkbIndicatorStateNotifyMask, XkbIndicatorStateNotifyMask))
		return FALSE;
	
	gdk_window_add_filter(NULL, (GdkFilterFunc)event_filter, applet);

	buf = g_strdup_printf(led_applet_menu_xml, _("_Preferences..."), _("_Help"), _("_About..."));
	
	panel_applet_setup_menu(applet_widget, buf,
                                 led_applet_menu_verbs,
                                 (gpointer)applet);
	g_free(buf);
	
	return TRUE;
}	

PANEL_APPLET_BONOBO_FACTORY("OAFIID:GNOME_LockKeysApplet_Factory", PANEL_TYPE_APPLET,
			PACKAGE, VERSION, (PanelAppletFactoryCallback)led_applet_factory, NULL)
