/*
   DumpToFile.c : Dump to file dialog window.
   Copyright (C) 1999 Jean-Hugues Deschenes, 2000 Karim Yaghmour.
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; version 2 of the License.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

   History :
      K.Y., 22/08/2000, Added RTAI support
      JH.D., 15/09/1999, Initial typing.
*/

#include <stdlib.h>
#include <string.h>
#include <malloc.h>

#include <Tables.h>

#include "MainWindow.h"

/**********************************************************************************/
/**************************** Internal signal handlers ****************************/
/**********************************************************************************/
/******************************************************************
 * Function :
 *    ISHSetEventInMask()
 * Description :
 * Parameters :
 * Return values :
 * History :
 * Note :
 ******************************************************************/
void ISHSetEventInMask(GtkWidget* pmWidget, gpointer pmData)
{
  gint                lEventID; /* the list item's associated event type */
  trace_event_mask*   lMask;    /* The mask to use to set selection/deselection information */

  /* Retrieve the vector pointer */
  if(!(lMask = (trace_event_mask*) pmData))
    {
    g_print("Invalid Mask pointer in ISHSetEventInMask! \n");
    exit(1);
    }

  /* retireve the event type index */
  lEventID = (gint) gtk_object_get_data(GTK_OBJECT(pmWidget), EVENT_TYPE_DATA);

  /* is this a valid index? */
  if(lEventID >= 0 && lEventID <= MaxEventID)
    {
    /* Is the event type to be selected? */
    if(ltt_test_bit(lEventID, lMask))
      /* Select it */
      gtk_widget_set_state(pmWidget, GTK_STATE_SELECTED);
    /* Otherwise, the item is to be deselected */
    else
      /* Deselect it */
      gtk_widget_set_state(pmWidget, GTK_STATE_NORMAL);
    }
}

/******************************************************************
 * Function :
 *    ISHSelectEventRadioButton()
 * Description :
 * Parameters :
 * Return values :
 * History :
 * Note :
 ******************************************************************/
void ISHSelectEventRadioButton(GtkWidget* pmWidget, gpointer pmData)
{
  dumpToFileWindow* pDumpWindow; /* The dump to file window */

  if(!(pDumpWindow = (dumpToFileWindow*) pmData))
    {
    g_print("Selected event radio without passing dump to file window! \n");
    exit(1);
    }

  /* Set initial values for the event type Tracing options */
  pDumpWindow->LocalOptions->Omit = pDumpWindow->LocalOptions->Trace = FALSE;

  /* Should we specify event types omitted? */
  if(((pDumpWindow->LocalOptions->Omit = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(pDumpWindow->OmitTheseEvents)) == TRUE))
     && (pDumpWindow->OmitTheseEvents == pmWidget))
    /* Go through the event type list to set the options accordingly */
    gtk_container_foreach(GTK_CONTAINER(pDumpWindow->EventList),
			  ISHSetEventInMask,
			  (gpointer) &(pDumpWindow->LocalOptions->OmitMask));

  /* Should we specify event types included? */
  if(((pDumpWindow->LocalOptions->Trace = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(pDumpWindow->TraceTheseEvents)) == TRUE)
      && pDumpWindow->TraceTheseEvents == pmWidget))
    /* Go through the event type list to set the options accordingly */
    gtk_container_foreach(GTK_CONTAINER(pDumpWindow->EventList),
			  ISHSetEventInMask,
			  (gpointer) &(pDumpWindow->LocalOptions->TraceMask));

  /* Make the event type list sensitive if either option is selected */
  gtk_widget_set_sensitive(pDumpWindow->ScrolledEventList, 
			   pDumpWindow->LocalOptions->Omit || pDumpWindow->LocalOptions->Trace);
}

/******************************************************************
 * Function :
 *    ISHSelectProcRadioButton()
 * Description :
 * Parameters :
 * Return values :
 * History :
 * Note :
 ******************************************************************/
void ISHSelectProcRadioButton(GtkWidget* pmWidget, gpointer pmData)
{
  dumpToFileWindow* pDumpWindow; /* The dump to file window */

  if(!(pDumpWindow = (dumpToFileWindow*) pmData))
    {
    g_print("Selected process radio without passing dump to file window! \n");
    exit(1);
    }

  /* Should we trace all PIDs? */
  pDumpWindow->LocalOptions->TracePID = 
    gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(pDumpWindow->JustThisProc));

  /* Make the option menu sensitive if process specification is selected */
  gtk_widget_set_sensitive(pDumpWindow->Proclist, 
			   pDumpWindow->LocalOptions->TracePID);

}

/******************************************************************
 * Function :
 *    ISHSelectSMPRadioButton()
 * Description :
 * Parameters :
 * Return values :
 * History :
 * Note :
 ******************************************************************/
void ISHSelectSMPRadioButton(GtkWidget* pmWidget, gpointer pmData)
{
  dumpToFileWindow* pDumpWindow; /* The dump to file window */

  if(!(pDumpWindow = (dumpToFileWindow*) pmData))
    {
    g_print("Selected SMP radio without passing dump to file window! \n");
    exit(1);
    }

  /* Should we trace all CPU IDs? */
  pDumpWindow->LocalOptions->TraceCPUID = 
    gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(pDumpWindow->JustThisCPU));

  /* Make the option menu sensitive if CPU specification is selected */
  gtk_widget_set_sensitive(pDumpWindow->CPUIDEntry, 
			   pDumpWindow->LocalOptions->TraceCPUID);

}

/******************************************************************
 * Function :
 *    ISHSelectEventCheckBox()
 * Description :
 * Parameters :
 * Return values :
 * History :
 * Note :
 ******************************************************************/
void ISHSelectEventCheckBox(GtkWidget* pmWidget, gpointer pmData)
{
  dumpToFileWindow* pDumpWindow; /* The dump to file window */

  if(!(pDumpWindow = (dumpToFileWindow*) pmData))
    {
    g_print("Selected event radio without passing dump to file window! \n");
    exit(1);
    }

  /* Shoule event types be printed? */
  pDumpWindow->LocalOptions->Summarize = 
    !gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(pDumpWindow->PrintEvents));

  /* Make the event type selection radio buttons unavailable */
  gtk_widget_set_sensitive(pDumpWindow->SelectEventFrame, 
			   !pDumpWindow->LocalOptions->Summarize);
}

/******************************************************************
 * Function :
 *    ISHSelectEventType()
 * Description :
 * Parameters :
 * Return values :
 * History :
 * Note :
 ******************************************************************/
void ISHSelectEventType(GtkWidget* pmWidget, GtkStateType state, gpointer pmData)
{
  options* lLocalOptions; /* Local option structure */
  gint     lEventID;      /* the list item's associated event type */

  /* don't bother executing this if the widget shouldn't be sensitive */
  if(pmWidget->state == GTK_STATE_INSENSITIVE) return;

  /* Was the data passed correstly? */
  if(!(lLocalOptions = (options*) pmData))
    {
    g_print("Selected event without passing local options! \n");
    exit(1);
    }

  /* retireve the event type index */
  lEventID = (gint) gtk_object_get_data(GTK_OBJECT(pmWidget), EVENT_TYPE_DATA);

  /* is this a valid index? */
  if(lEventID >= 0 && lEventID <= MaxEventID)
    {
    /* Are we indicating which event types to omit? */
    if(lLocalOptions->Omit)
      {
      /* Set the Omit vector accordingly */
      if(pmWidget->state == GTK_STATE_SELECTED)
	ltt_set_bit(lEventID, &(lLocalOptions->OmitMask));
      else
	ltt_clear_bit(lEventID, &(lLocalOptions->OmitMask));
      }

    /* Are we indicating which event types to include? */
    if(lLocalOptions->Trace)
      {
      /* Set the Trace vector accordingly */
      if(pmWidget->state == GTK_STATE_SELECTED)
	ltt_set_bit(lEventID, &(lLocalOptions->TraceMask));
      else
	ltt_clear_bit(lEventID, &(lLocalOptions->TraceMask));
      }
    }
}

/******************************************************************
 * Function :
 *    ISHSelectProc()
 * Description :
 * Parameters :
 * Return values :
 * History :
 * Note :
 ******************************************************************/
void ISHSelectProc(GtkWidget* pmWidget, gpointer pmData)
{
  options* lLocalOptions; /* System view pointer */
  process* pProc;        /* the menu item's associated process */

  if(!(lLocalOptions = (options*) pmData))
    {
    g_print("Selected process without passing local options! \n");
    exit(1);
    }

  /* retireve the process pointer */
  if(((pProc = (process*) gtk_object_get_data(GTK_OBJECT(pmWidget), OPT_MENU_DATA)) != NULL))
    /* Set it in our temporary options structure */
    lLocalOptions->PID = pProc->PID;
  else
    /* default to tracing the kernel */
    lLocalOptions->PID = 0;
}

/******************************************************************
 * Function :
 *    ISHSelectOptionsCheckBox()
 * Description :
 * Parameters :
 * Return values :
 * History :
 * Note :
 ******************************************************************/
void ISHSelectOptionsCheckBox(GtkWidget* pmWidget, gpointer pmData)
{
  gint* ItemToSet; /* Pointer to the item to be set in LocalOptions */

  if(!(ItemToSet = (gint*) pmData))
    {
    g_print("Internal error: select options check box called without ItemToSet \n");
    exit(1);
    }

  *ItemToSet = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(pmWidget));
}

/**********************************************************************************/
/******************************* Internal functions *******************************/
/**********************************************************************************/
/******************************************************************
 * Function :
 * Description :
 * Parameters :
 * Return values :
 * History :
 * Note :
 ******************************************************************/
void DTFIAddEventTypeItem(GtkWidget* List, gchar* Label, gint EventID, options* LocalOptions)
{
  GtkWidget* NewListItem; /* the new list itme to create */

  /* Create the new list item */
  NewListItem = gtk_list_item_new_with_label(Label);

  /* Memorize the EventID */
  gtk_object_set_data(GTK_OBJECT(NewListItem), EVENT_TYPE_DATA, (gpointer) EventID);

  /* Add the item to the list */
  gtk_container_add(GTK_CONTAINER(List), NewListItem);

  /* Make the menu item visible */
  gtk_widget_show(NewListItem);
}

/******************************************************************
 * Function :
 * Description :
 * Parameters :
 * Return values :
 * History :
 * Note :
 ******************************************************************/
void DTFIAddMenuItem(GtkWidget* pmMenu, options* LocalOptions, process* pProc)
{
  GtkWidget* NewMenuItem;             /* The new menu item to add */
  gchar      lString[MW_LONG_STRLEN]; /* Temporary string to build menu */

  /* Create list item text */
  if(pProc->Command != NULL)
    snprintf(lString, MW_LONG_STRLEN, "%s", pProc->Command);
  else 
    if(pProc->PID == 0)
      snprintf(lString, MW_LONG_STRLEN, "Kernel");
    else
      snprintf(lString, MW_LONG_STRLEN, "Unnamed child");

  snprintf(lString, MW_LONG_STRLEN, "%s (%d)", lString, pProc->PID);

  /* Create the new menu item */
  NewMenuItem = gtk_menu_item_new_with_label(lString);

  /* If it should be the  default menu item */
  if(pProc->PID == LocalOptions->PID)
    /* Prepend the menu item to the menu */
    gtk_menu_prepend(GTK_MENU(pmMenu), NewMenuItem);
  else
    /* Append the menu item to the menu */
    gtk_menu_append(GTK_MENU(pmMenu), NewMenuItem);

  /* Memorize the associated process */
  gtk_object_set_data(GTK_OBJECT(NewMenuItem), OPT_MENU_DATA, pProc);

  /* Connect selection signal */
  gtk_signal_connect(GTK_OBJECT(NewMenuItem),
		     "activate",
		     GTK_SIGNAL_FUNC(ISHSelectProc),
		     LocalOptions);

  /* Make the menu item visible */
  gtk_widget_show(NewMenuItem);

}

/******************************************************************
 * Function :
 * Description :
 * Parameters :
 * Return values :
 * History :
 * Note :
 ******************************************************************/
void DTFIFileSelectDestroy(GtkWidget* pmWidget, gpointer pmData)
{
  dumpToFileWindow*  pDTFWin;  /* The window for whom this was called */

  /* Do we have anything meaningfull */
  if((pDTFWin = (dumpToFileWindow*) pmData) == NULL)
    {
    g_print("Internal error: Destroy File Select without Dump To File Window \n");
    exit(1);
    }

  /* Delete the file selection  */
  gtk_widget_destroy(pDTFWin->FileSelect);

  pDTFWin->FileSelect = NULL;
}

/******************************************************************
 * Function :
 * Description :
 * Parameters :
 * Return values :
 * History :
 * Note :
 ******************************************************************/
void DTFIFileSelectOK(GtkWidget* pmWidget, gpointer pmData)
{
  dumpToFileWindow*  pDTFWin;  /* The window for whom this was called */

  /* Do we have anything meaningfull */
  if((pDTFWin = (dumpToFileWindow*) pmData) == NULL)
    {
    g_print("Internal error: OK File Select without Open Trace Window \n");
    exit(1);
    }

  /* Display the filename in the prof file text entry */
  gtk_entry_set_text(GTK_ENTRY(pDTFWin->FileTextEntry),
		     gtk_file_selection_get_filename(GTK_FILE_SELECTION(pDTFWin->FileSelect)));
}

/******************************************************************
 * Function :
 * Description :
 * Parameters :
 * Return values :
 * History :
 * Note :
 ******************************************************************/
void DTFIFileBrowse(dumpToFileWindow* pmDTFWin)
{
  GtkWidget*   pFileSelect;    /* The file select window */

  /* Don't open more than one file selection window */
  if(pmDTFWin->FileSelect != NULL)
    {
    gtk_widget_hide(pmDTFWin->FileSelect);
    gtk_widget_show(pmDTFWin->FileSelect);
    return;
    }

  /* Create a new file selection dialog */
  pFileSelect = pmDTFWin->FileSelect = gtk_file_selection_new("Dump to file...");

  /* Connect the correct signals */
  gtk_signal_connect(GTK_OBJECT(pFileSelect),
		     "destroy",
		     GTK_SIGNAL_FUNC(DTFIFileSelectDestroy),
		     pmDTFWin);
  gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(pFileSelect)->ok_button),
		     "clicked",
		     GTK_SIGNAL_FUNC(DTFIFileSelectOK),
		     pmDTFWin);
  gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(pFileSelect)->ok_button),
			    "clicked",
			    GTK_SIGNAL_FUNC(gtk_widget_destroy),
			    GTK_OBJECT(pFileSelect));
  gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(pFileSelect)->cancel_button),
			    "clicked",
			    GTK_SIGNAL_FUNC(gtk_widget_destroy),
			    GTK_OBJECT(pFileSelect));

  /* Display file selection window in middle of screen */
  gtk_window_set_position(GTK_WINDOW(pFileSelect), GTK_WIN_POS_CENTER);

  /* Keep file selection on the foreground */
  gtk_window_set_transient_for(GTK_WINDOW(pFileSelect), 
			       GTK_WINDOW(pmDTFWin->Window));

  /* Show the file selection window */
  gtk_widget_show(pFileSelect);
}

/******************************************************************
 * Function :
 * Description :
 * Parameters :
 * Return values :
 * History :
 * Note :
 ******************************************************************/
void DTFISetInitialValues(dumpToFileWindow* pmDTFWin)
{
  process* Navigator;                  /* Temporary pointer to navigate through process list */ 
  gchar    lString[MW_DEFAULT_STRLEN]; /* Temporary building string */

  /* Populate lists */

  /* Allocate the options menu list */
  for(Navigator = ((systemView*)(pmDTFWin->SysView))->System->Processes; Navigator != NULL; Navigator = Navigator->Next)
  {
    DTFIAddMenuItem(pmDTFWin->ProcListMenu, pmDTFWin->LocalOptions, Navigator);
  }

  /* Set the process tab's option menu */
  gtk_option_menu_set_menu (GTK_OPTION_MENU(pmDTFWin->Proclist), pmDTFWin->ProcListMenu);

  /* Populate event type list */
  DTFIAddEventTypeItem(pmDTFWin->EventList, "Trace beginning",
		       TRACE_START, pmDTFWin->LocalOptions);
  DTFIAddEventTypeItem(pmDTFWin->EventList, "System call entered",
		       TRACE_SYSCALL_ENTRY, pmDTFWin->LocalOptions);
  DTFIAddEventTypeItem(pmDTFWin->EventList, "System call exited",
		       TRACE_SYSCALL_EXIT, pmDTFWin->LocalOptions);
  DTFIAddEventTypeItem(pmDTFWin->EventList, "Trap entered",
		       TRACE_TRAP_ENTRY, pmDTFWin->LocalOptions);
  DTFIAddEventTypeItem(pmDTFWin->EventList, "Trap exited",
		       TRACE_TRAP_EXIT, pmDTFWin->LocalOptions);
  DTFIAddEventTypeItem(pmDTFWin->EventList, "IRQ entered",
		       TRACE_IRQ_ENTRY, pmDTFWin->LocalOptions);
  DTFIAddEventTypeItem(pmDTFWin->EventList, "IRQ exited",
		       TRACE_IRQ_EXIT, pmDTFWin->LocalOptions);
  DTFIAddEventTypeItem(pmDTFWin->EventList, "Scheduling change",
		       TRACE_SCHEDCHANGE, pmDTFWin->LocalOptions);
  DTFIAddEventTypeItem(pmDTFWin->EventList, "Kernel timer routine called",
		       TRACE_KERNEL_TIMER, pmDTFWin->LocalOptions);
  DTFIAddEventTypeItem(pmDTFWin->EventList, "Soft IRQ management",
		       TRACE_SOFT_IRQ, pmDTFWin->LocalOptions);
  DTFIAddEventTypeItem(pmDTFWin->EventList, "Process management",
		       TRACE_PROCESS, pmDTFWin->LocalOptions);
  DTFIAddEventTypeItem(pmDTFWin->EventList, "File system",
		       TRACE_FILE_SYSTEM, pmDTFWin->LocalOptions);
  DTFIAddEventTypeItem(pmDTFWin->EventList, "Timer management",
		       TRACE_TIMER, pmDTFWin->LocalOptions);
  DTFIAddEventTypeItem(pmDTFWin->EventList, "Memory management",
		       TRACE_MEMORY, pmDTFWin->LocalOptions);
  DTFIAddEventTypeItem(pmDTFWin->EventList, "Socket communication",
		       TRACE_SOCKET, pmDTFWin->LocalOptions);
  DTFIAddEventTypeItem(pmDTFWin->EventList, "Inter-process communication",
		       TRACE_IPC, pmDTFWin->LocalOptions);
  DTFIAddEventTypeItem(pmDTFWin->EventList, "Network communication",
		       TRACE_NETWORK, pmDTFWin->LocalOptions);
  DTFIAddEventTypeItem(pmDTFWin->EventList, "Creation of custom event",
		       TRACE_NEW_EVENT, pmDTFWin->LocalOptions);
  DTFIAddEventTypeItem(pmDTFWin->EventList, "Custom event occurrence",
		       TRACE_CUSTOM, pmDTFWin->LocalOptions);
#if SUPP_RTAI
  /* Is this an RTAI trace */
  if(((systemView*)(pmDTFWin->SysView))->EventDB->SystemType == TRACE_SYS_TYPE_RTAI_LINUX)
    {
    DTFIAddEventTypeItem(pmDTFWin->EventList, "RTAI mount",
			 TRACE_RTAI_MOUNT, pmDTFWin->LocalOptions);
    DTFIAddEventTypeItem(pmDTFWin->EventList, "RTAI un-mount",
			 TRACE_RTAI_UMOUNT, pmDTFWin->LocalOptions);
    DTFIAddEventTypeItem(pmDTFWin->EventList, "RTAI global IRQ entry",
			 TRACE_RTAI_GLOBAL_IRQ_ENTRY, pmDTFWin->LocalOptions);
    DTFIAddEventTypeItem(pmDTFWin->EventList, "RTAI global IRQ exit",
			 TRACE_RTAI_GLOBAL_IRQ_EXIT, pmDTFWin->LocalOptions);
    DTFIAddEventTypeItem(pmDTFWin->EventList, "RTAI CPU-own IRQ entry",
			 TRACE_RTAI_OWN_IRQ_ENTRY, pmDTFWin->LocalOptions);
    DTFIAddEventTypeItem(pmDTFWin->EventList, "RTAI CPU-own IRQ exit",
			 TRACE_RTAI_OWN_IRQ_EXIT, pmDTFWin->LocalOptions);
    DTFIAddEventTypeItem(pmDTFWin->EventList, "RTAI trap entry",
			 TRACE_RTAI_TRAP_ENTRY, pmDTFWin->LocalOptions);
    DTFIAddEventTypeItem(pmDTFWin->EventList, "RTAI trap exit",
			 TRACE_RTAI_TRAP_EXIT, pmDTFWin->LocalOptions);
    DTFIAddEventTypeItem(pmDTFWin->EventList, "RTAI SRQ entry",
			 TRACE_RTAI_SRQ_ENTRY, pmDTFWin->LocalOptions);
    DTFIAddEventTypeItem(pmDTFWin->EventList, "RTAI SRQ exit",
			 TRACE_RTAI_SRQ_EXIT, pmDTFWin->LocalOptions);
    DTFIAddEventTypeItem(pmDTFWin->EventList, "RTAI switch to Linux",
			 TRACE_RTAI_SWITCHTO_LINUX, pmDTFWin->LocalOptions);
    DTFIAddEventTypeItem(pmDTFWin->EventList, "RTAI switch to RT",
			 TRACE_RTAI_SWITCHTO_RT, pmDTFWin->LocalOptions);
    DTFIAddEventTypeItem(pmDTFWin->EventList, "RTAI scheduling change",
			 TRACE_RTAI_SCHED_CHANGE, pmDTFWin->LocalOptions);
    DTFIAddEventTypeItem(pmDTFWin->EventList, "RTAI task management",
			 TRACE_RTAI_TASK, pmDTFWin->LocalOptions);
    DTFIAddEventTypeItem(pmDTFWin->EventList, "RTAI timer management",
			 TRACE_RTAI_TIMER, pmDTFWin->LocalOptions);
    DTFIAddEventTypeItem(pmDTFWin->EventList, "RTAI semaphore communication",
			 TRACE_RTAI_SEM, pmDTFWin->LocalOptions);
    DTFIAddEventTypeItem(pmDTFWin->EventList, "RTAI message communication",
			 TRACE_RTAI_MSG, pmDTFWin->LocalOptions);
    DTFIAddEventTypeItem(pmDTFWin->EventList, "RTAI RPC communication",
			 TRACE_RTAI_RPC, pmDTFWin->LocalOptions);
    DTFIAddEventTypeItem(pmDTFWin->EventList, "RTAI message-box communication",
			 TRACE_RTAI_MBX, pmDTFWin->LocalOptions);
    DTFIAddEventTypeItem(pmDTFWin->EventList, "RTAI FIFO communication",
			 TRACE_RTAI_FIFO, pmDTFWin->LocalOptions);
    DTFIAddEventTypeItem(pmDTFWin->EventList, "RTAI shared-memory mangement",
			 TRACE_RTAI_SHM, pmDTFWin->LocalOptions);
    DTFIAddEventTypeItem(pmDTFWin->EventList, "RTAI POSIX layer",
			 TRACE_RTAI_POSIX, pmDTFWin->LocalOptions);
    DTFIAddEventTypeItem(pmDTFWin->EventList, "RTAI LXRT layer",
			 TRACE_RTAI_LXRT, pmDTFWin->LocalOptions);
    DTFIAddEventTypeItem(pmDTFWin->EventList, "RTAI LXRT-Informed layer",
			 TRACE_RTAI_LXRTI, pmDTFWin->LocalOptions);
    }
#endif

  /* Set initial values for the widgets */

  /* Process radio buttons */
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(pmDTFWin->JustThisProc), 
			       pmDTFWin->LocalOptions->TracePID);
  ISHSelectProcRadioButton(pmDTFWin->JustThisProc, pmDTFWin);

  /* Summarize CheckBox */
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(pmDTFWin->PrintEvents),
			       !pmDTFWin->LocalOptions->Summarize);
  ISHSelectEventCheckBox(pmDTFWin->PrintEvents, pmDTFWin);

  /* Event radio buttons */
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(pmDTFWin->OmitTheseEvents), 
			       pmDTFWin->LocalOptions->Omit);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(pmDTFWin->TraceTheseEvents), 
			       pmDTFWin->LocalOptions->Trace);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(pmDTFWin->TraceAllEvents),
			       !(pmDTFWin->LocalOptions->Omit || pmDTFWin->LocalOptions->Trace));

  /* Depending on which toggle button is selected, simulate the proper mouse click */
  if(pmDTFWin->LocalOptions->Omit)
    ISHSelectEventRadioButton(pmDTFWin->OmitTheseEvents, pmDTFWin);

  else
    ISHSelectEventRadioButton(pmDTFWin->TraceTheseEvents, pmDTFWin);

  /* SMP radio buttons */
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(pmDTFWin->JustThisCPU), 
			       pmDTFWin->LocalOptions->TraceCPUID);
  ISHSelectSMPRadioButton(pmDTFWin->JustThisCPU, pmDTFWin);

  /* Set the CPUid entry */
  snprintf(lString, MW_DEFAULT_STRLEN, "%d", pmDTFWin->LocalOptions->CPUID);
  gtk_entry_set_text(GTK_ENTRY(pmDTFWin->CPUIDEntry), lString);

  /* Misc. options */
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(pmDTFWin->AccountSysCalls),
			       pmDTFWin->LocalOptions->AcctSysCall);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(pmDTFWin->ForgetTime),
			       pmDTFWin->LocalOptions->ForgetTime);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(pmDTFWin->ForgetString),
			       pmDTFWin->LocalOptions->ForgetString);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(pmDTFWin->ForgetDataLen),
			       pmDTFWin->LocalOptions->ForgetDataLen);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(pmDTFWin->ForgetPID),
			       pmDTFWin->LocalOptions->ForgetPID);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(pmDTFWin->ForgetCPUID),
			       pmDTFWin->LocalOptions->ForgetCPUID);

}

/**********************************************************************************/
/**************************** Signal handling functions ***************************/
/**********************************************************************************/
/******************************************************************
 * Function :
 * Description :
 * Parameters :
 * Return values :
 * History :
 * Note :
 ******************************************************************/
gint DTFSHDeleteEvent(GtkWidget* pmWidget, GdkEvent* pmEvent, gpointer pmData)
{
  /* Close it */
  return(FALSE);
}

/******************************************************************
 * Function :
 * Description :
 * Parameters :
 * Return values :
 * History :
 * Note :
 ******************************************************************/
void DTFSHDestroy(GtkWidget* pmWidget, gpointer pmData)
{
  dumpToFileWindow*  pDTFWin;  /* The window for whom this was called */

  /* Do we have anything meaningfull */
  if((pDTFWin = (dumpToFileWindow*) pmData) == NULL)
    {
    g_print("Internal error: Callback without Open Trace Window \n");
    exit(1);
    }

  /* Call on the cancel callback */
  pDTFWin->CBCancel(pDTFWin->SysView);
}

/******************************************************************
 * Function :
 * Description :
 * Parameters :
 * Return values :
 * History :
 * Note :
 ******************************************************************/
void DTFSHFileBrowse(GtkWidget* pmWidget, gpointer pmData)
{
  DTFIFileBrowse(((dumpToFileWindow*) pmData));
}

/******************************************************************
 * Function :
 * Description :
 * Parameters :
 * Return values :
 * History :
 * Note :
 ******************************************************************/
void DTFSHInternalOK(GtkWidget* pmWidget, gpointer pmData)
{
  gchar*             pFileName;     /* Pointer to retrieve entry contents */
  gint               lFileNameLen;  /* Length of file name */
  dumpToFileWindow*  pDTFWin;       /* The window for whom this was called */

  /* Do we have anything meaningfull */
  if((pDTFWin = (dumpToFileWindow*) pmData) == NULL)
    {
    g_print("Internal error: Callback without Open Trace Window \n");
    exit(1);
    }

  /* free previously allocated string if there was one */
  free(pDTFWin->LocalOptions->OutputFileName);

  /* Set the new CPU id value */
  pDTFWin->LocalOptions->CPUID = atoi(gtk_entry_get_text(GTK_ENTRY(pDTFWin->CPUIDEntry)));

  /* Set the output filename in the options structure */
  pFileName = gtk_entry_get_text(GTK_ENTRY(pDTFWin->FileTextEntry));
  lFileNameLen = strlen(pFileName) + 1;
  pDTFWin->LocalOptions->OutputFileName = (char*) malloc(lFileNameLen);
  strncpy(pDTFWin->LocalOptions->OutputFileName, pFileName, lFileNameLen);

  /* Copy the temporary options to the permanent options */
  ((systemView*)(pDTFWin->SysView))->Options = 
    memcpy(((systemView*)(pDTFWin->SysView))->Options, pDTFWin->LocalOptions, sizeof(options));

  /* Call the OK Callback */
  pDTFWin->CBOK(pDTFWin->SysView);
}

/**********************************************************************************/
/******************************* Windowing functions ******************************/
/**********************************************************************************/
/******************************************************************
 * Function :
 * Description :
 * Parameters :
 * Return values :
 * History :
 * Note :
 ******************************************************************/
void DTFConnectSignals(dumpToFileWindow* pmDTFWin)
{
  GList* Navigator; /* Pointer to navigate through list items */

  /* Connect the generic signals */
  gtk_signal_connect(GTK_OBJECT(pmDTFWin->Window),
		     "delete_event",
		     GTK_SIGNAL_FUNC(DTFSHDeleteEvent),
		     pmDTFWin);
  gtk_signal_connect(GTK_OBJECT(pmDTFWin->Window),
		     "destroy",
		     GTK_SIGNAL_FUNC(DTFSHDestroy),
		     pmDTFWin);

  /* connect the event check box */
  gtk_signal_connect(GTK_OBJECT(pmDTFWin->PrintEvents),
		     "toggled",
		     GTK_SIGNAL_FUNC(ISHSelectEventCheckBox),
		     pmDTFWin);

  /* Connect the event radio buttons */
  gtk_signal_connect(GTK_OBJECT(pmDTFWin->TraceAllEvents),
		     "toggled",
		     GTK_SIGNAL_FUNC(ISHSelectEventRadioButton),
		     pmDTFWin);
  gtk_signal_connect(GTK_OBJECT(pmDTFWin->TraceTheseEvents),
		     "toggled",
		     GTK_SIGNAL_FUNC(ISHSelectEventRadioButton),
		     pmDTFWin);
  gtk_signal_connect(GTK_OBJECT(pmDTFWin->OmitTheseEvents),
		     "toggled",
		     GTK_SIGNAL_FUNC(ISHSelectEventRadioButton),
		     pmDTFWin);

  /* Navigate through event type list */
  for(Navigator = GTK_LIST(pmDTFWin->EventList)->children; Navigator; Navigator = Navigator->next)
    /* Connect selection signal */
    gtk_signal_connect(GTK_OBJECT(Navigator->data),
		       "state-changed",
		       GTK_SIGNAL_FUNC(ISHSelectEventType),
		       pmDTFWin->LocalOptions);

  /* Connect the Process radio buttons */
  gtk_signal_connect(GTK_OBJECT(pmDTFWin->TraceAllProc),
		     "toggled",
		     GTK_SIGNAL_FUNC(ISHSelectProcRadioButton),
		     pmDTFWin);
  gtk_signal_connect(GTK_OBJECT(pmDTFWin->JustThisProc),
		     "toggled",
		     GTK_SIGNAL_FUNC(ISHSelectProcRadioButton),
		     pmDTFWin);

  /* Connect the SMP radio buttons */
  gtk_signal_connect(GTK_OBJECT(pmDTFWin->TraceAllCPUs),
		     "toggled",
		     GTK_SIGNAL_FUNC(ISHSelectSMPRadioButton),
		     pmDTFWin);
  gtk_signal_connect(GTK_OBJECT(pmDTFWin->JustThisCPU),
		     "toggled",
		     GTK_SIGNAL_FUNC(ISHSelectSMPRadioButton),
		     pmDTFWin);

  /* Connect the option check boxes */
  gtk_signal_connect(GTK_OBJECT(pmDTFWin->AccountSysCalls),
		     "toggled",
		     GTK_SIGNAL_FUNC(ISHSelectOptionsCheckBox),
		     &pmDTFWin->LocalOptions->AcctSysCall);
  gtk_signal_connect(GTK_OBJECT(pmDTFWin->ForgetTime),
		     "toggled",
		     GTK_SIGNAL_FUNC(ISHSelectOptionsCheckBox),
		     &pmDTFWin->LocalOptions->ForgetTime);
  gtk_signal_connect(GTK_OBJECT(pmDTFWin->ForgetString),
		     "toggled",
		     GTK_SIGNAL_FUNC(ISHSelectOptionsCheckBox),
		     &pmDTFWin->LocalOptions->ForgetString);
  gtk_signal_connect(GTK_OBJECT(pmDTFWin->ForgetDataLen),
		     "toggled",
		     GTK_SIGNAL_FUNC(ISHSelectOptionsCheckBox),
		     &pmDTFWin->LocalOptions->ForgetDataLen);
  gtk_signal_connect(GTK_OBJECT(pmDTFWin->ForgetPID),
		     "toggled",
		     GTK_SIGNAL_FUNC(ISHSelectOptionsCheckBox),
		     &pmDTFWin->LocalOptions->ForgetPID);
  gtk_signal_connect(GTK_OBJECT(pmDTFWin->ForgetCPUID),
		     "toggled",
		     GTK_SIGNAL_FUNC(ISHSelectOptionsCheckBox),
		     &pmDTFWin->LocalOptions->ForgetCPUID);

  /* Connect the Browse buttons */
  gtk_signal_connect(GTK_OBJECT(pmDTFWin->FileBrowse),
		     "clicked",
		     GTK_SIGNAL_FUNC(DTFSHFileBrowse),
		     pmDTFWin);

  /* Connect the Ok and Cancel buttons */
  gtk_signal_connect(GTK_OBJECT(pmDTFWin->OKButton),
		     "clicked",
		     GTK_SIGNAL_FUNC(DTFSHInternalOK),
		     pmDTFWin);
  gtk_signal_connect_object(GTK_OBJECT(pmDTFWin->OKButton),
			    "clicked",
			    GTK_SIGNAL_FUNC(gtk_widget_destroy),
			    GTK_OBJECT(pmDTFWin->Window));
  gtk_signal_connect_object(GTK_OBJECT(pmDTFWin->CancelButton),
			    "clicked",
			    GTK_SIGNAL_FUNC(gtk_widget_destroy),
			    GTK_OBJECT(pmDTFWin->Window));
}

/******************************************************************
 * Function :
 * Description :
 * Parameters :
 * Return values :
 * History :
 * Note :
 ******************************************************************/
void DTFShowDumpToFileWindow(dumpToFileWindow* pmDTFWin)
{
  /* Make OK button default choice */
  gtk_widget_grab_default(pmDTFWin->OKButton);

  /* Show it all to the world */
  gtk_widget_show(pmDTFWin->VBox);
  gtk_widget_show(pmDTFWin->Notebook);
  gtk_widget_show(pmDTFWin->ProcessTable);
  gtk_widget_show(pmDTFWin->EventsVBox);
  gtk_widget_show(pmDTFWin->OptionsVBox);
  gtk_widget_show(pmDTFWin->SMPTable);
  gtk_widget_show(pmDTFWin->FileHBox);
  gtk_widget_show(pmDTFWin->ButtonHBox);
  gtk_widget_show(pmDTFWin->TraceAllProc);
  gtk_widget_show(pmDTFWin->JustThisProc);
  gtk_widget_show(pmDTFWin->Proclist);
  gtk_widget_show(pmDTFWin->PrintEvents);
  gtk_widget_show(pmDTFWin->SelectEventFrame);
  gtk_widget_show(pmDTFWin->SelectEventVBox);
  gtk_widget_show(pmDTFWin->TraceAllEvents);
  gtk_widget_show(pmDTFWin->TraceTheseEvents);
  gtk_widget_show(pmDTFWin->OmitTheseEvents);
  gtk_widget_show(pmDTFWin->ScrolledEventList);
  gtk_widget_show(pmDTFWin->EventList);
  gtk_widget_show(pmDTFWin->TraceAllCPUs);
  gtk_widget_show(pmDTFWin->JustThisCPU);
  gtk_widget_show(pmDTFWin->CPUIDEntry);
  gtk_widget_show(pmDTFWin->AccountSysCalls);
  gtk_widget_show(pmDTFWin->ForgetTime);
  gtk_widget_show(pmDTFWin->ForgetString);
  gtk_widget_show(pmDTFWin->ForgetDataLen);
  gtk_widget_show(pmDTFWin->ForgetPID);
  gtk_widget_show(pmDTFWin->ForgetCPUID);
  gtk_widget_show(pmDTFWin->FileLabel);
  gtk_widget_show(pmDTFWin->FileTextEntry);
  gtk_widget_show(pmDTFWin->FileBrowse);
  gtk_widget_show(pmDTFWin->OKButton);
  gtk_widget_show(pmDTFWin->CancelButton);

  /* Set title */
  gtk_window_set_title(GTK_WINDOW(pmDTFWin->Window), "Dump file options");

  /* Display in the center of screen */
  gtk_window_set_position(GTK_WINDOW(pmDTFWin->Window), GTK_WIN_POS_CENTER);

  /* Make Dump to file window always visible */
  gtk_window_set_transient_for(GTK_WINDOW(pmDTFWin->Window), 
			       GTK_WINDOW(pmDTFWin->ParentWin));

  /* Display it */
  gtk_widget_show(pmDTFWin->Window);

}

/******************************************************************
 * Function :
 * Description :
 * Parameters :
 * Return values :
 * History :
 * Note :
 ******************************************************************/
dumpToFileWindow* DTFCreateDumpToFileWindow(gpointer      pmSysView,
					    GtkWidget*    pmParentWin,
					    _DTFCBOK*     pmCBOK,
					    _DTFCBCancel* pmCBCancel)
{
  dumpToFileWindow* pDTFWin;     /* Dump to file window */

  /* Create a new dump to file window */
  pDTFWin = (dumpToFileWindow*) g_malloc(sizeof(dumpToFileWindow));

  /* Initialize basic variables */
  pDTFWin->FileSelect = NULL;

  /* Put the given data in the right place */
  pDTFWin->SysView  = pmSysView;
  pDTFWin->ParentWin= pmParentWin;
  pDTFWin->CBOK     = pmCBOK;
  pDTFWin->CBCancel = pmCBCancel;

  /* Create a local copy of the options to be configured */
  pDTFWin->LocalOptions = (options*) g_malloc(sizeof(options));
  pDTFWin->LocalOptions = memcpy(pDTFWin->LocalOptions, ((systemView*)(pmSysView))->Options, sizeof(options));

  /* Build the window */
  pDTFWin->Window = gtk_window_new(GTK_WINDOW_DIALOG);
  pDTFWin->VBox   = gtk_vbox_new(FALSE, TRUE);
  gtk_container_add(GTK_CONTAINER(pDTFWin->Window), pDTFWin->VBox);

  /* Build the notebook object and pack it */
  pDTFWin->Notebook = gtk_notebook_new();
  gtk_notebook_set_tab_pos(GTK_NOTEBOOK(pDTFWin->Notebook), GTK_POS_LEFT);
  gtk_box_pack_start(GTK_BOX(pDTFWin->VBox), pDTFWin->Notebook, TRUE, TRUE, 5);

  /* Build the notebook Tabs and containers and pack them */
  pDTFWin->ProcessTable = gtk_table_new(2, 2, FALSE);
  pDTFWin->EventsVBox   = gtk_vbox_new(FALSE, 0);
  pDTFWin->SMPTable     = gtk_table_new(2, 2, FALSE);
  pDTFWin->OptionsVBox  = gtk_vbox_new(FALSE, 5);
  gtk_notebook_append_page(GTK_NOTEBOOK(pDTFWin->Notebook), 
			   pDTFWin->ProcessTable,
			   gtk_label_new("Processes"));
  gtk_notebook_append_page(GTK_NOTEBOOK(pDTFWin->Notebook),
			   pDTFWin->EventsVBox,
			   gtk_label_new("Events"));
  gtk_notebook_append_page(GTK_NOTEBOOK(pDTFWin->Notebook),
			   pDTFWin->SMPTable,
			   gtk_label_new("SMP"));
  gtk_notebook_append_page(GTK_NOTEBOOK(pDTFWin->Notebook),
			   pDTFWin->OptionsVBox, 
			   gtk_label_new("Options"));

  /* Allocate HBoxes to put the rest of the widgets and pack them */
  pDTFWin->FileHBox   = gtk_hbox_new(FALSE, 0);
  pDTFWin->ButtonHBox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(pDTFWin->VBox), pDTFWin->FileHBox, FALSE, TRUE, 5);
  gtk_box_pack_start(GTK_BOX(pDTFWin->VBox), pDTFWin->ButtonHBox, FALSE, TRUE, 0);

  /* Build Processes tab widgets and pack them */
  pDTFWin->ProcRadioGroup = NULL;
  pDTFWin->TraceAllProc   = gtk_radio_button_new_with_label(pDTFWin->ProcRadioGroup, "Trace all processes");
  pDTFWin->ProcRadioGroup = gtk_radio_button_group(GTK_RADIO_BUTTON(pDTFWin->TraceAllProc));
  pDTFWin->JustThisProc   = gtk_radio_button_new_with_label(pDTFWin->ProcRadioGroup, "Only trace process");
  pDTFWin->ProcRadioGroup = gtk_radio_button_group(GTK_RADIO_BUTTON(pDTFWin->JustThisProc));
  pDTFWin->Proclist       = gtk_option_menu_new();
  pDTFWin->ProcListMenu   = gtk_menu_new();
  gtk_table_attach(GTK_TABLE(pDTFWin->ProcessTable), pDTFWin->TraceAllProc, 0, 2, 0, 1,
		   GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 5);
  gtk_table_attach(GTK_TABLE(pDTFWin->ProcessTable), pDTFWin->JustThisProc, 0, 1, 1, 2,
		   GTK_FILL, GTK_FILL, 0, 5);
  gtk_table_attach(GTK_TABLE(pDTFWin->ProcessTable), pDTFWin->Proclist, 1, 2, 1, 2,
		   GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 5);

  /* Build main Events tab widgets and pack them */
  pDTFWin->PrintEvents      = gtk_check_button_new_with_label("Print trace events");
  pDTFWin->SelectEventFrame = gtk_frame_new("Events to print");
  pDTFWin->SelectEventVBox  = gtk_vbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(pDTFWin->EventsVBox), pDTFWin->PrintEvents, FALSE, TRUE, 5);
  gtk_box_pack_start(GTK_BOX(pDTFWin->EventsVBox), pDTFWin->SelectEventFrame, TRUE, TRUE, 5);
  gtk_container_add(GTK_CONTAINER(pDTFWin->SelectEventFrame), pDTFWin->SelectEventVBox);

  /* Build Event selection frame widgets and pack them */
  pDTFWin->EventRadioGroup   = NULL;
  pDTFWin->TraceAllEvents    = gtk_radio_button_new_with_label(pDTFWin->EventRadioGroup, "Trace all events");
  pDTFWin->EventRadioGroup   = gtk_radio_button_group(GTK_RADIO_BUTTON(pDTFWin->TraceAllEvents));
  pDTFWin->TraceTheseEvents  = gtk_radio_button_new_with_label(pDTFWin->EventRadioGroup, "Trace these events:");
  pDTFWin->EventRadioGroup   = gtk_radio_button_group(GTK_RADIO_BUTTON(pDTFWin->TraceTheseEvents));
  pDTFWin->OmitTheseEvents   = gtk_radio_button_new_with_label(pDTFWin->EventRadioGroup, "Omit these events:");
  pDTFWin->EventRadioGroup   = gtk_radio_button_group(GTK_RADIO_BUTTON(pDTFWin->OmitTheseEvents));
  pDTFWin->ScrolledEventList = gtk_scrolled_window_new(NULL, NULL);
  pDTFWin->EventList         = gtk_list_new();
  gtk_container_set_border_width(GTK_CONTAINER(pDTFWin->ScrolledEventList), 5);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(pDTFWin->ScrolledEventList),
				 GTK_POLICY_AUTOMATIC,
				 GTK_POLICY_AUTOMATIC);
  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(pDTFWin->ScrolledEventList),
					pDTFWin->EventList);
  gtk_box_pack_start(GTK_BOX(pDTFWin->SelectEventVBox), pDTFWin->TraceAllEvents, FALSE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(pDTFWin->SelectEventVBox), pDTFWin->TraceTheseEvents, FALSE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(pDTFWin->SelectEventVBox), pDTFWin->OmitTheseEvents, FALSE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(pDTFWin->SelectEventVBox), pDTFWin->ScrolledEventList, TRUE, TRUE, 0);
  gtk_list_set_selection_mode(GTK_LIST(pDTFWin->EventList), GTK_SELECTION_MULTIPLE);

  /* Build SMP tab widgets and pack them */
  pDTFWin->SMPRadioGroup = NULL;
  pDTFWin->TraceAllCPUs  = gtk_radio_button_new_with_label(pDTFWin->SMPRadioGroup, "Trace all CPUs");
  pDTFWin->SMPRadioGroup = gtk_radio_button_group(GTK_RADIO_BUTTON(pDTFWin->TraceAllCPUs));
  pDTFWin->JustThisCPU   = gtk_radio_button_new_with_label(pDTFWin->SMPRadioGroup, "Only trace CPU no.:");
  pDTFWin->SMPRadioGroup = gtk_radio_button_group(GTK_RADIO_BUTTON(pDTFWin->JustThisCPU));
  pDTFWin->CPUIDEntry    = gtk_entry_new();
  gtk_table_attach(GTK_TABLE(pDTFWin->SMPTable), pDTFWin->TraceAllCPUs, 0, 2, 0, 1,
		   GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 5);
  gtk_table_attach(GTK_TABLE(pDTFWin->SMPTable), pDTFWin->JustThisCPU, 0, 1, 1, 2,
		   GTK_FILL, GTK_FILL, 0, 5);
  gtk_table_attach(GTK_TABLE(pDTFWin->SMPTable), pDTFWin->CPUIDEntry, 1, 2, 1, 2,
		   GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 5);

  /* Build Options tab widgets and pack them */
  pDTFWin->AccountSysCalls = gtk_check_button_new_with_label("Account time spent in system calls");
  pDTFWin->ForgetTime      = gtk_check_button_new_with_label("Do not print event times");
  pDTFWin->ForgetString    = gtk_check_button_new_with_label("Do not print event description strings");
  pDTFWin->ForgetDataLen   = gtk_check_button_new_with_label("Do not print length of event data entries");
  pDTFWin->ForgetPID       = gtk_check_button_new_with_label("Do not print process identifiers");
  pDTFWin->ForgetCPUID     = gtk_check_button_new_with_label("Do not print CPU identifiers");
  gtk_box_pack_start(GTK_BOX(pDTFWin->OptionsVBox), pDTFWin->AccountSysCalls, FALSE, TRUE, 5);
  gtk_box_pack_start(GTK_BOX(pDTFWin->OptionsVBox), pDTFWin->ForgetTime,      FALSE, TRUE, 5);
  gtk_box_pack_start(GTK_BOX(pDTFWin->OptionsVBox), pDTFWin->ForgetString,    FALSE, TRUE, 5);
  gtk_box_pack_start(GTK_BOX(pDTFWin->OptionsVBox), pDTFWin->ForgetDataLen,   FALSE, TRUE, 5);
  gtk_box_pack_start(GTK_BOX(pDTFWin->OptionsVBox), pDTFWin->ForgetPID,       FALSE, TRUE, 5);
  gtk_box_pack_start(GTK_BOX(pDTFWin->OptionsVBox), pDTFWin->ForgetCPUID,     FALSE, TRUE, 5);

  /* Create the label and pack it */
  pDTFWin->FileLabel = gtk_label_new("Dump to file: ");
  gtk_widget_set_usize(pDTFWin->FileLabel, 80, 20);
  gtk_box_pack_start(GTK_BOX(pDTFWin->FileHBox), pDTFWin->FileLabel, FALSE, TRUE, 0);

  /* Create text entry and pack it */
  pDTFWin->FileTextEntry = gtk_entry_new();
  gtk_box_pack_start(GTK_BOX(pDTFWin->FileHBox), pDTFWin->FileTextEntry, TRUE, TRUE, 0);

  /* Create browse button and pack it */
  pDTFWin->FileBrowse = gtk_button_new_with_label("Browse");
  gtk_box_pack_start(GTK_BOX(pDTFWin->FileHBox), pDTFWin->FileBrowse, FALSE, TRUE, 0);

  /* Create OK and cancel buttons and pack them */
  pDTFWin->OKButton     = gtk_button_new_with_label("OK");
  pDTFWin->CancelButton = gtk_button_new_with_label("Cancel");
  GTK_WIDGET_SET_FLAGS(pDTFWin->OKButton, GTK_CAN_DEFAULT);  /* Allow OK button to be default */
  gtk_box_pack_start(GTK_BOX(pDTFWin->ButtonHBox), pDTFWin->OKButton, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(pDTFWin->ButtonHBox), pDTFWin->CancelButton, TRUE, TRUE, 0);

  /* Set initial widget values and populate lists */
  DTFISetInitialValues(pDTFWin);

  /* Give it to the caller */
  return pDTFWin;
}

/******************************************************************
 * Function :
 * Description :
 * Parameters :
 * Return values :
 * History :
 * Note :
 ******************************************************************/
void DTFDestroyDumpToFileWindow(dumpToFileWindow* pmDTFWin)
{
  /* Is the file select window open */
  if(pmDTFWin->FileSelect != NULL)
    gtk_widget_destroy(pmDTFWin->FileSelect);


  /* Do we still hold a local copy of the options to set? */
  if(pmDTFWin->LocalOptions)
    g_free(pmDTFWin->LocalOptions);

  /* Destroy the window */
  gtk_widget_destroy(pmDTFWin->Window);

  /* Free space used by structure */
  g_free(pmDTFWin);
}
