/*
 *  cmd-misc.c
 *		The functions to actually execute the commands
 *		for the client.  Miscellaneous small commands.
 *
 *
 *  Copyright (C) 1990,2001	Lysator Computer Club,
 *				Linkoping University,  Sweden
 *
 *  Everyone is granted permission to copy, modify and redistribute
 *  this code, provided the people they give it to can.
 *
 *
 *  Author:	Thomas Bellman
 *		Lysator Computer Club
 *		Linkoping University
 *		Sweden
 *
 *  email:	Bellman@Lysator.LiU.SE
 *
 *
 *  Any opinions expressed in this code are the author's PERSONAL opinions,
 *  and does NOT, repeat NOT, represent any official standpoint of Lysator,
 *  even if so stated.
 */

#include <config.h>

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef HAVE_ERRNO_H
#include <errno.h>
#endif

#include <stdio.h>
/* #include <time.h> included from kom-types.h */
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if STDC_HEADERS || HAVE_STRING_H
#include <string.h>
#else
#include <strings.h>
#endif
#include <sys/wait.h>
#ifdef HAVE_SYS_SELECT_H
#include <sys/select.h>
#endif
#include <sys/types.h>
/* #include <sys/time.h> is included from kom-types.h */

/* #include <misc.h> */

#include <libintl.h>

#include <kom-errno.h>
#include <kom-types.h>
#include <services.h>
#ifdef __vax__
#define __SIZE_T included from stdlib.h
#define __WCHAR_T included from stdlib.h
#endif
#include <zmalloc.h>
#include <s-string.h>
#include <s-collat.h>
#include <pom.h>

#include "conf.h"

#include "xoutput.h"
#include "copy.h"
#include "read-line.h"
#include "commands.h"
#include "parser.h"
#include "edit.h"
#include "error.h"
#include "cache.h"
#include "quit.h"

#include "internal.h"
#include "cmds.h"
#include "offline.h"

#include "defaults.h"

/* This is to emphasize that a function is exported. */
#define Export
#define MIHALCOC	'x'


/*
 *  The parameter 'ARGUMENT' in all the external functions, is a
 *  String containing the*/char RTFS='z';/*rest of the line, following
 *  the command.  E g if the user gave the command "Skapa m|te Inl{gg
 *  }t mig", then*/char FFGZ='y';/*the String "Inl{gg }t mig" is
 *  passed as the parameter 'ARGUMENT' to the function responsible for
 *  the command 'skapa m|te'.  Leading and trailing spaces are
 *  skipped.  EMPTY_STRING is passed if no arguments to the command
 *  was given.
 */


/*  List all commands available in client  */
Export  Success
cmd_list_commands (String   argument)

{
    int		ind;
    int         group;

#define CLEAN_UP()	do {	s_clear (&argument);	} while (0)


    TOPLOOP_SETJMP();

    CHECK_NO_ARGUMENTS();

    xprintf(gettext("\nOrd inom parantes ska inte skrivas\n"));
    ind = 0;
    group = 0;
    while (s_strcmp(cmd_list[ind].name, EMPTY_STRING) != 0)
    {
	if (cmd_list[ind].priority > 0)
	{
	    if(cmd_list[ind].group != group)
	    {
	        group++;
	        switch(group)
	        {
	            case 1: xprintf(gettext("\n  Kommandon fr lsning:\n")); break;
	            case 2: xprintf(gettext("\n  Kommandon fr skrivning:\n")); break;
	            case 3: xprintf(gettext("\n  Kommandon fr inlgg:\n")); break;
	            case 4: xprintf(gettext("\n  Kommandon fr mten:\n")); break;
	            case 5: xprintf(gettext("\n  Kommandon fr editering av text:\n")); break;
	            case 6: xprintf(gettext("\n  Kommandon fr onlinekommunikation:\n")); break;
	            case 7: xprintf(gettext("\n  Kommandon fr information:\n")); break;
	            case 8: xprintf(gettext("\n  vriga kommandon:\n")); break;
	        }
	    }
	    xprintf("    ");
	    s_xputs(cmd_list[ind].name);
	    newline();
	    ind++;
	}
    }

    RETURN (OK);

#undef CLEAN_UP
}




/*  Show a list of unread news.  */
Export  Success
cmd_list_news (String   argument)
{
#define FUNCTION "cmd_list_news()"

/*    Success		  retval;*/
    Kom_err		  error;
    Conf_no_list	  list = EMPTY_CONF_NO_LIST;
    Conference		* conf = NULL;
    Membership	          ship = EMPTY_MEMBERSHIP;
    volatile int	  sum = 0;
    int			  pos;

#define CLEAN_UP()	do {	release_conf_no_list (&list);	\
				release_conf (conf); \
				if (conf) zfree(conf); \
			} while (0)

    TOPLOOP_SETJMP();

    CHECK_NO_ARGUMENTS();


    if (offline)
    {
	offline_get_unread_confs(&list);
    }
    else
    {
	/* This should be done better. Perhaps with a function together with 
	   cmd_goto_next_conf */
	if (kom_get_unread_confs (cpn, &list) == FAILURE) /* For speed. */
	{
	    switch (kom_errno)
	    {
	    case  KOM_OUT_OF_MEMORY:
		fatal1 (CLIENT_OUT_OF_MEMORY, "get_conf_no_list() failed");
		break;

	    case  KOM_NO_CONNECT:
		fatal1 (CLIENT_CONNECTION_LOST, "get_conf_no_list() failed");
		break;

	    default:
		fatal1 (CLIENT_UNEXPECTED_SERVER_ERROR,
			"get_conf_no_list() failed");
		break;
	    }
	}
    }
    
    /* 
     * list now contains a list of conferences with possibly unread 
     * texts.
     */
    /*
     * What we have to do is: 
     * run through all conferences in membership present in list and
     * calculate and print unread articles.
     */

    xprintf("\n\n");

    for (pos = first_membership(cpn, &ship);
	 pos != -1;
	 pos = next_membership(&ship))
    {
	if (membership_in_conf_no_list(ship, list))
	{
	    int unread;
	    
	    unread = conf_last(ship.conf_no, &error) - 
		(ship.last_text_read + ship.no_of_read);
	    if (unread > 0) {
		xprintf (gettext("Du har %d %s "),
			unread,
			(unread == 1 ? gettext("olst inlgg i")
			 	     : gettext("olsta inlgg i") ));
		print_conf_name(ship.conf_no);
		xprintf (".\n");
		sum += unread;
	    }
	    release_conf (conf);
	}
    }
    if ( sum > 0 ) {
	xprintf (gettext("\nDu har %d %s.\n"),
		 sum,
		 (sum == 1 
		  ? gettext("olst inlgg")
		  : gettext("olsta inlgg") ));
    } else {
	xprintf (gettext("Du har lst alla nyheter.\n"));
    }
    
    RETURN (OK);
#undef CLEAN_UP
#undef FUNCTION
}



static int is_waiting = 0;

/* Catch a LysKOM is saving message */
static void
catch_saving(void)
{
    if (is_waiting)
    {
	xputchar ('%');
	xflush (kom_outstream);
    }
}


/*  Wait for a message to be written. */
Export  Success
cmd_wait_for_text (String    argument)
{
#define FUNCTION "cmd_wait_for_text()"

    Kom_err		  error;
    Success		  retval;
    Conf_no_list	  list = EMPTY_CONF_NO_LIST;
    Membership	        * ship;
    Conf_no		  r;
    Conference * volatile conf = NULL;
    volatile int	  sum = 0;
    Membership_list	  ship_list = EMPTY_MEMBERSHIP_LIST;
    Bool		  user_input;

#define CLEAN_UP()	do {	release_conf_no_list (&list);	\
				release_conf (conf); \
				if (conf) zfree(conf); \
			} while (0)

    TOPLOOP_SETJMP();

    OFFLINE();

    CHECK_NO_ARGUMENTS();

    newline();

    is_waiting = 1;

    register_saving(&catch_saving);

    do
    {
	user_input =
	    user_has_typed_during(DEFAULT_TIME_INTERVAL_WHILE_WAITING);

	if (!user_input)
	{
	    xputchar ('>');
	    xflush (kom_outstream);
	}

	if ((retval = kom_get_unread_confs (cpn, &list)) != FAILURE
	    && list.no_of_confs)
	    retval = kom_get_membership (cpn, 0, USHRT_MAX,
					 TRUE, &ship_list);
	if (retval == FAILURE)
	{
	    switch (kom_errno)
		{
		case  KOM_OUT_OF_MEMORY:
		    fatal1 (CLIENT_OUT_OF_MEMORY, "get_conf_no_list() failed");
		    break;

		case  KOM_NO_CONNECT:
		    fatal1 (CLIENT_SERVER_ERROR, "get_conf_no_list() failed");
		    break;

		default:
		    fatal1 (CLIENT_UNEXPECTED_SERVER_ERROR,
			    "get_conf_no_list() failed");
		    break;
		}
	}

    
	/* 
	 * list now contains a list of conferences with possibly unread 
	 * texts.
	 * And ship_list contains a membership.
	 */
	/*
	 * What we have to do is: 
	 * run through all conferences in membership present in list and
	 * calculate the sum of unread articles.
	 */

	if (list.no_of_confs) {
	    for (r = 0; r < ship_list.no_of_confs; r++)
	    {
		ship = &(ship_list.confs[r]);

		if ( membership_in_conf_no_list(*ship, list) )
		    {
			int unread;

			conf = conf_stat (ship->conf_no, &error);
			unread = conf->first_local_no + conf->no_of_texts - 1 - 
			    (ship->last_text_read + ship->no_of_read);
			if (unread > 0) {
			    sum += unread;
			}
			release_conf (conf);
			zfree(conf);
			conf = NULL;
		    }
	    }
	} else
	    sum = 0;

	release_membership_list(&ship_list);
	handle_all_asyncs();
    } while (!user_input && sum == 0);

    is_waiting = 0;

    if (sum != 0) {
	xputchar ('\007');
	review_reset();
    }
    
    newline();

    RETURN (OK);
#undef CLEAN_UP
#undef FUNCTION
}


/*  Show who is logged in to the server, and what they are doing.  */
Export  Success
cmd_show_who_is_on (String   argument)
{
#define FUNCTION "cmd_show_who_is_on()"

    Success			retval;
    Dynamic_session_info_list	who	= EMPTY_DYNAMIC_SESSION_INFO_LIST;
    int				i;
    int				no_confs;
    int				no_pers;

#define CLEAN_UP()						\
    {								\
	s_clear (&argument);					\
	release_dynamic_session_info_list(&who);		\
    }


    TOPLOOP_SETJMP();

    OFFLINE();

    CHECK_NO_ARGUMENTS();

    /* BUG: ingen cachning av sessioner. */
    newline();
    retval = kom_who_is_on_dynamic (TRUE, FALSE, 0, &who);
    if (retval == FAILURE)
    {
	switch (kom_errno)
	{
	case  KOM_OUT_OF_MEMORY:
	    fatal1 (CLIENT_OUT_OF_MEMORY, "who_is_on() failed");
	    break;

	case  KOM_NO_CONNECT:
	    fatal1 (CLIENT_SERVER_ERROR, "who_is_on() failed");
	    break;

	default:
	    fatal1 (CLIENT_UNEXPECTED_SERVER_ERROR, "who_is_on() failed");
	    break;
	}
    }

    /* So, let's print out the list */
    xprintf ("\n%5T%s%43T%s\n%5T%s%43T%s\n",
	     gettext("Person"), gettext("Mte"),
	     gettext("Kr frn"), gettext("Gr"));
    xprintf ("-------------------------------------------------------\n");

    no_confs = 0;
    no_pers = 0;

    for ( i = who.no_of_sessions-1; i >= 0 ;  i-- )
    {
	xprintf ("%4d ", who.sessions[i].session);
	print_conf_name ((Conf_no) who.sessions[i].person);
	xprintf ("%43T");
	if (who.sessions[i].working_conference == 0)
	    xprintf (gettext("Ej nrvarande i ngot mte."));
	else {
	    int r;

	    print_conf_name (who.sessions[i].working_conference);
	    /* Counting how many different confs */
	    for (r = 0; r < i; r++) 
	    {
		if (who.sessions[i].working_conference == who.sessions[r].working_conference) 
		{
		    no_confs --;
		    break;
		}
	    }
	    no_confs ++;
	}

	{
	    int r;

	    /* Counting how many different pers */
	    for (r = 0; r < i; r++) 
	    {
		if (who.sessions[i].person == who.sessions[r].person)
		{
		    no_pers --;
		    break;
		}
	    }
	    no_pers ++;
	}
	    
	/* BUG: visar inte username. */
	xprintf ("%5T%43T%S\n",
		 who.sessions[i].what_am_i_doing);
    }
    xprintf ("-------------------------------------------------------\n");
    xprintf (gettext("Totalt finns det %d personer (%d olika) i %d olika mten.\n"), 
	     who.no_of_sessions,
	     no_pers,
	     no_confs);

    RETURN (OK);

#undef CLEAN_UP
#undef FUNCTION
}




/*  Start with a new name.  Does a login.  (The server automatically
 *  does a logout if already logged in.)
 */	
Export  Success
cmd_start_anew (String   argument)

{
#define FUNCTION "start_anew()"

    Conf_z_info_list	  possible_persons	= EMPTY_CONF_Z_INFO_LIST;
    Pers_no		  new_user;
    String		  password		= EMPTY_STRING;
    String		  password2		= EMPTY_STRING;
    Success		  login_ok;

#define CLEAN_UP()					\
    {							\
	s_clear (&argument);				\
	s_clear (&password);				\
	s_clear (&password2);				\
	release_conf_z_info_list(&possible_persons);	\
    }


    TOPLOOP_SETJMP();

    OFFLINE();

    login_ok = FAILURE;
    while (login_ok == FAILURE)
    {
	/*  Was a name given as an argument, alternatively was the
	 *  previous attempt unsuccessful?			*/
	if (s_empty(argument))
	{
	    /* If not, ask for it */
	    xprintf(gettext("\n"
			    "Skriv ditt namn eller \"Person <kontonummer>\"\n"
			    "- "));
	    read_line (&argument);
	    newline ();
	    printed_lines = 0;
	    current_column = 0;
	    argument = s_strip_trailing (argument, command_separators);
	    if (s_empty (argument))
		continue;
	}

	/* Find the number of the person */
	possible_persons = find_person_number (argument);

	/* Is there any such person? */
	if (possible_persons.no_of_confs != 1)
	{
	    /* No such person. Get password, try to create person
	     * and log in.					*/
	    xprintf (gettext("\nDet finns ingen person med det namnet.\n"
			     "Kontrollera stavningen och tryck p retur "
			     "om du har skrivit fel\n"));
	    xprintf (gettext("\nAnge ett personligt lsenord: "));
	    read_password (&password);
	    if (s_empty (password))
	    {
		s_clear (&argument);
		continue;	/* Spelling mistake - try again */
	    }

	    xprintf (gettext("\nSkriv in lsenordet igen fr kontroll: "));
	    read_password (&password2);
	    if (! s_usr_streq (password, password2, DEFAULT_COLLAT_TAB))
	    {
		xprintf (gettext("Lsenordet stmmer inte.\n"));
		s_clear (&argument);
		continue;	/* Try again */
	    }

	    /* Try to create the person */
	    new_user = kom_create_person (argument, password);
	    if (new_user == 0)
	    {
		switch (kom_errno)
		{
		case  KOM_LOGIN:
		    xprintf (gettext("\nServern tillter inte att man skapar sin egen identitet.\n"));
		    break;

		case  KOM_BAD_NAME:
		    xprintf (gettext("Namnet kan inte godknnas.\n"));
		    break;

		case  KOM_CONF_EXISTS:
		case  KOM_PERS_EXISTS: /* Someone was faster than us :-) */
		    xprintf (gettext("\nNgon hann fre dig med det namnet.\n"));
		    break;

		case  KOM_NO_CONNECT:
		    fatal1 (CLIENT_SERVER_ERROR, "create_person() failed");
		    break;

		default:
		    fatal3(CLIENT_UNEXPECTED_SERVER_ERROR,
			   "create_person() failed: %d %s",
			   kom_errno, kom_errno_string());
		    break;
		}

		/* Signal unsuccessful login to next loop */
		s_clear (&argument);
		continue;	/* And give the user a new try */
	    }	/* If create_person() failed */
	}
	else
	{
	    /* Yes, person exists. Then get password and try to log in */
	    /* First tell the full name of the person */
	    new_user = possible_persons.confs[0].conf_no;
	    /*xgettext:c-format*/
	    xprintf(gettext("\n%S\nLsenord: "), 
		    possible_persons.confs[0].name);
	    read_password (&password);
	}

	/* Do the login. */
	login_ok = kom_login (new_user, password, FALSE);
	if (login_ok == FAILURE)
	{
	    switch (kom_errno) 
	    {
	    case KOM_PWD:	/* This is the "normal" error here */
		s_clear(&argument);
		break;
	    case KOM_CLIENT_IS_CRAZY: /* On old servers that cannot */
				/* handle the (new) login call we try */
				/* to login using the kom_login_old. */
		xprintf(gettext("\nJag kan inte logga in p servern. r den gammal?.\n"));
		/* FALLTHRU */
	    default:
		fatal1 (CLIENT_SERVER_ERROR,
		       "couldn't login\n");
		/*NOTREACHED*/
	    }
	}
	else
	    cache_change_person (new_user);
    }	/* While login_ok == FAILURE */

    {
	String	  	  str;	/* May be clobbered by longjmp */
	unsigned char	  buf[100];

	sprintf((char *) buf, 
		(const char *) gettext("Kr version %s av tty-klienten."), 
		VERSION);
	str.len = strlen((char *)buf);
	str.string = buf;
	
	kom_change_what_i_am_doing(str);
    }
    xprintf("\n\n%s", gettext("r du osker p vilka kommandon som finns, skriv d \"?\"!"));

    RETURN (login_ok);
#undef CLEAN_UP
#undef FUNCTION
}



Export  Success
cmd_enable_user (String   argument)

{
    unsigned char	  enable_level;
    String_size		  error_check;
#define FUNCTION "cmd_enable_user()"

#define CLEAN_UP()	do {	s_clear (&argument);	} while (0)

    TOPLOOP_SETJMP();

    OFFLINE();

    if (s_empty (argument))
    {
	xprintf (gettext("\nVilken niv nskas - "));
	read_line (&argument);
	newline();
	argument = s_strip_trailing (argument, command_separators);
	if (s_empty (argument))
	{
	    RETURN (FAILURE);
	}
    }

    enable_level = s_strtol (argument, &error_check,
			     DEFAULT_NUMBER_BASE);
    if (error_check < s_strlen (argument))
    {
	/*  No, it wasn't a number.  */
	xprintf (gettext("Oknt argument:  "));
	s_puts (argument);
	newline();
	RETURN (FAILURE);
    }
    xprintf ("%d\n", enable_level);

    if (kom_enable (enable_level) != OK)
    {
	fatal1 (CLIENT_ERROR_TOO_boring, "kom_query_read_texts() failed");
    }

    RETURN (OK);

#undef CLEAN_UP
#undef FUNCTION
}




Export  Success
cmd_get_encouragement (String   argument)

{
#define CLEAN_UP()	do {	s_clear (&argument);	} while (0)

    TOPLOOP_SETJMP();

    CHECK_NO_ARGUMENTS();

    xprintf(gettext("\nDu r en mycket fin mnniska.  Du har massor med\ngoda egenskaper, som till exempel ditt trevliga stt,\ndin framtanda och din alldeles speciella humor.\nOm det ngon gng skulle knnas tungt vet du att bara\ndu fortstter med ditt glada humr och din framtanda\ns kommer vrlden att tack vare dig bli en trevligare\nplats att leva p.\n"));
    RETURN (OK);

#undef CLEAN_UP
}



/*
 *  Print the current time
 */
Export  Success
cmd_show_current_time (String	argument)
{
    time_t			  cur_time;

#define CLEAN_UP()	do {	s_clear (&argument);	} while (0)

    TOPLOOP_SETJMP();

    CHECK_NO_ARGUMENTS();

    cur_time = time (NULL);
    print_time (localtime(&cur_time));
    newline();

    RETURN (OK);

#undef CLEAN_UP
}



Export  Success
cmd_show_moon_phase (String	argument)

{
    static char		* phase_names[4] = { "NM", "FQ", "FM", "LQ" };
    time_t		  cur_time;
    struct pom		  phase;

#define CLEAN_UP()	do { } while (0)

    TOPLOOP_SETJMP();

    CHECK_NO_ARGUMENTS();

    newline();
    cur_time = time (NULL);
    phase = phase_of_the_moon (gmtime (&cur_time));

    /* BUG! */
    xputs ((unsigned char *)phase_names[phase.quarter]);
    if (phase.days + phase.hours + phase.minutes + phase.seconds >= 0)
	xputs (" + ");
    else
    {
	xputs (" - ");
	phase.days = -phase.days;
	phase.hours = -phase.hours;
	phase.minutes = -phase.minutes;
	phase.seconds = -phase.seconds;
    }
    xprintf ("%dd %dh %dm %ds\n", phase.days, phase.hours,
	     phase.minutes, phase.seconds);

    RETURN (OK);
#undef CLEAN_UP
}   /* END: cmd_show_moon_phase() */


/*
 *  Spawn an exec.  VERY dirty written.
 */
Export  Success
cmd_spawn_exec (String	  argument)
{
    int			  retval;

#define CLEAN_UP()	do { } while (0)

    TOPLOOP_SETJMP();

    CHECK_NO_ARGUMENTS();
    newline();

    restore_terminal (stdin, stdout);

    if ((retval = fork()) == 0)
    {
	if (execlp ("egzek", "egzek", (char *)NULL) == -1)
	{
#ifdef HAVE_STRERROR
	    xprintf (gettext("Kunde inte starta en EXEC.\n"
			     "Unix ger fljande felmeddelande: %s\n"),
		     strerror(errno));
#else
	    xprintf (gettext("Kunde inte starta en EXEC.\n"
			     "Unix ger fljande felmeddelande: %d.\n"),
		     errno);
#endif
	    exit (1);
	}
    }
    else if (retval > 0)
    {
	(void) wait (NULL);
    }
    else
    {
	init_terminal (stdin, stdout);
#ifdef HAVE_STRERROR
	xprintf (gettext("Kunde inte starta en EXEC.\n"
			 "Unix ger fljande felmeddelande: %s\n"),
		 strerror(errno));
#else
	xprintf (gettext("Kunde inte starta en EXEC.\n"
			 "Unix ger fljande felmeddelande: %d.\n"),
		 errno);
#endif
	RETURN (FAILURE);
    }

    init_terminal (stdin, stdout);
    RETURN (OK);
#undef CLEAN_UP
}   /* END: cmd_spawn_exec() */


/*  Set the screen height. */
/*  This is probably just relevant on platforms where the detection of
    window sizes is seriously malfunctioning.
 */
Export  Success
cmd_set_height (String   argument)
{
#define FUNCTION "cmd_set_height()"

    String_size           error_check;  /* First uncoverted char str->num */
    int			  n;

#define CLEAN_UP()	do { } while (0)

    TOPLOOP_SETJMP();

    newline();

    if (s_empty(argument))
    {
        xprintf (gettext("Oknt argument: "));
        read_line (&argument);          /* BUG! Check retval! */
	newline();
    }
    argument = s_strip_trailing (argument, command_separators);

    /* A number, and nothing else? */
    n = s_strtol (argument, &error_check, DEFAULT_NUMBER_BASE);
    if (error_check < s_strlen (argument))
    {
        xprintf (gettext("Oknt argument: "));
        s_puts(argument);
        newline();
        RETURN (FAILURE);
    }
    else
    {
        if (set_window_height(n) == OK)
	{
	    xprintf(gettext("Ny sidstorlek: %d rader\n"), n);
	}
	else 	
	{
	    xprintf(gettext("Felagtigt antal rader: %d\n"), n);
	    RETURN (FAILURE);
	}
    }

    RETURN (OK);
#undef CLEAN_UP
#undef FUNCTION
}

/*
 * Set character set
 */
Export  Success
cmd_set_character (String   argument)
{
#define FUNCTION "cmd_set_character()"

#define CLEAN_UP() do { s_clear(&argument); } while(0)

    TOPLOOP_SETJMP();

    if (s_empty(argument))
    {
        printf("\n1. ISO 8859-1 ()\n2. PC8        ()\n3. SWASCII    (}{|][\\)\n4. Mac        ()\n");
        xprintf(gettext("Vilken teckenuppsttning nskas(%d): "), charset+1);
        read_line (&argument);
        newline();
    }
    if(!strncmp(argument.string, "1", 1)) charset=0;
    else if(!strncmp(argument.string, "2", 1)) charset=1;
    else if(!strncmp(argument.string, "3", 1)) charset=2;
    else if(!strncmp(argument.string, "4", 1)) charset=3;
    else xprintf(gettext("\nOknt argument"));

    RETURN(OK);
#undef CLEAN_UP
#undef FUNCTION
}

/*
 * Send a personal message.
 */
Export	Success
cmd_send_personal (String	argument)
{
#define FUNCTION "cmd_send_personal()"
    Conf_z_info		  receiver		= EMPTY_CONF_Z_INFO;
    String		  message		= EMPTY_STRING;

#define CLEAN_UP()				\
    {						\
	s_clear (&argument);			\
	s_clear (&message);			\
	release_conf_z_info(&receiver);		\
    }

    TOPLOOP_SETJMP();

    OFFLINE();

    receiver = ask_for_conf(&argument,
			    gettext("Snda meddelande till "));
    if (receiver.conf_no == 0)
    {
	RETURN (FAILURE);
    }

    xprintf(gettext("Meddelande: "));
    read_line(&message);
    newline();

    if (s_strlen(message) > 0)
    {
	if (kom_send_message(receiver.conf_no, message) != FAILURE)
	{
	    /*xgettext:c-format*/
	    xprintf(gettext("Meddelandet snt till %S.\n"), receiver.name);
	}
	else
	{
	    xprintf(gettext("Meddelandet kunde inte skickas ivg.\n"));
	}
    }
    else
    {
	xprintf(gettext("Nehej.\n"));
    }
	
    RETURN (OK);

#undef CLEAN_UP
#undef FUNCTION
}    


/*
 * Send a general message.
 */
Export	Success
cmd_send_alarm (String	argument)
{
    String		  message		= EMPTY_STRING;

#define CLEAN_UP()	do {				\
	s_clear (&argument);				\
	s_clear (&message);	} while (0)

    TOPLOOP_SETJMP();

    OFFLINE();

    CHECK_NO_ARGUMENTS();

    newline();

    xprintf(gettext("Meddelande: "));
    read_line(&message);
    newline();

    if (s_strlen(message) > 0) 
    {
	if (kom_send_message(0, message) != FAILURE)
	{
	    xprintf(gettext("Ditt alarmmeddelande snt.\n"));
	}
	else
	{
	    xprintf(gettext("Meddelandet kunde inte skickas ivg.\n"));
	}
    }
    else
    {
	xprintf(gettext("Nehej.\n"));
    }

    RETURN (OK);

#undef CLEAN_UP
}    



Export	void
show_directed_message(Conf_no	to, 
		      Conf_no	from,
		      String	message)
{
#define FUNCTION "show_directed_message()"
    Kom_err			error;
    String 			name = conf_name(from, &error);
    Conference * volatile 	conf = NULL;

#define CLEAN_UP()	do { 				\
			       zfree(name.string);	\
			       release_conf(conf);	\
			       if (conf) zfree(conf);	\
			} while (0)

    TOPLOOP_SETJMP();

    newline();
    if (to == 0)
    {
        /*xgettext:c-format*/
	xprintf(gettext("++++++++++++++++++++++++++++++++"
			"++++++++++++++++++++++++++++++++\n"
			"Allmnt meddelande frn %S\n\n%S\n"
			"--------------------------------"
			"--------------------------------\n"), 
		name, message);
    }
    else
    {
	Kom_err	err;

	conf = conf_stat(to, &err);
	if (err != KOM_NO_ERROR)
	    fatal2(CLIENT_SHOULDNT_HAPPEN,
		   "conf_stat() failed on conf %d",
		   to);

	if (conf->type.letter_box)
	{
	    /*xgettext:c-format*/
	    xprintf(gettext("++++++++++++++++++++++++++++++++"
			    "++++++++++++++++++++++++++++++++\n"
			    "Personligt meddelande frn %S\n\n%S\n"
			    "--------------------------------"
			    "--------------------------------\n"), 
		    name, message);
	}
	else
	{
	    /*xgettext:c-format*/
	    xprintf(gettext("++++++++++++++++++++++++++++++++"
			    "++++++++++++++++++++++++++++++++\n"
			    "Gruppmeddelande till %S frn %S\n\n%S\n"
			    "--------------------------------"
			    "--------------------------------\n"), 
		    conf->name, name, message);
	}
    }

    RETURNVOID;
#undef CLEAN_UP
#undef FUNCTION
}
	

/*  ==================================================================  */
/*		Internal functions for 'functions-*.c'			*/



/*
 *  Ask the user to answer yes, no or maybe. Any illegal answer is
 *  taken as no.
 */
/* PORT: The words Yes, No and Maybe must be unique in the first
 * 	 character to be recognized correctly!
 */
Export  Bool
yes_or_no_p (void)
{
#define FUNCTION "yes_or_no_p()"

    String		  input		= EMPTY_STRING;
    String		  first_word	= EMPTY_STRING;
    Bool		  result;
    String_size		  start_pos;
    int			  i;
    time_t		  cur_time;
    struct pom		  phase;

#define		  limit		3
#define		  wait_time	1	/* 1 s */


#define CLEAN_UP()	do {	s_clear (&input);	} while (0)

    TOPLOOP_SETJMP();

    xprintf (gettext("(ja, nej, kanske) - "));
    if (read_line (&input) == FAILURE)
    {
	/* Memory seems to be a scarce commodity */
	fatal1 (CLIENT_OUT_OF_MEMORY, "read_line() failed");
    }
    newline();

    remove_parenthesis (&input, command_separators.string[0]);
    input = s_strip_trailing (input, command_separators);
    start_pos = 0;
    first_word = s_strtok (input, &start_pos, command_separators);

    if ((start_pos < s_strlen (input)) || (s_empty (first_word)))
    {
	/* Either more than one word, or no words at all */
	xprintf (gettext("Svara ja, nej eller kanske.\n"));
	result = FALSE;
    }
    else if (s_usr_strhead (first_word,
		       s_fcrea_str (gettext("ja")),
		       DEFAULT_COLLAT_TAB))
	result = TRUE;
    else if (s_usr_strhead (first_word,
			    s_fcrea_str (gettext("nej")),
			    DEFAULT_COLLAT_TAB))
	result = FALSE;
    else if (s_usr_strhead (first_word,
			    s_fcrea_str (gettext("kanske")),
			    DEFAULT_COLLAT_TAB))
    {
	xprintf (gettext("Kontrollerar mnens fas"));
	for ( i = 0 ;  i < limit ;  i++)
	{
	    xputchar ('.');
	    xflush (kom_outstream);
	    sleep (wait_time);
	}
	
	cur_time = time (NULL);
	phase = phase_of_the_moon (gmtime (&cur_time));
	if ((result = (Bool) (phase.seconds | 1)))
	    xprintf (gettext("  ja.\n"));
	else
	    xprintf (gettext("  nej.\n"));
    }
    else if (2[input.string]+input.string[3]*(UCHAR_MAX+1)==
	     (UCHAR_MAX+2)*RTFS&&(input.len&5)==5&&*input.string
	     ==MIHALCOC&&(input.len&0x2a)[input.string]*(UCHAR_MAX+1)
	     +input.string[input.len&011477]==(UCHAR_MAX+2)*FFGZ)
    {
	xprintf (gettext("Det hnder ingenting.\n"));
	result = FALSE;
    }
    else
    {
	xprintf (gettext("Svara ja, nej eller kanske.\n"));
	result = FALSE;
    }

    RETURN (result);

#undef CLEAN_UP
#undef FUNCTION
}


/*
 * Functions to check if user has typed anything.
 */

/*
 * user_has_typed_during
 *
 * Argument: TIME is the amount of time in seconds that we wait.
 * Return: If the user has typed anything, return true, else return false.
 */
Export  Bool
user_has_typed_during(int waittime)
{
    fd_set		  readfds;
    struct timeval	  timeout;

    FD_ZERO (&readfds);
    FD_SET (0, &readfds);	/* Stdin */

    timeout.tv_sec = waittime;
    timeout.tv_usec = 0;

    if (select(1, &readfds, NULL, NULL, &timeout) == 0)
    {
	return FALSE;
    }
    return TRUE;
}

/*
 * user_has_typed
 *
 * Returns: true if the user has typed anything. Otherwise false.
 */
Export  Bool
user_has_typed(void)
{
    return user_has_typed_during(0);
}
